/*	$Causality: serial.c,v 1.6 1997/12/23 23:57:08 ebsa Exp $	*/

/*-
 * Copyright (c) 1997 Causality Limited.
 * All rights reserved.
 *
 * This code was written by Mark Brinicombe <mark@causality.com>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Causality Limited.
 * 4. The name of Causality Limited may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY CAUSALITY LIMITED ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL CAUSALITY LIMITED BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * Additions from Deborah A. Wallach (kerr@pa.dec.com) 12/22/97
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <termios.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <errno.h>

#include "serial.h"

struct {
	unsigned int baudrate;
	speed_t speed;
} baudrates[] = {
	{ 50,	 B50	},
	{ 75,	 B75	},
	{ 110,	 B110	},
	{ 134,	 B134	},
	{ 150,	 B150	},
	{ 200,	 B200	},
	{ 300,	 B300	},
	{ 600,	 B600	},
	{ 1200,	 B1200	},
	{ 1800,	 B1800	},
	{ 2400,	 B2400	},
	{ 4800,	 B4800	},
	{ 9600,	 B9600	},
	{ 19200, B19200	},
	{ 38400, B38400	},
	{ 57600, B57600	},
	{ 115200,B115200},
	{ 0,	 0	}	/*
				 * Technically this is speed B0 but we
				 * will not bother to support that and
				 * use it for table termination instead
				 */
};		


/*
 * Setup of the communication port. Hope it will work
 */

int serial_set_tty(int fd, char *options)
{
	struct termios serial_tio;
	unsigned int baudrate;
	int databits;
	char parity;
	int stopbits;

	/* Process the options string */
	if (sscanf(options, "%d %d%c%d", &baudrate, &databits, &parity,
	    &stopbits) != 4) {
		fprintf(stderr, "Invalid serial options - %s\n", options);
		return(EINVAL);
	}

	/* flush the terminal */
	tcflush(fd, TCIFLUSH);

	/* Get the terminal attributes */
	tcgetattr(fd, &serial_tio);

	/* Set terminal for raw I/O */
	cfmakeraw(&serial_tio);

	/* Set states of various flags */
	serial_tio.c_cflag &= ~(CSTOPB | CSIZE | PARENB | PARODD);
	serial_tio.c_cflag |= CLOCAL | CREAD;
	serial_tio.c_iflag &= ~(IGNPAR | PARMRK);

	/* Update flags for the number of data bits required */
	switch (databits) {
	case 6:
		serial_tio.c_cflag |= CS6;
		break;
	case 7:
		serial_tio.c_cflag |= CS7;
		break;
	case 8:
		serial_tio.c_cflag |= CS8;
		break;
	default:
		fprintf(stderr, "Invalid data bits %d\n", databits);
		return(EINVAL);
		break;
	}

	/* Update flags for the parity required */
	switch (parity) {
	case 'N':
		serial_tio.c_iflag |= IGNPAR;
		break;
	case 'O':
		serial_tio.c_cflag |= (PARENB | PARODD);
		break;
	case 'E':
		serial_tio.c_cflag |= (PARENB);
		break;
	default:
		fprintf(stderr, "Invalid parity %c\n", parity);
		return(EINVAL);
		break;
	}

	/* Update flags for the number of stop bits required */
	switch (stopbits) {
	case 1:
		serial_tio.c_cflag &= ~CSTOPB;
		break;
	case 2:
		serial_tio.c_cflag |= CSTOPB;
		break;
	default:
		fprintf(stderr, "Invalid stopbits %d\n", stopbits);
		return(EINVAL);
		break;
	}

	serial_tio.c_cc[VMIN] = 1;
	serial_tio.c_cc[VTIME] = 0;

	/* Set terminal state */
	tcsetattr(fd, TCSANOW, &serial_tio);

	/* Set the baudrate and return */
	return(serial_setbaudrate(fd, baudrate));
}


/*
 * Set the baudrate
 */

int serial_setbaudrate(int fd, unsigned int baudrate)
{
	struct termios serial_tio;
	int loop;

	/* Validate the baudrate by looking up in baudrate table */
	loop = 0;
	while (baudrates[loop].baudrate != 0) {
		if (baudrates[loop].baudrate == baudrate) {
			baudrate = baudrates[loop].speed;
			break;
		}
		++loop;
	}

	/* Fail if invalid */
	if (baudrates[loop].baudrate == 0) {
		fprintf(stderr, "Invalid baudrate %d\n", baudrate);
		return(EINVAL);
	}

	/* flush the terminal */
	tcflush(fd, TCIFLUSH);

	/* Get the terminal attributes */
	tcgetattr(fd, &serial_tio);

	/* Set the input and output speeds */

	/*
	 * BSD can do this with just cfsetspeed() but Linux cannot
	 * so we set the speeds individually to avoid using #ifdefs
	 */
	cfsetispeed(&serial_tio, baudrate);
	cfsetospeed(&serial_tio, baudrate);

	
	tcsetattr(fd, TCSANOW, &serial_tio);
	return(0);
}


int serial_open(char *name, char *options)
{
	int fd;

/*	fd = open(name, O_RDWR | O_NOCTTY | O_NONBLOCK);*/
	fd = open(name, O_RDWR | O_NOCTTY);
	if (fd == -1) {
		fprintf(stderr, "Unable to open %s\n", name);
		return(-1);
	}
	if (serial_set_tty(fd, options))
		return(-1);
	return(fd);
}


int serial_close(int fd)
{
	close(fd);
	return(0);
}


int serial_putc(int fd, int byte)
{
	unsigned char data;

	data = byte;
	return(write(fd, &data, 1));
}

int serial_puts(int fd, int count, unsigned char *buf)
{
	return(write(fd, buf, (size_t)count));
}

int serial_getc(int fd)
{
	unsigned char data;

	if (read(fd, &data, 1) == 1)
		return(data);
	return(-1);
}

int serial_getc_timeout(int fd, int timeout)
{
	fd_set fdset;
	int fdsetsize;
	struct timeval timeval;
	int ready;

	fdsetsize = getdtablesize();
	if (fdsetsize > FD_SETSIZE)
		fdsetsize = FD_SETSIZE;

	FD_ZERO(&fdset);
	FD_SET(fd, &fdset);

	timeval.tv_sec = timeout;
	timeval.tv_usec = 0;
	
	ready = select(fdsetsize, &fdset, NULL, NULL,
	    &timeval);

	return(serial_getc(fd));
}

/*
 * Additions from Deborah A. Wallach (kerr@pa.dec.com) 12/22/97
 */

int serial_gets(int fd, unsigned char *buf, int count)
{
	int bytes_read;

	bytes_read = read(fd, buf, (size_t)count);

	return(bytes_read);
}

/*
 * Because the sa1100-eval board does not do flow control, the
 * host side ends up occasionally losing the later parts of some
 * acknowledgement packets from the angel.  Since there are just
 * acknowledgements anyway, we pretend to have received the entire
 * packet when this happens.
 */

int serial_gets_timeout(int fd, unsigned char *buf, int count, int timeout)
{
	int fdsetsize;
	struct timeval timeval;
	int ready;
	fd_set fdset;

	fdsetsize = getdtablesize();
	if (fdsetsize > FD_SETSIZE)
		fdsetsize = FD_SETSIZE;

	FD_ZERO(&fdset);
	FD_SET(fd, &fdset);

	timeval.tv_sec = timeout;
	timeval.tv_usec = 0;

	while (1) {
		ready = select(fdsetsize, &fdset, NULL, NULL, &timeval);
		if (ready)
			return(serial_gets(fd, buf, count));
		else {
			putchar('*');
			/* XXX - hack!, fake the end of the packet */
			buf[0] = 0x1d;
			return(1);
		}
	}
}
