/*
 * arch/arm/kernel/dma-sa1100.c
 *
 * Copyright (C) 2000 Nicolas Pitre
 *
 * DMA functions specific to SA1100 architectures
 * (this is work in progress)
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/wait.h>
#include <linux/spinlock.h>
#include <linux/errno.h>

#include <asm/system.h>
#include <asm/irq.h>
#include <asm/hardware.h>
#include <asm/io.h>
#include <asm/dma.h>


// #define DEBUG
#ifdef DEBUG
#define DPRINTK( x... )  printk( ##x )
#else
#define DPRINTK( x... )
#endif


spinlock_t dma_spin_lock = SPIN_LOCK_UNLOCKED;

typedef struct {
	unsigned int	lock;		/* Device is allocated */
	const char	*device_id;	/* Device name */
} dma_t;

static dma_t dma_chan[MAX_DMA_CHANNELS];

/* 
 * Get dma list
 * for /proc/dma
 */
int get_dma_list(char *buf)
{
	int i, len = 0;

	for (i = 0; i < MAX_DMA_CHANNELS; i++) {
		if (dma_chan[i].lock)
			len += sprintf(buf + len, "%2d: %s\n",
				       i, dma_chan[i].device_id);
	}
	return len;
}

/* 
 * Request DMA channel
 */
int request_dma(dmach_t channel, const char *device_id)
{
	if (channel < MAX_DMA_CHANNELS) {
		if (xchg(&dma_chan[channel].lock, 1) != 0)
			return -EBUSY;

		dma_chan[channel].device_id = device_id;
		return 0;
	} else {
		printk (KERN_ERR "Trying to allocate DMA%d\n", channel);
		return -EINVAL;
	}
}

/* 
 * Free DMA channel
 */
void free_dma(dmach_t channel)
{
	if (channel >= MAX_DMA_CHANNELS) {
		printk (KERN_ERR "Trying to free DMA%d\n", channel);
		return;
	}

	if (xchg(&dma_chan[channel].lock, 0) == 0) {
		printk (KERN_ERR "Trying to free free DMA%d\n", channel);
		return;
	}
}


/*
 * DMA processing...
 */

int sa1100_start_dma( dma_regs_t *DMA, void *dma_ptr, int size )
{
	int status;
	int use_bufa;

	status = DMA->RdDCSR;
	DPRINTK(" dma st %#x at %p", status, dma_ptr);

	/* If both DMA channels are started, there's nothing else we can do. */
	if( (status & DCSR_STRTA) && (status & DCSR_STRTB) ){
		DPRINTK(" busy\n");
		return -EBUSY;
	}

	use_bufa = ( ((status & DCSR_BIU) && (status & DCSR_STRTB)) ||
		     (!(status & DCSR_BIU) && !(status & DCSR_STRTA)) );
	if( use_bufa ){
		DMA->ClrDCSR = DCSR_DONEA|DCSR_STRTA;
		DMA->DBSA = dma_ptr;
		DMA->DBTA = size-1;
		DMA->SetDCSR = DCSR_STRTA|DCSR_IE|DCSR_RUN;
		DPRINTK(" with a\n");
	}else{
		DMA->ClrDCSR = DCSR_DONEB|DCSR_STRTB;
		DMA->DBSB = dma_ptr;
		DMA->DBTB = size-1;
		DMA->SetDCSR = DCSR_STRTB|DCSR_IE|DCSR_RUN;
		DPRINTK(" with b\n");
	}

	return 0;
}

EXPORT_SYMBOL(sa1100_start_dma);


#if 0

int arch_request_dma(dmach_t channel, dma_t *dma, const char *dev_id)
{
	dma_regs_t *reg = (dma_regs_t*)dma->iobase;

	int err = request_irq( irq, sa1100_dma_irq, SA_INTERRUPT, 
			       dev_id, (void*)s );
	if( err )
		return err;
	s->DMA = DMA;
	reg->ClrDCSR = (DCSR_DONEA|DCSR_DONEB|DCSR_STRTA|DCSR_STRTB|
	 		DCSR_IE|DCSR_ERROR|DCSR_RUN);
	reg->DDAR = 0;
	return 0;
}

void arch_free_dma(dmach_t channel, dma_t *dma)
{
	free_irq( dma->dma_irq, (void*)s );
}

void arch_enable_dma(dmach_t channel, dma_t *dma)
{
}

void arch_disable_dma(dmach_t channel, dma_t *dma)
{
}

int arch_get_dma_residue(dmach_t channel, dma_t *dma)
{
	return 0;
}

int arch_set_dma_speed(dmach_t channel, dma_t *dma, int cycle)
{
}

void __init arch_dma_init(dma_t *dma)
{
	int channel;
	for( channel = 0; channel < MAX_DMA_CHANNELS; channel++ ){
		dma[i].dma_iobase = io_p2v(_DDAR(channel));
		dma[i].dma_irq = IRQ_DMA0 + channel;
	}
}

#else

int no_dma(void)
{
	return 0;
}

#define GLOBAL_ALIAS(_a,_b) asm (".set " #_a "," #_b "; .globl " #_a)
GLOBAL_ALIAS(disable_dma, no_dma);
GLOBAL_ALIAS(enable_dma, no_dma);
GLOBAL_ALIAS(get_dma_residue, no_dma);
GLOBAL_ALIAS(set_dma_mode, no_dma);
GLOBAL_ALIAS(set_dma_count, no_dma);
GLOBAL_ALIAS(set_dma_addr, no_dma);
GLOBAL_ALIAS(init_dma, no_dma);

#endif
