/*
 * scale.c: A clock scaling implementation for the SA-1100 CPU
 *
 * Copyright (C) 2000, The Delft University of Technology
 *   portions Copyright 2000, Lernout & Hauspie Speech Products, N.V.
 *
 * Authors: 
 * - Johan Pouwelse (J.A.Pouwelse@its.tudelft.nl): initial version
 * - Erik Mouw (J.A.K.Mouw@its.tudelft.nl): major rewrite for linux-2.3.99
 *
 * Portions by:
 * - Eik Bunce (ebunce@lhsl.com): Added support for Intel Assabet
 * 
 * This software has been developed for the LART computing board
 * (http://www.lart.tudelft.nl/). The development has been sponsored by
 * the Mobile MultiMedia Communications (http://www.mmc.tudelft.nl/)
 * and Ubiquitous Communications (http://www.ubicom.tudelft.nl/)
 * projects.
 *
 * The authors can be reached at:
 *
 *  Erik Mouw
 *  Information and Communication Theory Group
 *  Faculty of Information Technology and Systems
 *  Delft University of Technology
 *  P.O. Box 5031
 *  2600 GA Delft
 *  The Netherlands
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *
 * Theory of operations
 * ====================
 * 
 * Clock scaling can be used to lower the power consumption of the CPU
 * core. This will give you a somewhat 

 * The SA-1100 has a single register to change the core clock speed:
 *
 *   PPCR      0x90020014    PLL config
 *
 * However, the DRAM timings are closely related to the core clock
 * speed, so we need to change these, too. The used registers are:
 *
 *   MDCNFG    0xA0000000    DRAM config
 *   MDCAS0    0xA0000004    Access waveform
 *   MDCAS1    0xA0000008    Access waveform
 *   MDCAS2    0xA000000C    Access waveform 
 * The SA-1110 also has the DRAM timing register
 *   MDREFR    0xA000001C    DRAM Refresh control register
 *
 * Care must be taken to change the DRAM parameters the correct way,
 * because otherwise the DRAM becomes unusable and the kernel will
 * crash. 
 *
 * The simple solution to avoid a kernel crash is to put the actual
 * clock change in ROM and jump to that code from the kernel. The main
 * disadvantage is that the ROM has to be modified, which is not
 * possible on all SA-1100 platforms. Another disadvantage is that
 * jumping to ROM makes clock switching unecessary complicated.
 *
 * The idea behind this driver is that the memory configuration can be
 * changed while running from DRAM (even with interrupts turned on!)
 * as long as all re-configuration steps yield a valid DRAM
 * configuration. The advantages are clear: it will run on all SA-1100
 * platforms, and the code is very simple.
 * 
 * If you really want to understand what is going on in
 * change_clock_speed(), you'll have to read sections 8.2, 9.5.7.3,
 * and 10.2 from the "Intel StrongARM SA-1100 Microprocessor
 * Developers Manual" or sections 8.2, 10.2, 10.4 from the "Intel
 * StrongARM SA-1110 Microprocessor Developers Manual" or
 * (both available for free from Intel) 
 *
 *
 * Notes
 * =====
 *
 * This driver works directly on the LART. If your system happens to
 * use 50ns EDO DRAM, it will work, too.
 *
 * This driver works directly on the Assabet. If your system happens to
 * use 125Mhz SDRAM in banks 0/1, it will work, too.
 *
 * The SA-1100 can be overclocked. CPUs specced at 190 MHz will safely
 * run at 250 MHz; CPUs specced at 220 MHz can do 280 MHz. If you want
 * more performance, you can increase the core voltage, but this is
 * not without risk.
 *
 * The SA-1110 can be overclocked. CPUs specced at 206 MHz will safely
 * run at up to 236 MHz.
 *
 * To Do
 * =====
 *
 * - Test it when linked statically into the kernel
 * - Change the kernel timing loop
 * - Add proper voltage scaling for the LART
 * - Change the PCMCIA timings as well
 *
 *
 * Usage
 * =====
 *
 * insmod scale.o [max_speed=xx] [debug=1]
 * 
 * Where max_speed is the maximum allowed CPU speed (9 = 190 MHz, 11 =
 * 220HMz, 15 = 280MHz), and debug is to enable debug information.
 *
 */


#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/module.h>		/* because we are a module */
#include <linux/init.h>                 /* for the __init macros */
#include <linux/proc_fs.h>		/* all the /proc functions */
#include <linux/ioport.h>
#include <asm/uaccess.h>		/* to copy to/from userspace */
#include <asm/arch/hardware.h> 

#define MODULE_NAME "scale"

/* maximum speed setting in PPCR*/
#define SCALE_MAX_SPEED (21)
/* only the lower five bits are defined in the PPCR */
#define PPCR_MASK (0x0000001f)

typedef struct 
{
	u32 mdcnfg;
	u32 mdcas0; 
	u32 mdcas1;
	u32 mdcas2;
} sa1100_freq_regs_t;

typedef struct {
	u32 mdrefr;
	u32 mdcas0_2;
} sa1110_freq_regs_t;

typedef struct {
	int cpu_is_sa1110;
	int * speeds;
	union {
		const sa1100_freq_regs_t*	settings_1100;
		const sa1110_freq_regs_t*	settings_1110;
	} platform_settings;
} scale_platform_data_t;

/* This is for boards using a 3.6864 MHz Crystal */
static int sa11x0_3_6864_speeds[SCALE_MAX_SPEED + 1] = {
  59,
  74,
  89, 
  103,
  118,
  133,
  148,
  162,
  177,
  192,
  206,
  221,
  236,
  251,
  265, 
  280,
  295,
  310,
  324,
  339,
  354,
  369,
};

#if 0
/* This is for boards using a 3.5795 MHz Crystal */
static int sa11x0_3_5795_speeds[SCALE_MAX_SPEED + 1] = {
  57,
  72,
  86,
  100,
  115,
  129,
  143,
  158,
  172,
  186,
  201,
  215,
  229,
  243,
  258,
  272,
  286,
  301,
  315,
  329,
  344,
  358,
};
#endif

static const sa1100_freq_regs_t lart_freq_regs[SCALE_MAX_SPEED + 1] =
{
	/* Clock frequency:  59.0 MHz */
	{
		0x00dc88a3,
		0xcccccccf,
		0xfffffffc,
		0xffffffff
	},
	/* Clock frequency:  73.7 MHz */
	{      
		0x011490a3,
		0xcccccccf,
		0xfffffffc,
		0xffffffff
	},
	/* Clock frequency:  88.5 MHz */
	{	
		0x014e90a3,
		0xcccccccf,
		0xfffffffc,
		0xffffffff
	},
	/* Clock frequency: 103.2 MHz */
	{	
		0x01889923,
		0xcccccccf,
		0xfffffffc,
		0xffffffff
	},
	/* Clock frequency: 118.0 MHz */
	{	
		0x01c29923,
		0x9999998f,
		0xfffffff9,
		0xffffffff
	},
	/* Clock frequency: 132.7 MHz */
	{	
		0x01fb2123,
		0x9999998f,
		0xfffffff9,
		0xffffffff
	},
	/* Clock frequency: 147.5 MHz */
	{	
		0x02352123,
		0x3333330f,
		0xfffffff3,
		0xffffffff
	},
	/* Clock frequency: 162.2 MHz */
	{	
		0x026b29a3,
		0x38e38e1f,
		0xfff8e38e,
		0xffffffff
	},
	/* Clock frequency: 176.9 MHz */
	{	
		0x02a329a3,
		0x71c71c1f,
		0xfff1c71c,
		0xffffffff
	},
	/* Clock frequency: 191.7 MHz */
	{	
		0x02dd31a3,
		0xe38e383f,
		0xffe38e38,
		0xffffffff
	},
	/* Clock frequency: 206.4 MHz */
	{	
		0x03153223,
		0xc71c703f,
		0xffc71c71,
		0xffffffff
	},
	/* Clock frequency: 221.2 MHz */
	{	
		0x034fba23,
		0xc71c703f,
		0xffc71c71,
		0xffffffff
	},
	/* Clock frequency: 235.9 MHz */
	{	
		0x03853a23,
		0xe1e1e07f,
		0xe1e1e1e1,
		0xffffffe1
	},
	/* Clock frequency: 250.7 MHz */
	{	
		0x03bf3aa3,
		0xc3c3c07f,
		0xc3c3c3c3,
		0xffffffc3
	},
	/* Clock frequency: 265.4 MHz */
	{	
		0x03f7c2a3,
		0xc3c3c07f,
		0xc3c3c3c3,
		0xffffffc3
	},
	/* Clock frequency: 280.2 MHz */
	{	
		0x0431c2a3,
		0x878780ff,
		0x87878787,
		0xffffff87
	},
	/* Clock frequency: 294.9 MHz */
	{	
		0x0469caa3,
		0x1e0f00ff,
		0xe0f0783c,
		0xffff83c1
	},
	/* Clock frequency: 309.7 MHz */
	{	
		0x049f4b23,
		0xf07c01ff,
		0x7c1f07c1,
		0xff07c1f0
	},
	/* Clock frequency: 324.4 MHz */
	{	
		0x04d9d323,
		0xf07c01ff,
		0x7c1f07c1,
		0xff07c1f0
	},
	/* Clock frequency: 339.1 MHz */
	{	
		0x0511d323,
		0xc0f803ff,
		0x81f03e07,
		0x03e07c0f
	},
	/* Clock frequency: 353.9 MHz */
	{	
		0x05475be3,	/* now we use clock divide 2 */
		0xe38e383f,
		0xffe38e38,
		0xffffffff
	},
	/* Clock frequency: 368.6 MHz */
	{	
		0x0581dbe3,	/* now we use clock divide 2 */
		0xe38e383f,
		0xffe38e38,
		0xffffffff
	},
};

static const scale_platform_data_t lart_platform_data =
{
	0,
	sa11x0_3_6864_speeds,
	{ lart_freq_regs }
};

#if defined(CONFIG_SA1100_ASSABET)
static const sa1110_freq_regs_t assabet_freq_regs[SCALE_MAX_SPEED + 1] =
{
	/* Clock frequency:  59.0 MHz */
	{
		0x00000001,
		0xaaaaaa7f
	},
	/* Clock frequency:  73.7 MHz */
	{      
		0x00000001,
		0xaaaaaa7f,
	},
	/* Clock frequency:  88.5 MHz */
	{	
		0x00000001,
		0xaaaaaa7f,
	},
	/* Clock frequency: 103.2 MHz */
	{	
		0x00000001,
		0xaaaaaa7f,
	},
	/* Clock frequency: 118.0 MHz */
	{	
		0x00000001,
		0xaaaaaa7f,
	},
	/* Clock frequency: 132.7 MHz */
	{	
		0x00000001,
		0xaaaaaa7f,
	},
	/* Clock frequency: 147.5 MHz */
	{	
		0x00000001,
		0xaaaaaa7f,
	},
	/* Clock frequency: 162.2 MHz */
	{	
		0x00000000,
		0xaaaaaa9f,
	},
	/* Clock frequency: 176.9 MHz */
	{	
		0x00000000,
		0xaaaaaa9f,
	},
	/* Clock frequency: 191.7 MHz */
	{	
		0x00000000,
		0xaaaaaa9f,
	},
	/* 10 Clock frequency: 206.4 MHz */
	{	
		0x00000000,
		0xaaaaaa9f,
	},
	/* 11 Clock frequency: 221.2 MHz */
	{	
		0x00000000,
		0xaaaaaa9f,
	},
	/* 12 Clock frequency: 235.9 MHz */
	{	
		0x00000000,
		0xaaaaaa9f,
	},
	/* 13 Clock frequency: 250.7 MHz */
	{	
		0x00000001,
		0xaaaaaa7f,
	},
	/* 14 Clock frequency: 265.4 MHz */
	{	
		0x00000001,	/* now we use clock divide 2 */
		0xaaaaaa7f,
	},
	/* 15 Clock frequency: 280.2 MHz */
	{	
		0x00000001,	/* now we use clock divide 2 */
		0xaaaaaa7f,
	},
	/* Clock frequency: 294.9 MHz */
	{	
		0x00000001,	/* now we use clock divide 2 */
		0xaaaaaa7f,
	},
	/* Clock frequency: 309.7 MHz */
	{	
		0x00000001,	/* now we use clock divide 2 */
		0xaaaaaa7f,
	},
	/* Clock frequency: 324.4 MHz */
	{	
		0x00000001,	/* now we use clock divide 2 */
		0xaaaaaa7f,
	},
	/* Clock frequency: 339.1 MHz */
	{	
		0x00000001,	/* now we use clock divide 2 */
		0xaaaaaa7f,
	},
	/* Clock frequency: 353.9 MHz */
	{	
		0x00000001,	/* now we use clock divide 2 */
		0xaaaaaa7f,
	},
	/* Clock frequency: 368.6 MHz */
	{	
		0x00000001,	/* now we use clock divide 2 */
		0xaaaaaa7f,
	},
};

static const scale_platform_data_t assabet_platform_data =
{
	1,
	sa11x0_3_6864_speeds,
	{  assabet_freq_regs }
};
#endif

static const scale_platform_data_t *current_platform;
static int cpu_is_sa1110 = 0;
static int debug = 0;
static u32 current_speed;
static int max_speed = 11; /* 11 = 220MHz, safe for all CPUs */
extern unsigned long loops_per_sec;


static int change_clock_speed(u32 new_speed)
{
	/* sanity check */
	if((new_speed < 0) || (new_speed > max_speed)) {
		if(debug)
			printk(KERN_ERR MODULE_NAME 
			       ": clock speed out of range\n");

		return(-EINVAL);
	}

	/* don't do anything if the speed is the same */
	if(new_speed == current_speed) {
		if(debug)
			printk(KERN_INFO MODULE_NAME
			       ": keeping current clock speed\n");

		return(0);
	}

	/* Is this an SA1110, it has to be handled differently... */
	if (cpu_is_sa1110) {
		const sa1110_freq_regs_t *new_settings;
		u32 newMDREFR;

		new_settings =
			&current_platform->platform_settings.settings_1110[new_speed];

		if (new_settings->mdrefr)
			newMDREFR = MDREFR | MDREFR_K1DB2;
		else
			newMDREFR = MDREFR & ~MDREFR_K1DB2;

		/* No risk, no fun: run with interrupts on! */
		if(new_speed > current_speed) {
			/* We're going FASTER, so first relax the memory
                	 * timings before changing the core frequency 
			 */
			MDREFR = newMDREFR;

			/* The order of these statements IS important
		 	*/
			MDCAS2 = new_settings->mdcas0_2;
			MDCAS0 = new_settings->mdcas0_2;

			/* change the clock frequency */
			PPCR = new_speed & PPCR_MASK;
		} else {
			/* We're going SLOWER: first decrease the core
		 	* frequency and then tighten the memory settings.
		 	*/

			/* change the clock frequency */
			PPCR = new_speed & PPCR_MASK;
		
			/* Set the memory access clock */
			MDREFR = newMDREFR;

			/* The order of these statements IS important
		 	*/
			MDCAS0 = new_settings->mdcas0_2;
			MDCAS2 = new_settings->mdcas0_2;
		}
	}
	else {
		const sa1100_freq_regs_t *new_settings;
		new_settings =
			&(current_platform->platform_settings.settings_1100[new_speed]);
		
		/* No risk, no fun: run with interrupts on! */
		if(new_speed > current_speed) {
			/* We're going FASTER, so first relax the memory
                	 * timings before changing the core frequency 
			 */
		
			/* Half the memory access clock */
			MDCNFG |= MDCNFG_CDB2;

			/* The order of these statements IS important, keep 8
                 	* pulses!!
		 	*/
			MDCAS2 = new_settings->mdcas2;
			MDCAS1 = new_settings->mdcas1;
			MDCAS0 = new_settings->mdcas0;
			MDCNFG = new_settings->mdcnfg;

			/* change the clock frequency */
			PPCR = new_speed & PPCR_MASK;
		} else {
			/* We're going SLOWER: first decrease the core
		 	* frequency and then tighten the memory settings.
		 	*/

			/* change the clock frequency */
			PPCR = new_speed & PPCR_MASK;
		
			/* Half the memory access clock */
			MDCNFG |= MDCNFG_CDB2;

			/* The order of these statements IS important, keep 8
                 	* pulses!!
		 	*/
			MDCAS0 = new_settings->mdcas0;
			MDCAS1 = new_settings->mdcas1;
			MDCAS2 = new_settings->mdcas2;
			MDCNFG = new_settings->mdcnfg;
		}
	}

	if(debug)
		printk(KERN_INFO MODULE_NAME
		       ": old speed %d (%d MHz), new speed %d (%d MHz)\n",
		       current_speed,
		       current_platform->speeds[current_speed],
		       new_speed,
		       current_platform->speeds[new_speed]);

	/* update the current speed */
	current_speed = new_speed;

	return(0);
}




static int read_proc_scale(char *page, char **start, off_t off,
			       int count, int *eof, void *data)
{
	int len;

	MOD_INC_USE_COUNT;

	if(debug)
		len = sprintf(page, "%d (SA11%d0 at %d MHz)\n", 
			      current_speed,  cpu_is_sa1110,
			      current_platform->speeds[current_speed]);
	else
		len = sprintf(page, "%d\n", current_speed);

	MOD_DEC_USE_COUNT;

	return(len);
}




static int write_proc_scale(struct file *file, const char *buffer,
				unsigned long count, void *data)
{
	u32 new_speed;
	char *endp;
	unsigned long res;

	MOD_INC_USE_COUNT;

	/* get the value, automatically detect the base */
	new_speed = simple_strtoul(buffer, &endp, 0);

	/* set the new speed */
	res = change_clock_speed(new_speed);

	MOD_DEC_USE_COUNT;

	if(res < 0)
		return(res);
	else
		return(count + (endp - buffer));
}




static int __init init_clock_scaling(void)
{
	struct proc_dir_entry *entry;

	if (machine_is_lart())
		current_platform = &lart_platform_data;
	else
#if defined(CONFIG_SA1100_ASSABET)
		if (machine_is_assabet()) {
			current_platform = &assabet_platform_data;
		}
	else
#endif
		current_platform = &lart_platform_data;

	/* Is the machine a 1110 based board? */
	cpu_is_sa1110 = current_platform->cpu_is_sa1110;
	
	/* sanity check on the max_speed variable */
	if((max_speed < 0) || (max_speed > SCALE_MAX_SPEED)) {
		printk(KERN_ERR MODULE_NAME ": max_speed out of range\n");
		return(-EINVAL);
	}
		
	/* get the current speed */
	current_speed = PPCR & PPCR_MASK;

	/* sanity check on current speed */
	if(current_speed > max_speed) {
		printk(KERN_ERR MODULE_NAME 
		       ": current clock speed out of range\n");
		return(-ENOENT);
	}

		
	/* create proc entry */
	entry = create_proc_entry(MODULE_NAME, 
				  S_IWUSR |S_IRUSR | S_IRGRP | S_IROTH, 
				  &proc_root);
	if(entry) {
		entry->read_proc = read_proc_scale;
		entry->write_proc = write_proc_scale;
	} else {
		printk(KERN_ERR MODULE_NAME 
		       ": can't create /proc/" MODULE_NAME "\n");
		return(-ENOMEM);
	}

	/* reserve I/O regions */
	request_region((unsigned long)&PPCR,   0x04, MODULE_NAME);
	request_region((unsigned long)&MDCNFG, 0x10, MODULE_NAME);
	if (machine_is_assabet())
		request_region((unsigned long)&MDREFR, 0x04, MODULE_NAME);

	printk(KERN_INFO MODULE_NAME
	       ": SA-11%d0 clock scaling initialized, clock speed is %d MHz\n",
	       cpu_is_sa1110, current_platform->speeds[current_speed]);

	/* everything went OK */
	return(0);
}




static void __exit cleanup_clock_scaling(void)
{
	/* release I/O regions */
	release_region((unsigned long)&MDCNFG, 0x10);
	release_region((unsigned long)&PPCR,   0x04);
	if (machine_is_assabet())
	  release_region((unsigned long)&MDREFR, 0x04);

	/* remove proc entry */
	remove_proc_entry(MODULE_NAME, &proc_root);

	printk(KERN_INFO MODULE_NAME
	       ": SA-11%d0 clock scaling released, clock speed is %d MHz\n",
	       cpu_is_sa1110, current_platform->speeds[current_speed]);
}




module_init(init_clock_scaling);
module_exit(cleanup_clock_scaling);

MODULE_AUTHOR("Johan Pouwelse and Erik Mouw");
MODULE_DESCRIPTION("SA-1100 clock scaling");
MODULE_PARM(debug, "i");
MODULE_PARM_DESC(debug, "Enable debug mode");
MODULE_PARM(max_speed, "i");
MODULE_PARM_DESC(max_speed, "Maximum allowed clock speed");

EXPORT_NO_SYMBOLS;
