/*
 * Flash block driver
 * Copyright (C) 1999 Nicolas Pitre <nico@cam.org>
 *
 * Portions are:
 *    Copyright (C) 2000 Lernout & Hauspie Speech Products, N.V.
 *    released under GNU Public Licence (GPL) version 2.
 * 
 * This is a block device driver for Intel Flash memory.
 *
 * 1999-02-21	Stephane Dalton		Added write functions
 * 1999-11-22	Nicolas Pitre		Multiple access arrangement support
 *					Added EBSA285-like board  support
 * 2000-03-26	Nicolas Pitre		Write buffer support
 *					ADS ThinClient parallel Flash support
 * 2000-04-15	Nicolas Pitre		Fixed write corruption problem when
 *					partitions weren't aligned on 
 *					flash sector boundaries.
 * 2000-05-25	Erik Bunce		Modify to use CFI commands to 
 *					retrieve chip type and verify
 *					flash sector size.
 */


#include <linux/config.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/ioctl.h>
#include <linux/blkpg.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/delay.h>

#include <asm/system.h>
#include <asm/segment.h>
#include <asm/uaccess.h>
#include <asm/delay.h>
#include <asm/arch/hardware.h>

#define MAJOR_NR 60
#define DEVICE_NAME "flash"
#define DEVICE_REQUEST flash_request
#define DEVICE_NR(device) (MINOR(device))
#define DEVICE_ON(device)
#define DEVICE_OFF(device)
#define DEVICE_NO_RANDOM
#include <linux/blk.h>

#include "flash_mem.h"


/* Flash mapping start */
#define FLASH_START		0xd0000000


static unsigned char *flash_start[FLASH_PARTITIONS];
static int flash_hardsec[FLASH_PARTITIONS];
static int flash_blocksizes[FLASH_PARTITIONS];
static int flash_sizes[FLASH_PARTITIONS];
static int flash_sectsize = FLASH_SECTSIZE;

/* features*/
#define USE_WRITE_BUFFER
#define WRITE_VERIFY
#if 1
#define FLASH_VERBOSE
#undef FLASH_DEBUG
#endif

/* cache structure */
static struct flash_sect_cache_struct {
	enum { UNUSED, CLEAN, DIRTY, BAD } state;
	char *buf;
	char *src;
} flash_cache;


/* Flash commands.. */
#define FLASH_CMD_READ		_CMD( 0x00FF )
#define FLASH_CMD_READ_ID_CODES _CMD( 0x0090 )
#define FLASH_CMD_READ_QUERY	_CMD( 0x0098 )
#define FLASH_CMD_ERASE		_CMD( 0x0020 )
#define FLASH_CMD_CONFIRM	_CMD( 0x00D0 )
#define FLASH_CMD_CLEAR		_CMD( 0x0050 )
#define FLASH_CMD_WRITE		_CMD( 0x0040 )
#define FLASH_CMD_WR_BUFF	_CMD( 0x00e8 )
#define FLASH_CMD_STATUS	_CMD( 0x0070 )

#define FLASH_STATUS_READY	_CMD( 0x0080 )

#define FLASH_WR_BUFF_SIZE	16

#define ERROR_VOLTAGE_RANGE	_CMD( 0x0008 )
#define ERROR_DEVICE_PROTECT	_CMD( 0x0002 )
#define ERROR_PROGRAMMING	_CMD( 0x0010 )

#define USEC		1000000		/* usec per sec */
#define POLL_TIME	10		/* usec */
#define POLL_LOOP_LIMIT	3*USEC/POLL_TIME  /* 3 sec */


#define SECTOR(d)	((d) & ~(flash_sectsize-1))
#define OFFSET(d)	((d) & (flash_sectsize-1))


#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif



/*
 *  Function:   full_status_check
 *  Author:     Stephane Dalton
 *  Parameters: in->    flash status value
 *              out->   TRUE: status ok, FALSE: problem
 */
static inline int full_status_check( unsigned long status_reg )
{
   if( status_reg & ERROR_VOLTAGE_RANGE ){
      printk("Flash driver: programming voltage error!\n");
      return FALSE;
   }
   if( status_reg & ERROR_DEVICE_PROTECT ){
      printk("Flash driver: device is write protected!\n");
      return FALSE;
   }
   if( status_reg & ERROR_PROGRAMMING ){
      printk("Flash driver: programming error!\n");
      return FALSE;
   }
   return TRUE;
}                     


/* 
 * Function: flash_status_check
 * 	Poll for status until it's ready.  Return FALSE if some error 
 * 	detected, TRUE otherwise.
 * Note: This really really has to be redesigned to avoid busy waits.
 */

static int flash_status_check( volatile FLASH_t *flash_ptr )
{
	FLASH_t status;
	int timeout = POLL_LOOP_LIMIT;

	/* just to be sure status is actually updated */
	{
		FLASH_t x;
		x = *flash_ptr;
		x = *flash_ptr;
		x = *flash_ptr;
	}

	for( status = *flash_ptr;
	     (status & FLASH_STATUS_READY) != FLASH_STATUS_READY;
	     status = *flash_ptr ) 
	{
		if( timeout-- < 0 ){
			printk( "flash_mem: flash seems dead!\n" );
			goto error;
		}
		udelay(POLL_TIME);
	} 
	if( !full_status_check(status) ) {
		error:
		*flash_ptr = FLASH_CMD_CLEAR;
		*flash_ptr = FLASH_CMD_READ;
		return FALSE;
	}
	return TRUE;
}


/*
 * Function: verify_data
 * 	Verify flash data correctness, displaying bad data if there is some.
 */

static inline int verify_data(	char *flash_addr, 
				char *data_addr, 
				unsigned long data_len )
{
	int i;
	volatile unsigned long *l_flash_ptr = (unsigned long *)flash_addr;
	unsigned long *l_data_ptr  = (unsigned long *)data_addr;

	for( i = 0; i < data_len; i += sizeof(long) ) {
		if( *l_flash_ptr++ != *l_data_ptr++ ) {
			l_flash_ptr--; l_data_ptr--;
			printk( "flash_mem: data error @ %p: data is 0x%08lX, "
				"should be 0x%08lX from %p\n",
				l_flash_ptr, *l_flash_ptr, 
				*l_data_ptr, l_data_ptr );
			return FALSE;
		}
	}
	return TRUE;
}


/*
 *  Function:   erase_sector
 *  Author:     Stephane Dalton
 *  Parameters: in->    address within the flash sector to erase
 *              out->   TRUE: sector erased, FALSE otherwise
 */
static int erase_sector( char *flash_addr )
{
	volatile FLASH_t *flash_ptr;
	int res;

#ifdef FLASH_VERBOSE
	printk("Erasing sector %p\n", flash_addr);
#endif

	WP_VPP_ON();
	flash_ptr = FLASH_PTR(flash_addr, 0);
	*flash_ptr = FLASH_CMD_CLEAR;
	*flash_ptr = FLASH_CMD_ERASE;
	*flash_ptr = FLASH_CMD_CONFIRM;
	res = flash_status_check( flash_ptr );
	*flash_ptr = FLASH_CMD_READ;
	WP_VPP_OFF();
	return res;
}


/*
 *  Function:   write_sector
 *  Author:     Stephane Dalton
 *  Parameters: in->    flash addr to write to
 *			data addr to read from
 *              out->   TRUE: sector written, FALSE otherwise
 */
static int write_sector( char *flash_addr, char *data_addr )
{
	volatile FLASH_t *flash_ptr;
	FLASH_t *data_ptr;
	unsigned long i;
	int error = FALSE;

#ifdef FLASH_VERBOSE
	printk( "Writing sector %p with data at %p\n", 
		flash_addr, data_addr);
#endif

	WP_VPP_ON();
	data_ptr = (FLASH_t *)data_addr;

	flash_ptr = FLASH_PTR(flash_addr, 0);
	*flash_ptr = FLASH_CMD_CLEAR;

	i = 0;
	while( i < flash_sectsize ) {
#ifdef USE_WRITE_BUFFER
		int pollmax = 1<<20;
		int cnt = FLASH_WR_BUFF_SIZE;

		flash_ptr = FLASH_PTR(flash_addr, i);
		do {
			*flash_ptr = FLASH_CMD_WR_BUFF;
			pollmax--;
		} while( pollmax && (*flash_ptr & FLASH_STATUS_READY) != FLASH_STATUS_READY );
		error = !flash_status_check( flash_ptr );
		if( error )  break;

		*flash_ptr = _CMD(cnt-1);
		while( cnt-- ) {
			flash_ptr = FLASH_PTR(flash_addr, i);
			i += sizeof(FLASH_t);
			*flash_ptr = *data_ptr++;
		}
		*flash_ptr = FLASH_CMD_CONFIRM;
		error = !flash_status_check( flash_ptr );
		if( error )  break;
#else
		flash_ptr = FLASH_PTR(flash_addr, i);
		i += sizeof(FLASH_t);
		*flash_ptr = FLASH_CMD_WRITE;
		*flash_ptr = *data_ptr++;
		error = !flash_status_check( flash_ptr );
		if( error )  break;
#endif
	}

	flash_ptr = FLASH_PTR(flash_addr, 0);
	*flash_ptr = FLASH_CMD_READ;
	WP_VPP_OFF();

	if( error ) return FALSE;

#ifdef WRITE_VERIFY
	return verify_data(flash_addr, data_addr, flash_sectsize);
#else
	return TRUE;
#endif
}


/*
 *  Function:   write_cached_data
 *  Author:     Stephane Dalton
 *  Parameters: 
 *		out ->	status code
 *  Abstract:   Write back the data cached by the driver to flash
 */
static int write_cached_data(void)
{
	if(flash_cache.state == DIRTY){
		if( !erase_sector(flash_cache.src) ||
		    !write_sector(flash_cache.src, flash_cache.buf) )
		{
		    flash_cache.state = BAD;
		    return -EIO;
		}

		flash_cache.state = CLEAN;
	}
	return 0;
}


/*
 *  Function:  	flash_cached_read 
 *  Author:     Stephane Dalton
 *  Parameters: 
 *		in ->	buf: 	where to put read data;
 *			minor: 	minor number aka partition we have to read from;
 *			offset:	data offset in this partition;
 *			len:	the size of the read;
 *  Abstract:   If the requested data is already in the cache,
 *		the driver read from there instead of the flash itself
 */
static int flash_cached_read(	char *buf, 
				int minor,
				int offset,
				int len )
{
    while( len > 0 ){
	char *flash_abspos = flash_start[minor] + offset;
	char *flash_sect = (char*)((long)flash_abspos & ~(flash_sectsize-1));
	int sect_offset = flash_abspos - flash_sect;
	int size = flash_sectsize - sect_offset;
	if( size > len ) size = len;

	/*
	 * Check if the requested data is already cached
	 * Read the requested amount of data from our internal cache if it
	 * contains what we want, otherwise we read the data directly 
	 * from flash.
	 */
	if( flash_cache.src != flash_sect) {
		copy_from_flash:
		memcpy( buf, flash_abspos, size );
	}
	/* From here, the cache should contain our data */
	else if( flash_cache.state == DIRTY || flash_cache.state == CLEAN ) {
		memcpy(buf, flash_cache.buf + sect_offset, size);
	}else if( flash_cache.state == BAD ) {
		return -EIO;
	}else{
		goto copy_from_flash;
	}

	len -= size;
	buf += size;
	offset += size;
    }
    return 0;
}


/*
 *  Function:   flash_cached_write
 *  Author:     Stephane Dalton
 *  Parameters: 
 *		in ->	buf: 	where to get the data to be written;
 *			minor: 	minor number aka partition to write to;
 *			offset: where within this partition start the writing;
 *			len:	the size of the write;
 *  Abstract:   We have to cache written date to prevent overerasing 
 *			the flash.  The typical exemple is when using a 
 *			blocksize of 4k and consecutively writing a 64k block 
 *			which would generate 16 erase/write cycles for a same 
 *			flash sector.
 *			A better solution is to cache the flash sector 
 *			currently being written.  To do so, if the sector 
 *			requested is different from the previous one, 
 *			write the cached sector and read the requested one.
 *			Then the block to write is copied in the cache buffer.
 */
static int flash_cached_write(	const char *buf,
				int minor,
				int offset,
				int len )
{
    while( len > 0 ) {
	int err;
	char *flash_abspos = flash_start[minor] + offset;
	char *flash_sect = (char*)((long)flash_abspos & ~(flash_sectsize-1));
	int sect_offset = flash_abspos - flash_sect;
	int size = flash_sectsize - sect_offset;
	if( size > len ) size = len;

	if( flash_cache.src != flash_sect) {
		/*
		 * We have to write previously cached data to the flash 
		 * and read the requested sector in the cache.
		 */	
		if( flash_cache.state == DIRTY ) {
			err = write_cached_data();
			if( err ) return err;
		}

		/* 
		 * Get the correct flash sector corresponding to the 
		 * requested offset 
		 */
		memcpy(	flash_cache.buf, flash_sect, flash_sectsize );
		flash_cache.src = flash_sect;
		flash_cache.state = CLEAN;
	}

	/* Write the requested amount of data to our internal cache */	
	memcpy( flash_cache.buf + sect_offset, buf, size );
	flash_cache.state = DIRTY;

	len -= size;
	buf += size;
	offset += size;
    }
    return 0;
}


/*
 *  Function:  	flash_request
 *  Author:     Nicola Pitre
 *  Abstract:	Flash block request routine
 */
static void flash_request(request_queue_t * q)
{
    unsigned int minor;
    int offset, len;

    for(;;) {
	INIT_REQUEST;

	minor = MINOR(CURRENT->rq_dev);
	if (minor >= FLASH_PARTITIONS) {
	    printk( "flash: out of partition range (minor = %d)\n", minor );
	    end_request(0);
	    continue;
	}

	offset = CURRENT->sector << 9;
	len = CURRENT->current_nr_sectors << 9;
	if ((offset + len) > flash_length[minor]) {
	    printk( "flash_request: access beyond end of partition\n" );
	    end_request(0);
	    continue;
	}

#ifdef FLASH_DEBUG
	printk( "flash_request: %s for part %d at %lX, size %ld from 0x%08X\n", 
		CURRENT->cmd == READ ? "read" : 
		CURRENT->cmd == WRITE ? "write" : "unknown", 
		minor,
		flash_start[minor] + offset, len,
		CURRENT->buffer);
#endif

	switch( CURRENT->cmd ) {
	    case READ:
		if( flash_cached_read(	CURRENT->buffer,
					minor, 
					offset, 
					len) != 0) 
		{
		    end_request(0);
		}else{
		    end_request(1);
		}
		break;

	    case WRITE:
		if( flash_cached_write(	CURRENT->buffer, 
					minor,
					offset, 
					len) != 0) 
		{
		    end_request(0);
		}else{
		    end_request(1);	
		}
		break;

	    default:
		end_request(0);
		break;
	}
    }
} 


/*
 *  Function:  	flash_ioctl	
 *  Author:     Nicola Pitre
 */
static int flash_ioctl(	struct inode *inode, 
			struct file *file, 
			unsigned int cmd, 
			unsigned long arg)
{
    switch (cmd) {
      case BLKFLSBUF:
	if (!capable(CAP_SYS_ADMIN)) return -EACCES;
	invalidate_buffers(inode->i_rdev);
	break;

      case BLKGETSIZE:
	/* Return device size */
	return put_user( flash_length[MINOR(inode->i_rdev)] / 512, 
			 (long *) arg );

      case BLKROSET:
      case BLKROGET:
      case BLKSSZGET:
	return blk_ioctl(inode->i_rdev, cmd, arg);

      default:
	printk( "flash: unimplemented ioctl(0x%08X)\n", cmd );
	return -EINVAL;
    }

    return 0;
}


/*
 *  Function:	flash_open	
 *  Author:     Nicola Pitre
 */
static int flash_open(struct inode * inode, struct file * filp)
{
    if (DEVICE_NR(inode->i_rdev) >= FLASH_PARTITIONS) return -ENXIO;
    MOD_INC_USE_COUNT;
    return 0;
}


/*
 *  Function:	flash_release
 *  Author:     Nicola Pitre
 *  Note:	We sync kernel buffers as well as our local cache too.
 */
static int flash_release(struct inode * inode, struct file * filp)
{
    int ret;

    sync_dev(inode->i_rdev);
    ret = write_cached_data();
#ifdef FLASH_DEBUG
    printk("flash_release() called\n");
#endif
    MOD_DEC_USE_COUNT;
    return ret;
}


/*
 *  Function:   flash_mem_init
 *  Author:     Nicola Pitre
 *  Abstract:   This is the registration and initialization 
 *		function for the flash driver
 */

static struct block_device_operations flash_fops = {
	ioctl:	flash_ioctl,
	open:	flash_open,
	release:	flash_release,
};

#ifdef FLASH_HAS_CFI
static struct flash_cfi_info flash_info;

/*
 *  Function:  	flash_retrieve_cfi_info
 *  Author:     Erik Bunce (ebunce@lhsl.com)
 *  Abstract:   Attempt to retrieve information about the current 
 *		flash devices using the Common Flash Interface (CFI).
 *  Notes: This has currently only been tested with Intel parts...
 */
static void flash_retrieve_cfi_info(void)
{
	FLASH_t* flash_cmd_ptr = FLASH_PTR((char*)FLASH_START, 0);
	FLASH_t* flash_ptr = FLASH_PTR((char*)FLASH_START, 0 * sizeof(FLASH_t));
#define FLASH_LOBYTE(x) (flash_ptr[(x)] & 0x000000FFL)

#ifdef FLASH_DEBUG
	u8 testmanid;
	u8 testdevid;
#endif
	
	/* Clear the information structure */
	memset((void*)&flash_info, 0, sizeof(flash_info));
	
	/* Clear any status register info */
	*flash_cmd_ptr = FLASH_CMD_CLEAR;

	/* Put flash into READ ID code mode */
	*flash_cmd_ptr = FLASH_CMD_READ_ID_CODES;
	udelay(POLL_TIME);

	/* Get the manufacturer code */
	flash_info.manufacturer_code = FLASH_LOBYTE(0x00);

	flash_ptr = FLASH_PTR((char*)FLASH_START, 1 * sizeof(FLASH_t));
	/* Get the device code */
	flash_info.device_code = FLASH_LOBYTE(0x00);

	/* Put flash into READ QUERY mode */
	*flash_cmd_ptr = FLASH_CMD_READ_QUERY;
	udelay(POLL_TIME);

	flash_ptr = FLASH_PTR((char*)FLASH_START, 0 * sizeof(FLASH_t));

#ifdef FLASH_DEBUG
	testmanid = FLASH_LOBYTE(0x00);
	testdevid = FLASH_LOBYTE(0x01);
#endif
	
	/* Retrieve CFI Ident string */
	flash_info.cfi_ident[0] = (char)FLASH_LOBYTE(0x10);
	flash_info.cfi_ident[1] = (char)FLASH_LOBYTE(0x11);
	flash_info.cfi_ident[2] = (char)FLASH_LOBYTE(0x12);
	flash_info.cfi_ident[3] = (char)0;

	flash_info.is_cfi_supported = ((flash_info.cfi_ident[0] == 'Q') &&
				       (flash_info.cfi_ident[1] == 'R') &&
				       (flash_info.cfi_ident[2] == 'Y'));
		
	if (flash_info.is_cfi_supported != 0) {
		/* Retrieve CFI Ident extended command set info */
		flash_info.primary_vendor_command_code = FLASH_LOBYTE(0x13);
		flash_info.primary_vendor_command_code |=
			(FLASH_LOBYTE(0x14) << 8);

		flash_info.primary_extended_query_table = 
			FLASH_LOBYTE(0x15);
		flash_info.primary_extended_query_table |=
			(FLASH_LOBYTE(0x16) << 8);
		
		flash_info.alternate_vendor_command_code = 
			FLASH_LOBYTE(0x17);
		flash_info.alternate_vendor_command_code |=
			(FLASH_LOBYTE(0x18) << 8);
		
		flash_info.alternate_extended_query_table = 
			FLASH_LOBYTE(0x19);
		flash_info.alternate_extended_query_table |=
			(FLASH_LOBYTE(0x1A) << 8);
		
		/* Retrieve System Interface Information */
		flash_info.vcc_min_program_erase = FLASH_LOBYTE(0x1B);
		flash_info.vcc_max_program_erase = FLASH_LOBYTE(0x1C);
		flash_info.vpp_min_program_erase = FLASH_LOBYTE(0x1D);
		flash_info.vpp_max_program_erase = FLASH_LOBYTE(0x1E);
		flash_info.typ_single_word_timeout = FLASH_LOBYTE(0x1F);
		flash_info.typ_max_buffer_write_timeout =
			FLASH_LOBYTE(0x20);
		flash_info.typ_block_erase_timeout = FLASH_LOBYTE(0x21);
		flash_info.typ_full_chip_erase_timeout = FLASH_LOBYTE(0x22);
		flash_info.max_word_program_timeout = FLASH_LOBYTE(0x23);
		flash_info.max_buffer_write_timeout = FLASH_LOBYTE(0x24);
		flash_info.max_block_erase_timeout = FLASH_LOBYTE(0x25);
		flash_info.max_chip_erase_timeout = FLASH_LOBYTE(0x26);
		
		/* Retrieve Device Geometry Info */
		flash_info.device_size = FLASH_LOBYTE(0x27);
		flash_info.x8_async_ifc = FLASH_LOBYTE(0x28);
		flash_info.x16_async_ifc = FLASH_LOBYTE(0x29);
		flash_info.max_write_buffer_size = FLASH_LOBYTE(0x2A);
		flash_info.max_write_buffer_size |= 
			(FLASH_LOBYTE(0x2B) << 8);
		flash_info.num_erase_block_regions = FLASH_LOBYTE(0x2C);
		flash_info.num_erase_blocks_r1 = FLASH_LOBYTE(0x2D);
		flash_info.num_erase_blocks_r1 |= 
			(FLASH_LOBYTE(0x2E) << 8);
		flash_info.size_erase_blocks_r1 = FLASH_LOBYTE(0x2F);
		flash_info.size_erase_blocks_r1 |= 
			(FLASH_LOBYTE(0x30) << 8);
	}
		
#ifdef FLASH_DEBUG
	printk("Flash CFI Info: ****************************\n");
	printk("\tmanufacturer_code = %#04x (%#04x)\n", 
	       flash_info.manufacturer_code, testmanid);
	printk("\tdevice_code = %#04x (%#04x)\n", 
	       flash_info.device_code, testdevid);
	printk("\tcfi_ident = %s\n", flash_info.cfi_ident);
	printk("\tis_cfi_supported = %d\n", flash_info.is_cfi_supported);
	if (flash_info.is_cfi_supported != 0) {
		int tmpval1, tmpval2;

		/* Print rest of CFI Ident */
		printk("\tprimary_vendor_command_code = %#04x\n", 
		       flash_info.primary_vendor_command_code);
		printk("\tprimary_extended_query_table = %#04x\n",
		       flash_info.primary_extended_query_table);
		printk("\talternate_vendor_command_code = %#04x\n",
		       flash_info.alternate_vendor_command_code);
		printk("\talternate_extended_query_table = %#04x\n",
		       flash_info.alternate_extended_query_table);
		
		/* Print CFI Geometry Info */
		tmpval1 = 1 << flash_info.device_size;
		tmpval2 = tmpval1 / 1024 / 1024;
		printk("\tdevice_size = %#02x (%d bytes, %d MB, %d Mb)\n",
		       flash_info.device_size,
		       tmpval1, tmpval2, tmpval2 * 8);
		printk("\tx8_async_ifc = %#02x\n", 
		       flash_info.x8_async_ifc);
		printk("\tx16_async_ifc = %#02x\n", 
		       flash_info.x16_async_ifc);
		printk("\tmax_write_buffer_size = %d (%d bytes)\n", 
		       flash_info.max_write_buffer_size,
		       1 << flash_info.max_write_buffer_size);
		printk("\tnum_erase_block_regions = %d\n",
		       flash_info.num_erase_block_regions);
		printk("\tnum_erase_blocks_r1 = %#04x (%d blocks)\n",
		       flash_info.num_erase_blocks_r1,
		       flash_info.num_erase_blocks_r1);
		printk("\tsize_erase_blocks_r1 = %#04x (%d bytes)\n",
		       flash_info.size_erase_blocks_r1,
		       flash_info.size_erase_blocks_r1 * 256);
		
		/* System Interface Information */
		printk("\tvcc_min_program_erase = %d.%d V\n",
		       (flash_info.vcc_min_program_erase & 0xF0) >> 4,
		       flash_info.vcc_min_program_erase & 0x0F);
		printk("\tvcc_max_program_erase = %d.%d V\n",
		       (flash_info.vcc_max_program_erase & 0xF0) >> 4,
		       flash_info.vcc_max_program_erase & 0x0F);
		printk("\tvpp_min_program_erase = %d.%d V\n",
		       (flash_info.vpp_min_program_erase & 0xF0) >> 4,
		       flash_info.vpp_min_program_erase & 0x0F);
		printk("\tvpp_max_program_erase = %d.%d V\n",
		       (flash_info.vpp_max_program_erase & 0xF0) >> 4,
		       flash_info.vpp_max_program_erase & 0x0F);

		printk("\ttyp_single_word_timeout = %d (%d usec)\n",
		       flash_info.typ_single_word_timeout,
		       1 << flash_info.typ_single_word_timeout);
		printk("\ttyp_max_buffer_write_timeout = %d (%d usec)\n",
		       flash_info.typ_max_buffer_write_timeout,
		       1 << flash_info.typ_max_buffer_write_timeout);
		printk("\ttyp_block_erase_timeout = %d (%d msec)\n",
		       flash_info.typ_block_erase_timeout,
		       1 << flash_info.typ_block_erase_timeout);
		printk("\ttyp_full_chip_erase_timeout = %d (%d msec)\n",
		       flash_info.typ_full_chip_erase_timeout,
		       1 << flash_info.typ_full_chip_erase_timeout);

		printk("\tmax_word_program_timeout = %d (%d usec)\n",
		       flash_info.max_word_program_timeout,
		       ((1 << flash_info.max_word_program_timeout) *
			(1 << flash_info.typ_single_word_timeout)));
		printk("\tmax_buffer_write_timeout = %d (%d usec)\n",
		       flash_info.max_buffer_write_timeout,
		       ((1 << flash_info.max_buffer_write_timeout) *
			(1 << flash_info.typ_max_buffer_write_timeout)));
		printk("\tmax_block_erase_timeout = %d (%d msec)\n",
		       flash_info.max_block_erase_timeout,
		       ((1 << flash_info.max_block_erase_timeout) * 
			(1 << flash_info.typ_block_erase_timeout)));
		printk("\tmax_chip_erase_timeout = %d (%d msec)\n",
		       flash_info.max_chip_erase_timeout,
		       ((1 << flash_info.max_chip_erase_timeout) *
			(1 << flash_info.typ_full_chip_erase_timeout)));
	}
	
#endif /* FLASH_DEBUG */
	
	/* Clear any status register info */
	*flash_cmd_ptr = FLASH_CMD_CLEAR;

	/* Return flash to read mode */
	*flash_cmd_ptr = FLASH_CMD_READ;
}
#endif /* FLASH_HAS_CFI */

int __init flash_mem_init(void)
{
	int		i;

#ifdef FLASH_HAS_CFI
	flash_retrieve_cfi_info();
	
	if (flash_info.is_cfi_supported != 0) {
		/* Set true flash sector size */
		flash_sectsize = SECTSIZE_FROM_CFI( FLASH_CHIPS_PER_BUS,
						    flash_info.size_erase_blocks_r1);

		/* Attempt to determine chip used if on Assabet */
		if (machine_is_assabet()) {
			/* determine type of chip on Assabet */
			if ((flash_info.manufacturer_code == 0xB0) &&
			    (flash_info.device_code == 0xD0)) {
				/* is Intel 28F160S3 FastFlash chips,
				 * use their partition table.
				 */
				memcpy((void*)flash_length,
				       (void*)flash_length_28F160S3,
				       sizeof(flash_length));
#ifdef FLASH_DEBUG
				printk("FLASH Driver: using 2 x 28F160S3 16Mb devices\n");
#endif
			}
			else if ((flash_info.manufacturer_code == 0x89) &&
				 (flash_info.device_code == 0x18)) {
				/* is Intel 28F128J3A StrataFlash chips,
				 * use their partition table.
				 */
				memcpy((void*)flash_length,
				       (void*)flash_length_28F128J3A,
				       sizeof(flash_length));
#ifdef FLASH_DEBUG
				printk("FLASH Driver: using 2 x 28F128J3A 128Mb devices\n");
#endif
			}
			else
				printk(KERN_WARNING
				       "FLASH Driver: Unknown flash type (manufacturer = %#04x, device_code = %#04x!\n",
				       flash_info.manufacturer_code,
				       flash_info.device_code);
		}

#ifdef FLASH_DEBUG
		printk("FLASH Driver: flash_sectsize = %d bytes (%d KB)\n",
		       flash_sectsize, flash_sectsize / 1024);
#endif
	}	
#endif /* FLASH_HAS_CFI */

	if (register_blkdev(MAJOR_NR, DEVICE_NAME, &flash_fops)) {
		printk("FLASHDISK: Could not get major %d", MAJOR_NR);
		return -EIO;
	}

	/*
	 * We allocate the cache here, even if it might never get used, 
	 * because memory fragmentation could make the allocation fail 
	 * at a later time.
	 */
	flash_cache.buf = (char *)__get_free_pages( GFP_KERNEL,
						    get_order(flash_sectsize));
	if( flash_cache.buf == 0 ) {
		printk( "Flash driver: mem allocation error\n" );
		unregister_blkdev(MAJOR_NR, DEVICE_NAME);
		return -ENOMEM;
	}

	flash_cache.state = UNUSED;
	flash_cache.src = 0;

	blk_init_queue(BLK_DEFAULT_QUEUE(MAJOR_NR), &flash_request);

	for (i = 0; i < FLASH_PARTITIONS; i++) {
		flash_start[i] = i ? (flash_start[i-1] + flash_length[i-1]) 
				 : (unsigned char *)FLASH_START;
		flash_hardsec[i] = BLOCK_SIZE;
		flash_blocksizes[i] = BLOCK_SIZE;
		flash_sizes[i] = flash_length[i] / 1024;
	}

	hardsect_size[MAJOR_NR] = flash_hardsec;
	blksize_size[MAJOR_NR] = flash_blocksizes;
	blk_size[MAJOR_NR] = flash_sizes;

	for (i = 0; i < FLASH_PARTITIONS; i++)
		register_disk(  NULL, MKDEV(MAJOR_NR,i), 1, 
				&flash_fops, flash_length[i]>>9 );

#ifdef FLASH_DEBUG
	printk("Flash Driver Partition Table\n");
	for (i = 0; i < FLASH_PARTITIONS; i++)
		printk("\t%d\tstart = %#08x\tlength = %#08x\n",
		       i, flash_start[i], flash_length[i]);
#endif

#ifdef CONFIG_SA1100_VICTOR
/*
 *	Initialize GPIO to communicate with the flash WPP
 */
	GPDR |= GPIO_FLASH_WP;
	GPCR = GPIO_FLASH_WP;
	MSC0 =  0x5389538c;			/* clock speed */
#endif

	printk("FLASH driver initialized\n" );

	return(0);
}


#ifdef MODULE

/*
 *  Function:   init_module
 *  Author:     Stephane Dalton
 *  Abstract:   Flash module initialisation function
 */
int init_module(void)
{
    return( flash_mem_init() );
}


/*
 *  Function:  	cleanup_module 
 *  Author:     Stephane Dalton
 *  Abstract:   Clean the structure used by the initialisation 
 *		function and frees the memory associated with the 
 *		caching system
 */
void cleanup_module(void)
{
    int i;
 
    for (i = 0 ; i < FLASH_PARTITIONS; i++)
      destroy_buffers(MKDEV(MAJOR_NR, i));
    unregister_blkdev(MAJOR_NR, DEVICE_NAME);
    blk_cleanup_queue(BLK_DEFAULT_QUEUE(MAJOR_NR));
    flash_hardsec[i] = NULL;
    blk_size[MAJOR_NR] = NULL;
    blksize_size[MAJOR_NR] = NULL;

    if( flash_cache.buf )
	free_pages(flash_cache.buf, get_order(flash_sectsize));
    flash_cache.buf = NULL;
}

#endif
