/*======================================================================

  $Id: mtdcore.c,v 1.1 2000/06/09 18:58:50 jamey Exp $

    A general driver for accessing Memory Technology Devices

    This driver provides the equivalent of /dev/mem for a MTD
    device. It includes a character device - a block device is
    implemented by mtdblock.c
    
======================================================================*/
#define MTD_DEBUG 3
#ifdef MTD_DEBUG
#define DEBUGLVL debug
#endif

#include <linux/module.h>
#include <asm/uaccess.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/ptrace.h>
#include <linux/malloc.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/major.h>
#include <linux/fs.h>
#include <linux/ioctl.h>
#include <asm/io.h>
#include <asm/system.h>
#include <asm/segment.h>
#include <stdarg.h>
#include <linux/init.h>
#include <linux/compatmac.h>
#ifdef CONFIG_PROC_FS
#include <linux/proc_fs.h>
#endif

#include <linux/mtd/mtd.h>

#ifdef MTD_DEBUG
static int debug = MTD_DEBUG;
MODULE_PARM(debug, "i");
#else
#endif

/* Init code required for 2.2 kernels */

/* Major device #'s for memory device */

static DECLARE_MUTEX(mtd_table_mutex);
static struct mtd_info *mtd_table[MAX_MTD_DEVICES];



static loff_t mtd_lseek (struct file *file, loff_t offset, int orig);
static int mtd_ioctl(struct inode *inode, struct file *file,
		     u_int cmd, u_long arg);
static ssize_t mtd_read(struct file *file, char *buf, size_t count,
		     loff_t *ppos);
static ssize_t mtd_write(struct file *file, const char *buf, size_t count,
			 loff_t *ppos);
static int mtd_open(struct inode *inode, struct file *file);
static int mtd_close(struct inode *inode,
		     struct file *file);

static struct file_operations mtd_fops = {
	mtd_lseek,     	/* lseek */
	mtd_read,	/* read */
	mtd_write, 	/* write */
	NULL,		/* readdir */
	NULL,		/* poll */
	mtd_ioctl,	/* ioctl */
	NULL,		/* mmap */
	mtd_open,	/* open */
	NULL,		/* flush */
	mtd_close,	/* release */
	NULL		/* fsync */
};

static struct mtd_notifier *mtd_notifiers = NULL;

static loff_t mtd_lseek (struct file *file, loff_t offset, int orig)
{
	struct mtd_info *mtd=(struct mtd_info *)file->private_data;

	switch (orig) {
	case 0: 
		/* SEEK_SET */
		file->f_pos = offset;
		break;
	case 1: 
		/* SEEK_CUR */
		file->f_pos += offset;
		break;
	case 2:
		/* SEEK_END */
		file->f_pos =mtd->size + offset;
		break;
	default: 
		return -EINVAL;
	}

	if (file->f_pos < 0) 
		file->f_pos = 0;
	else if (file->f_pos >= mtd->size)
		file->f_pos = mtd->size - 1;

	return file->f_pos;
}

static int mtd_open(struct inode *inode, struct file *file)
{
	int minor = MINOR(inode->i_rdev);
	int devnum = minor >> 1;
	struct mtd_info *mtd;

	DEBUG(0, "MTD_open\n");

	if (devnum >= MAX_MTD_DEVICES)
		return -ENODEV;

	/* You can't open the RO devices RW */
	if ((file->f_mode & 2) && (minor & 1))
		return -EACCES;

	MOD_INC_USE_COUNT;

	down(&mtd_table_mutex); {
		
		mtd = mtd_table[devnum];	
		
		if (!mtd) {
			MOD_DEC_USE_COUNT;
			up(&mtd_table_mutex);
			return -ENODEV;
		}
		/* You can't open it RW if it's not a writeable device */
		if ((file->f_mode & 2) && !(mtd->flags & MTD_WRITEABLE)) {
			MOD_DEC_USE_COUNT;
			up(&mtd_table_mutex);
			return -EACCES;
		}
		
		if (mtd->module)
			__MOD_INC_USE_COUNT(mtd->module);
	}
	up(&mtd_table_mutex);
	
	file->private_data = mtd;
	return 0;
} /* mtd_open */

/*====================================================================*/

static int mtd_close(struct inode *inode,
				 struct file *file)
{
	struct mtd_info *mtd;

	DEBUG(0, "MTD_close\n");

	mtd = (struct mtd_info *)file->private_data;
	
	if (mtd->sync)
		mtd->sync(mtd);
	
	/* This can safely be done without the mutex */
	if (mtd->module)
		__MOD_DEC_USE_COUNT(mtd->module);

	MOD_DEC_USE_COUNT;
	return 0;
} /* mtd_close */


static ssize_t mtd_read(struct file *file, char *buf, size_t count,
			loff_t *ppos)
{
	struct mtd_info *mtd = (struct mtd_info *)file->private_data;
	u_char *mtdbuf;
	size_t retlen=0;
	u_long count1;
	int ret=0;
	
	DEBUG(0,"MTD_read\n");

	if (*ppos + count > mtd->size)
		count = mtd->size - *ppos;

	if (!count)
		printk("Count is zero.\n");

	count1 = count;

	while (count && !ret)
	{
		if (!mtd->point || (ret = MTD_POINT(mtd, *ppos, count, &retlen, &mtdbuf)) != 0)
		{
			/* mtd->point() failed; use mtd->read instead */
		
			ret = MTD_READ(mtd, *ppos, count, &retlen, buf);
			if (retlen)
			{
				*ppos += retlen;
				return retlen;
			}
			else 
				return ret;
		}

		if (!retlen)
		{
			printk(KERN_NOTICE "mtd_read: device returned no error, but no data either!\n");
			ret = -EIO;
			break;
		}
		
		if (copy_to_user(buf,mtdbuf,retlen) < 0) 
			ret = -EFAULT;
		else
		{
			count -= retlen;
			*ppos += retlen;
		}
		MTD_UNPOINT(mtd, mtdbuf);
	}
	
	if (count < count1)
		return count1-count;
	else 
		return ret;
} /* mtd_read */


static ssize_t mtd_write(struct file *file, const char *buf, size_t count,
			 loff_t *ppos)
{
	struct mtd_info *mtd = (struct mtd_info *)file->private_data;
	u_long count1;
	size_t retlen;
	int ret=0;

	DEBUG(0,"MTD_write\n");
	
	if (*ppos == mtd->size)
		return -ENOSPC;
	
	if (*ppos + count > mtd->size)
		count = mtd->size - *ppos;
	
	count1 = count;

	/* TPW: while(x) => while x != 0, not while x > 0 */	
	while (count > 0) {
		ret = (*(mtd->write))(mtd, *ppos, count, &retlen, buf);
		
		if (ret)
			break;
		
		if (!retlen) {
			printk(KERN_NOTICE "mtd_read: device returned no error, but no data either!\n");
			ret = -EIO;
			break;
		}

		count -= retlen;
	}

	if (count < count1) {
		*ppos += count1-count;
		return count1-count;
	}
	else 
		return ret;
} /* mtd_write */

/*======================================================================

    IOCTL calls for getting device parameters.

======================================================================*/
static void mtd_erase_callback (struct erase_info *instr)
{
	wake_up((wait_queue_head_t *)instr->priv);
}

static int mtd_ioctl(struct inode *inode, struct file *file,
		     u_int cmd, u_long arg)
{
	struct mtd_info *mtd = (struct mtd_info *)file->private_data;
	int ret = 0;
	u_long size;
	
	DEBUG(0, "MTD_ioctl\n");

	size = (cmd & IOCSIZE_MASK) >> IOCSIZE_SHIFT;
	if (cmd & IOC_IN) {
		ret = verify_area(VERIFY_READ, (char *)arg, size);
		if (ret) return ret;
	}
	if (cmd & IOC_OUT) {
		ret = verify_area(VERIFY_WRITE, (char *)arg, size);
		if (ret) return ret;
	}
	
	switch (cmd) {
	case MEMGETINFO:
		copy_to_user((struct mtd_info *)arg, mtd,
			     sizeof(struct mtd_info_user));
		break;

	case MEMERASE:
	{
		struct erase_info *erase=kmalloc(sizeof(struct erase_info),GFP_KERNEL);
		if (!erase)
			ret = -ENOMEM;
		else {
			wait_queue_head_t waitq;
			DECLARE_WAITQUEUE(wait, current);


			init_waitqueue_head(&waitq);

			memset (erase,0,sizeof(struct erase_info));
			copy_from_user(&erase->addr, (u_long *)arg,
				       2 * sizeof(u_long));
			erase->mtd = mtd;
			erase->callback = mtd_erase_callback;
			erase->priv = (unsigned long)&waitq;
			
			/* FIXME: Allow INTERRUPTIBLE. Which means
			   not having the wait_queue head on the stack
			*/
			current->state = TASK_UNINTERRUPTIBLE;
			add_wait_queue(&waitq, &wait);
			ret = MTD_ERASE(mtd, erase);
			schedule();
			remove_wait_queue(&waitq, &wait);
			kfree(erase);
#if 0
			if (signal_pending(current))
				return -EINTR;
			else
#endif
				return (erase->state==MTD_ERASE_DONE);
		}
		break;
	}

	case MEMWRITEOOB:
	{
		struct mtd_oob_buf buf;
		void *databuf;
		ssize_t retlen;
		
		copy_from_user(&buf, (struct mtd_oob_buf *)arg, sizeof(struct mtd_oob_buf));
		
		if (buf.length > 0x4096)
			return -EINVAL;

		if (!mtd->write_oob)
			ret = -EOPNOTSUPP;
		else
			ret = verify_area(VERIFY_READ, (char *)buf.ptr, buf.length);

		if (ret)
			return ret;

		databuf = kmalloc(buf.length, GFP_KERNEL);
		if (!databuf)
			return -ENOMEM;
		
		copy_from_user(databuf, buf.ptr, buf.length);

		ret = (mtd->write_oob)(mtd, buf.start, buf.length, &retlen, databuf);

		copy_to_user((void *)arg + sizeof(loff_t), &retlen, sizeof(ssize_t));

		kfree(databuf);
		break;

	}

	case MEMREADOOB:
	{
		struct mtd_oob_buf buf;
		void *databuf;
		ssize_t retlen;

		copy_from_user(&buf, (struct mtd_oob_buf *)arg, sizeof(struct mtd_oob_buf));
		
		if (buf.length > 0x4096)
			return -EINVAL;

		if (!mtd->read_oob)
			ret = -EOPNOTSUPP;
		else
			ret = verify_area(VERIFY_WRITE, (char *)buf.ptr, buf.length);

		if (ret)
			return ret;

		databuf = kmalloc(buf.length, GFP_KERNEL);
		if (!databuf)
			return -ENOMEM;
		
		ret = (mtd->read_oob)(mtd, buf.start, buf.length, &retlen, databuf);

		copy_to_user((void *)arg + sizeof(loff_t), &retlen, sizeof(ssize_t));

		if (retlen)
			copy_to_user(buf.ptr, databuf, retlen);

		kfree(databuf);
		break;
	}
			     
			     
		
		

	default:
	  printk("Invalid ioctl %x (MEMGETINFO = %x)\n",cmd, MEMGETINFO);
		ret = -EINVAL;
	}
	
	return ret;
} /* memory_ioctl */




/*====================================================================*/

int add_mtd_device(struct mtd_info *mtd)
{
	int i;

	down(&mtd_table_mutex);

	for (i=0; i< MAX_MTD_DEVICES; i++)
		if (!mtd_table[i])
		{
			struct mtd_notifier *not=mtd_notifiers;

			mtd_table[i] = mtd;
			DEBUG(0, "mtd: Giving out device %d to %s\n",i, mtd->name);
			while (not)
			{
				(*(not->add))(mtd);
				not = not->next;
			}
			up(&mtd_table_mutex);
			return 0;
		}
	
	up(&mtd_table_mutex);
	return 1;
}


int del_mtd_device (struct mtd_info *mtd)
{
	struct mtd_notifier *not=mtd_notifiers;
	int i;
	
	down(&mtd_table_mutex);

	for (i=0; i < MAX_MTD_DEVICES; i++)
	{
		if (mtd_table[i] == mtd)
		{
			while (not)
			{
				(*(not->remove))(mtd);
				not = not->next;
			}
			mtd_table[i] = NULL;
			up (&mtd_table_mutex);
			return 0;
		}
	}

	up(&mtd_table_mutex);
	return 1;
}



void register_mtd_user (struct mtd_notifier *new)
{
	int i;

	down(&mtd_table_mutex);

	new->next = mtd_notifiers;
	mtd_notifiers = new;
	
	for (i=0; i< MAX_MTD_DEVICES; i++)
		if (mtd_table[i])
			new->add(mtd_table[i]);

	up(&mtd_table_mutex);
}



int unregister_mtd_user (struct mtd_notifier *old)
{
	struct mtd_notifier **prev = &mtd_notifiers;
	struct mtd_notifier *cur;
	int i;

	down(&mtd_table_mutex);

	while ((cur = *prev)) {
		if (cur == old) {
			*prev = cur->next;
			
			for (i=0; i< MAX_MTD_DEVICES; i++)
				if (mtd_table[i])
					old->remove(mtd_table[i]);
			
			up(&mtd_table_mutex);
			return 0;
		}
		prev = &cur->next;
	}
	up(&mtd_table_mutex);
	return 1;
}


/* get_mtd_device(): 
 * Prepare to use an MTD device referenced either by number or address.
 *
 * If <num> == -1, search the table for an MTD device located at <mtd>.
 * If <mtd> == NULL, return the MTD device with number <num>.
 * If both are set, return the MTD device with number <num> _only_ if it
 *     is located at <mtd>.
 */
	
struct mtd_info *__get_mtd_device(struct mtd_info *mtd, int num)
{
	struct mtd_info *ret = NULL;
	int i;

	down(&mtd_table_mutex);

	if (num == -1) {
		for (i=0; i< MAX_MTD_DEVICES; i++)
			if (mtd_table[i] == mtd)
				ret = mtd_table[i];
	} else if (num < MAX_MTD_DEVICES) {
		ret = mtd_table[num];
		if (mtd && mtd != ret)
			ret = NULL;
	}
	
	up(&mtd_table_mutex);
	return ret;
}

/*====================================================================*/
/* /proc/mtd support */

#ifdef CONFIG_PROC_FS

struct proc_dir_entry *proc_mtd;

static inline int mtd_proc_info (char *buf, int i)
{
	struct mtd_info *this = mtd_table[i];

	if (!this) 
		return 0;

	return sprintf(buf, "mtd%d: %8.8lx \"%s\"\n", i, this->size, 
		       this->name);
}

static int mtd_read_proc ( char *page, char **start, off_t off, 
		       int count, int *eof, void *data)
{
	int len = 0, l, i;
        off_t   begin = 0;
      
	down(&mtd_table_mutex);

        for (i=0; i< MAX_MTD_DEVICES; i++) {

                l = mtd_proc_info(page + len, i);
                len += l;
                if (len+begin > off+count)
                        goto done;
                if (len+begin < off) {
                        begin += len;
                        len = 0;
                }
        }

        *eof = 1;
done:
	up(&mtd_table_mutex);
        if (off >= len+begin)
                return 0;
        *start = page + (begin-off);
        return ((count < begin+len-off) ? count : begin+len-off);
}
#endif

/*====================================================================*/

#if LINUX_VERSION_CODE < 0x20300
#ifdef MODULE
#define init_mtd init_module
#define cleanup_mtd cleanup_module
#endif
#define __exit
#endif

int __init init_mtd(void)
{
	int i;
	DEBUG(1, "INIT_MTD:\n");	
	for (i=0; i<MAX_MTD_DEVICES; i++)
		mtd_table[i]=NULL;

	if (register_chrdev(MTD_CHAR_MAJOR,"mtd",&mtd_fops))
	{
		printk(KERN_NOTICE "Can't allocate major number %d for Memory Technology Devices.\n",
		       MTD_CHAR_MAJOR);
		return EAGAIN;
	}
#ifdef CONFIG_PROC_FS
	if ((proc_mtd = create_proc_entry( "mtd", 0, 0 )))
	  proc_mtd->read_proc = mtd_read_proc;
#endif



#if LINUX_VERSION_CODE < 0x20300
// Initialise all other MTD drivers
#ifdef CONFIG_MTD_DOC1000
	init_doc1000();
#endif
#ifdef CONFIG_MTD_DOC2000
	init_doc2000();
#endif
#ifdef CONFIG_MTD_OCTAGON
	init_octagon5066();
#endif
#ifdef CONFIG_MTD_VMAX
	init_vmax301();
#endif
#ifdef CONFIGF_MTD_MIXMEM
	init_mixmem();
#endif
#ifdef CONFIG_MTD_PMC551
	init_pmc551();
#endif
#ifdef CONFIG_FTL
	init_ftl();
#endif
#ifdef CONFIG_NFTL
	init_nftl();
#endif
#ifdef CONFIG_MTD_BLOCK
	init_mtdblock();
#endif
#endif /* LINUX_VERSION_CODE < 0x20300 */
	return 0;
}

static void __exit cleanup_mtd(void)
{
	unregister_chrdev(MTD_CHAR_MAJOR, "mtd");
#ifdef CONFIG_PROC_FS
	if (proc_mtd)
	  remove_proc_entry( "mtd", 0);
#endif
}

      
#if LINUX_VERSION_CODE > 0x20300
module_init(init_mtd);
module_exit(cleanup_mtd);
#endif


