/*
 * Itsy Pocket Computer Host Driver based on Armin Fuerst's Abstract Control 
 * Model which is apparently based on Brad Keryan's USB busmouse driver .
 *
 * version 0.5: Rewrite for URB interface.
 * version 0.2: Improved Bulk transfer. TX led now flashes every time data is
 * sent. Send Encapsulated Data is not needed, nor does it do anything.
 * Why's that ?!? Thanks to Thomas Sailer for his close look at the bulk code.
 * He told me about some important bugs. (5/21/99)
 *
 * version 0.1: Bulk transfer for uhci seems to work now, no dangling tds any
 * more. TX led of the ISDN TA flashed the first time. Does this mean it works?
 * The interrupt of the ctrl endpoint crashes the kernel => no read possible
 * (5/19/99)
 *
 * version 0.0: Driver sets up configuration, sets up data pipes, opens misc
 * device. No actual data transfer is done, since we don't have bulk transfer,
 * yet. Purely skeleton for now. (5/8/99)
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/signal.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/malloc.h>

#include <linux/spinlock.h>

#include <linux/netdevice.h>   /* struct net_device, and other headers */
#include <linux/etherdevice.h> /* eth_type_trans */
#include <linux/ip.h>          /* struct iphdr */
#include <linux/tcp.h>         /* struct tcphdr */
#include <linux/skbuff.h>

#include <linux/usb.h>

#define USB_VENDOR_COMPAQ  1183
#define COMPAQ_ITSY_ID     0x505A

struct itsy_state {
    int present; /* this acm is plugged in */
    int active; /* someone is has this acm's device open */
    
    struct net_device *net_dev;
    struct enet_statistics eth_stats;
    char client_addr[ETH_ALEN];
    
    struct sk_buff *send_skb;
    struct sk_buff *recv_skb;

    struct tq_struct ctrl_task;
    struct usb_device *usb_dev;
    char *readbuffer;	/*buffer for control messages*/
    char *writebuffer;
    int writing, reading;
    void *readtransfer, *writetransfer;
    unsigned int readendp,writeendp,ctrlendp;
    unsigned int readpipe,writepipe,ctrlpipe;
    char buffer;
};

spinlock_t usb_itsy_lock = SPIN_LOCK_UNLOCKED;
char itsy_name[16] = " ";

/*
 * Open and close
 */
int 
itsy_open(struct net_device *net_dev)
{
    int retval = 0;

    netif_start_queue(net_dev);

    return retval;
}

int 
itsy_release(struct net_device *net_dev)
{
    printk("usbd_release\n");
    /* release ports, irq and such -- like fops->close */
    netif_stop_queue(net_dev);
    return 0;
}

/*
 * Configuration changes (passed on by ifconfig)
 */
int 
itsy_config(struct net_device *net_dev, struct ifmap *map)
{
    printk("usbd_config\n");
    
    /* can't act on a running interface */
    if (net_dev->flags & IFF_UP)
    {
        return -EBUSY;
    }
    
    /* ignore other fields */
    return 0;
}

/*
 * Handles control requests, which are initiated in irq handlers. 
 * Because the control request runs synchronously and requires the interrupt
 * this needs to be done outside the context of an irq handler.
 *
 * The data is an integer USB status code.  So far the endpoint may be determined
 * from the status code.
 */
static void
itsy_ctrl(void *data)
{
    struct itsy_state *itsy = &static_itsy_state;
/*    int status = (int) data;*/

    printk("itsy_ctrl %p %d\n", itsy->usb_dev, itsy->writeendp);

/*    switch (status)
      {
      case USB_ST_STALL :*/
    printk("clearing\n");
    usb_clear_halt(itsy->usb_dev, itsy->writeendp | USB_DIR_OUT);
    printk("cleared\n");
#if OLD_VERSION
    itsy->net_dev->tbusy = 0;
#endif
    itsy->eth_stats.tx_packets++;
    itsy->writing = 0;
    itsy->send_skb = NULL;
    mark_bh(NET_BH);
/*            break;
	      default :
	      break;
	      }*/
}

/*
 * Transmit interrupt handler, checks the status and potentially clears the pipe
 * if an error occurred.
 */
static int 
itsy_tx_irq(int state, 
	    void *__buffer, 
	    int count, 
	    void *dev_id)
{
    struct itsy_state *itsy = (struct itsy_state *) dev_id; 
#if 0
    printk("itsy_WRITE_IRQ %d itsy %p skb %p net_dev %p\n", count, itsy, 
           itsy->send_skb, itsy->net_dev);
#endif
    
#if 1
    if (state == USB_ST_STALL)
#else
        if (1)
#endif
	{
	    printk("state in tx irq %x\n", state);
	    /* reset the pipe, I cannot make the synchronous call from
	     * inside the irq handler so I shedule the task for later.
	     */
	    if (itsy->ctrl_task.sync == 0)
	    {
		itsy->ctrl_task.routine = itsy_ctrl;
		itsy->ctrl_task.data = NULL; /*(void *) USB_ST_STALL;*/
		queue_task(&itsy->ctrl_task, &tq_scheduler);
	    }
	    else
	    {
		printk("couldn't clear tx stall\n");
	    }
	}
	else
	{
#if OLD_VERSION
	    itsy->net_dev->tbusy = 0;
#endif
	    itsy->eth_stats.tx_packets++;
	    itsy->writing = 0;
	    itsy->send_skb = NULL;
	    mark_bh(NET_BH);
	}
    
/*	printk("itsy tx irq terminated\n");
	usb_terminate_bulk(itsy->usb_dev, itsy->writetransfer);
	printk("itsy tx transfer terminated\n");*/
    return 0; /* stop tranfer */
}

/*
 * Transmit a packet (called by the kernel)
 */
int 
itsy_tx(struct sk_buff *skb, struct net_device *net_dev)
{
    struct itsy_state *itsy = &static_itsy_state;
    unsigned int send_len;

    if (net_dev->tbusy)
    {
        /* XXX use jiffies to time out the previous send. */
        return -EBUSY;
    }

    if (itsy->present)
    {
#if 0
        printk("itsy_tx about to transmit itsy %p skb %p skb->net_dev %p net_dev %p\n", 
               itsy, skb, itsy->net_dev, net_dev);
#endif
        net_dev->tbusy = 1; /* transmission is busy */
        net_dev->trans_start = jiffies; /* save the timestamp */
        itsy->send_skb = skb;
        /* copy the buffer into the send page...
         * I can't corrupt the actual skb because
         * it is used after I am finished with it, so I make
         * a copy.
         */
        send_len = skb->len - 10;
        if (send_len > PAGE_SIZE)
        {
            printk("copying more than a page ...\n");
        }
        memcpy(itsy->writebuffer, (skb->data+10), send_len);

        itsy->writebuffer[0] = (char) (send_len & 0x00ff);
        itsy->writebuffer[1] = (char) ((send_len & 0xff00) >> 8) ;

        itsy->writetransfer = usb_request_bulk(itsy->usb_dev,
                                               itsy->writepipe, 
                                               itsy_tx_irq,
                                               itsy->writebuffer,
                                               send_len, 
                                               itsy);
        itsy->eth_stats.tx_packets++;
    }
    dev_kfree_skb(skb);
    return 0;
}

/*
 * Ioctl commands 
 */
int
itsy_ioctl(struct net_device *net_dev, struct ifreq *rq, int cmd)
{
 
    printk("usbd_ioctl\n");
    return 0;
}

/*
 * Return statistics to the caller
 */
struct enet_statistics *
itsy_stats(struct net_device *net_dev)
{
    struct itsy_state *priv =  (struct itsy_state *) net_dev->priv;
    struct enet_statistics *stats=NULL;
    
    printk("usbd_stats\n");
    if (priv)
    {
        stats = &priv->eth_stats;
    }
    return stats;
}

/*
 * The init function (sometimes called probe).
 * It is invoked by register_netdev()
 */
int
itsy_init(struct net_device *net_dev)
{
   struct itsy_state *itsy = (struct itsy_state *)net_dev->priv;
    
    /*
     * Then, assign other fields in dev, using ether_setup() and some
     * hand assignments
     */

    /*
     * Assign the hardware address of the board: use
     * 40 00 00 00 00 XX, where XX is the USB address of the
     * device
     */
    net_dev->dev_addr[0] = 0x40;
    net_dev->dev_addr[1] = 0;
    net_dev->dev_addr[2] = 0;
    net_dev->dev_addr[3] = 0;
    net_dev->dev_addr[4] = 0xff;
    net_dev->dev_addr[5] = 1;

   
    itsy->client_addr[0] = 0x40;
    itsy->client_addr[1] = 0;
    itsy->client_addr[2] = 0;
    itsy->client_addr[3] = 0;
    itsy->client_addr[4] = 0;
    itsy->client_addr[5] = 1;
   
    net_dev->open            = itsy_open;
    net_dev->stop            = itsy_release;
    net_dev->set_config      = itsy_config;
    net_dev->hard_start_xmit = itsy_tx;
    net_dev->do_ioctl        = itsy_ioctl;
    net_dev->get_stats       = itsy_stats;

    /* clear the statistics */
    ether_setup(net_dev);
    return 0;
}

/*
 * if buffer is NULL, then we should just kick off a recv.
 */
static int 
itsy_rx_irq(int state, 
            void *__buffer, 
            int len, 
            void *dev_id)
{
    unsigned char *data = __buffer;
    struct itsy_state *itsy = dev_id;
    unsigned char *mac_header;
    int idx;
    
#if 0
    printk("ITSY receive IRQ\n");
    printk("Status %x Length: %p %d\n", state, __buffer, len);
#endif
    /* allocate a skbuf if needed. */
    if (itsy->recv_skb == NULL)
    {
        itsy->recv_skb = dev_alloc_skb(1518);
        /* want the ip packet aligned on a 16 byte boundary so
         * reserve 12 bytes, 2 empty, and 10 for the address.
         * 2 bytes are used for packet length on the USB.
         */
        skb_reserve(itsy->recv_skb, 12);
    }

    /* if the buffer is null no processing to do just start a receive. */
    if ((data != NULL) && (len > 0))
    {
#if 0
        printk("processing receive\n");
#endif
    
        if (len > 0)
        {
            if (skb_tailroom(itsy->recv_skb) < len)
            {
                printk("itsy RECV : no room skb %p"
                       " length %d copy %d\n",
                       itsy->recv_skb, itsy->recv_skb->len, len);
            }
            /* data already there just allocate it. */
            memcpy(skb_put(itsy->recv_skb, len), data, len);
            
            /* packet size shoudln't be hard coded here XXX */
#if 0
            printk("max pkt size %d\n", usb_maxpacket(itsy->usb_dev, itsy->readpipe, 0));
#endif
            if (len < 255)
            {
                /* finished receiving. */
#if 0
                printk("finished packet\n");
#endif
                if (itsy->net_dev->start == 1)
                {
                    /* put the mac addresses back in, over the top of the
                     * size that was sent over.
                     */
                    mac_header = skb_push(itsy->recv_skb, (2*ETH_ALEN)-2);
                    for (idx=0;idx<ETH_ALEN;idx++)
                    {
                        mac_header[idx] = itsy->net_dev->dev_addr[idx];
                        mac_header[idx+ETH_ALEN] = itsy->client_addr[idx];
                    }
                    itsy->recv_skb->dev = itsy->net_dev;
                    itsy->recv_skb->protocol = eth_type_trans(itsy->recv_skb,
                                                              itsy->net_dev);
                    itsy->recv_skb->ip_summed = CHECKSUM_UNNECESSARY;
                
                    itsy->eth_stats.rx_packets++;
                    netif_rx(itsy->recv_skb);
                    itsy->recv_skb = NULL;
                }
                else
                {
                    dev_kfree_skb(itsy->recv_skb);
                    itsy->recv_skb = NULL;
                }
            }
        }
    }
    return 1;  /* continue transfer */
}

static void * 
itsy_probe(struct usb_device *usb_dev, unsigned int ifnum)
{
  struct usb_interface_descriptor *interface;
  void *retval= NULL;
  int cfgnum;

  printk("itsy_probe usb_dev=%p ifnum=%d\n", usb_dev, ifnum);

  /* With the itsy all we care about is the vendor ID
   * and product ID. After that we know it's an itsy 
   * and don't need to check the configuration.
   */
  if (usb_dev->descriptor.idVendor == USB_VENDOR_COMPAQ ||
      usb_dev->descriptor.idProduct == COMPAQ_ITSY_ID)
    {
      struct net_device *netdev = init_etherdev(NULL, sizeof(struct itsy_state));
      struct itsy_state *itsy = (struct itsy_state *)netdev->priv;

      /* have got an itsy, set configuration 0. */
      printk("have got an itsy plugged in\n");
      
      interface = &usb_dev->config[0].interface[0].altsetting[0];
      usb_set_configuration(usb_dev, usb_dev->config[0].bConfigurationValue);
        
      /* now find which endpoint is which */
      for (cfgnum=0; 
           cfgnum < interface->bNumEndpoints;
           cfgnum++) {
        switch (interface->endpoint[cfgnum].bEndpointAddress) {
        case 0x01 :
          /* this is out write endpoint */
          itsy->writeendp = interface->endpoint[cfgnum].bEndpointAddress;
          itsy->writepipe = usb_sndbulkpipe(usb_dev, itsy->writeendp);
          itsy->writebuffer = (char *)__get_dma_pages(GFP_KERNEL, 0);
          break;
        case 0x82 :
          itsy->readendp = interface->endpoint[cfgnum].bEndpointAddress;
          itsy->readpipe = usb_rcvbulkpipe(usb_dev, itsy->readendp);
          printk("interval %d\n", interface->endpoint[cfgnum].bInterval);
          itsy->readbuffer = (char *)__get_dma_pages(GFP_KERNEL, 0);
          break;
        default:
          break;
        }
      }

      printk("read %d write %d\n", itsy->readendp, itsy->writeendp);

      /* we are always receiving */
      usb_request_irq(itsy->usb_dev, itsy->readpipe, itsy_rx_irq, 
                      1, itsy, &itsy->readtransfer);

      retval = itsy;
    }

  return retval;
}

static void 
itsy_disconnect(struct usb_device *usb_dev, void *foo)
{
   struct usb_interface_descriptor *interface = usb_dev->config[0].interface[0].altsetting[0];
   struct itsy_state *itsy = (struct itsy_state *)interface->private_data;

   printk("itsy_disconnect\n");
    
   /* cancel any transactions */
   if (itsy->writetransfer) {
      usb_terminate_bulk(itsy->usb_dev, itsy->writetransfer);
   }
   usb_release_irq(itsy->usb_dev, itsy->readtransfer, itsy->readpipe);
}

static struct usb_driver itsy_driver = {
    "itsy",
    itsy_probe,
    itsy_disconnect,
    { NULL, NULL }
};

int 
usb_itsy_init(void)
{
    printk("itsy_init\n");

    /* initialise USB */
    usb_register(&itsy_driver);
    
    printk(KERN_INFO "USB Itsy registered.\n");
    return 0;
}

#ifdef MODULE

int init_module(void)
{
    return usb_itsy_init();
}

void cleanup_module(void)
{
    struct itsy_state *itsy = &static_itsy_state;

    /* this, too, probably needs work */
    if (itsy->net_dev)
    {
        unregister_netdev(itsy->net_dev);
        kfree(itsy->net_dev);
        itsy->net_dev = NULL;
    }
    usb_deregister(&itsy_driver);
}

#endif


