/*
*
* Driver for the h3600 Touch Screen and other Atmel controlled devices.
*
* Copyright 2000 Compaq Computer Corporation.
*
* Use consistent with the GNU GPL is permitted,
* provided that this copyright notice is
* preserved in its entirety in all copies and derived works.
*
* COMPAQ COMPUTER CORPORATION MAKES NO WARRANTIES, EXPRESSED OR IMPLIED,
* AS TO THE USEFULNESS OR CORRECTNESS OF THIS CODE OR ITS
* FITNESS FOR ANY PARTICULAR PURPOSE.
*
* Author: Charles Flynn.
*
*/

#ifndef __KERNEL__
#  define __KERNEL__
#endif
#ifndef MODULE
#  define MODULE
#endif

/* TODO TODO This version stuff is done ONCE or there are errors */
#if 0
#define __NO_VERSION__ /* don't define kernel_verion in module.h */
#endif
#include <linux/module.h>
#include <linux/version.h>
#if 0
/* UTS_RELEASE is defined in version.h */
char kernel_version [] = UTS_RELEASE;
#endif


#include <linux/init.h>
#include <linux/fs.h>
#include <linux/delay.h>
#include <linux/poll.h>
#include <asm/uaccess.h>        /* get_user,copy_to_user */
#include <linux/string.h>

/* SA1100 serial defines */
#include <asm/arch/hardware.h>
#include <asm/arch/serial_reg.h>
#include <asm/arch/irqs.h>

#include <linux/h3650_ts.h>

#define _INLINE_ inline
/* Note this define only works if TXBUF_MAX is a power of 2 */
#define INCBUF(x,mod) ( (++x) & ( mod - 1) )

/* ++++++++++++local function defines ++++++++++++*/
static int checkDevice(struct inode *pInode);

static int initSerial( void );

#if 0
/* These are the stub functions until I get around to implementing them */
static int xxIoctl(void * , unsigned int , unsigned long );
static int xxRead(ID *);
static int xxRes(ID *);
#endif

static int xxInit(ID *);
/* These are the touch screen functions */
static int tsEvent(ID *);	/* event handler */
static int tsRead(ID *);	/* file operations _read() */
static int tsReadRaw(ID *);	/* file operations _read() */
static int tsIoctl(void * , unsigned int , unsigned long );
static int tsInit(ID *);

static int keyEvent(ID *);	/* buttons event handler */
static int keyRead(ID *);	/* file operations _read() */

static int ackRes(ID *);	/* generic cmd Acknowledgement */
static int verEvent(ID * );	/* Version response handler */
static int epromRdEvent(ID * );/* read eeprom */

/* ++++++++++++local function defines ++++++++++++*/


/* +++++++++++++globals ++++++++++++++++*/
int gMajor = 0;			/* TODO dynamic major for now */
/* This is the pointer to the UART receiving the ATMEL serial data */
unsigned long   port = (unsigned long)&Ser1UTCR0;


/* TODO declare these static */
VER_RET verRet;		/* The structure returned to the client */

KEY_DEV keyDev;		/*  bit7=1=on bit7=0=off bits0:3=key num */
unsigned char keyRet;

TS_DEV tsDev;
TS_RET tsRet;

EPROM_READ epromRdRet;

EPROM_WRITE epromWrRet;

BAT_DEV battRet;

IICRD_DEV iicRdRet;

IICWR_DEV iicWrRet;


/*
	++++++++++++++ start of driver structures ++++++++++++++
	Each Atmel microcontroller device is represented in this array.
*/
/* TODO TODO put g_Events into file private data */
EVENT g_Events[MAX_MINOR] = {
{ tsRead,tsIoctl,TOUCHS_ID,0,0},	/* TS minor = 0 */
{ tsReadRaw,tsIoctl,TOUCHS_ID,0,0},	/* TSRAW minor = 1 */
{ keyRead,0,KEYBD_ID,0,0}  		/* KEY minor = 2 */
};

ID g_Ids[MAX_ID] = {

{
0, (void *)&verRet, verEvent,xxInit,0,sizeof(VER_RET),0
}, /*ID=0 VERSION*/

{ 0,0,0,0,0,0,0 },	/* ID=1 N/A */

{	/* ID=2 keybd */
(void *)&keyDev,(void *)&keyRet,keyEvent,xxInit,sizeof(KEY_DEV),sizeof(keyRet),0
},

{	/* ID=3 Touch Screen */
(void *)&tsDev,(void *)&tsRet,tsEvent,tsInit,sizeof(TS_DEV),sizeof(tsRet),0
},

{	/* ID=4 EEPROM READ */
0, (void *)&epromRdRet,epromRdEvent,xxInit, 0,sizeof(epromRdRet),0
},

{	/* ID=5 EEPROM WRITE */
0, (void *)&epromWrRet,0,xxInit, 0,sizeof(epromWrRet),0
},

{ 0,0,0,0,0,0,0 },	/* ID=6 N/A */
{ 0,0,0,0,0,0,0 },	/* ID=7 N/A */

{ 0, 0, ackRes ,xxInit, 0,0,0 }, /* ID=8 LED */

{	/* ID=9 Battery Status */
0, (void *)&battRet,0,xxInit, 0,sizeof(battRet),0
},

{ 0,0,0,0,0,0,0 },	/* ID=10 N/A */

{	/* ID=11 IIC Read */
0, (void *)&iicRdRet,0,xxInit, 0,sizeof(iicRdRet),0
},

{	/* ID=12 IIC Write */
0, (void *)&iicWrRet,0,xxInit, 0,sizeof(iicWrRet),0
},

{ 0, 0, ackRes ,xxInit, 0,0,0 } /* ID=13 Front Light */

};

/* ++++++++++++++ end driver structures ++++++++++++++*/

static RXDEV rxdev;		/* receive ISR state */
static TXDEV txdev;		/* send */
/* +++++++++++++globals ++++++++++++++++*/


/* +++++++++++++File operations ++++++++++++++*/
static int h3600_fasync(int fd, struct file *filp, int mode);
static ssize_t h3600_read(struct file * , char * , size_t , loff_t * l);
#if 0
ssize_t h3600_write(struct file * , const char * , size_t , loff_t *);
static int h3600_open(struct inode * inode, struct file * filp);
static int h3600_release(struct inode * inode, struct file * filp);
#endif

static int h3600_ioctl(struct inode * inode, struct file *filp
				   ,unsigned int cmd , unsigned long arg);
static unsigned int h3600_poll(struct file * filp, struct poll_table_struct * wait);

static void eventIsr(int , void *, struct pt_regs *);	/* ISR :*/

#if 1
struct file_operations ts_fops = {
	read:           h3600_read,
        poll:           h3600_poll,
	ioctl:		h3600_ioctl,
        fasync:         h3600_fasync,
};

#else
struct file_operations ts_fops = {
		NULL,			/* LSEEK */
		h3600_read,		/* READ  */
		NULL,		/* WRITE */
		NULL,			/* READDIR */
		h3600_poll,		/* SELECT */
		h3600_ioctl,		/* IOCTL */
		NULL,			/* MMAP */
		NULL,		/* OPEN */
		NULL,		
		NULL,		/* CLOSE */
		NULL,
		NULL
};
#endif

/*
	sendBytes
*/
static _INLINE_ void sendBytes(const char * buff, size_t count,int id)
{
    unsigned char chksum;
    unsigned int space,head,tail;

    /* TODO TODO TODO (event << 4) compiles even though event is NOT defined */
    chksum = ( (unsigned char)((id << 4) | count ) );

    while(1)	/* this loop guarantees the payload is sent */
    {
	/* check if enough space - if not wait until the ISR frees some */
	head = txdev.head;
	tail = txdev.tail;
	/* size is the number of bytes waiting to be transmitted */
	space = ( head > tail ) ? head - tail : tail - head;
	space = TXBUF_MAX - space;
	if ( space >= ( count + FRAME_OVERHEAD ) )
	{
	    unsigned int cr3;
	    unsigned cnt = count;
	    txdev.buf[head]= (unsigned char)CHAR_SOF;
	    head=INCBUF(head,TXBUF_MAX);
	    txdev.buf[head]= chksum;
	    head=INCBUF(head,TXBUF_MAX);
	    while(cnt--)
	    {
		txdev.buf[head]= *buff;	
		head=INCBUF(head,TXBUF_MAX);
		chksum += (unsigned char)(*buff++);
	    }
	    txdev.buf[head]=chksum;
	    txdev.head=INCBUF(head,TXBUF_MAX);
#if 0
	    {
	    unsigned i;
	    printk("\nsendBytes: head=%d tail=%d\n",txdev.head,txdev.tail);
		for( i=tail ; i != txdev.head ; i = INCBUF(i,TXBUF_MAX) )
		{
		    printk("%d:%02x\n",i,txdev.buf[i]);
		}
	    }
#endif
	    /* if not enabled then enable Tx IRQs */
	    cr3 = ( (volatile unsigned long *)port)[UTCR3];
	    if ( !( cr3 & UTCR3_TIE) )
	        ( (volatile unsigned long *)port)[UTCR3] = cr3 | UTCR3_TIE;
	    break;	/* all done */	
	}
    }
}

/* +++++++++++++File operations ++++++++++++++*/

#if 1
static int h3600_fasync(int fd, struct file *filp, int mode)
{
    /* TODO TODO put this data into file private data */
    int minor = checkDevice( filp->f_dentry->d_inode);
    if ( minor == - 1)
    {
        printk("h3600_fasyn:bad minor\n");
	return -ENODEV;
    }
    printk("fasync:called minor=%d\n",minor);

    return( fasync_helper(fd, filp, mode, &g_Events[minor].aq) );
}
#else
int h3600_fasync (struct inode *inode, struct file *filp, int mode)
{
    /* TODO TODO put this data into file private data */
    int minor = checkDevice( inode );
    if ( minor == - 1)
    {
        printk("h3600_fasyn:bad minor\n");
	return -ENODEV;
    }
    printk("fasync:called minor=%d\n",minor);
    return fasync_helper(inode, filp, mode, &g_Events[minor].aq);
}
#endif

int h3600_ioctl(struct inode * inode, struct file *filp,
				   unsigned int cmd , unsigned long arg)
{
    char buff[28];	/* TODO magic number see sizeof(EPROM_READ/WRITE) */
    int minor;
    unsigned rxucnt,txcnt=0,sn,id;
    int status = 0;	/* TODO success */
    ID * pid;


    minor = checkDevice( inode );
    if ( minor == - 1)
    {
        printk("h3600_ioctl:bad minor\n");
	return -ENODEV;
    }

#if 0
    printk("h3600_ioctl:minor=%08x cmd=%d\n",minor,cmd);
#endif


    /*
	We transmit the 1st 'txcnt' bytes from this buffer to the Atmel as data
	pid->lenRet = size of buffer copied from user.
	if pid->lenRet == 0 then copy 'txcnt' bytes from user.
	We return the Atmel response in pid->pReturn;
	sizeof(pid-<pReturn) = pid->lenRet.

	This means the 1st txcnt bytes of the copied buffer MUST
		be the ATMEl cmd sequence.
    */
    switch (cmd)
    {
	case LED_ON:
	    txcnt=sizeof(LED_IN);
#if 1
		id=NLED_ID;
#else
		id=7;		/* test -guarantees no response and timeout */
#endif
	    break;
	case GET_VERSION:
	    id=VERSION_ID;
	    break;
	case READ_EPROM:
	    txcnt=2;  /* TODO 2=magic */
	    id=EEPROM_READ_ID;
	    break;
	case BLITE_ON:
	    txcnt=sizeof(BLITE_IN);
	    id=13;
	    break;
	default:
	{
	    EVENT * pev = &g_Events[minor];
    	    ID * pid = &g_Ids[pev->id];
	    if( pev->processIoctl )
		return ( (*pev->processIoctl)(pid->pdev,cmd,arg) );
	    return 0;
	}
    }

    pid = &g_Ids[id];
    rxucnt = pid->lenRet;
    if( txcnt > rxucnt )
	rxucnt = txcnt;
    if( rxucnt )
	__copy_from_user(buff,(char *)arg,rxucnt );
    txcnt &= (0x0f);
#if 0
{
    unsigned i;
    printk("copy_from_user: rxcnt=%d\n",rxucnt);
    for(i=0 ; i < rxucnt ;i++)
	printk("%02x",buff[i]);
}
#endif
    
    sendBytes( buff, txcnt, id );
    /* now wait on response */
    sn = pid->sn;
    interruptible_sleep_on_timeout(&pid->wq, (2 * HZ) ); /*wait 2 secs*/
#if 0
    printk("TxSN=%d ESN=%d lenRet=%d\n",sn, pid->sn,pid->lenRet );
#endif
    if ( pid->sn == ( sn + 1 ) )
    {
	if( rxucnt)
	    __copy_to_user((char *)arg,(char *)pid->pReturn,rxucnt);
    }
    else
	status = -EIO;

    return status;
}

/*
	This IS generic.
*/
unsigned int h3600_poll(struct file * filp, struct poll_table_struct * wait)
{
    int minor;
    EVENT * pev;
    ID * pid;


    minor = checkDevice( filp->f_dentry->d_inode);
    if ( minor == - 1)
	return -ENODEV;


#ifdef DEBUG
    printk("h3600_poll:\n");
#endif

    pev = &g_Events[minor];
    pid = &g_Ids[pev->id];
    poll_wait(filp,&pid->wq,wait);
    /*
	The reason head & tail are in the generic structure is
	to allow us to fast poll
    */
    return ( pev->head == pev->tail) ? 0 : (POLLIN | POLLRDNORM);
}

/*
	This IS generic
*/

ssize_t h3600_read(struct file * filp, char * buf, size_t count, loff_t * l)
{
    int nonBlocking = filp->f_flags & O_NONBLOCK;
    int minor;
    EVENT * pev;
    ID * pid;


    minor = checkDevice( filp->f_dentry->d_inode );
    if ( minor == - 1)
	return -ENODEV;


    pev = &g_Events[minor];
    pid = &g_Ids[pev->id];

#if 0
    printk("h3600_read: minor=%d id=%d\n",minor,pev->id);
#endif

    if( nonBlocking )
    {
	if( ( pev->head != pev->tail ) && pev->processRead )
	{
	    int count;
	    count = (*pev->processRead)(pid);
	    /* returns length to be copied otherwise errno -Exxx */
	    if( count > 0 )
		/* TODO TODO use __copy or copy ? */
    		__copy_to_user(buf,(char *)pid->pReturn,count );
	    return count;
	}
	else
	    return -EINVAL;

    }
    else
    {
#if 0
	int i;
	struct task_struct * curr = get_current();

#ifdef DEBUG
	printk("ts_read: waiting for I/O to complete[%p]\n",&dev->wq);		
#endif

	/* TODO TODO TODO check this code */
	for (i=0 ; i < _NSIG_WORDS; i++)
	{
		if (curr->signal.sig[i] & ~curr->blocked.sig[i])
		{
			printk("ts_read: -ERESTARTSYS\n");
			return -ERESTARTSYS;
		}
	}
#endif
	/* check , when woken, there is a complete event to read */
retry:
	if( ( pev->head != pev->tail ) && pev->processRead )
	{
	    int rcount;
	    rcount = (*pev->processRead)(pid);
	    /* returns length to be copied otherwise errno -Exxx */
	    if( rcount > 0 )
	    {
    		__copy_to_user(buf,(char *)pid->pReturn,rcount );
	    }
	    return rcount;	/* TODO fmtLen if success -Exxx if error */
	}
	else
	{
     	    interruptible_sleep_on(&pid->wq);
	    goto retry;
	}

	
    }
}


#if 0
/*
	h3600_write
	Writes 'count' bytes from buff to transmitter fifo.
*/
ssize_t h3600_write(struct file * filp, const char * buff,
		size_t count, loff_t * off)
{
    int minor;

    minor = checkDevice( filp->f_dentry->d_inode );
    if ( minor == - 1)
	return -ENODEV;

#if 0
    {
    DEV * dev;
    unsigned sn;	/* TODO TODO what type SNs?? */
    /* test code */
    static unsigned dummy=0;
    if( ++dummy & 0x01 )
	minor = 7;	/* guarantees no response */
    }
    sendBytes( buff, count, minor );
    /* now wait on response */

    dev = &g_Devs[minor];
    sn = dev->sn;
    interruptible_sleep_on_timeout(&dev->wq, (2 * HZ) );	/*wait 2 secs*/
    printk("TxSN=%d ESN=%d\n",sn, dev->sn );
    return ( dev->sn == ( sn + 1 ) ) ? count : -EIO;
#endif
    return 0;

}

/*
	TODO TODO do we need an open or release since
	IRQs are enabled from init_module?
*/
int h3600_open(struct inode * inode, struct file * filp)
{
    int minor;

    minor = checkDevice( inode );
    if ( minor == - 1)
	return -ENODEV;

#ifdef DEBUG
    printk("h3600_open:minor=%d\n",minor);
#endif

    return 0;
}

int h3600_release(struct inode * inode, struct file * filp)
{
    int minor;
    minor = checkDevice( inode );
    if ( minor == - 1)
	return -ENODEV;

#ifdef DEBUG
    printk("h3600_release:minor=%d\n",minor);
#endif
    return 0;
}

#endif

/* int __init h3600_init(void) */
int init_module(void)
{
	/* TODO TODO what happens if there is an error - see pdriver */
	int result;
	unsigned i;
	ID * pid;
	printk("init_module:\n");

        /* register our character device */
        printk("ts_dev: init_module registering char device\n");
        result = register_chrdev(0, MOD_NAME, &ts_fops);
        if (result < 0) {
            printk("ts_dev:can't get major number\n");
            return result;
        }

	if( gMajor == 0 )
		gMajor = result;

	initSerial();		/* do this first before irq management */
	
	/* Now register our IRQ - ID must be unique if sharing an IRQ line*/
	/* TODO TODO IRQ not shared tale out */
#if 0
	free_irq(DEV_IRQ, NULL);
	result = request_irq(DEV_IRQ, eventIsr, SA_SHIRQ | SA_INTERRUPT,
				DEV_IRQ_NAME, DEV_IRQ_ID);
#endif
	result = request_irq(DEV_IRQ, eventIsr, SA_SHIRQ | SA_INTERRUPT,
				DEV_IRQ_NAME, DEV_IRQ_ID);
	if (!result)
	{	
       	    printk("init_module successful init major= %d irq=%d\n",
		gMajor,DEV_IRQ);

	    for(i=0 ; i < MAX_ID ; i++)
	    {
		/* Zero device specific structures */
		pid = &g_Ids[i];
		pid->sn=0;
		memset(pid->pdev,0,pid->lenDev);		
		init_waitqueue_head(&pid->wq);
		memset(pid->pReturn,0,pid->lenRet);		
		if( pid->initDev )
			(*pid->initDev)( pid );
	    }
	}
	else
		printk("init_module error from irq %d\n", result);
	return 0;
}

void cleanup_module(void)
{
	printk("cleanup_module:\n");

	free_irq(DEV_IRQ, DEV_IRQ_ID);
	unregister_chrdev(gMajor, MOD_NAME);
}


/*
	Returns the Minor number from the inode.
*/
static int checkDevice(struct inode *pInode)
{
	int minor;
	kdev_t dev = pInode->i_rdev;

	if( MAJOR(dev) != gMajor)
	{
		printk("checkDevice bad major = %d\n",MAJOR(dev) );
		return -1;
	}
	minor = MINOR(dev);

	if ( minor < MAX_ID )
		return minor;
	else
	{
		printk("checkDevice bad minor = %d\n",minor );
		return -1;
	}
}

/* +++++++++++++++++ Start of ISR code ++++++++++++++++++++++++++ */


/*
	Sets up serial port 1
	baud rate = 115k
	8 bits no parity 1 stop bit
*/

int initSerial( )
{
    unsigned int status;

    /* TODO registers read in 32bit chunks - need to do this for the sa1110? */

    /* Clean up CR3 for now */
    ( (volatile unsigned long *)port)[UTCR3]= 0;

    /* 8 bits no parity 1 stop bit */
    ( (volatile unsigned long *)port)[UTCR0] = 0x08;

    /* Set baud rate MS nibble to zero. */
    ( (volatile unsigned long *)port)[UTCR1] = 0;

    /* Set baud rate LS nibble to 115K bits/sec */
    ( (volatile unsigned long *)port)[UTCR2]=0x1;

    /*
	  SR0 has sticky bits, must write ones to them to clear.
	    the only way you can clear a status reg is to write a '1' to clear.
    */
    ( (volatile unsigned long *)port)[UTSR0]=0xff;

    /*
	    enable rx fifo interrupt (SAM 11-116/7)
	    If RIE=0 then sr0.RFS and sr0.RID are ignored ELSE
	    if RIE=1 then whenever sro.RFS OR sr0.RID are high an
		interrupt will be generated.
	    Also enable the transmitter and receiver.
    */
    status = UTCR3_TXE | UTCR3_RXE | UTCR3_RIE;
    ( (volatile unsigned long *)port)[UTCR3]=status;


    /*
	    initialize Serial channel protocol frame
    */
    rxdev.state = STATE_SOF;
    rxdev.event = rxdev.len = 0;
    memset(&rxdev.buf[0],0,sizeof(rxdev.buf));
    txdev.head = txdev.tail=0;
    return 0;	/* TODO how do we denote an error */

} // End of function initSerial

static  _INLINE_ void ClearInterrupt(void)
{
	/* SR0 has sticky bits, must write ones to them to clear */
	( (volatile unsigned long *)port)[UTSR0]=0xff;
}


/*
	This function takes a single byte and detects the frame
*/
static _INLINE_ void processChar(unsigned char rxchar)
{
    switch ( rxdev.state )
    {
		/*
			We are looking for the SOF in this state.
		*/
		case STATE_SOF:
			if ( rxchar == CHAR_SOF )
			{
				/* found it - next byte is the id and len */
				rxdev.state=STATE_ID;
			}
			break;
		/*
			We now expect the id and len byte
		*/
		case STATE_ID:
			rxdev.event = (rxchar & 0xf0) >> 4 ;
			rxdev.len = (rxchar & 0x0f);
			rxdev.idx=0;
			if (rxdev.event >= MAX_ID )
			{
				/* increment error counter 1 */
				printk("Error1\n");
				++rxdev.counter[0];
				rxdev.state = STATE_SOF;
				break;
			}
			rxdev.chksum = rxchar;
			rxdev.state=( rxdev.len > 0 ) ? STATE_DATA : STATE_EOF;
			break;
		/*
			We now expect 'len' data bytes.
		*/
		case STATE_DATA:
			rxdev.chksum += rxchar;
			rxdev.buf[rxdev.idx]= rxchar;
			if( ++rxdev.idx == rxdev.len )
			{
				/* look for the EOF char next */
				rxdev.state = STATE_EOF;
			}
			break;
		/*
			We now expect to find the STATE_EOF
			TODO TODO not sure if this is an STATE_EOF marker
			or simply the checksum byte
		*/
		case STATE_EOF:
			rxdev.state = STATE_SOF;
			if (rxchar == CHAR_EOF || rxchar == rxdev.chksum )
			{
				ID * pid = &g_Ids[(unsigned)rxdev.event];
				if( pid->processEvent )
				{
				  /*printk(" [%d]\n",(unsigned)rxdev.event);*/
				  if ( (*pid->processEvent)(pid) == 0 )
				      wake_up_interruptible(&pid->wq);
				}
			}
			else
			{
			    ++rxdev.counter[1];
			    printk("\nbadFrame ");
			}
			break;
		default:
			++rxdev.counter[2];
			printk("Error3\n");
			break;

    }	/* end switch */

}  /* end of in-line function */


/*

	Frame format
  byte	  1	  2		  3 		 len + 4
	+-------+---------------+---------------+--=------------+
	|SOF	|id	|len	| len bytes	| Chksum	|
	+-------+---------------+---------------+--=------------+
  bit	0     7  8    11 12   15 16

	+-------+---------------+-------+
	|SOF	|id	|0	|Chksum	| - Note Chksum does not include SOF
	+-------+---------------+-------+
  bit	0     7  8    11 12   15 16

*/

static void eventIsr(int irq, void *dev_id, struct pt_regs *regs)
{
    unsigned status0;	/* UTSR0 */
    int rxval;

    /* TODO check if we need to share */
    if ( strcmp( (char *)dev_id, DEV_IRQ_ID) )
	goto exit;;

    /* ascertain reason for interrupt and handle accordingly */
    status0 = ( (volatile unsigned long *)port)[UTSR0];
    if( (status0 & UTSR0_RID) || (status0 & UTSR0_RFS) )
    {
	/* Rx idle or RxFIFO above high water mark and needs service */
	unsigned status1;
#ifdef DEBUG
	printk("IRQ Rx - ");
#endif
	do
	{
	    /* process all bytes in Rx FIFO */
	    rxval = ( (volatile unsigned long *)port)[UART_RX];
	    if (!(rxval & 0x700)) 
	    {
#ifdef DEBUG
		printk("%02x",(unsigned char)rxval );
#else
		processChar( (unsigned char)rxval);
#endif
	    }
	    else
	    {
		/* found parity,frame or overrun error */
		printk("Bad Char\n");
	    }

            status1 = ( (volatile unsigned long *)port)[UTSR1];
	
	} while (status1 & UTSR1_RNE );
#ifdef DEBUG
	printk("\n");
#endif

    }
    else if ( status0 & UTSR0_TFS )
    {
	unsigned int cr3;
	unsigned int tail=txdev.tail;
	unsigned int head=txdev.head;
	/* Tx FIFO below low water mark and needs replenished */
#ifdef DEBUG
	printk("IRQ Tx - ");
#endif
	while( ( (volatile unsigned long *)port)[UTSR1] & UTSR1_TNF )
	{
	    /* load another char into the Tx FIFO */
            ( (volatile unsigned long *)port)[UART_TX] = txdev.buf[tail];
#ifdef DEBUG
	    printk("%02x",txdev.buf[tail]);
#endif
	    if ( (tail=INCBUF(tail,TXBUF_MAX) ) == head )
	    {
		/* nothing more to Tx - disable interrupts */
                cr3 = ( (volatile unsigned long *)port)[UTCR3] & ~UTCR3_TIE;
                ( (volatile unsigned long *)port)[UTCR3] = cr3;
		/* printk(" TxIRQ off head=%d tail=%d\n",head,tail); */
		break;
	    }
	}
	txdev.tail=tail;
#ifdef DEBUG
	printk("\n");
#endif
	
    }
    else if ( status0 & UTSR0_EIF )
    {
	/* Detected a Parity,Frame or Overrun error */
	printk("IRQ EIF\n");
    }
    else
    {
	/* must be a Begin/End of break that caused IRQ */
	printk("IRQ RBB | REB\n");
    }


exit:

    ClearInterrupt();

}	//end of function



/* +++++++++++++++++ end of ISR code ++++++++++++++++++++++++++ */
/****************** start of device handlers section ***********************/

/*
	Buttons - returned as a single byte
	7 6 5 4 3 2 1 0
	S x x x N N N N

	S	switch state ( 0=pressed 1=released)
	x	Unused.
	NNNN	switch number 0-15

*/
static int keyEvent(ID * pDev )		/* touch screen event handler */
{
  KEY_DEV * pKeyDev = (KEY_DEV *)pDev->pdev;
  EVENT * pev = &g_Events[KEY_MINOR];
  unsigned char * pKey = &pKeyDev->buf[pev->head];

  *pKey=rxdev.buf[0];	/* store current key state */

#if 0
  /* debug only */
{
  unsigned char * state;
  state = ( *pKey & KEY_OFF ) ? "OFF" : "ON";
  printk("KERNEL[%02x]KEY %d is %s\n", *pKey, *pKey & KEY_NUM, state );
}
#endif

    pev->head=INCBUF(pev->head,MAX_KEY_EVENTS );
    /* Send an interrupt (SIGIO) to the client's signal handler */
    if (pev->aq)
	kill_fasync(&pev->aq, SIGIO, POLL_IN);

  return 0;

}

static int keyRead(ID * pDev)
{
    /* generic line for all readers */
    KEY_DEV * pKeyDev = (KEY_DEV *)pDev->pdev;
    EVENT * pev = &g_Events[KEY_MINOR];
    unsigned char * pKey = &pKeyDev->buf[pev->tail];
    keyRet = *pKey;
    pev->tail=INCBUF(pev->tail,MAX_KEY_EVENTS );
    return 1;	/* TODO make this a sizeof() */
}


/*	ID=3 TOUCHSCREEN EVENTS
	TouchScreen event data is formatted as shown below:-

	+-------+-------+-------+-------+
	| Xmsb	| Xlsb	| Ymsb	| Ylsb	|
	+-------+-------+-------+-------+
	byte 0	  1	  2	  3

*/

static int tsEvent(ID * pDev )		/* touch screen event handler */
{
  unsigned head;
  /* This is where we will store the event */
  EVENT * pev = &g_Events[TS_MINOR];
  EVENT * pevraw = &g_Events[TSRAW_MINOR];
  TS_DEV * pTsDev = ( TS_DEV * )pDev->pdev;
  TS_EVENT * pEvent;

  head = pev->head;
  pEvent = &pTsDev->events[head];

  if( rxdev.len == TS_DATA_LEN )
  {
    /*
	There are 3 sub-events within the TS_EVENT
	PEN_UP		- user has just lifted pen off screen.
	PEN_FLEETING	- user has just put pen DOWN for the first time.
			- we ignore this event.
	PEN_DOWN	- this is a true sample.
    */
    switch(pTsDev->penStatus)
    {
	case PEN_FLEETING:
	    /* TODO how many fleeting before we pronounce a PEN_DOWN? */
	    pTsDev->penStatus = PEN_DOWN;
	    /* OK to use this data -  pass thru to next case*/
	case PEN_DOWN:
	    {
		/* store raw x,y & pressure to ts_dev.event[head] */
		unsigned  short x,y;
		/* Extract RAW coords from event data  */
		/* TODO TODO check this!! */
		x = rxdev.buf[0]; x <<= 8; x += rxdev.buf[1];
		y = rxdev.buf[2]; y <<= 8; y += rxdev.buf[3];
		/* dont calibrate it until the user asks for it */
		pEvent->x = x;
		pEvent->y = y;
		pEvent->pressure = 1;
		/* increment heads */
		pev->head = INCBUF(head,MAX_TS_EVENTS);
		pevraw->head=pev->head;
#if 0
	        printk("PEN_DOWN: x=%d y=%d p=%d ++h=%d:%d\n",
			pEvent->x,pEvent->y,
			pEvent->pressure,pev->head,pevraw->head);
#endif
	    }
	    break;
	case PEN_UP:
	    /* Pen was previously UP dont trust this sample */
	    pTsDev->penStatus = PEN_FLEETING;
#if 0
	    printk("PEN_FLEETING\n");
#endif
	    return 1; 
	default:
	    pTsDev->penStatus = PEN_UP;
	    printk("PEN_UNKNOWN\n");
	    return -1;
    }

  }
  else if (rxdev.len == 0)
  {
    /* PEN_UP - since we have no data in the frame */
    pTsDev->penStatus = PEN_UP;
    pEvent->x = pEvent->y = pEvent->pressure = 0;
    /* increment heads */
    pev->head = INCBUF(head,MAX_TS_EVENTS);
    pevraw->head=pev->head;
#if 0
    printk("PEN_UP head=%d\n",pev->head);
#endif
  }
  else
  {
    /* we have a length over or under run */
    printk("LENGTH_UNKNOWN\n");
    return -1;
  }

  /* There is an event to report - Async notification */
  if (pev->aq)
	kill_fasync(&pev->aq, SIGIO, POLL_IN);
  if (pevraw->aq)
	kill_fasync(&pevraw->aq, SIGIO, POLL_IN);

  return 0;	/* event to report */

} /* end of tsEvent() */


/*
	Read() function for the touch screen.
	IN:  raw data
	OUT: Calibrated data pointed to by pReturn.
*/
static int tsRead(ID * pDev)
{
	/* generic line for all readers */
    TS_DEV * pTsDev = (TS_DEV *)pDev->pdev;
    EVENT * pev = &g_Events[TS_MINOR];
    TS_EVENT raw = pTsDev->events[pev->tail];
    TS_RET * fmt = (TS_RET *)pDev->pReturn;

    /*
	calibrate the raw data
    */
    if (  raw.pressure )
    {
	/* do this only if x,y and p are non-zero */
        fmt->x = ( (pTsDev->cal.xscale * raw.x) >> 8 ) + pTsDev->cal.xtrans ;
        fmt->y = ( (pTsDev->cal.yscale * raw.y) >> 8 ) + pTsDev->cal.ytrans ;
        fmt->pressure = raw.pressure;
#if 0
	printk("tsRead: x=%d y=%d rawX=%d rawY=%d\n",
		fmt->x,fmt->y,raw.x,raw.y );
#endif
    }
    else
    {
	fmt->x = fmt->y = fmt->pressure = 0;	/* PEN UP */
    }
    /* TODO there may be other stuff to copy to the return buffer */

    pev->tail=INCBUF(pev->tail,MAX_TS_EVENTS);

#if 0
    printk("\ntsRead: h=%d ++t=%d x=%d y=%d p=%d\n",
	pDev->head,pDev->tail,fmt->x,fmt->y,fmt->pressure);
#endif

    return sizeof(TS_RET);		/* success */
}

/*
	RAW Touch screen data - used by the calibration utility.
	Read() function for the touch screen.
	IN:  raw data
	OUT: Calibrated data pointed to by pReturn.
*/
static int tsReadRaw(ID * pDev)
{
	/* generic line for all readers */
    TS_DEV * pTsDev = (TS_DEV *)pDev->pdev;
    EVENT * pev = &g_Events[TSRAW_MINOR];
    TS_EVENT raw = pTsDev->events[pev->tail];
    TS_RET * ret = (TS_RET *)pDev->pReturn;

#if 0
	printk("tsReadRaw:rawX=%d rawY=%d\n", raw.x,raw.y );
#endif
	/* Raw data including PEN_UP */
    ret->x = raw.x;
    ret->y = raw.y;
    ret->pressure = raw.pressure;

    /* TODO there may be other stuff to copy to the return buffer */

    pev->tail=INCBUF(pev->tail,MAX_TS_EVENTS);

#if 0
    printk("\ntsReadRaw:h=%d ++t=%d p=%d\n",pev->head,pev->tail,ret->pressure);
#endif

    return sizeof(TS_RET);		/* success */
}
/*
	ts_ioctl - ioctl function for touch screen device
*/
static int tsIoctl( void * pDevice, unsigned int cmd, unsigned long arg)
{
    TS_DEV * pdev = (TS_DEV *)pDevice;

    printk("tsIoctl: cmd %04x pdev=%p\n",cmd,pdev);

    switch(cmd)
    {
	case TS_GET_RATE:	/* TODO TODO */
	    return pdev->rate;
	    break;
	case TS_SET_RATE:	/* TODO TODO */
	    pdev->rate = arg;
	    break;
	case TS_GET_CAL:
	    printk("TS_GET_CAL\n");
	    __copy_to_user((char *)arg, (char *)&pdev->cal, sizeof(TS_CAL) );
	    break;
	case TS_SET_CAL:
	    printk("\n\nTS_SET_CAL: ");
	    __copy_from_user( (char *)&pdev->cal,(char *)arg ,sizeof(TS_CAL) );
	    
	    printk("xscale=%d xtrans=%d yscale=%d ytrans=%d\n\n",
		pdev->cal.xscale,pdev->cal.xtrans,
		pdev->cal.yscale,pdev->cal.ytrans);
	    break;
	default:
	    printk("IOCTL: unknowncmd %04x\n",cmd);
	    return -EINVAL;
    }

    return 0;
}

static int tsInit(ID * pDev)
{
    TS_DEV * pTsDev = (TS_DEV *)pDev->pdev;

    printk("tsInit: not implemented yet\n");

    /* TODO this is zero'd at init_module - need to do it again? */

    pTsDev->penStatus=PEN_DOWN;

    /* default calibration */
    pTsDev->cal.xscale = -93;
    pTsDev->cal.xtrans = 346;
    pTsDev->cal.yscale = -64;
    pTsDev->cal.ytrans = 251;

    return 0;
}

/* ID=xx
	Stub functions 
*/
#if 0
static int xxIoctl( void * pDevice, unsigned int cmd, unsigned long arg)
{
	printk("xxIoctl: not implemented\n");
	return 0;
}
static int xxRead(ID * pdev)
{
	printk("xxRead: not implemented\n");
	return 0;
}

/* Generic RES function */
static int xxRes(ID * pdev )		/* GET VERSION handler */
{
	printk("xxRes: not implemented\n");
	return 0;
}

#endif

/* Init Stub */ 
static int xxInit(ID * pdev )
{
	printk("xxInit: not implemented\n");
	return 0;
}

/*
	generic response handler - assumes no data - simply an Ack
	All it does is increment a MOD255 sequence number.
*/
static int ackRes(ID * pDev )
{
  ++pDev->sn;
#if 0
  printk("ackRes sn=%d\n",pDev->sn);
#endif
  return 0;
}

/*
	Response handler for GET version number.
	Ver Number returned as 4 bytes XX.YY
	NOTE: Implemented only for the 4-byte version format.
*/
static int verEvent(ID * pDev )
{
  VER_RET * pVer = (VER_RET *)pDev->pReturn;
  unsigned short x;
  /* TODO TODO this may be reversed chack!! */
  x = rxdev.buf[0]; x <<= 8; x += rxdev.buf[1];
  pVer->maj = x;
  x = rxdev.buf[2]; x <<= 8; x += rxdev.buf[3];
  pVer->min = x;
  ++pDev->sn;
  printk("Version %d.%d sn=%d len=%d\n",pVer->maj,pVer->min,pDev->sn,rxdev.len);
#if 0
  printk("rx0=%02x,rx1=%02x,rx2=%02x,rx3=%02x\n",
  rxdev.buf[0], rxdev.buf[1],
  rxdev.buf[2], rxdev.buf[3] );
#endif
 return 0;
}

static int epromRdEvent(ID * pDev)
{
  EPROM_READ *pER = (EPROM_READ *)pDev->pReturn;
 {
	unsigned i;
	printk("epromRd: len=%d\n",rxdev.len);
	for (i=0 ; i < rxdev.len ; i++ )
		printk("%02x",rxdev.buf[i]);
	
	printk("\n\n");
    /* TEST ONLY */
    pER->len=2;
    pER->addr=1;
    pER->buff[0]=0x0304;
    pER->buff[1]=0x0405;
 }

    ++pDev->sn;
    return 0;
}
