/* 
 * mtdflash - an onboard flash mtd device
 * Author: Jamey Hicks <jamey@crl.dec.com>
 * Original Author: Alexander Larsson <alex@cendio.se> 
 *
 * Copyright (c) 2000 Compaq Computer Corp
 *
 */

#include <linux/module.h>
#include <linux/malloc.h>
#include <linux/ioport.h>
#include <linux/init.h>
#include <linux/vmalloc.h>
#include <linux/mtd/mapped.h>
#include <asm/arch/hardware.h>

struct flash_mtd_info {
  struct mtd_info mtd;
  u_long base;
  u_long length;
};


/* Flash mapping start */
#define FLASH_START		0xd0000000
#define FLASH_PARTITIONS	7
static int flash_length[FLASH_PARTITIONS] = 
{ 0x00040000, 0x00080000, 0x00040000, 0x00700000, 0x00800000, 0x00800000, 0x00800000 };

/* Flash commands.. */
#define _CMD( x )  ((x)|((x)<<16))

#define FLASH_CMD_READ		_CMD( 0x00FF )
#define FLASH_CMD_ERASE		_CMD( 0x0020 )
#define FLASH_CMD_CONFIRM	_CMD( 0x00D0 )
#define FLASH_CMD_CLEAR		_CMD( 0x0050 )
#define FLASH_CMD_WRITE		_CMD( 0x0040 )
#define FLASH_CMD_WR_BUFF	_CMD( 0x00e8 )
#define FLASH_CMD_STATUS	_CMD( 0x0070 )
#define FLASH_CMD_ID            _CMD( 0x0090 )

#define FLASH_STATUS_READY	_CMD( 0x0080 )

#define FLASH_WR_BUFF_SIZE	16

#define ERROR_VOLTAGE_RANGE	_CMD( 0x0008 )
#define ERROR_DEVICE_PROTECT	_CMD( 0x0002 )
#define ERROR_PROGRAMMING	_CMD( 0x0010 )

#define USEC		1000000		/* usec per sec */
#define POLL_TIME	10		/* usec */
#define POLL_LOOP_LIMIT	3*USEC/POLL_TIME  /* 3 sec */

static struct flash_mtd_info *mtd_info[FLASH_PARTITIONS];

int mtdflash_point (struct mtd_info *mtd, loff_t from, size_t len, size_t *retlen, u_char **mtdbuf)
{
  struct flash_mtd_info *priv = (struct flash_mtd_info *)mtd;

  if (from + len > mtd->size)
    return -EINVAL;
  
  *mtdbuf = (char*)FLASH_START + priv->base + from;
  *retlen = len;
  return 0;
}

void mtdflash_unpoint (struct mtd_info *mtd, u_char *addr)
{
}

int mtdflash_read(struct mtd_info *mtd, loff_t from, size_t len,
	     size_t *retlen, u_char *buf)
{
  struct flash_mtd_info *priv = (struct flash_mtd_info *)mtd;

  if (from + len > mtd->size)
    return -EINVAL;

  memcpy (buf, (char*)FLASH_START + priv->base + from, len);

  *retlen=len;
  return 0;
}

/*
 *  Function:   full_status_check
 *  Author:     Stephane Dalton
 *  Parameters: in->    flash status value
 *              out->   0: status ok, statuscode: problem
 */
static inline int full_status_check( unsigned long status_reg )
{
   if( status_reg & ERROR_VOLTAGE_RANGE ){
      printk("Flash driver: programming voltage error!\n");
      return status_reg;
   }
   if( status_reg & ERROR_DEVICE_PROTECT ){
      printk("Flash driver: device is write protected!\n");
      return status_reg;
   }
   if( status_reg & ERROR_PROGRAMMING ){
      printk("Flash driver: programming error!\n");
      return status_reg;
   }
   return 0;
}

/* 
 * Function: flash_status_check
 * 	Poll for status until it's ready.  Return error code if some error 
 * 	detected, 0 otherwise.
 * Note: This really really has to be redesigned to avoid busy waits.
 */

static int flash_status_check( volatile u_long *flash_ptr )
{
  u_long status;
  int timeout = POLL_LOOP_LIMIT;

  for( status = *flash_ptr;
       (status & FLASH_STATUS_READY) != FLASH_STATUS_READY;
       status = *flash_ptr ) 
    {
      if( timeout-- < 0 ){
	printk( "flash_mem: flash seems dead!\n" );
	goto error;
      }
      udelay(POLL_TIME);
    } 
  if( full_status_check(status) != 0 ) {
  error:
    *flash_ptr = FLASH_CMD_CLEAR;
    *flash_ptr = FLASH_CMD_READ;
    return status;
  }
  return 0;
}

#define FLASH_PTR(priv, offset) (volatile u_long *)(FLASH_START + (priv)->base + (long)(offset))

int mtdflash_write (struct mtd_info *mtd, loff_t to, size_t len, size_t *retlen, const u_char *buf)
{
  struct flash_mtd_info *priv = (struct flash_mtd_info *)mtd;
  loff_t end = to+len;
  volatile u_long *flash_ptr = FLASH_PTR(priv, 0);
  u_long *data_ptr = (u_long *)buf;

  if (to + len > mtd->size)
    return -EINVAL;
  /* check alignment of target */
  if (to & (FLASH_WR_BUFF_SIZE-1))
    return -EINVAL;
  
  printk("mtdflash_write: beginning of flash=%p to=%llx len=%ld\n", flash_ptr, to, (long)len);
  *flash_ptr = FLASH_CMD_ID;
  printk("mtdflash_write: id data=%08lx\n", *flash_ptr);
  *flash_ptr = FLASH_CMD_CLEAR;

#ifdef CONFIG_SA1100_BITSY
  set_bitsy_egpio(EGPIO_BITSY_VPP_ON);
#endif
  
  while (to < end) {
    int pollmax = 1<<20;
    int cnt = FLASH_WR_BUFF_SIZE;
    u_long error;
    u_long flags;
    flash_ptr = FLASH_PTR(priv, to);


    *flash_ptr = FLASH_CMD_ID;
    printk("mtdflash_write: head of buffer=%p *flash_ptr=%08lx\n", flash_ptr, *flash_ptr);
    *flash_ptr = FLASH_CMD_CLEAR;

    /* Once we start selecting the erase sectors the delay between each 
     * command must not exceed 50us or it will immediately start erasing 
     * and ignore the other sectors */
    save_flags(flags);
    cli();

    do {
      *flash_ptr = FLASH_CMD_WR_BUFF;
      pollmax--;
    } while( pollmax && (*flash_ptr & FLASH_STATUS_READY) != FLASH_STATUS_READY );
    error = flash_status_check( flash_ptr );
    if( error )  {
      printk("mtdflash_write: FLASH_CMD_WR_BUFF error=%08lx\n", error);
      break;
    }

    *flash_ptr = _CMD(cnt-1);
    while( cnt-- ) {
      flash_ptr = FLASH_PTR(priv, to);
      to += sizeof(u_long);
      *flash_ptr = *data_ptr++;
    }
    *flash_ptr = FLASH_CMD_CONFIRM;

    restore_flags(flags);

    error = flash_status_check( flash_ptr );
    printk("mtd_flash_write: wrote buffer: error=%08lx\n", error);
    if( error ) {
      printk("  got an error: break\n");
      break;
    }
  }
  flash_ptr = FLASH_PTR(priv, 0);
  *flash_ptr = FLASH_CMD_READ;
#ifdef CONFIG_SA1100_BITSY
  clr_bitsy_egpio(EGPIO_BITSY_VPP_ON);
#endif

  return 0;
}

int mtdflash_erase (struct mtd_info *mtd, struct erase_info *instr)
{
  int res = 0;
  struct flash_mtd_info *priv = (struct flash_mtd_info *)mtd;
  if (instr->addr + instr->len > mtd->size)
    return -EINVAL;

  /* do the erase */

#ifdef CONFIG_SA1100_BITSY
  set_bitsy_egpio(EGPIO_BITSY_VPP_ON);
#endif

  {
    u_long sector_mask = ~(mtd->erasesize-1);
    u_long start_sector = instr->addr & sector_mask;
    u_long end_sector = (instr->addr + instr->len + mtd->erasesize-1)&sector_mask;
    u_long sector;
    for (sector = start_sector; sector < end_sector; sector += mtd->erasesize) {
      volatile u_long *flash_ptr = (u_long *)(FLASH_START + priv->base + sector);

      *flash_ptr = FLASH_CMD_READ;
      printk("mtd_erase: erasing sector=%08lx flash_ptr=%p *flash_ptr=%08lx\n", 
	     sector, flash_ptr, *flash_ptr);

      *flash_ptr = FLASH_CMD_CLEAR;
      *flash_ptr = FLASH_CMD_ERASE;
      *flash_ptr = FLASH_CMD_CONFIRM;
      res = flash_status_check(flash_ptr);
      *flash_ptr = FLASH_CMD_READ;
      if (res != 0) {
	printk("status=%08x\n", res);
	return -EIO;
      }
    }
  }

#ifdef CONFIG_SA1100_BITSY
  clr_bitsy_egpio(EGPIO_BITSY_VPP_ON);
#endif

  /* now do the other stuff */
  instr->state = MTD_ERASE_DONE;
  if (instr->callback)
    (*instr->callback)(instr);
  return 0;
}

static void __exit cleanup_mtdflash(void)
{
  int i;
  for (i = 0; i < FLASH_PARTITIONS; i++) {
    if (mtd_info[i]) {
      del_mtd_device(&mtd_info[i]->mtd);
      kfree(mtd_info[i]);
    }
  }
}


static int __init init_mtdflash(void)
{
  int i;
  u_long base = 0;
  for (i = 0; i < FLASH_PARTITIONS; i++) {
    // Allocate some memory
    mtd_info[i] = (struct flash_mtd_info *)kmalloc(sizeof(struct flash_mtd_info), GFP_KERNEL);
    if (mtd_info[i] == 0)
      return 0;
   
    memset(mtd_info[i], 0, sizeof(struct flash_mtd_info));

    // Setup the MTD structure
    strcpy(mtd_info[i]->mtd.name,"mtdflash test device");
    mtd_info[i]->mtd.type = MTD_NORFLASH;
    mtd_info[i]->mtd.flags = MTD_CAP_NORFLASH;
    mtd_info[i]->mtd.size = flash_length[i];
    mtd_info[i]->mtd.erasesize = 256*1024;
    mtd_info[i]->base = base;
    mtd_info[i]->length = flash_length[i];
    mtd_info[i]->mtd.module = THIS_MODULE;

    base += flash_length[i];

    mtd_info[i]->mtd.erase = mtdflash_erase;
    mtd_info[i]->mtd.point = mtdflash_point;
    mtd_info[i]->mtd.unpoint = mtdflash_unpoint;
    mtd_info[i]->mtd.read = mtdflash_read;
    mtd_info[i]->mtd.write = mtdflash_write;

    if (add_mtd_device(&mtd_info[i]->mtd)) {
      /* Clean up */
    }
  }   
  return 0;
}



module_init(init_mtdflash);
module_exit(cleanup_mtdflash);
