/*======================================================================

    Device driver for the PCMCIA control functionality of StrongARM
    SA-1100 microprocessors.

    The contents of this file are subject to the Mozilla Public
    License Version 1.1 (the "License"); you may not use this file
    except in compliance with the License. You may obtain a copy of
    the License at http://www.mozilla.org/MPL/

    Software distributed under the License is distributed on an "AS
    IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
    implied. See the License for the specific language governing
    rights and limitations under the License.

    The initial developer of the original code is John G. Dorsey
    <john+@cs.cmu.edu>.  Portions created by John G. Dorsey are
    Copyright (C) 1999 John G. Dorsey.  All Rights Reserved.

    Alternatively, the contents of this file may be used under the
    terms of the GNU Public License version 2 (the "GPL"), in which
    case the provisions of the GPL are applicable instead of the
    above.  If you wish to allow the use of your version of this file
    only under the terms of the GPL and not to allow others to use
    your version of this file under the MPL, indicate your decision
    by deleting the provisions above and replace them with the notice
    and other provisions required by the GPL.  If you do not delete
    the provisions above, a recipient may use your version of this
    file under either the MPL or the GPL.
    
======================================================================*/

#include <linux/module.h>
#include <linux/init.h>
#include <linux/config.h>
#include <linux/delay.h>
#include <linux/ioport.h>
#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/proc_fs.h>
#include <linux/version.h>

#include <pcmcia/version.h>
#include <pcmcia/cs_types.h>
#include <pcmcia/cs.h>
#include <pcmcia/ss.h>
#include <pcmcia/bus_ops.h>

#include <asm/hardware.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/pgtable.h>
#include <asm/procinfo.h>
#include <asm/segment.h>
#include <asm/system.h>

#include "sa1100.h"

#ifdef PCMCIA_DEBUG
static int pc_debug=PCMCIA_DEBUG;
MODULE_PARM(pc_debug, "i");
#define DEBUG(n, args...) if (pc_debug>(n)) printk(KERN_DEBUG args)
#else
#define DEBUG(n, args...) do { } while (0)
#endif

MODULE_AUTHOR("John Dorsey <john+@cs.cmu.edu>");
MODULE_DESCRIPTION("Linux PCMCIA Card Services: SA-1100 Socket Controller");



/* This structure maintains housekeeping state for each socket, such
 * as the last known values of the card detect pins, or the Card Services
 * callback value associated with the socket:
 */
static struct sa1100_pcmcia_socket 
sa1100_pcmcia_socket[SA1100_PCMCIA_MAX_SOCK];

static int sa1100_pcmcia_socket_count=0;


/* We currently don't do anything interesting in this driver with I/O
 * or memory maps, as on the Itsy, this has been taken care of during
 * kernel initialization. These structures are basically just regurgitated
 * back to Card Services:
 */
static pccard_io_map  sa1100_pcmcia_io_map[SA1100_PCMCIA_IO_MAP_COUNT];
static pccard_mem_map sa1100_pcmcia_mem_map[SA1100_PCMCIA_MEM_MAP_COUNT];


/* Returned by the low-level PCMCIA interface: */
static struct pcmcia_low_level *pcmcia_low_level;
static struct pcmcia_maps kernel_maps;


/* Prototypes for routines which are used internally: */

static int  sa1100_pcmcia_driver_init(void);
static void sa1100_pcmcia_driver_shutdown(void);
static void sa1100_pcmcia_interrupt(int irq, void *dev,
				    struct pt_regs *regs);
static int  sa1100_pcmcia_proc_status(char *buf, char **start, off_t pos,
				      int count, int *eof, void *data);


/* Prototypes for operations which are exported to the
 * new-and-impr^H^H^H^H^H^H^H^H^H^H in-kernel PCMCIA core:
 */

static int sa1100_pcmcia_init(unsigned int sock);
static int sa1100_pcmcia_suspend(unsigned int sock);
static int sa1100_pcmcia_register_callback(unsigned int sock,
					   void (*handler)(void *, 
							   unsigned int),
					   void *info);
static int sa1100_pcmcia_inquire_socket(unsigned int sock, 
					socket_cap_t *cap);
static int sa1100_pcmcia_get_status(unsigned int sock, u_int *value);
static int sa1100_pcmcia_get_socket(unsigned int sock, 
				    socket_state_t *state);
static int sa1100_pcmcia_set_socket(unsigned int sock,
				    socket_state_t *state);
static int sa1100_pcmcia_get_io_map(unsigned int sock,
				    struct pccard_io_map *io);
static int sa1100_pcmcia_set_io_map(unsigned int sock,
				    struct pccard_io_map *io);
static int sa1100_pcmcia_get_mem_map(unsigned int sock,
				     struct pccard_mem_map *mem);
static int sa1100_pcmcia_set_mem_map(unsigned int sock,
				     struct pccard_mem_map *mem);
static void sa1100_pcmcia_proc_setup(unsigned int sock,
				     struct proc_dir_entry *base);

static struct pccard_operations sa1100_pcmcia_operations = {
  sa1100_pcmcia_init,
  sa1100_pcmcia_suspend,
  sa1100_pcmcia_register_callback,
  sa1100_pcmcia_inquire_socket,
  sa1100_pcmcia_get_status,
  sa1100_pcmcia_get_socket,
  sa1100_pcmcia_set_socket,
  sa1100_pcmcia_get_io_map,
  sa1100_pcmcia_set_io_map,
  sa1100_pcmcia_get_mem_map,
  sa1100_pcmcia_set_mem_map,
  sa1100_pcmcia_proc_setup
};


/* Virtual Bus Operations
 * ^^^^^^^^^^^^^^^^^^^^^^
 * The following sa1100_bus_* operations are used by other components
 * in the PCMCIA implementation to perform actions like card I/O,
 * IRQ requests, and memory mapping. See the "Linux PCMCIA Programmer's
 * Guide", section 9.3, for more information.
 */

static u32 sa1100_bus_in(void *bus, u32 port, s32 sz){
  DEBUG(5, "%s(%08x, %d)\n", __FUNCTION__, port, sz);

  switch(sz){
  case  0: return inb(port);
  case  1: return le16_to_cpu(inw(port));
  case -1: return inw(port);
  case  2: return le32_to_cpu(inl(port));
  case -2: return inl(port);
  }

  printk(KERN_ERR "%s(): invalid size %d\n", __FUNCTION__, sz);
  return 0;
}

static void sa1100_bus_out(void *bus, u32 val, u32 port, s32 sz){
  DEBUG(5, "%s(%08x, %08x, %d)\n", __FUNCTION__, val, port, sz);

  switch(sz){
  case  0: outb(val, port);               break;
  case  1: outw(cpu_to_le16(val), port);  break;
  case -1: outw(val, port);               break;
  case  2: outl(cpu_to_le32(val), port);  break;
  case -2: outl(val, port);               break;
  default: printk(KERN_ERR "%s(): invalid size %d\n", __FUNCTION__, sz);
  }
}

static void sa1100_bus_ins(void *bus, u32 port, void *buf, 
			   u32 count, s32 sz){
  DEBUG(5, "%s(%08x, %p, %u, %d)\n", __FUNCTION__, port, buf, count, sz);

  switch(sz){
  case 0:
    while((sz--)>0)
      *(((unsigned char *)buf)++)=inb(port);
    break;

  case 1:
    while((sz--)>0)
      *(((unsigned short *)buf)++)=le16_to_cpu(inw(port));
    break;

  case -1:
    while((sz--)>0)
      *(((unsigned short *)buf)++)=inw(port);
    break;

  case 2:
    while((sz--)>0)
      *(((unsigned int *)buf)++)=le32_to_cpu(inl(port));
    break;

  case -2:
    while((sz--)>0)
      *(((unsigned int *)buf)++)=inl(port);
    break;

  default: printk(KERN_ERR "%s(): invalid size %d\n", __FUNCTION__, sz);
  }
}

static void sa1100_bus_outs(void *bus, u32 port, void *buf,
			    u32 count, s32 sz){
  DEBUG(5, "%s(%08x, %p, %u, %d)\n", __FUNCTION__, port, buf, count, sz);

  switch(sz){
  case 0:
    while((sz--)>0)
      outb(*(((unsigned char *)buf)++), port);
    break;
    
  case 1:
    while((sz--)>0)
      outw(cpu_to_le16(*(((unsigned short *)buf)++)), port);
    break;

  case -1:
    while((sz--)>0)
      outw(*(((unsigned short *)buf)++), port);
    break;

  case 2:
    while((sz--)>0)
      outl(cpu_to_le32(*(((unsigned int *)buf)++)), port);
    break;

  case -2:
    while((sz--)>0)
      outl(*(((unsigned int *)buf)++), port);
    break;

  default: printk(KERN_ERR "%s(): invalid size %d\n", __FUNCTION__, sz);
  }
}

static void *sa1100_bus_ioremap(void *bus, u_long ofs, u_long sz){
  DEBUG(3, "%s(%08lx, %lu)\n", __FUNCTION__, ofs, sz);
  return (void *)ofs;
}

static void sa1100_bus_iounmap(void *bus, void *addr){
  DEBUG(3, "%s(%p)\n", __FUNCTION__, addr);
}

static u32 sa1100_bus_read(void *bus, void *addr, s32 sz){
  DEBUG(5, "%s(%p, %d)\n", __FUNCTION__, addr, sz);

  switch(sz){
  case  0: return *((unsigned char *)addr);
  case  1: return le16_to_cpu(*((unsigned short *)addr));
  case -1: return *((unsigned short *)addr);
  case  2: return le32_to_cpu(*((unsigned int *)addr));
  case -2: return *((unsigned int *)addr);
  }

  printk(KERN_ERR "%s(): invalid size %d\n", __FUNCTION__, sz);
  return 0;
}

static void sa1100_bus_write(void *bus, u32 val, void *addr, s32 sz){
  DEBUG(5, "%s(%x, %p, %d)\n", __FUNCTION__, val, addr, sz);

  switch(sz){
  case  0: *((unsigned char *)addr)=(unsigned char)val;                break;
  case  1: *((unsigned short *)addr)=cpu_to_le16((unsigned short)val); break;
  case -1: *((unsigned short *)addr)=(unsigned short)val;              break;
  case  2: *((unsigned int *)addr)=cpu_to_le32((unsigned int)val);     break;
  case -2: *((unsigned int *)addr)=(unsigned int)val;                  break;
  default: printk(KERN_ERR "%s(): invalid size %d\n", __FUNCTION__, sz);
  }
}

static void sa1100_bus_copy_from(void *bus, void *d, void *s, u32 count){
  DEBUG(5, "%s(%p, %p, %u)\n", __FUNCTION__, d, s, count);
  _memcpy_fromio(d, (unsigned long)s, count);
}

static void sa1100_bus_copy_to(void *bus, void *d, void *s, u32 count){
  DEBUG(5, "%s(%p, %p, %u)\n", __FUNCTION__, d, s, count);
  _memcpy_toio((unsigned long)d, s, count);
}

static int sa1100_bus_request_irq(void *bus, u_int irq,
				  void (*handler)(int, void *,
						  struct pt_regs *),
				  u_long flags, const char *device,
				  void *dev_id){

  DEBUG(2, "%s(%u, %p, %08lx, \"%s\", %p)\n", __FUNCTION__, irq, handler,
	flags, device, dev_id);

  return(request_irq(irq, handler, flags, device, dev_id));
}

static void sa1100_bus_free_irq(void *bus, u_int irq, void *dev_id){

  DEBUG(2, "%s(%u, %p)\n", __FUNCTION__, irq, dev_id);

  free_irq(irq, dev_id);
}


/* sa1100_pcmcia_driver_init()
 * ^^^^^^^^^^^^^^^^^^^^^^^^^^^
 * This routine used to be named sa1100_pcmcia_init().
 *
 * Here, we perform three tasks:
 *
 *   1. Initialize the kernel low-level PCMCIA mechanism.
 *   2. Determine initial state by issuing a fake IRQ sample.
 *   3. Register with Card Services.
 *
 * Please see linux/Documentation/arm/SA1100/PCMCIA for more information
 * on the low-level kernel interface.
 *
 * Returns: 0 on success, -1 on error
 */
static int __init sa1100_pcmcia_driver_init(void){
  int i;
  struct pcmcia_init pcmcia_init;
  struct pcmcia_state state[SA1100_PCMCIA_MAX_SOCK];
  struct pcmcia_state_array state_array;
  servinfo_t info;

  printk(KERN_INFO "SA-1100 PCMCIA (CS release %s)\n", CS_RELEASE);

  CardServices(GetCardServicesInfo, &info);

  if(info.Revision!=CS_RELEASE_CODE){
    printk(KERN_ERR "Card Services release codes do not match\n");
    return -1;
  }

  if(machine_is_assabet()){
    if(machine_has_neponset()){
#ifdef CONFIG_ASSABET_NEPONSET
      pcmcia_low_level=&neponset_pcmcia_ops;
#else
      printk(KERN_ERR "Card Services disabled: missing Neponset support\n");
      return -1;
#endif
    }else{
      pcmcia_low_level=&assabet_pcmcia_ops;
    }
  } else if (machine_is_bitsy()) {
#ifdef CONFIG_SA1100_BITSY
    pcmcia_low_level = &bitsy_pcmcia_ops;
#endif    
  } else {
    printk(KERN_ERR "This hardware is not supported by the SA1100 Card Service driver\n");
    return -1;
  }

  pcmcia_init.handler=sa1100_pcmcia_interrupt;
  pcmcia_init.maps=&kernel_maps;

  if((sa1100_pcmcia_socket_count=pcmcia_low_level->init(&pcmcia_init))<0){
    printk(KERN_ERR "Unable to initialize kernel PCMCIA service.\n");
    return -1;
  }

  state_array.size=sa1100_pcmcia_socket_count;
  state_array.state=state;


  /* These values are from arch/arm/mm/mm-sa1100.c: */

  kernel_maps.sock[0].io.start=0xe0000000;
  kernel_maps.sock[0].io.stop=0xe3ffffff;

  kernel_maps.sock[0].attr.start=0xe8000000;
  kernel_maps.sock[0].attr.stop=0xebffffff;

  kernel_maps.sock[0].mem.start=0xf0000000;
  kernel_maps.sock[0].mem.stop=0xf3ffffff;

  kernel_maps.sock[1].io.start=0xe4000000;
  kernel_maps.sock[1].io.stop=0xe7ffffff;

  kernel_maps.sock[1].attr.start=0xec000000;
  kernel_maps.sock[1].attr.stop=0xefffffff;

  kernel_maps.sock[1].mem.start=0xf4000000;
  kernel_maps.sock[1].mem.stop=0xf7ffffff;


  if(pcmcia_low_level->socket_state(&state_array)<0){
    printk(KERN_ERR "Unable to get PCMCIA status from kernel.\n");
    return -1;
  }

  memset(sa1100_pcmcia_socket, 0, sizeof(sa1100_pcmcia_socket));

  for(i=0; i<sa1100_pcmcia_socket_count; ++i){
    sa1100_pcmcia_socket[i].k_state=state[i];

    /* As suggested, our "private data" is just a label to indicate the
     * socket:
     */
    sa1100_pcmcia_socket[i].bus_ops.priv=(void *)i;

    sa1100_pcmcia_socket[i].bus_ops.b_in=sa1100_bus_in;
    sa1100_pcmcia_socket[i].bus_ops.b_ins=sa1100_bus_ins;
    sa1100_pcmcia_socket[i].bus_ops.b_out=sa1100_bus_out;
    sa1100_pcmcia_socket[i].bus_ops.b_outs=sa1100_bus_outs;
    sa1100_pcmcia_socket[i].bus_ops.b_ioremap=sa1100_bus_ioremap;
    sa1100_pcmcia_socket[i].bus_ops.b_iounmap=sa1100_bus_iounmap;
    sa1100_pcmcia_socket[i].bus_ops.b_read=sa1100_bus_read;
    sa1100_pcmcia_socket[i].bus_ops.b_write=sa1100_bus_write;
    sa1100_pcmcia_socket[i].bus_ops.b_copy_from=sa1100_bus_copy_from;
    sa1100_pcmcia_socket[i].bus_ops.b_copy_to=sa1100_bus_copy_to;
    sa1100_pcmcia_socket[i].bus_ops.b_request_irq=sa1100_bus_request_irq;
    sa1100_pcmcia_socket[i].bus_ops.b_free_irq=sa1100_bus_free_irq;

  }

  /* Only advertise as many sockets as we can detect: */
  if(register_ss_entry(sa1100_pcmcia_socket_count, 
		       &sa1100_pcmcia_operations)<0){
    printk(KERN_ERR "Unable to register socket service routine\n");
    return -1;
  }

  DEBUG(1, "sa1100: initialization complete\n");

  return 0;

}  /* sa1100_pcmcia_driver_init() */

module_init(sa1100_pcmcia_driver_init);


/* sa1100_pcmcia_driver_shutdown()
 * ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
 * Invokes the low-level kernel service to free IRQs associated with this
 * socket controller and reset GPIO edge detection.
 */
static void __exit sa1100_pcmcia_driver_shutdown(void){

  unregister_ss_entry(&sa1100_pcmcia_operations);

  pcmcia_low_level->shutdown();

  DEBUG(1, "sa1100: shutdown complete\n");
}

module_exit(sa1100_pcmcia_driver_shutdown);


/* sa1100_pcmcia_init()
 * ^^^^^^^^^^^^^^^^^^^^
 * We perform all of the interesting initialization tasks in 
 * sa1100_pcmcia_driver_init().
 *
 * Returns: 0
 */
static int sa1100_pcmcia_init(unsigned int sock){
  
  DEBUG(2, "%s(): initializing socket %u\n", __FUNCTION__, sock);

  return 0;
}


/* sa1100_pcmcia_suspend()
 * ^^^^^^^^^^^^^^^^^^^^^^^
 * We don't currently perform any actions on a suspend.
 *
 * Returns: 0
 */
static int sa1100_pcmcia_suspend(unsigned int sock){

  DEBUG(2, "%s(): suspending socket %u\n", __FUNCTION__, sock);

  return 0;
}


/* sa1100_pcmcia_interrupt()
 * ^^^^^^^^^^^^^^^^^^^^^^^^^
 * Service routine for card detect interrupts (requested in 
 * sa1100_pcmcia_init()).
 *
 * Whenever an interruptable PCMCIA card event occurs, this routine is 
 * entered to update internal driver state, and possibly to call back
 * Card Services. The routine loops over all of the slots it knows about, 
 * and performs the following steps for each slot which has generated an
 * interrupt:
 *
 *   1. Construct a bitmask of events to report back to Card Services.
 *      Events that can be reported include card detection, card
 *      status change, low battery voltage, and so forth.
 *
 *   2. Invoke the Card Services callback for the slot we're servicing,
 *      passing the event mask we just constructed.
 *
 */
static void sa1100_pcmcia_interrupt(int irq, void *dev,
				    struct pt_regs *regs){
  unsigned int i, events;
  int irq_status;
  struct pcmcia_state state[SA1100_PCMCIA_MAX_SOCK];
  struct pcmcia_state_array state_array;

  state_array.size=sa1100_pcmcia_socket_count;
  state_array.state=state;

  if((irq_status=pcmcia_low_level->socket_state(&state_array))<0){
    printk(KERN_ERR "Error in kernel PCMCIA interrupt service.\n");
    return;
  }

  if(irq_status==0)  /* Interrupt wasn't for us. */
    return;

  for(i=0; i<sa1100_pcmcia_socket_count; ++i){

    events=0;

    if(state[i].detect!=sa1100_pcmcia_socket[i].k_state.detect){

      DEBUG(2, "%s(): card detect value %u (IRQ %d)\n", __FUNCTION__,
	    state[i].detect, irq);

      events|=sa1100_pcmcia_socket[i].cs_state.csc_mask&SS_DETECT;

      if(state[i].detect==0){
	sa1100_pcmcia_socket[i].cs_state.Vcc=0;
	sa1100_pcmcia_socket[i].cs_state.Vpp=0;
      }
    }

    if(state[i].ready!=sa1100_pcmcia_socket[i].k_state.ready){

      DEBUG(2, "%s(): card ready value %u (IRQ %d)\n", __FUNCTION__,
	    state[i].ready, irq);

      events|=sa1100_pcmcia_socket[i].cs_state.csc_mask&
	((sa1100_pcmcia_socket[i].cs_state.flags&SS_IOCARD)?0:SS_READY);
    }

    if(state[i].bvd1!=sa1100_pcmcia_socket[i].k_state.bvd1){

      DEBUG(2, "%s(): card BVD1 value %u (IRQ %d)\n", __FUNCTION__,
	    state[i].bvd1, irq);

      events|=sa1100_pcmcia_socket[i].cs_state.csc_mask&
	(sa1100_pcmcia_socket[i].cs_state.flags&SS_IOCARD)?\
	SS_STSCHG:SS_BATDEAD;
    }

    if(state[i].bvd2!=sa1100_pcmcia_socket[i].k_state.bvd2){

      DEBUG(2, "%s(): card BVD2 value %u (IRQ %d)\n", __FUNCTION__,
	    state[i].bvd2, irq);

      events|=sa1100_pcmcia_socket[i].cs_state.csc_mask&
	(sa1100_pcmcia_socket[i].cs_state.flags&SS_IOCARD)?0:SS_BATWARN;
    }

    DEBUG(2, "events[%d]: %s%s%s%s%s%s\n", i,
	  (events==0)?"<NONE>":"",
	  (events&SS_DETECT)?"DETECT ":"",
	  (events&SS_READY)?"READY ":"", 
	  (events&SS_BATDEAD)?"BATDEAD ":"",
	  (events&SS_BATWARN)?"BATWARN ":"",
	  (events&SS_STSCHG)?"STSCHG ":"");

    sa1100_pcmcia_socket[i].k_state=state[i];

    if(events!=0 && sa1100_pcmcia_socket[i].handler!=NULL)
      sa1100_pcmcia_socket[i].handler(sa1100_pcmcia_socket[i].handler_info,
				      events);

  }  /* for(i=0; ...) */

}  /* sa1100_pcmcia_interrupt() */


/* sa1100_pcmcia_register_callback()
 * ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
 * Implements the register_callback() operation for the in-kernel
 * PCMCIA service (formerly SS_RegisterCallback in Card Services). If 
 * the function pointer `handler' is not NULL, remember the callback 
 * location in the state for `sock', and increment the usage counter 
 * for the driver module. (The callback is invoked from the interrupt
 * service routine, sa1100_pcmcia_interrupt(), to notify Card Services
 * of interesting events.) Otherwise, clear the callback pointer in the
 * socket state and decrement the module usage count.
 *
 * Returns: 0
 */
static int sa1100_pcmcia_register_callback(unsigned int sock,
					   void (*handler)(void *,
							   unsigned int),
					   void *info){
  if(handler==NULL){
    sa1100_pcmcia_socket[sock].handler=NULL;
    MOD_DEC_USE_COUNT;
  } else {
    MOD_INC_USE_COUNT;
    sa1100_pcmcia_socket[sock].handler=handler;
    sa1100_pcmcia_socket[sock].handler_info=info;
  }

  return 0;
}


/* sa1100_pcmcia_inquire_socket()
 * ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
 * Implements the inquire_socket() operation for the in-kernel PCMCIA
 * service (formerly SS_InquireSocket in Card Services). Of note is
 * the setting of the SS_CAP_PAGE_REGS bit in the `features' field of
 * `cap' to "trick" Card Services into tolerating large "I/O memory" 
 * addresses. Also set is SS_CAP_STATIC_MAP, which disables the memory
 * resource database check. (Mapped memory is set up within the socket
 * driver itself.)
 *
 * Returns: 0 on success, -1 if no pin has been configured for `sock'
 */
static int sa1100_pcmcia_inquire_socket(unsigned int sock,
					socket_cap_t *cap){
  struct pcmcia_irq_info irq_info;

  DEBUG(3, "%s() for sock %u\n", __FUNCTION__, sock);

  if(sock>=sa1100_pcmcia_socket_count){
    printk(KERN_ERR "sa1100: socket %u not configured\n", sock);
    return -1;
  }

  /* SS_CAP_PAGE_REGS: used by setup_cis_mem() in cistpl.c to set the
   *   force_low argument to validate_mem() in rsrc_mgr.c -- since in
   *   general, the mapped * addresses of the PCMCIA memory regions
   *   will not be within 0xffff, setting force_low would be
   *   undesirable.
   *
   * SS_CAP_VIRTUAL_BUS: readb(), &c., are defined to be troublemaking
   *   stubs if the target architecture doesn't support PCI. 
   *   Unfortunately, Card Services is littered with these calls.
   *   Fortunately, we can override the literal invocations by
   *   supplying our own table of routines.
   *
   * SS_CAP_STATIC_MAP: don't bother with the (user-configured) memory
   *   resource database; we'll handle the memory mapping in this
   *   driver.
   *
   * SS_CAP_PCCARD: we can deal with 16-bit PCMCIA & CF cards, but
   *   not 32-bit CardBus devices.
   */
  cap->features=(SS_CAP_PAGE_REGS  | SS_CAP_VIRTUAL_BUS | 
		 SS_CAP_STATIC_MAP | SS_CAP_PCCARD);

  irq_info.sock=sock;
  irq_info.irq=0;

  if(pcmcia_low_level->get_irq_info(&irq_info)<0){
    printk(KERN_ERR "Error obtaining IRQ info from kernel for socket %u\n",
	   sock);
    return -1;
  }

  if(irq_info.irq!=0)
    IRQ_SET(&cap->irq_mask, irq_info.irq);
  cap->map_size=PAGE_SIZE;
  cap->pci_irq=0;
  cap->bus=&(sa1100_pcmcia_socket[sock].bus_ops);

  return 0;

}  /* sa1100_pcmcia_inquire_socket() */


/* sa1100_pcmcia_get_status()
 * ^^^^^^^^^^^^^^^^^^^^^^^^^^
 * Implements the get_status() operation for the in-kernel PCMCIA
 * service (formerly SS_GetStatus in Card Services). Essentially just
 * fills in bits in `status' according to internal driver state or
 * the value of the voltage detect chipselect register.
 *
 * As a debugging note, during card startup, the PCMCIA core issues
 * three set_socket() commands in a row the first with RESET deasserted,
 * the second with RESET asserted, and the last with RESET deasserted
 * again. Following the third set_socket(), a get_status() command will
 * be issued. The kernel is looking for the SS_READY flag (see
 * setup_socket(), reset_socket(), and unreset_socket() in cs.c).
 *
 * Returns: 0
 */
static int sa1100_pcmcia_get_status(unsigned int sock,
				    unsigned int *status){
  struct pcmcia_state state[SA1100_PCMCIA_MAX_SOCK];
  struct pcmcia_state_array state_array;

  DEBUG(3, "%s() for sock %u\n", __FUNCTION__, sock);

  state_array.size=sa1100_pcmcia_socket_count;
  state_array.state=state;

  if((pcmcia_low_level->socket_state(&state_array))<0){
    printk(KERN_ERR "Unable to get PCMCIA status from kernel.\n");
    return -1;
  }

  sa1100_pcmcia_socket[sock].k_state=state[sock];

  *status=state[sock].detect?SS_DETECT:0;

  *status|=state[sock].ready?SS_READY:0;

  /* The power status of individual sockets is not available
   * explicitly from the hardware, so we just remember the state
   * and regurgitate it upon request:
   */
  *status|=sa1100_pcmcia_socket[sock].cs_state.Vcc?SS_POWERON:0;

  if(sa1100_pcmcia_socket[sock].cs_state.flags&SS_IOCARD)
    *status|=state[sock].bvd1?SS_STSCHG:0;
  else {
    if(state[sock].bvd1==0)
      *status|=SS_BATDEAD;
    else if(state[sock].bvd2==0)
      *status|=SS_BATWARN;
  }

  *status|=state[sock].vs_3v?SS_3VCARD:0;

  *status|=state[sock].vs_Xv?SS_XVCARD:0;

  DEBUG(3, "\tstatus: %s%s%s%s%s%s%s%s\n",
	(*status&SS_DETECT)?"DETECT ":"",
	(*status&SS_READY)?"READY ":"", 
	(*status&SS_BATDEAD)?"BATDEAD ":"",
	(*status&SS_BATWARN)?"BATWARN ":"",
	(*status&SS_POWERON)?"POWERON ":"",
	(*status&SS_STSCHG)?"STSCHG ":"",
	(*status&SS_3VCARD)?"3VCARD ":"",
	(*status&SS_XVCARD)?"XVCARD ":"");

  return 0;

}  /* sa1100_pcmcia_get_status() */


/* sa1100_pcmcia_get_socket()
 * ^^^^^^^^^^^^^^^^^^^^^^^^^^
 * Implements the get_socket() operation for the in-kernel PCMCIA
 * service (formerly SS_GetSocket in Card Services). Not a very 
 * exciting routine.
 *
 * Returns: 0
 */
static int sa1100_pcmcia_get_socket(unsigned int sock,
				    socket_state_t *state){

  DEBUG(3, "%s() for sock %u\n", __FUNCTION__, sock);

  /* This information was given to us in an earlier call to set_socket(),
   * so we're just regurgitating it here:
   */
  *state=sa1100_pcmcia_socket[sock].cs_state;

  return 0;
}


/* sa1100_pcmcia_set_socket()
 * ^^^^^^^^^^^^^^^^^^^^^^^^^^
 * Implements the set_socket() operation for the in-kernel PCMCIA
 * service (formerly SS_SetSocket in Card Services). We more or
 * less punt all of this work and let the kernel handle the details
 * of power configuration, reset, &c. We also record the value of
 * `state' in order to regurgitate it to the PCMCIA core later.
 *
 * Returns: 0
 */
static int sa1100_pcmcia_set_socket(unsigned int sock,
				    socket_state_t *state){
  struct pcmcia_configure configure;

  DEBUG(3, "%s() for sock %u\n", __FUNCTION__, sock);

  DEBUG(3, "\tmask:  %s%s%s%s%s%s\n\tflags: %s%s%s%s%s%s\n"
	"\tVcc %d  Vpp %d  irq %d\n",
	(state->csc_mask==0)?"<NONE>":"",
	(state->csc_mask&SS_DETECT)?"DETECT ":"",
	(state->csc_mask&SS_READY)?"READY ":"",
	(state->csc_mask&SS_BATDEAD)?"BATDEAD ":"",
	(state->csc_mask&SS_BATWARN)?"BATWARN ":"",
	(state->csc_mask&SS_STSCHG)?"STSCHG ":"",
	(state->flags==0)?"<NONE>":"",
	(state->flags&SS_PWR_AUTO)?"PWR_AUTO ":"",
	(state->flags&SS_IOCARD)?"IOCARD ":"",
	(state->flags&SS_RESET)?"RESET ":"",
	(state->flags&SS_SPKR_ENA)?"SPKR_ENA ":"",
	(state->flags&SS_OUTPUT_ENA)?"OUTPUT_ENA ":"",
	state->Vcc, state->Vpp, state->io_irq);

  configure.sock=sock;
  configure.vcc=state->Vcc;
  configure.vpp=state->Vpp;
  configure.output=(state->flags&SS_OUTPUT_ENA)?1:0;
  configure.speaker=(state->flags&SS_SPKR_ENA)?1:0;
  configure.reset=(state->flags&SS_RESET)?1:0;

  if(pcmcia_low_level->configure_socket(&configure)<0){
    printk(KERN_ERR "Unable to configure socket %u\n", sock);
    return -1;
  }

  sa1100_pcmcia_socket[sock].cs_state=*state;
  
  return 0;

}  /* sa1100_pcmcia_set_socket() */


/* sa1100_pcmcia_get_io_map()
 * ^^^^^^^^^^^^^^^^^^^^^^^^^^
 * Implements the get_io_map() operation for the in-kernel PCMCIA
 * service (formerly SS_GetIOMap in Card Services). Just returns an
 * I/O map descriptor which was assigned earlier by a set_io_map().
 *
 * Returns: 0 on success, -1 if the map index was out of range
 */
static int sa1100_pcmcia_get_io_map(unsigned int sock,
				    struct pccard_io_map *map){

  DEBUG(4, "%s() for sock %u\n", __FUNCTION__, sock);

  if(map->map>=SA1100_PCMCIA_IO_MAP_COUNT){
    printk(KERN_ERR "%s(): map (%d) out of range\n", __FUNCTION__,
	   map->map);
    return -1;
  }

  *map=sa1100_pcmcia_io_map[map->map];

  return 0;
}


/* sa1100_pcmcia_set_io_map()
 * ^^^^^^^^^^^^^^^^^^^^^^^^^^
 * Implements the set_io_map() operation for the in-kernel PCMCIA
 * service (formerly SS_SetIOMap in Card Services). We record the
 * map descriptor for use in a subsequent get_io_map() request, and
 * set up the I/O memory map.
 *
 * Returns: 0 on success, -1 on error
 */
static int sa1100_pcmcia_set_io_map(unsigned int sock,
				    struct pccard_io_map *map){
  unsigned int clock, speed;
  unsigned long mecr;

  DEBUG(4, "%s() for sock %u\n", __FUNCTION__, sock);

  DEBUG(4, "\tmap %u  speed %u\n\tstart 0x%08lx  stop 0x%08lx\n"
	"\tflags: %s%s%s%s%s%s%s%s\n",
	map->map, map->speed, map->start, map->stop,
	(map->flags==0)?"<NONE>":"",
	(map->flags&MAP_ACTIVE)?"ACTIVE ":"",
	(map->flags&MAP_16BIT)?"16BIT ":"",
	(map->flags&MAP_AUTOSZ)?"AUTOSZ ":"",
	(map->flags&MAP_0WS)?"0WS ":"",
	(map->flags&MAP_WRPROT)?"WRPROT ":"",
	(map->flags&MAP_USE_WAIT)?"USE_WAIT ":"",
	(map->flags&MAP_PREFETCH)?"PREFETCH ":"");

  if(map->map>=SA1100_PCMCIA_IO_MAP_COUNT){
    printk(KERN_ERR "%s(): map (%d) out of range\n", __FUNCTION__,
	   map->map);
    return -1;
  }

  if(map->flags&MAP_ACTIVE){

    speed=(map->speed>0)?map->speed:SA1100_PCMCIA_IO_ACCESS;
    
    if((clock=SA1100_PCMCIA_CCF())==0){
      printk(KERN_ERR "sa1100: invalid CCF code (0x%02x)\n",
	     FExtr(PPCR, PPCR_CCF));
      return -1;
    }

    mecr=MECR;

    MECR_BSIO_SET(mecr, sock, sa1100_pcmcia_mecr_bs(speed, clock));

    DEBUG(4, "%s(): FAST%u %lx  BSM%u %lx  BSA%u %lx  BSIO%u %lx\n",
	  __FUNCTION__, sock, MECR_FAST_GET(mecr, sock), sock,
	  MECR_BSM_GET(mecr, sock), sock, MECR_BSA_GET(mecr, sock), 
	  sock, MECR_BSIO_GET(mecr, sock));

    MECR=mecr;

  } else 
    /* This is extremely sketchy. What's going on here is that we're
     * causing Card Services to change its mind about where it thinks
     * the I/O base should be. We don't fiddle with the `stop' field
     * because it's not really important; CS will remember the
     * NumPorts value it used to originally calculate the `stop'
     * address anyway. Right now we're taking it on faith that CS
     * won't be asking for more space than is actually mapped for
     * PCMCIA I/O (64MB).
     */
    map->start=kernel_maps.sock[sock].io.start;

  sa1100_pcmcia_io_map[map->map]=*map;

  return 0;

}  /* sa1100_pcmcia_set_io_map() */


/* sa1100_pcmcia_get_mem_map()
 * ^^^^^^^^^^^^^^^^^^^^^^^^^^^
 * Implements the get_mem_map() operation for the in-kernel PCMCIA
 * service (formerly SS_GetMemMap in Card Services). Just returns a
 *  memory map descriptor which was assigned earlier by a
 *  set_mem_map() request.
 *
 * Returns: 0 on success, -1 if the map index was out of range
 */
static int sa1100_pcmcia_get_mem_map(unsigned int sock,
				     struct pccard_mem_map *map){

  DEBUG(4, "%s() for sock %u\n", __FUNCTION__, sock);

  if(map->map>=SA1100_PCMCIA_MEM_MAP_COUNT){
    printk(KERN_ERR "%s(): map (%d) out of range\n", __FUNCTION__,
	   map->map);
    return -1;
  }

  *map=sa1100_pcmcia_mem_map[map->map];

  return 0;
}


/* sa1100_pcmcia_set_mem_map()
 * ^^^^^^^^^^^^^^^^^^^^^^^^^^^
 * Implements the set_mem_map() operation for the in-kernel PCMCIA
 * service (formerly SS_SetMemMap in Card Services). We record the
 * map descriptor for use in a subsequent get_mem_map() request,
 * and set up the memory map.
 *
 * Returns: 0 on success, -1 on error
 */
static int sa1100_pcmcia_set_mem_map(unsigned int sock,
				     struct pccard_mem_map *map){
  unsigned int clock=0;
  unsigned long mecr;

  DEBUG(4, "%s() for sock %u\n", __FUNCTION__, sock);

  DEBUG(4, "\tmap %u  speed %u\n\tsys_start  %#lx\n"
	"\tsys_stop   %#lx\n\tcard_start %#x\n"
	"\tflags: %s%s%s%s%s%s%s%s\n",
	map->map, map->speed, map->sys_start, map->sys_stop,
	map->card_start, (map->flags==0)?"<NONE>":"",
	(map->flags&MAP_ACTIVE)?"ACTIVE ":"",
	(map->flags&MAP_16BIT)?"16BIT ":"",
	(map->flags&MAP_AUTOSZ)?"AUTOSZ ":"",
	(map->flags&MAP_0WS)?"0WS ":"",
	(map->flags&MAP_WRPROT)?"WRPROT ":"",
	(map->flags&MAP_ATTRIB)?"ATTRIB ":"",
	(map->flags&MAP_USE_WAIT)?"USE_WAIT ":"");

  if(map->map>=SA1100_PCMCIA_MEM_MAP_COUNT){
    printk(KERN_ERR "%s(): map (%d) out of range\n", __FUNCTION__,
	   map->map);
    return -1;
  }

  if(map->flags&MAP_ACTIVE && map->speed>0){
    
    if((clock=SA1100_PCMCIA_CCF())==0){
      printk(KERN_ERR "sa1100: invalid CCF code (0x%02x)\n",
	     FExtr(PPCR, PPCR_CCF));
      return -1;
    }
    
    mecr=MECR;

    /* SetMemMap is called so many times, we can just do this here on
     * behalf of the IO space as well: 
     */
    MECR_FAST_SET(mecr, sock, 0);

    if(map->flags&MAP_ATTRIB)
      MECR_BSA_SET(mecr, sock, sa1100_pcmcia_mecr_bs(map->speed, clock));
    else
      MECR_BSM_SET(mecr, sock, sa1100_pcmcia_mecr_bs(map->speed, clock));

    DEBUG(4, "%s(): FAST%u %lx  BSM%u %lx  BSA%u %lx  BSIO%u %lx\n",
	  __FUNCTION__, sock, MECR_FAST_GET(mecr, sock), sock,
	  MECR_BSM_GET(mecr, sock), sock, MECR_BSA_GET(mecr, sock), 
	  sock, MECR_BSIO_GET(mecr, sock));
    
    MECR=mecr;
  }

  /* As with set_io_map() above, here we're going to fool the PCMCIA
   * core by changing the `sys_start' address. We don't bother with
   * `sys_stop' because the core can just keep the size it was using 
   * when it invoked us.
   */
  if(map->flags&MAP_ATTRIB)
    map->sys_start=kernel_maps.sock[sock].attr.start;
  else
    map->sys_start=kernel_maps.sock[sock].mem.start;
  
  sa1100_pcmcia_mem_map[map->map]=*map;

  return 0;

}  /* sa1100_pcmcia_set_mem_map() */


/* sa1100_pcmcia_proc_setup()
 * ^^^^^^^^^^^^^^^^^^^^^^^^^^
 * Implements the proc_setup() operation for the in-kernel PCMCIA
 * service (formerly SS_ProcSetup in Card Services).
 *
 * Returns: 0 on success, -1 on error
 */
static void sa1100_pcmcia_proc_setup(unsigned int sock,
				     struct proc_dir_entry *base){
  struct proc_dir_entry *entry;

  DEBUG(4, "%s() for sock %u\n", __FUNCTION__, sock);

  if((entry=create_proc_entry("status", 0, base))==NULL){
    printk(KERN_ERR "Unable to install \"status\" procfs entry\n");
    return;
  }

  entry->read_proc=sa1100_pcmcia_proc_status;
  entry->data=(void *)sock;
}


/* sa1100_pcmcia_proc_status()
 * ^^^^^^^^^^^^^^^^^^^^^^^^^^^
 * Implements the /proc/bus/pccard/??/status file.
 *
 * Returns: the number of characters added to the buffer
 */
static int sa1100_pcmcia_proc_status(char *buf, char **start, off_t pos,
				     int count, int *eof, void *data){
  char *p=buf;
  unsigned int sock=(unsigned int)data;

  p+=sprintf(p, "k_flags  : %s%s%s%s%s%s%s\n", 
	     sa1100_pcmcia_socket[sock].k_state.detect?"detect ":"",
	     sa1100_pcmcia_socket[sock].k_state.ready?"ready ":"",
	     sa1100_pcmcia_socket[sock].k_state.bvd1?"bvd1 ":"",
	     sa1100_pcmcia_socket[sock].k_state.bvd2?"bvd2 ":"",
	     sa1100_pcmcia_socket[sock].k_state.wrprot?"wrprot ":"",
	     sa1100_pcmcia_socket[sock].k_state.vs_3v?"vs_3v ":"",
	     sa1100_pcmcia_socket[sock].k_state.vs_Xv?"vs_Xv ":"");

  p+=sprintf(p, "status   : %s%s%s%s%s%s%s%s%s\n",
	     sa1100_pcmcia_socket[sock].k_state.detect?"SS_DETECT ":"",
	     sa1100_pcmcia_socket[sock].k_state.ready?"SS_READY ":"",
	     sa1100_pcmcia_socket[sock].cs_state.Vcc?"SS_POWERON ":"",
	     sa1100_pcmcia_socket[sock].cs_state.flags&SS_IOCARD?\
	     "SS_IOCARD ":"",
	     (sa1100_pcmcia_socket[sock].cs_state.flags&SS_IOCARD &&
	      sa1100_pcmcia_socket[sock].k_state.bvd1)?"SS_STSCHG ":"",
	     ((sa1100_pcmcia_socket[sock].cs_state.flags&SS_IOCARD)==0 &&
	      (sa1100_pcmcia_socket[sock].k_state.bvd1==0))?"SS_BATDEAD ":"",
	     ((sa1100_pcmcia_socket[sock].cs_state.flags&SS_IOCARD)==0 &&
	      (sa1100_pcmcia_socket[sock].k_state.bvd2==0))?"SS_BATWARN ":"",
	     sa1100_pcmcia_socket[sock].k_state.vs_3v?"SS_3VCARD ":"",
	     sa1100_pcmcia_socket[sock].k_state.vs_Xv?"SS_XVCARD ":"");

  p+=sprintf(p, "mask     : %s%s%s%s%s\n",
	     sa1100_pcmcia_socket[sock].cs_state.csc_mask&SS_DETECT?\
	     "SS_DETECT ":"",
	     sa1100_pcmcia_socket[sock].cs_state.csc_mask&SS_READY?\
	     "SS_READY ":"",
	     sa1100_pcmcia_socket[sock].cs_state.csc_mask&SS_BATDEAD?\
	     "SS_BATDEAD ":"",
	     sa1100_pcmcia_socket[sock].cs_state.csc_mask&SS_BATWARN?\
	     "SS_BATWARN ":"",
	     sa1100_pcmcia_socket[sock].cs_state.csc_mask&SS_STSCHG?\
	     "SS_STSCHG ":"");

  p+=sprintf(p, "cs_flags : %s%s%s%s%s\n",
	     sa1100_pcmcia_socket[sock].cs_state.flags&SS_PWR_AUTO?\
	     "SS_PWR_AUTO ":"",
	     sa1100_pcmcia_socket[sock].cs_state.flags&SS_IOCARD?\
	     "SS_IOCARD ":"",
	     sa1100_pcmcia_socket[sock].cs_state.flags&SS_RESET?\
	     "SS_RESET ":"",
	     sa1100_pcmcia_socket[sock].cs_state.flags&SS_SPKR_ENA?\
	     "SS_SPKR_ENA ":"",
	     sa1100_pcmcia_socket[sock].cs_state.flags&SS_OUTPUT_ENA?\
	     "SS_OUTPUT_ENA ":"");

  p+=sprintf(p, "Vcc      : %d\n", sa1100_pcmcia_socket[sock].cs_state.Vcc);

  p+=sprintf(p, "Vpp      : %d\n", sa1100_pcmcia_socket[sock].cs_state.Vpp);

  p+=sprintf(p, "irq      : %d\n", sa1100_pcmcia_socket[sock].cs_state.io_irq);

  return p-buf;
}
