/*
 * UDA1341 Audio Device Driver for SA1100
 *
 * Copyright (c) 2000 Nicolas Pitre <nico@cam.org>
 *
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/types.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/malloc.h>
#include <linux/delay.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/errno.h>
#include <linux/sound.h>
#include <linux/soundcard.h>

#include <asm/system.h>
#include <asm/uaccess.h>
#include <asm/io.h>
#include <asm/hardware.h>
#include <asm/irq.h>


#undef  DEBUG
#ifdef DEBUG
#define DPRINTK( x... )  printk( ##x )
#else
#define DPRINTK( x... )
#endif


/*
 * GPIO based L3 bus support.
 *
 * This provides control of Philips L3 type devices. 
 * GPIO lines are used for clock, data and mode pins.
 *
 * Note: The L3 pins are shared with I2C devices. This should not present
 * any problems as long as an I2C start sequence is not generated. This is
 * defined as a 1->0 transition on the data lines when the clock is high.
 * It is critical this code only allow data transitions when the clock
 * is low. This is always legal in L3.
 *
 * The IIC interface requires the clock and data pin to be LOW when idle. We
 * must make sure we leave them in this state.
 *
 * It appears the read data is generated on the falling edge of the clock
 * and should be held stable during the clock high time.
 */

/* 
 * L3 bus GPIO pin definitions
 */
/* I believe that these ifdef are in error and should be machin_is_?() */

#ifdef CONFIG_SA1100_ASSABET
    #define L3_DataPin	GPIO_GPIO(15)
    #define L3_ClockPin	GPIO_GPIO(18)
    #define L3_ModePin	GPIO_GPIO(17)
#endif
#ifdef CONFIG_SA1100_BITSY
    #define L3_DataPin	GPIO_BITSY_L3_DATA
    #define L3_ClockPin	GPIO_BITSY_L3_CLOCK
    #define L3_ModePin	GPIO_BITSY_L3_MODE
#endif


/* 
 * L3 setup and hold times (expressed in us)
 */
#define L3_DataSetupTime 1 /* 190 ns */
#define L3_DataHoldTime  1 /*  30 ns */
#define L3_ModeSetupTime 1 /* 190 ns */
#define L3_ModeHoldTime  1 /* 190 ns */
#define L3_ClockHighTime 100 /* 250 ns (min is 64*fs, 35us @ 44.1 Khz) */
#define L3_ClockLowTime  100 /* 250 ns (min is 64*fs, 35us @ 44.1 Khz) */
#define L3_HaltTime      1 /* 190 ns */

/*
 * Grab control of the IIC/L3 shared pins
 */
static inline void L3_acquirepins(void)
{
	GPSR = (L3_ClockPin | L3_DataPin);
	GPDR |=  (L3_ClockPin | L3_DataPin);
}

/*
 * Release control of the IIC/L3 shared pins
 */
static inline void L3_releasepins(void)
{
	GPDR &= ~(L3_ClockPin | L3_DataPin);
	GPCR = (L3_ClockPin | L3_DataPin);
}

/*
 * Initialize the interface
 */
static void __init L3_init(void)
{
	GAFR &= ~(L3_DataPin | L3_ClockPin | L3_ModePin);
	GPSR = L3_ModePin;
	GPDR |= L3_ModePin;
	L3_releasepins();
}

/*
 * Get a bit. The clock is high on entry and on exit. Data is read after
 * the clock low time has expired.
 */
static inline int L3_getbit(void)
{
	int data;

	GPCR = L3_ClockPin;
	udelay(L3_ClockLowTime);

	data = (GPLR & L3_DataPin) ? 1 : 0;

 	GPSR = L3_ClockPin;
	udelay(L3_ClockHighTime);

	return data;
}

/*
 * Send a bit. The clock is high on entry and on exit. Data is sent only
 * when the clock is low (I2C compatibility).
 */
static inline void L3_sendbit(int bit)
{
	GPCR = L3_ClockPin;

	if (bit & 1)
		GPSR = L3_DataPin;
	else
		GPCR = L3_DataPin;

	/* Assumes L3_DataSetupTime < L3_ClockLowTime */
	udelay(L3_ClockLowTime);

	GPSR = L3_ClockPin;
	udelay(L3_ClockHighTime);
}

/*
 * Send a byte. The mode line is set or pulsed based on the mode sequence
 * count. The mode line is high on entry and exit. The mod line is pulsed
 * before the second data byte and before ech byte thereafter.
 */
static void L3_sendbyte(char data, int mode)
{
	int i;

	L3_acquirepins();

	switch(mode) {
	    case 0: /* Address mode */
		GPCR = L3_ModePin;
		break;
	    case 1: /* First data byte */
		break;
	    default: /* Subsequent bytes */
		GPCR = L3_ModePin;
		udelay(L3_HaltTime);
		GPSR = L3_ModePin;
		break;
	}

	udelay(L3_ModeSetupTime);

	for (i = 0; i < 8; i++)
		L3_sendbit(data >> i);

	if (mode == 0)  /* Address mode */
		GPSR = L3_ModePin;

	udelay(L3_ModeHoldTime);

	L3_releasepins();
}

/*
 * Get a byte. The mode line is set or pulsed based on the mode sequence
 * count. The mode line is high on entry and exit. The mod line is pulsed
 * before the second data byte and before each byte thereafter. This
 * function is never valid with mode == 0 (address cycle) as the address
 * is always sent on the bus, not read.
 */
static char L3_getbyte(int mode)
{
	char data = 0;
	int i;

	L3_acquirepins();
	GPDR &= ~(L3_DataPin);

	switch(mode) {
	    case 0: /* Address mode - never valid */
		break;
	    case 1: /* First data byte */
		break;
	    default: /* Subsequent bytes */
		GPCR = L3_ModePin;
		udelay(L3_HaltTime);
		GPSR = L3_ModePin;
		break;
	}

	udelay(L3_ModeSetupTime);

	for (i = 0; i < 8; i++)
		data |= (L3_getbit() << i);

	udelay(L3_ModeHoldTime);

	L3_releasepins();

	return data;
}

/*
 * Write data to a device on the L3 bus. The address is passed as well as
 * the data and length. The length written is returned. The register space
 * is encoded in the address (low two bits are set and device address is
 * in the upper 6 bits).
 */
static int L3_write(char addr, char *data, int len)
{
	int mode = 0;
	int bytes = len;

	L3_sendbyte(addr, mode++);
	while(len--)
		L3_sendbyte(*data++, mode++);

	return bytes;
}

/*
 * Read data from a device on the L3 bus. The address is passed as well as
 * the data and length. The length read is returned. The register space
 * is encoded in the address (low two bits are set and device address is
 * in the upper 6 bits).
 */
static int L3_read(char addr, char * data, int len)
{
	int mode = 0;
	int bytes = len;

	L3_sendbyte(addr, mode++);
	while(len--)
		*data++ = L3_getbyte(mode++);

	return bytes;
}


/*
 * UDA1341 L3 address and command types
 */
#define UDA1341_L3Addr	5
#define UDA1341_DATA0	0
#define UDA1341_DATA1	1
#define UDA1341_STATUS	2


/* Definitions */

#define AUDIO_NAME		"UDA1341"
#define AUDIO_NAME_VERBOSE	"UDA1341 audio driver"
#define AUDIO_VERSION_STRING	"version 0.2"

#define AUDIO_FMT_MASK		(AFMT_S16_LE)
#define AUDIO_FMT_DEFAULT	(AFMT_S16_LE)
#define AUDIO_CHANNELS_DEFAULT	2
#define AUDIO_RATE_DEFAULT	44100
#define AUDIO_NBFRAGS_DEFAULT	8
#define AUDIO_FRAGSIZE_DEFAULT	8192


/* 
 * UDA1341 internal state variables.  
 * Those are initialized according to what it should be after a reset.
 */

/* UDA1341 status settings */

#define UDA_STATUS0_IF_I2S      0
#define UDA_STATUS0_IF_LSB16    1
#define UDA_STATUS0_IF_LSB18    2
#define UDA_STATUS0_IF_LSB20    3
#define UDA_STATUS0_IF_MSB      4
#define UDA_STATUS0_IF_MSB16    5
#define UDA_STATUS0_IF_MSB18    6
#define UDA_STATUS0_IF_MSB20    7

#define UDA_STATUS0_SC_512FS   0
#define UDA_STATUS0_SC_384FS   1
#define UDA_STATUS0_SC_256FS   2

static struct {
	u_int	DC_filter	: 1;	/* DC filter */
	u_int	input_fmt	: 3;	/* data input format */
	u_int	system_clk	: 2;	/* system clock frequency */
	u_int	reset		: 1;	/* reset */
  const u_int	select		: 1;	/* must be set to 0 */
} STATUS_0 = { 0, UDA_STATUS0_IF_LSB16, UDA_STATUS0_SC_256FS, 0, 0 };

static struct {
	u_int	DAC_on		: 1;	/* DAC powered */
	u_int	ADC_on		: 1;	/* ADC powered */
	u_int	double_speed	: 1;	/* double speed playback */
	u_int	DAC_pol		: 1;	/* polarity of DAC */
	u_int	ADC_pol		: 1;	/* polarity of ADC */
	u_int	ADC_gain	: 1;	/* gain of ADC */
	u_int	DAC_gain	: 1;	/* gain of DAC */
  const u_int	select		: 1;	/* must be set to 1 */
} STATUS_1 = { 1, 1, 0, 0, 0, 0, 0, 1 };

/* UDA1341 direct control settings */

static struct {
	u_int	volume		: 6;	/* volume control */
  const u_int	select		: 2;	/* must be set to 0 */
} DATA0_0 = { 0, 0 };

static struct {
	u_int	treble		: 2;
	u_int	bass		: 4;
  const u_int	select		: 2;	/* must be set to 1 */
} DATA0_1 = { 0, 0, 1 };

static struct {
	u_int	mode		: 2;	/* mode switch */
	u_int	mute		: 1;
	u_int	deemphasis	: 2;
	u_int	peak_detect	: 1;
  const u_int	select		: 2;	/* must be set to 2 */
} DATA0_2 = { 0, 0, 0, 1, 2 };

/* DATA0 extended programming registers */

static struct {
  const u_int	ext_addr	: 3;	/* must be set to 0 */
  const u_int	select1		: 5;	/* must be set to 24 */
	u_int	ch1_gain	: 5;	/* mixer gain channel 1 */
  const u_int	select2		: 3;	/* must be set to 7 */
} DATA0_ext0 = { 0, 24, 0, 7 };

static struct {
  const u_int	ext_addr	: 3;	/* must be set to 1 */
  const u_int	select1		: 5;	/* must be set to 24 */
	u_int	ch2_gain	: 5;	/* mixer gain channel 2 */
  const u_int select2		: 3;	/* must be set to 7 */
} DATA0_ext1 = { 1, 24, 0, 7 };

static struct {
  const u_int	ext_addr	: 3;	/* must be set to 2 */
  const u_int	select1		: 5;	/* must be set to 24 */
	u_int	mixer_mode	: 2;
	u_int	mic_level	: 3;	/* MIC sensitivity level */
  const u_int	select2		: 3;	/* must be set to 7 */
} DATA0_ext2 = { 2, 24, 0, 1, 7 };

static struct {
  const u_int	ext_addr	: 3;	/* must be set to 4 */
  const u_int	select1		: 5;	/* must be set to 24 */
	u_int	ch2_igain_l	: 2;	/* input amplifier gain channel 2 (bits 1-0) */
  const u_int	reserved	: 2;	/* must be set to 0 */
	u_int	AGC_ctrl	: 1;	/* AGC control */
  const u_int	select2		: 3;	/* must be set to 7 */
} DATA0_ext4 = { 4, 24, 0 & 3, 0, 0, 7 };

static struct {
  const u_int	ext_addr	: 3;	/* must be set to 5 */
  const u_int	select1		: 5;	/* must be set to 24 */
	u_int	ch2_igain_h	: 5;	/* input amplifier gain channel 2 (bits 6-2) */
  const u_int	select2		: 3;	/* must be set to 7 */
} DATA0_ext5 = { 5, 24, 0 >> 2, 7 };

static struct {
  const u_int	ext_addr	: 3;	/* must be set to 6 */
  const u_int	select1		: 5;	/* must be set to 24 */
	u_int	AGC_level	: 2;	/* AGC output level */
	u_int	AGC_const	: 3;	/* AGC time constant */
  const u_int	select2		: 3;	/* must be set to 7 */
} DATA0_ext6 = { 6, 24, 0, 0, 7 };


/*
 * DMA register structure
 */

typedef struct {
	volatile u_long DDAR;
	volatile u_long SetDCSR;
	volatile u_long ClrDCSR;
	volatile u_long RdDCSR;
	volatile void * DBSA;
	volatile u_long DBTA;
	volatile void * DBSB;
	volatile u_long DBTB;
} dma_regs_t;


/* 
 * Buffer Management
 */

typedef struct {
	volatile int ref;		/* number of dma references */
	char *ptr;			/* current outgoing data position */
	int size;			/* buffer size */
	char *start;			/* points to actual buffer */
	char *dma;			/* physical dma address */
	int master;			/* master owner for dma space */
} audio_buf_t;

typedef struct {
	audio_buf_t *buffers;		/* pointer to audio buffer structures */
	audio_buf_t *buf;		/* current buffer used by read/write */
 	audio_buf_t *dma_head;		/* buffer to start next dma with */
	audio_buf_t *dma_tail;		/* buffer currently dma'ed */
	u_int buf_idx;			/* index for those pointers above... */
	u_int dma_head_idx;
	u_int dma_tail_idx;
	u_int fragsize;			/* fragment i.e. buffer size */
	u_int nbfrags;			/* nbr of fragments i.e. buffers */
	dma_regs_t *DMA;		/* Pointer to DMA regs */
	volatile int dma_active;	/* 1 if DMA transfer activated */
	volatile int dma_spin;		/* 1 if DMA should spin with all zeros */
	volatile int dma_spin_ref;	/* number of spinning DMAs */
} audio_stream_t;

static audio_stream_t output_stream;
static audio_stream_t input_stream;

#define NEXT_BUF(_s_,_b_) { \
	(_s_)->_b_##_idx++; \
	(_s_)->_b_##_idx %= (_s_)->nbfrags; \
	(_s_)->_b_ = (_s_)->buffers + (_s_)->_b_##_idx; }


/* Current specs for incoming audio data */
static u_int audio_rate;		
static int audio_channels;
static int audio_fmt;
static u_int audio_fragsize;
static u_int audio_nbfrags;


static wait_queue_head_t audio_waitq;	/* ... to wait for DMA completion */
static int audio_refcount;		/* nbr of concurrent open() */
static int audio_dev_dsp;		/* registered ID for DSP device */
static int audio_dev_mixer;		/* registered ID for mixer device */



/* This function frees all buffers */

static void audio_clear_buf( audio_stream_t *s )
{
	DPRINTK("audio_clear_buf\n");

	/* ensure DMA won't run anymore */
	s->DMA->ClrDCSR = DCSR_STRTA|DCSR_STRTB|DCSR_IE|DCSR_RUN;
	s->dma_active = 0;

	if( s->buffers ){
		int frag;
		for( frag = 0; frag < s->nbfrags; frag++ ){
			if( !s->buffers[frag].master ) continue;
			consistent_free( s->buffers[frag].start );
		}
		kfree( s->buffers );
		s->buffers = NULL;
	}

	s->buf_idx = s->dma_head_idx = s->dma_tail_idx = 0;
	s->buf = s->dma_head = s->dma_tail = NULL;
}


/* This function allocates the buffer structure array and buffer data space
 * according to the current number of fragments and fragment size.
 */

static int audio_setup_buf( audio_stream_t *s )
{
	int frag; 
	int dmasize = 0;
	char *dmabuf = 0;
	dma_addr_t dmaphys = 0;

	if( s->buffers ) return -EBUSY;

	s->nbfrags = audio_nbfrags;
	s->fragsize = audio_fragsize;

	s->buffers = (audio_buf_t *)
		kmalloc( sizeof(audio_buf_t) * s->nbfrags, GFP_KERNEL );
	if( !s->buffers) goto err;
	memset( s->buffers, 0, sizeof(audio_buf_t) * s->nbfrags );

	for( frag = 0; frag < s->nbfrags; frag++ ){
		audio_buf_t *b = &s->buffers[frag];

		/*
		 * Let's allocate non-cached memory for DMA buffers.
		 * We try to allocate all memory at once.
		 * If this fails (a common reason is memory fragmentation), 
		 * then we allocate more smaller buffers.
		 */
		if( !dmasize ){
			dmasize = (s->nbfrags - frag) * s->fragsize;
			do {
				dmabuf = consistent_alloc(
						GFP_KERNEL, 
						dmasize,
						&dmaphys );
				if( !dmabuf )
					dmasize -= s->fragsize;
			} while( !dmabuf && dmasize );
			if( !dmabuf ) goto err;
			b->master = 1;
		} else b->master = 0;

		b->start = dmabuf;
		b->dma = (char*)dmaphys;
		b->ptr = b->dma;
		b->ref = (s == &output_stream) ? 0 : 1;
		b->size = (s == &output_stream) ? 0 : s->fragsize;
		DPRINTK("buf %d: start %p dma %p\n", frag, b->start, b->dma);

		dmabuf += s->fragsize;
		dmaphys += s->fragsize;
		dmasize -= s->fragsize;
	}

	s->buf_idx = s->dma_head_idx = s->dma_tail_idx = 0;
	s->buf = s->dma_head = s->dma_tail = &s->buffers[0];

	return 0;

err:
	printk( AUDIO_NAME ": unable to allocate audio memory\n ");
	audio_clear_buf(s);
	return -ENOMEM;
}


/*
 * DMA processing...
 */

#define DMA_MAX 8192

static inline int audio_set_dma( dma_regs_t *DMA, void *dma_ptr, int size )
{
	int status;
	int use_bufa;

	status = DMA->RdDCSR;
	DPRINTK(" dma st %#x at %p", status, dma_ptr);

	/* If both DMA channels are started, there's nothing else we can do. */
	if( (status & DCSR_STRTA) && (status & DCSR_STRTB) ){
		DPRINTK(" busy\n");
		return -1;
	}

	use_bufa = ( ((status & DCSR_BIU) && (status & DCSR_STRTB)) ||
		     (!(status & DCSR_BIU) && !(status & DCSR_STRTA)) );
	if( use_bufa ){
		DMA->ClrDCSR = DCSR_DONEA|DCSR_STRTA;
		DMA->DBSA = dma_ptr;
		DMA->DBTA = size-1;
		DMA->SetDCSR = DCSR_STRTA|DCSR_IE|DCSR_RUN;
		DPRINTK(" with a\n");
	}else{
		DMA->ClrDCSR = DCSR_DONEB|DCSR_STRTB;
		DMA->DBSB = dma_ptr;
		DMA->DBTB = size-1;
		DMA->SetDCSR = DCSR_STRTB|DCSR_IE|DCSR_RUN;
		DPRINTK(" with b\n");
	}

	return 0;
}


static void audio_process_dma( audio_stream_t *s )
{
    long flags;
    int chunksize;

    save_flags_cli(flags);

    for(;;){
	audio_buf_t *b = s->dma_head;

	if( !s->buffers || b->ref == 0 ) {
		/* no more data available */
		if( s->dma_active && b == s->dma_tail ){
			s->dma_active = 0;
			wake_up_interruptible(&audio_waitq);
		}
		DPRINTK("no %s dma buf (%d)\n", 
			(s == &output_stream) ? "out" : "in", s->dma_active);
		if( s == &input_stream ){
			/*
			 * Input overrun: We better stop the output spin.
			 */
			output_stream.dma_spin = 0;
		}
		if( s->dma_spin ){
			while( audio_set_dma(s->DMA, (void*)FLUSH_BASE_PHYS, DMA_MAX) == 0 )
				s->dma_spin_ref++;
		}
		break;
	}

	/* 
	 * Let's try to start DMA on the current buffer.  
	 * If no DMA is available then we stop here.
	 */
	chunksize = (b->size < DMA_MAX) ? b->size : DMA_MAX;
	DPRINTK("buf %d (%d) chnk %d", s->dma_head_idx, b->ref, chunksize);
	if( audio_set_dma( s->DMA, b->ptr, chunksize ) )  break;
	s->dma_active = 1;
	b->ref++;
	b->ptr += chunksize;
	b->size -= chunksize;
	if( b->size == 0 ){
		/* current buffer is done: go to the next one */
		NEXT_BUF( s, dma_head );
	}
    }

    restore_flags(flags);
}

static void audio_irq(int irq, void *dev_id, struct pt_regs *regs)
{
	audio_stream_t *s = (audio_stream_t *)dev_id;
	int status = s->DMA->RdDCSR;

	DPRINTK("irq: %s st %#x buf %d", (s == &output_stream) ? "out" : "in",
		status, s->dma_tail_idx);
	s->DMA->ClrDCSR = DCSR_ERROR|DCSR_DONEA|DCSR_DONEB;
	if( !(status & (DCSR_DONEA|DCSR_DONEB)) )  return;

	if( s->dma_spin_ref ){
		s->dma_spin_ref--;
		DPRINTK(" spin\n");
	}else{
		audio_buf_t *b = s->dma_tail;
		b->ref--;
		if( b->size == 0 && b->ref == 1 ) {
			/* 
			 * Current buffer is done: recycle it and wake up any 
			 * process waiting for it.
			 */
			DPRINTK(" recyc");
			b->ref = 0;
			NEXT_BUF( s, dma_tail );
			wake_up_interruptible(&audio_waitq);
		}
		DPRINTK("\n");
	}

	audio_process_dma(s);
}


static int audio_sync( struct file *file )
{
	audio_stream_t *s = &output_stream;

	DPRINTK("audio_sync\n");

	if (s->buffers) {
		audio_buf_t *b = s->buf;

		/* Send half-full buffers */
		if (b->ref == 0 && b->size != 0) {
			b->ptr = b->dma;
			b->ref = 1;
			audio_process_dma(s);
			NEXT_BUF( s, buf );
		}
	}

	while( s->dma_active ){
		interruptible_sleep_on( &audio_waitq );
		if( signal_pending(current) )
			return -EINTR;
	}
	return 0;
}


static int audio_write( struct file *file, const char *buffer, 
			size_t count, loff_t *ppos )
{
	const char *buffer0 = buffer;
	audio_stream_t *s = &output_stream;
	int chunksize;

	DPRINTK("audio_write: count=%d\n", count);

	if( !s->buffers && audio_setup_buf(s) ) return -ENOMEM;

	while( count > 0 ) {
		audio_buf_t *b = s->buf;

		if( b->ref != 0 ){
			/* 
			 * This buffer is being sent to dma already.
			 * It means that there is no free buffer anymore and 
			 * we must wait for one.
			 */
			if( file->f_flags & O_NONBLOCK ){
				/* non-blocking  so avoid sleeping */
				if (buffer > buffer0) break;
				return -EAGAIN;
			}
			/* Wait for a buffer to become free */
			interruptible_sleep_on( &audio_waitq );
			if( signal_pending(current) ){
				if (buffer > buffer0) break;
				return -EINTR;
			}
			continue;
		}

		/* feed the current buffer */
		chunksize = s->fragsize - b->size;
		if( chunksize > count ) chunksize = count;
		DPRINTK("write %d to %d\n", chunksize, s->buf_idx);
		if(copy_from_user( b->start + b->size, buffer, chunksize ))
			return -EFAULT;
		buffer += chunksize;
		count -= chunksize;
		b->size += chunksize;
		if( b->size < s->fragsize ) break;

		/* Send current buffer to dma */
		b->ptr = b->dma;
		b->ref = 1;
		audio_process_dma(s);
		NEXT_BUF( s, buf );
	}

	DPRINTK("audio_write: return=%d\n", buffer - buffer0);
	return buffer - buffer0;
}


static int audio_read( struct file *file, char *buffer, 
			size_t count, loff_t *ppos )
{
	char *buffer0 = buffer;
	audio_stream_t *s = &input_stream;
	int chunksize;

	DPRINTK("audio_read: count=%d\n", count);

	if( !s->buffers && audio_setup_buf(s) ) return -ENOMEM;

	/*
	 * We must ensure there is an output stream at any time while 
	 * recording since this is how the UDA1341 gets its clock.
	 * So if there is no playback data to send, the output DMA will
	 * spin with all zeros.
	 */
	if( !output_stream.dma_spin ){
		output_stream.dma_spin = 1;
		audio_process_dma(&output_stream);
	}

	audio_process_dma(s);

	while( count > 0 ) {
		audio_buf_t *b = s->buf;

		if( b->ref != 0 ){
			/* 
			 * This buffer hasn't been filled by dma yet.
			 * It means that there is no available data anymore.
			 */
			if( file->f_flags & O_NONBLOCK ){
				/* non-blocking  so avoid sleeping */
				if (buffer > buffer0) break;
				return -EAGAIN;
			}
			/* Wait for a buffer to become full */
			interruptible_sleep_on( &audio_waitq );
			if( signal_pending(current) ){
				if (buffer > buffer0) break;
				return -EINTR;
			}
			continue;
		}

		/* The DMA code decremented b->size down to 0.
		 * We can retrieve it with the position of ptr.
		 * We then use ptr in the data buffer.
		 */
		if( b->size == 0 ){
			b->size = b->ptr - b->dma;
			b->ptr = b->start;
		}

		/* Grab data from the current buffer */
		chunksize = b->size;
		if( chunksize > count ) chunksize = count;
		DPRINTK("read %d from %d\n", chunksize, s->buf_idx);
		if(copy_to_user( buffer, b->ptr, chunksize ))
			return -EFAULT;
		buffer += chunksize;
		count -= chunksize;
		b->size -= chunksize;
		b->ptr += chunksize;
		if( b->size > 0 ) break;

		/* Make current buffer available for dma */
		b->ptr = b->dma;
		b->size = s->fragsize;
		b->ref = 1;
		NEXT_BUF( s, buf );
	}

	DPRINTK("audio_read: return=%d\n", buffer - buffer0);
	return buffer - buffer0;
}


static loff_t audio_llseek( struct file *file, loff_t offset, int origin )
{
	return -ESPIPE;
}


static int audio_ioctl( struct inode *inode, struct file *file,
			uint cmd, ulong arg )
{
    long val;

    /* dispatch based on command */
    switch( cmd ) {
      case SNDCTL_DSP_SETFMT:
	get_user( val, (long*) arg );
	if( val & AUDIO_FMT_MASK ) {
	    audio_fmt = val;
	    break;
	}else return -EINVAL;

      case SNDCTL_DSP_CHANNELS:
      case SNDCTL_DSP_STEREO:
	get_user( val, (long*) arg );
	if( cmd == SNDCTL_DSP_STEREO ) val = val ? 2 : 1;
	if( val == 2 ) {
	    audio_channels = val;
	    break;
	}else return -EINVAL;

      case SNDCTL_DSP_SPEED:
	get_user( val, (long*) arg );
	/* FIXME: we need to modify the clock rate for DRAM bank 2 
	 * (our clock source) accordingly.
	 */
	val = 44100;
	put_user( val, (long*) arg );
	break;

      case SNDCTL_DSP_GETFMTS:
	put_user( AUDIO_FMT_MASK, (long*)arg );
	break;

      case SNDCTL_DSP_GETBLKSIZE:
	put_user( 4096, (long*)arg );
	break;

      case SNDCTL_DSP_SETFRAGMENT:
	if( output_stream.buffers ) return -EBUSY;
	get_user( val, (long*)arg );
	audio_fragsize = 1 << (val & 0xFFFF);
	if( audio_fragsize < 16 ) audio_fragsize = 16;
	if( audio_fragsize > 16384 ) audio_fragsize = 16384;
	audio_nbfrags = (val >> 16) & 0x7FFF;
	if( audio_nbfrags < 2 ) audio_nbfrags = 2;
	if( audio_nbfrags * audio_fragsize > 128*1024 ) 
	    audio_nbfrags = 128*1024 / audio_fragsize;
	if( audio_setup_buf( &output_stream ) ) return -ENOMEM;
	break;

      case SNDCTL_DSP_SYNC:
	return audio_sync( file );

      case SNDCTL_DSP_GETOSPACE:
	{
	   audio_stream_t *s = &output_stream;
	   audio_buf_info *inf = (audio_buf_info *)arg;
	   int err = verify_area( VERIFY_READ, inf, sizeof(*inf) );
	   int i;
	   int frags = 0, bytes = 0;

	   if( err ) return err;
	   for( i = 0; i < s->nbfrags; i++ ){
	      if( s->buffers[i].ref == 0 ){
		   if( s->buffers[i].size ) frags++;
		   bytes += s->fragsize - s->buffers[i].size;
	      }
	   }
	   put_user( frags, &inf->fragments );
	   put_user( s->nbfrags, &inf->fragstotal );
	   put_user( s->fragsize, &inf->fragsize );
	   put_user( bytes, &inf->bytes );
	}
	break;

      case SNDCTL_DSP_RESET:
         audio_uda1341_reset();
         return 0;

      case SNDCTL_DSP_POST:
      case SNDCTL_DSP_SUBDIVIDE:
      case SNDCTL_DSP_GETISPACE:
      case SNDCTL_DSP_NONBLOCK:
      case SNDCTL_DSP_GETCAPS:
      case SNDCTL_DSP_GETTRIGGER:
      case SNDCTL_DSP_SETTRIGGER:
      case SNDCTL_DSP_GETIPTR:
      case SNDCTL_DSP_GETOPTR:
      case SNDCTL_DSP_MAPINBUF:
      case SNDCTL_DSP_MAPOUTBUF:
      case SNDCTL_DSP_SETSYNCRO:
      case SNDCTL_DSP_SETDUPLEX:
      default:
	return -EINVAL;
    }

    return 0;
}


static int mixer_ioctl( struct inode *inode, struct file *file,
			uint cmd, ulong arg )
{
    long val;

    /* dispatch based on command */
    switch( cmd ) {
      case SOUND_MIXER_READ_DEVMASK:
	val = (	SOUND_MASK_VOLUME |
		SOUND_MASK_TREBLE |
		SOUND_MASK_BASS |
		SOUND_MASK_LINE1 |
		SOUND_MASK_LINE2 |
		SOUND_MASK_MIC |
		SOUND_MASK_RECLEV );
	put_user( val, (long*)arg );
	break;
      case SOUND_MIXER_READ_RECMASK:
	val = (	SOUND_MASK_MIC |
		SOUND_MASK_RECLEV );
	put_user( val, (long*)arg );
	break;
      case SOUND_MIXER_READ_STEREODEVS:
	put_user( 0, (long*)arg );
	break;
      case SOUND_MIXER_READ_CAPS:
	val = SOUND_CAP_EXCL_INPUT;
	put_user( val, (long*)arg );
	break;
      case SOUND_MIXER_WRITE_RECSRC:
	get_user( val, (long*) arg );	
	DATA0_ext4.AGC_ctrl = (val & SOUND_MASK_MIC) ? 1 : 0;
	L3_write( (UDA1341_L3Addr<<2)|UDA1341_DATA0, (char*)&DATA0_ext4, 2 );
	break;
      case SOUND_MIXER_READ_RECSRC:
	val = DATA0_ext4.AGC_ctrl ? SOUND_MASK_MIC : SOUND_MASK_RECLEV;
	put_user( val, (long*)arg );
	break;

      case SOUND_MIXER_WRITE_VOLUME:
	get_user( val, (long*) arg );	
	DATA0_0.volume = 63 - (((val&0xff) + 1) * 63) / 100;
	L3_write( (UDA1341_L3Addr<<2)|UDA1341_DATA0, (char*)&DATA0_0, 1 );
	break;
      case SOUND_MIXER_READ_VOLUME:
	val = ((63 - DATA0_0.volume) * 100) / 63;
	val |= val << 8;
	put_user( val, (long*)arg );
	break;
      case SOUND_MIXER_WRITE_TREBLE:
	get_user( val, (long*) arg );	
	DATA0_1.treble = (((val&0xff) + 1) * 3) / 100;
	L3_write( (UDA1341_L3Addr<<2)|UDA1341_DATA0, (char*)&DATA0_1, 1 );
	break;
      case SOUND_MIXER_READ_TREBLE:
	val = (DATA0_1.treble * 100) / 3;
	val |= val << 8;
	put_user( val, (long*)arg );
	break;
      case SOUND_MIXER_WRITE_BASS:
	get_user( val, (long*) arg );	
	DATA0_1.bass = (((val&0xff) + 1) * 15) / 100;
	L3_write( (UDA1341_L3Addr<<2)|UDA1341_DATA0, (char*)&DATA0_1, 1 );
	break;
      case SOUND_MIXER_READ_BASS:
	val = (DATA0_1.bass * 100) / 15;
	val |= val << 8;
	put_user( val, (long*)arg );
	break;
      case SOUND_MIXER_WRITE_LINE1:
	get_user( val, (long*) arg );	
	DATA0_ext0.ch1_gain = (((val&0xff) + 1) * 31) / 100;
	L3_write( (UDA1341_L3Addr<<2)|UDA1341_DATA0, (char*)&DATA0_ext0, 2 );
	break;
      case SOUND_MIXER_READ_LINE1:
	val = (DATA0_ext0.ch1_gain * 100) / 31;
	val |= val << 8;
	put_user( val, (long*)arg );
	break;
      case SOUND_MIXER_WRITE_LINE2:
	get_user( val, (long*) arg );	
	DATA0_ext1.ch2_gain = (((val&0xff) + 1) * 31) / 100;
	L3_write( (UDA1341_L3Addr<<2)|UDA1341_DATA0, (char*)&DATA0_ext1, 2 );
	break;
      case SOUND_MIXER_READ_LINE2:
	val = (DATA0_ext1.ch2_gain * 100) / 31;
	val |= val << 8;
	put_user( val, (long*)arg );
	break;
      case SOUND_MIXER_WRITE_MIC:
	get_user( val, (long*) arg );	
	/* even if 3 bits, value 7 is not used */
	DATA0_ext2.mic_level = (((val&0xff) + 1) * 6) / 100;
	L3_write( (UDA1341_L3Addr<<2)|UDA1341_DATA0, (char*)&DATA0_ext2, 2 );
	break;
      case SOUND_MIXER_READ_MIC:
	val = (DATA0_ext2.mic_level * 100) / 6;
	val |= val << 8;
	put_user( val, (long*)arg );
	break;
      case SOUND_MIXER_WRITE_RECLEV:
	get_user( val, (long*) arg );	
	val = ((val&0xff) * 127) / 100;
	DATA0_ext4.ch2_igain_l = val & 3;
	DATA0_ext5.ch2_igain_h = val >> 2;
	L3_write( (UDA1341_L3Addr<<2)|UDA1341_DATA0, (char*)&DATA0_ext4, 2 );
	L3_write( (UDA1341_L3Addr<<2)|UDA1341_DATA0, (char*)&DATA0_ext5, 2 );
	break;
      case SOUND_MIXER_READ_RECLEV:
	val = DATA0_ext4.ch2_igain_l + (DATA0_ext5.ch2_igain_h << 2);
	val = (val * 100) / 127;
	val |= val << 8;
	put_user( val, (long*)arg );
	break;

#if 0 /* Experimental.  What those should produce is till not obvious to me. */
      case SOUND_MIXER_WRITE_OGAIN:
	get_user( val, (long*) arg );	
	DATA0_ext6.AGC_level = (((val&0xff) + 1) * 3) / 100;
	L3_write( (UDA1341_L3Addr<<2)|UDA1341_DATA0, (char*)&DATA0_ext6, 2 );
	break;
      case SOUND_MIXER_READ_OGAIN:
	val = (DATA0_ext6.AGC_level * 100) / 3;
	val |= val << 8;
	put_user( val, (long*)arg );
	break;
      case SOUND_MIXER_WRITE_IMIX:
	get_user( val, (long*) arg );	
	DATA0_ext2.mixer_mode = val;
	L3_write( (UDA1341_L3Addr<<2)|UDA1341_DATA0, (char*)&DATA0_ext2, 2 );
	break;
      case SOUND_MIXER_READ_IMIX:
	val = DATA0_ext2.mixer_mode;
	put_user( val, (long*)arg );
	break;
#endif

      default:
	return -EINVAL;
    }
    return 0;
}


static int audio_open( struct inode *inode, struct file *file )
{
	DPRINTK("audio_open\n");

	if( audio_refcount == 0 ){
		audio_rate = AUDIO_RATE_DEFAULT;
		audio_fragsize = AUDIO_FRAGSIZE_DEFAULT;
		audio_nbfrags = AUDIO_NBFRAGS_DEFAULT;
		audio_clear_buf(&output_stream);
		audio_clear_buf(&input_stream);
	}
	audio_refcount++;

	MOD_INC_USE_COUNT;
	return 0;
}


static int mixer_open( struct inode *inode, struct file *file )
{
	MOD_INC_USE_COUNT;
	return 0;
}


static int audio_release( struct inode *inode, struct file *file )
{
	DPRINTK("audio_release\n");

	audio_refcount--;
	if( audio_refcount == 0 ){
		audio_sync( file );
		audio_clear_buf(&output_stream);
		audio_clear_buf(&input_stream);
		output_stream.dma_spin = output_stream.dma_spin_ref = 0;
	}

	MOD_DEC_USE_COUNT;
	return 0;
}


static int mixer_release( struct inode *inode, struct file *file )
{
	MOD_DEC_USE_COUNT;
	return 0;
}


static struct file_operations UDA1341_dsp_fops =
{
	llseek:		audio_llseek,
	write:		audio_write,
	read:		audio_read,
	ioctl:		audio_ioctl,
	open:		audio_open,
	release:	audio_release
};

static struct file_operations UDA1341_mixer_fops =
{
	ioctl:		mixer_ioctl,
	open:		mixer_open,
	release:	mixer_release
};


static inline void audio_ssp_init( void )
{
	/* Setup the uarts */
	PPAR |= PPC_TXD4;
	GAFR |= (GPIO_SSP_TXD | GPIO_SSP_RXD | GPIO_SSP_SCLK | 
		 GPIO_SSP_SFRM | GPIO_SSP_CLK);
	GPDR |= (GPIO_SSP_TXD | GPIO_SSP_SCLK | GPIO_SSP_SFRM );
	GPDR &= ~(GPIO_SSP_RXD | GPIO_SSP_CLK);
	Ser4SSCR0 = SSCR0_SSE + SSCR0_DataSize(16) + SSCR0_TI;
	Ser4SSCR1 = SSCR1_SClkIactL + SSCR1_SClk1P + SSCR1_ExtClk;

	/* Enable the audio power */
        if (machine_is_assabet()) {
#ifdef CONFIG_SA1100_ASSABET
          BCR_clear(BCR_STEREO_LB|BCR_QMUTE|BCR_SPK_OFF);
          BCR_set(BCR_CODEC_RST|BCR_AUDIO_ON);
#endif
        } else if (machine_is_bitsy()) {
#ifdef CONFIG_SA1100_BITSY
          set_bitsy_egpio(EGPIO_BITSY_CODEC_NRESET|EGPIO_BITSY_AUD_AMP_ON|EGPIO_BITSY_AUD_PWR_ON);
          clr_bitsy_egpio(EGPIO_BITSY_QMUTE);
	  GPDR |=  (GPIO_BITSY_CLK_SET0|GPIO_BITSY_CLK_SET1);
          GPCR =   GPIO_BITSY_CLK_SET0;
          GPSR =   GPIO_BITSY_CLK_SET1;
#endif
        }

        if (machine_is_assabet()) {
           /* 
            * The assabet board uses the SDRAM clock as the source clock for 
            * audio. This is supplied to the SA11x0 from the CPLD on pin 19. 
            * At 206Mhz we need to run the audio clock (SDRAM bank 2) 
            * at half speed. This clock will scale with core frequency so 
            * the audio sample rate will also scale. The CPLD on Assabet 
            * will need to be programmed to match the core frequency.
	 */
           MDREFR |= (MDREFR_K2DB2 | MDREFR_K2RUN);
           /* Make sure EAPD and KAPD are clear to run the clocks at all times. */
           MDREFR &= ~(MDREFR_EAPD | MDREFR_KAPD);
	}

	/* Wait for the UDA1341 to wake up */
	mdelay(100);
}


static int __init register_dma( audio_stream_t *s, char *desc )
{
	int i;

	for( i = 0; i < 6; i++ ){
		dma_regs_t *DMA = (dma_regs_t*)io_p2v(_DDAR(i));
		int irq = IRQ_DMA0 + i;
		int err = request_irq( irq, audio_irq, SA_INTERRUPT, 
				       desc, (void*)s );
		if( err ) continue;
		s->DMA = DMA;
		DMA->ClrDCSR = (DCSR_DONEA|DCSR_DONEB|DCSR_STRTA|DCSR_STRTB|
		 		DCSR_IE|DCSR_ERROR|DCSR_RUN);
		DMA->DDAR = (s == &output_stream) ? 
				(DDAR_Ser4SSPWr) : (DDAR_Ser4SSPRd);
		break;
	}

	return (i == 6) ? -EBUSY : 0;
}


static int unregister_dma( audio_stream_t *s )
{
	dma_regs_t *DMA0 = (dma_regs_t*)io_p2v(_DDAR(0));
	int irq = (s->DMA - DMA0) + IRQ_DMA0;
	if( s->DMA ){
		s->DMA->ClrDCSR = DCSR_STRTA|DCSR_STRTB|DCSR_IE|DCSR_RUN;
		free_irq( irq, (void*)s );
		s->DMA = NULL;
	}
	return 0;
}


static void audio_uda1341_reset()
{
	/* Reset the chip */
	STATUS_0.reset = 1;
	L3_write( (UDA1341_L3Addr<<2)|UDA1341_STATUS, (char*)&STATUS_0, 1 );

        if (machine_is_bitsy()) {
#ifdef CONFIG_BITSY
           clr_bitsy_egpio(EGPIO_BITSY_CODEC_NRESET);
           set_bitsy_egpio(EGPIO_BITSY_CODEC_NRESET);
#endif           
        }

	STATUS_0.reset = 0;
	L3_write( (UDA1341_L3Addr<<2)|UDA1341_STATUS, (char*)&STATUS_0, 1 );

}

/* TODO:  error validation */
int __init audio_uda1341_init( void )
{
	int err;
   
	if( !(machine_is_assabet() || machine_is_bitsy()) )
		return -ENODEV;

	L3_init();
	audio_ssp_init();

        audio_uda1341_reset();

	init_waitqueue_head(&audio_waitq);

	/* Initialize DMA */
	register_dma( &output_stream, "UDA1341 DMA out" );
	register_dma( &input_stream, "UDA1341 DMA in" );

	/* Set some default mixer values... */
	STATUS_1.DAC_gain = 1;
	STATUS_1.ADC_gain = 1;
	L3_write( (UDA1341_L3Addr<<2)|UDA1341_STATUS, (char*)&STATUS_1, 1 );
	DATA0_0.volume = 15;
	L3_write( (UDA1341_L3Addr<<2)|UDA1341_DATA0, (char*)&DATA0_0, 1 );
	DATA0_2.mode = 3;
	L3_write( (UDA1341_L3Addr<<2)|UDA1341_DATA0, (char*)&DATA0_2, 1 );
	DATA0_ext2.mixer_mode = 2;
	DATA0_ext2.mic_level = 4;
	L3_write( (UDA1341_L3Addr<<2)|UDA1341_DATA0, (char*)&DATA0_ext2, 2 );
	DATA0_ext4.AGC_ctrl = 1;
	L3_write( (UDA1341_L3Addr<<2)|UDA1341_DATA0, (char*)&DATA0_ext4, 2 );
	DATA0_ext6.AGC_level = 3;
	L3_write( (UDA1341_L3Addr<<2)|UDA1341_DATA0, (char*)&DATA0_ext6, 2 );

	/* register devices */
	audio_dev_dsp = register_sound_dsp(&UDA1341_dsp_fops, -1);
	audio_dev_mixer = register_sound_mixer(&UDA1341_mixer_fops, -1);

	printk( AUDIO_NAME_VERBOSE " initialized\n" );

	return 0;
}

module_init(audio_uda1341_init);


void audio_uda1341_exit(void)
{
	/* unregister driver and IRQ */
	unregister_sound_dsp( audio_dev_dsp );
	unregister_sound_mixer( audio_dev_mixer );
	unregister_dma( &output_stream );
	unregister_dma( &input_stream );
	printk(AUDIO_NAME_VERBOSE " unloaded\n");
}

module_exit(audio_uda1341_exit);

