/*
 * SA1100 Power Management Routines
 *
 * Copyright (c) 2001 Cliff Brake <cbrake@accelent.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License.
 *
 * History:
 *
 * 2001-02-06:	Cliff Brake         Initial code
 *
 * 2001-02-25:	Sukjae Cho <sjcho@east.isi.edu> &
 * 		Chester Kuo <chester@linux.org.tw>
 * 			Save more value for the resume function! Support
 * 			Bitsy/Assabet/Freebird board
 *
 * 2001-08-29:	Nicolas Pitre <nico@cam.org>
 * 			Cleaned up, pushed platform dependent stuff
 * 			in the platform specific files.
 */

/*
 * Debug macros
 */
#define DEBUG 1
#ifdef DEBUG
#  define DPRINTK(fmt, args...)	printk("%s: " fmt, __FUNCTION__ , ## args)
#else
#  define DPRINTK(fmt, args...)
#endif

#include <linux/config.h>
#include <linux/init.h>
#include <linux/pm.h>
#include <linux/slab.h>
#include <linux/sysctl.h>

#include <asm/hardware.h>
#include <asm/memory.h>
#include <asm/system.h>

#include "sleep.h"

/*
 * ARGH!  Stupid ACPI people.  They should define this in linux/sysctl.h,
 * NOT linux/acpi.h.
 *
 * This means our interface here won't survive long - it needs a new
 * interface.  Quick hack to get this working - use sysctl id 9999.
 */
#warning ACPI broke the kernel, this interface needs to be fixed up.
#define CTL_ACPI 9999
#define ACPI_S1_SLP_TYP 19

extern void sa1100_cpu_suspend(void);
extern void sa1100_cpu_resume(void);

extern unsigned long *sleep_save;	/* virtual address */
extern unsigned long  sleep_save_p;	/* physical address */

#define SAVE(x)		sleep_save[SLEEP_SAVE_##x] = x
#define RESTORE(x)	x = sleep_save[SLEEP_SAVE_##x]

int sa1110_suspend(void)
{
	int retval;

	/* set up pointer to sleep parameters */
	sleep_save = kmalloc (SLEEP_SAVE_SIZE*sizeof(long), GFP_ATOMIC);
	if (!sleep_save)
		return -ENOMEM;
	sleep_save_p = virt_to_phys(sleep_save);

	retval = pm_send_all(PM_SUSPEND, (void *)2);
	if (retval) {
		kfree(sleep_save);
		return retval;
	}

#ifdef CONFIG_SA1100_H3XXX
	retval = h3600_power_management(PM_SUSPEND);
	if (retval) {
		kfree(sleep_save);
		return retval;
	}
#endif
	cli();

	/* preserve current time */
	RCNR = xtime.tv_sec;

	/* save vital registers */
	SAVE(OSCR);
	SAVE(OSMR0);
	SAVE(OSMR1);
	SAVE(OSMR2);
	SAVE(OSMR3);
	SAVE(OIER);

	SAVE(GPDR);
	SAVE(GRER);
	SAVE(GFER);
	SAVE(GAFR);

	SAVE(PPDR);
	SAVE(PPSR);
	SAVE(PPAR);
	SAVE(PSDR);

	SAVE(Ser1SDCR0);

	SAVE(ICMR);

	/* ... maybe a global variable initialized by arch code to set this? */
	GRER = PWER;
	GFER = 0;
	GEDR = GEDR;

	/* Clear previous reset status */
	RCSR = RCSR_HWR | RCSR_SWR | RCSR_WDR | RCSR_SMR;

	/* set resume return address */
	PSPR = virt_to_phys(sa1100_cpu_resume);

	/* go zzz */
	sa1100_cpu_suspend();

	/* ensure not to come back here if it wasn't intended */
	PSPR = 0;

	DPRINTK("*** made it back from resume\n");

	/* restore registers */
	RESTORE(GPDR);
	RESTORE(GRER);
	RESTORE(GFER);
	RESTORE(GAFR);

	/* clear any edge detect bit */
	GEDR = GEDR;

	RESTORE(PPDR);
	RESTORE(PPSR);
	RESTORE(PPAR);
	RESTORE(PSDR);

	RESTORE(Ser1SDCR0);

	PSSR = PSSR_PH;

	RESTORE(OSMR0);
	RESTORE(OSMR1);
	RESTORE(OSMR2);
	RESTORE(OSMR3);
	RESTORE(OSCR);
	RESTORE(OIER);

	ICLR = 0;
	ICCR = 1;
	RESTORE(ICMR);

	/* restore current time */
	xtime.tv_sec = RCNR;

	sti();

	kfree (sleep_save);

#ifdef CONFIG_CPU_FREQ
	cpufreq_restore();
#endif
	/* Run our special "fixup" here */
#ifdef CONFIG_SA1100_H3XXX
	h3600_power_management(PM_RESUME);  /* Ignoring return value for the moment */
#endif
	return pm_send_all(PM_RESUME, (void *)0);
}



static char pm_helper_path[128] = "/sbin/pm_helper";

static void
run_sbin_pm_helper( pm_request_t action )
{
	int i;
	char *argv[3], *envp[8];

	if (!pm_helper_path[0])
		return;

	if ( action != PM_SUSPEND && action != PM_RESUME )
		return;

	i = 0;
	argv[i++] = pm_helper_path;
	argv[i++] = (action == PM_RESUME ? "resume" : "suspend");
	argv[i] = 0;

        printk(KERN_CRIT __FUNCTION__ ":%d pm_helper_path=%s\n", __LINE__, pm_helper_path);
	i = 0;
	/* minimal command environment */
	envp[i++] = "HOME=/";
	envp[i++] = "PATH=/sbin:/bin:/usr/sbin:/usr/bin";
	envp[i] = 0;

	/* other stuff we want to pass to /sbin/hotplug */
	call_usermodehelper (argv [0], argv, envp);
}

int pm_do_suspend(void)
{
	DPRINTK("suggest\n");
	run_sbin_pm_helper(PM_SUSPEND);
	return 0;
}

int pm_force_suspend(void)
{
	int retval;
	
	DPRINTK("yea\n");
	retval = sa1110_suspend();
	run_sbin_pm_helper(PM_RESUME);
	return retval;
}

static struct ctl_table pm_table[] =
{
	{ACPI_S1_SLP_TYP, "suspend", NULL, 0, 0644, NULL, (proc_handler *)&pm_force_suspend},
	{2, "helper", pm_helper_path, sizeof(pm_helper_path), 0644, NULL, (proc_handler *)&proc_dostring},
	{0}
};

static struct ctl_table pm_dir_table[] =
{
	{CTL_ACPI, "pm", NULL, 0, 0555, pm_table},
	{0}
};

/*
 * Initialize power interface
 */
static int __init pm_init(void)
{
	register_sysctl_table(pm_dir_table, 1);
	return 0;
}

__initcall(pm_init);

