/*
 * drivers/pcmcia/sa1100_backpaq.c
 *
 * PCMCIA implementation routines for Compaq CRL Mercury BackPAQ
 *
 */
#include <linux/config.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/sysctl.h>
#include <linux/pm.h>

#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/mtd/partitions.h>
#include <linux/proc_fs.h> 
#include <linux/ctype.h>

#include <asm/uaccess.h>
#include <asm/hardware.h>
#include <asm/irq.h>
#include <asm/arch/pcmcia.h>
#include <asm/arch/h3600-sleeve.h>
#include <linux/h3600_ts.h>
#include <asm/arch/backpaq.h>

#define BACKPAQ_EEPROM_PROC_NAME "backpaq/eeprom"
#define BACKPAQ_PROC_DIR         "backpaq"

static struct h3600_backpaq_eeprom dummy_eeprom_values = {
	major_revision : 0,
	minor_revision : 0,
	fpga_version   : BACKPAQ_EEPROM_FPGA_VERTEX_100,
	camera         : 0,
	accel_offset_x : 0x8000,
	accel_scale_x  : 0x8000,
	accel_offset_y : 0x8000,
	accel_scale_y  : 0x8000,
	serial_number  : 0,
	flash_start    : 0x00000000,
	flash_length   : 0x02000000,
	sysctl_start   : 0x02000000
};

static struct proc_dir_entry *proc_backpaq_eeprom = NULL;
struct h3600_backpaq_eeprom   h3600_backpaq_eeprom_shadow;
static int                    h3600_backpaq_eeprom_offset = 0;       // 0 indicates invalid offset

EXPORT_SYMBOL(h3600_backpaq_eeprom_shadow);


extern void sa1100_bitsy_change_sleeves(enum sleeve_device new_sleeve_type) ;
int pcmcia_sleeve_attach_flash(void (*set_vpp)(struct map_info *, int), unsigned long map_size);
void pcmcia_sleeve_detach_flash(void);

static void backpaq_set_vpp(struct map_info *map, int vpp)
{
	if (vpp)
          BackpaqSysctlFlashControl |= BACKPAQ_FLASH_VPPEN;
	else
          BackpaqSysctlFlashControl &= ~BACKPAQ_FLASH_VPPEN;
}


/************************************************************************************/
/* Register devices on the backpaq which need to know about power/insertion events  */

static LIST_HEAD(h3600_backpaq_devices);
static DECLARE_MUTEX(h3600_backpaq_devs_lock);

struct h3600_backpaq_device* h3600_backpaq_register_device( h3600_backpaq_dev_t type,
							    unsigned long id,
							    h3600_backpaq_callback callback )
							  
{
	struct h3600_backpaq_device *dev = kmalloc(sizeof(struct h3600_backpaq_device), GFP_KERNEL);

	printk(__FUNCTION__ ": backpaq register device %d %ld %p\n", type, id, callback);

	if (dev) {
		memset(dev, 0, sizeof(*dev));
		dev->type = type;
		dev->id = id;
		dev->callback = callback;

		down(&h3600_backpaq_devs_lock);
		if ( type == H3600_BACKPAQ_FPGA_DEV ) {
			list_add(&dev->entry, &h3600_backpaq_devices);
		} else {
			list_add_tail(&dev->entry, &h3600_backpaq_devices);
		}
		up(&h3600_backpaq_devs_lock);
	}
	return dev;
}

void h3600_backpaq_unregister_device(struct h3600_backpaq_device *device)
{
	printk(__FUNCTION__ ": backpaq unregister device %d %ld %p\n", 
	       device->type, device->id, device->callback);

	if (device) {
		down(&h3600_backpaq_devs_lock);
		list_del(&device->entry);
		up(&h3600_backpaq_devs_lock);

		kfree(device);
	}
}

int h3600_backpaq_present( void )
{
	return h3600_current_sleeve() == MERCURY_BACKPAQ;
}


EXPORT_SYMBOL(h3600_backpaq_present);
EXPORT_SYMBOL(h3600_backpaq_register_device);
EXPORT_SYMBOL(h3600_backpaq_unregister_device);

int h3600_backpaq_send(h3600_backpaq_request_t request )
{
        struct list_head *entry;
	struct h3600_backpaq_device *device;
	int status; /* We're not doing anything with this */

	printk(__FUNCTION__ ": sending to everyone\n");

	down(&h3600_backpaq_devs_lock);
	entry = h3600_backpaq_devices.next;
	while ( entry != &h3600_backpaq_devices ) {
		device = list_entry(entry, struct h3600_backpaq_device, entry );
		if ( device->callback ) {
			printk(__FUNCTION__ ": sending to %p\n", device->callback);
			status = (*device->callback)(device, request);
		}
		entry = entry->next;
	}
	up(&h3600_backpaq_devs_lock);
	return 0;
}

/***********************************************************************************/
/*    Power management callbacks  */

static int suspended = 0;

static void backpaq_suspend_sleeve( struct sleeve_dev *sleeve_dev )
{
	h3600_backpaq_send( H3600_BACKPAQ_SUSPEND );
	suspended = 1;
}

static void backpaq_resume_sleeve( struct sleeve_dev *sleeve_dev )
{
	if ( suspended ) {
		h3600_backpaq_send( H3600_BACKPAQ_RESUME );
		suspended = 0;
	}

}


/***********************************************************************************/
/*                         Proc file system                                        */
/***********************************************************************************/


static char * parse_eeprom( char *p )
{
	struct h3600_backpaq_eeprom *t = &h3600_backpaq_eeprom_shadow;

	p += sprintf(p, "EEPROM status   : %s\n", 
		     (h3600_backpaq_eeprom_offset ? "Okay" : "Not programmed" ));
	p += sprintf(p, "Major revision  : 0x%02x\n", t->major_revision );
	p += sprintf(p, "Minor revision  : 0x%02x\n", t->minor_revision );
	p += sprintf(p, "FPGA version    : 0x%02x\n", t->fpga_version );
	p += sprintf(p, "Camera          : 0x%02x\n", t->camera );
	p += sprintf(p, "Accel offset x  : 0x%04x\n", t->accel_offset_x );
	p += sprintf(p, "Accel scale x   : 0x%04x\n", t->accel_scale_x );
	p += sprintf(p, "Accel offset y  : 0x%04x\n", t->accel_offset_y );
	p += sprintf(p, "Accel scale y   : 0x%04x\n", t->accel_scale_y );
	p += sprintf(p, "Serial number   : %d\n",     t->serial_number );
	p += sprintf(p, "Flash start     : 0x%08x\n", t->flash_start );
	p += sprintf(p, "Flash length    : 0x%08x\n", t->flash_length );
	p += sprintf(p, "Sysctl start    : 0x%08x\n", t->sysctl_start );

	return p;
}

static int proc_backpaq_eeprom_read(char *page, char **start, off_t off,
			  int count, int *eof, void *data)
{
	char *p = page;
	int len;

	if (!h3600_backpaq_present()) {
		p += sprintf(p,"No backpaq present\n");
	}
	else {
		p = parse_eeprom( p );
	}

	len = (p - page) - off;
	if (len < 0)
		len = 0;

	*eof = (len <= count) ? 1 : 0;
	*start = page + off;

	return len;
}

static int backpaq_find_oem_offset( void )
{
	short vendor, device;
	char d;
	short offset;

	if (!h3600_backpaq_present())
		return 0;

	h3600_spi_read( 6, (char *)&vendor, 2 );
	h3600_spi_read( 8, (char *)&device, 2 );

	if ( vendor != 0x1125 && device != 0x100 )
		return 0;

	offset = 10;
	do {
		h3600_spi_read( offset++, &d, 1 );
	} while ( d != 0 );
	offset += 11;
	h3600_spi_read( offset, (char *)&offset, 2 );
	offset += 4;
	// Now offset points to the oem table
	return offset;
}


static void load_eeprom( void )
{
	int offset, len;
	unsigned char *p;

	h3600_backpaq_eeprom_offset = backpaq_find_oem_offset();
	if ( h3600_backpaq_eeprom_offset ) {
		offset = h3600_backpaq_eeprom_offset;
		len = sizeof(struct h3600_backpaq_eeprom);
		p = (unsigned char *) &h3600_backpaq_eeprom_shadow;

		while ( len > 0 ) {
			int chunk = ( len > 4 ? 4 : len );
			h3600_spi_read( offset, p, chunk );
			p += chunk;
			offset += chunk;
			len -= chunk;
		}
        } else {   /* No EEPROM present : load dummy values */
		memcpy(&h3600_backpaq_eeprom_shadow, 
		       &dummy_eeprom_values,
		       sizeof(h3600_backpaq_eeprom_shadow));
	}
}


int proc_eeprom_update(ctl_table *table, int write, struct file *filp,
		       void *buffer, size_t *lenp)
{
	unsigned long val;
	size_t left, len;
#define TMPBUFLEN 20
	char buf[TMPBUFLEN];
	
	if (!table->data || !table->maxlen || !*lenp ||
	    (filp->f_pos && !write)) {
		*lenp = 0;
		return 0;
	}
	
	left = *lenp;   // How many character we have left
	
	printk(__FUNCTION__ ": called\n");
	if (write) {
		/* Skip over leading whitespace */
		while (left) {
			char c;
			if(get_user(c,(char *) buffer))
				return -EFAULT;
			if (!isspace(c))
				break;
			left--;
			((char *) buffer)++;
		}
		if ( left ) {
			len = left;
			if (len > TMPBUFLEN-1)
				len = TMPBUFLEN-1;
			if(copy_from_user(buf, buffer, len))
				return -EFAULT;
			buf[len] = 0;
			val = simple_strtoul(buf, NULL, 0);
			printk(__FUNCTION__ ": writing value 0x%lx\n", val );

			switch (table->maxlen) {
			case 1:	*(( u8 *) (table->data)) = val;  break;
			case 2: *((u16 *) (table->data)) = val; break;
			case 4: *((u32 *) (table->data)) = val; break;
			default:
				printk(__FUNCTION__ ": illegal table entry size\n");
				break;
			}
			
			if ( h3600_backpaq_eeprom_offset ) {
				int offset = (int) (table->data - (void *) &h3600_backpaq_eeprom_shadow);
				printk(__FUNCTION__ ": writing offset %d\n", offset );
				h3600_spi_write( h3600_backpaq_eeprom_offset + offset, 
						 table->data, table->maxlen);
			}
			else {
				printk(__FUNCTION__ ": unable to write to Backpaq - EEPROM not initialized\n");
			}
		}
		filp->f_pos += *lenp;
	} else { /* Reading */
		if ( !h3600_backpaq_eeprom_offset ) {
			printk(__FUNCTION__ ": unable to read Backpaq - EEPROM not initialized\n");
			return 0;
		}
		switch (table->maxlen) {
		case 1: val = *((u8 *)(table->data)); break;
		case 2: val = *((u16 *)(table->data)); break;
		case 4: val = *((u32 *)(table->data)); break;
		default:
			val = 0;
			printk(__FUNCTION__ ": illegal table entry size\n");
			break;
		}

		sprintf(buf, "0x%lx\n", val);
		len = strlen(buf);
		if (len > left)
			len = left;
		if(copy_to_user(buffer, buf, len))
			return -EFAULT;

		*lenp = len;
		filp->f_pos += len;
	}

	return 0;
}


/***********************************************************************************/


#define EESHADOW(x) &(h3600_backpaq_eeprom_shadow.x), sizeof(h3600_backpaq_eeprom_shadow.x)

static struct ctl_table backpaq_accel_table[] = 
{
	{10, "xoffset", EESHADOW(accel_offset_x), 0644, NULL, &proc_eeprom_update },
	{11, "xscale",  EESHADOW(accel_scale_x),  0644, NULL, &proc_eeprom_update },
	{12, "yoffset", EESHADOW(accel_offset_y), 0644, NULL, &proc_eeprom_update },
	{13, "yscale",  EESHADOW(accel_scale_y),  0644, NULL, &proc_eeprom_update },
	{0}
};

static struct ctl_table backpaq_eeprom_table[] = 
{
	{1, "major",  EESHADOW(major_revision), 0644, NULL, &proc_eeprom_update },
	{2, "minor",  EESHADOW(minor_revision), 0644, NULL, &proc_eeprom_update },
	{3, "serial", EESHADOW(serial_number),  0644, NULL, &proc_eeprom_update },
	{4, "fpga",   EESHADOW(fpga_version),   0644, NULL, &proc_eeprom_update },
	{5, "camera", EESHADOW(camera),         0644, NULL, &proc_eeprom_update },
	{6, "accel",  NULL, 0, 0555, backpaq_accel_table},
	{7, "flashstart",  EESHADOW(flash_start), 0644, NULL, &proc_eeprom_update },
	{8, "flashlen",    EESHADOW(flash_length), 0644, NULL, &proc_eeprom_update },
	{9, "sysctlstart", EESHADOW(sysctl_start), 0644, NULL, &proc_eeprom_update },
	{0}
};

static struct ctl_table backpaq_table[] = 
{
	{5, "eeprom", NULL, 0, 0555, backpaq_eeprom_table},
	{0}
};
static struct ctl_table backpaq_dir_table[] = 
{
	{22, "backpaq", NULL, 0, 0555, backpaq_table},
        {0}
};
static struct ctl_table_header *backpaq_ctl_table_header = NULL;


static int __devinit backpaq_probe_sleeve(struct sleeve_dev *sleeve_dev, const struct sleeve_device_id *ent)
{
        sprintf(sleeve_dev->name, "Compaq Mercury Backpaq");
        printk(__FUNCTION__ ": dev->name=%s\n", sleeve_dev->name);

        sa1100_bitsy_change_sleeves(MERCURY_BACKPAQ);

        BackpaqSysctlFlashControl = (BACKPAQ_FLASH_ENABLE0 | BACKPAQ_FLASH_ENABLE1);

	/* This was breaking stuff */
//        pcmcia_sleeve_attach_flash(backpaq_set_vpp, 0x02000000);

        MSC2 = 0x431a1290;  /* 32-bit, non-burst, wait states: recover =0,
                               first access = 18, next access = 18 */
        /* Don't change the MSC2 values without checking if the camera works correctly */
 
        printk(__FUNCTION__ ": setting MSC2=0x%x (&MSC2=%p)\n",MSC2, &MSC2);
	h3600_backpaq_send( H3600_BACKPAQ_INSERT );

	// Set up the EEPROM program area
	load_eeprom();
#ifdef CONFIG_PROC_FS
        backpaq_ctl_table_header = register_sysctl_table(backpaq_dir_table, 0);
	proc_backpaq_eeprom = create_proc_entry(BACKPAQ_EEPROM_PROC_NAME, 0, NULL);
	if ( !proc_backpaq_eeprom ) {
		/* We probably need to create the "backpaq" directory first */
		proc_mkdir(BACKPAQ_PROC_DIR,0);
		proc_backpaq_eeprom = create_proc_entry(BACKPAQ_EEPROM_PROC_NAME, 0, NULL);
	}
	
	if ( proc_backpaq_eeprom )
		proc_backpaq_eeprom->read_proc = proc_backpaq_eeprom_read;    
	else {
		printk(KERN_ALERT __FILE__ ": unable to create proc entry %s\n", 
		       BACKPAQ_EEPROM_PROC_NAME);
	}
#endif	
        return 0;
}

static void __devexit backpaq_remove_sleeve(struct sleeve_dev *sleeve_dev)
{
        printk(__FUNCTION__ ": dev->name=%s\n", sleeve_dev->name);
#ifdef CONFIG_PROC_FS
        unregister_sysctl_table(backpaq_ctl_table_header);
	if ( proc_backpaq_eeprom ) {
		remove_proc_entry(BACKPAQ_EEPROM_PROC_NAME, 0);
		proc_backpaq_eeprom = NULL;
	}
#endif

	h3600_backpaq_eeprom_offset = 0;       /* No backpaq */
	h3600_backpaq_send( H3600_BACKPAQ_EJECT );
        sa1100_bitsy_change_sleeves(SLEEVE_NO_DEVICE);
	printk(__FUNCTION__ ": changed to no sleeve\n");

	/* This was breaking stuff */
//        pcmcia_sleeve_detach_flash();
}

static struct sleeve_device_id backpaq_tbl[] __devinitdata = {
        { COMPAQ_VENDOR_ID, MERCURY_BACKPAQ, 0, 0, 0 },
        { CRL_VENDOR_ID, MERCURY_BACKPAQ_UNPROGRAMMED, 0, 0, 0 },
        { CRL_VENDOR_ID_2, MERCURY_BACKPAQ_UNPROGRAMMED, 0, 0, 0 },
        { 0, }
};

static struct sleeve_driver backpaq_driver = {
        name:   "Compaq Mercury Backpaq",
        id_table: backpaq_tbl,
        probe: backpaq_probe_sleeve,
        remove: backpaq_remove_sleeve,
	suspend: backpaq_suspend_sleeve,
	resume: backpaq_resume_sleeve
};


int __init backpaq_init_module(void)
{
	printk(__FUNCTION__ ":\n");
        h3600_sleeve_register_driver(&backpaq_driver);
	return 0;
}

void backpaq_cleanup_module(void)
{
	printk(__FUNCTION__ ":\n");
        h3600_sleeve_unregister_driver(&backpaq_driver);
}

module_init(backpaq_init_module);
module_exit(backpaq_cleanup_module);

