/*
 *
 * Definitions for Mercury BackPAQ
 *
 * Copyright 2000 Compaq Computer Corporation.
 *
 * Copying or modifying this code for any purpose is permitted,
 * provided that this copyright notice is preserved in its entirety
 * in all copies or modifications.  COMPAQ COMPUTER CORPORATION
 * MAKES NO WARRANTIES, EXPRESSED OR IMPLIED, AS TO THE USEFULNESS
 * OR CORRECTNESS OF THIS CODE OR ITS FITNESS FOR ANY PARTICULAR
 * PURPOSE.
 *
 * Author: Jamey Hicks.
 *
 * Change: 6/8/01   Power management callback interface added.
 *                  Andrew Christian
 *
 */

#ifndef _INCLUDE_BACKPAQ_H_
#define _INCLUDE_BACKPAQ_H_

#include <linux/types.h>

/********** Device management support **************/

enum {
	H3600_BACKPAQ_SUSPEND,
	H3600_BACKPAQ_RESUME,
	H3600_BACKPAQ_EJECT,
	H3600_BACKPAQ_INSERT
};

typedef int h3600_backpaq_request_t;  /* Mostly mapped directly from pm.h */

enum {
	H3600_BACKPAQ_UNKNOWN_DEV = 0,
	H3600_BACKPAQ_FPGA_DEV,
	H3600_BACKPAQ_CAMERA_DEV,
	H3600_BACKPAQ_AUDIO_DEV,
	H3600_BACKPAQ_ACCEL_DEV,
	H3600_BACKPAQ_GASGAUGE_DEV,
};
typedef int h3600_backpaq_dev_t;

struct h3600_backpaq_device;
typedef int (*h3600_backpaq_callback)(struct h3600_backpaq_device *dev, h3600_backpaq_request_t rqst);

struct h3600_backpaq_device {
	h3600_backpaq_dev_t       type;
	unsigned long             id;
	h3600_backpaq_callback    callback;
	void                     *data;

	struct list_head          entry;
};

int h3600_backpaq_present( void );
struct h3600_backpaq_device* h3600_backpaq_register_device( h3600_backpaq_dev_t type,
							    unsigned long id,
							    h3600_backpaq_callback callback );
void h3600_backpaq_unregister_device(struct h3600_backpaq_device *device);

/********** OEM area of the EEPROM **********/

struct h3600_backpaq_eeprom {
	u8  major_revision;
	u8  minor_revision;
	u8  fpga_version; 
	u8  camera;
	u16 accel_offset_x;
	u16 accel_scale_x;
	u16 accel_offset_y;
	u16 accel_scale_y;
	u32 serial_number;
	u32 flash_start;
	u32 flash_length;
	u32 sysctl_start;
};

#define BACKPAQ_EEPROM_CAMERA_ORIENTATION  0x80     /* High order bit set for landscape cameras */
#define BACKPAQ_EEPROM_CAMERA_TYPE_MASK    0x7f     /* All other bits for camera type */

/********** Memory-mapped CPLD interface **********/

struct mercury_backpaq_sysctl {
	volatile u16 hardware_version;   /* 0x00 */
	  u16 pad1;
	volatile u16 firmware_version;   /* 0x04 */
	  u16 pad2;
	volatile u16 fpga_ctl;           /* 0x08 */
	  u16 pad3;
	volatile u16 fpga_status;        /* 0x0c */
	  u16 pad4;
	volatile u16 fpga_program;       /* 0x10 */
	  u16 pad5;
	volatile u16 pcmcia_power;       /* 0x14 */
	  u16 pad6;
	volatile u16 flash_control;      /* 0x18 */ 
	  u16 pad7;
	volatile u16 gencontrol;         /* 0x1c */ 
	  u16 pad8;
	  u32 pad[4];  /* 0x20, 0x24, 0x28, 0x2c */
	long socket[2]; /* readable image at 0x30 and 0x34 */ 
};

struct mercury_backpaq_socket {
	u32 pad[32];     /* 0x00 */
	volatile u16 fpga_firmware_version;    /*  0x80 r/w */  	u16 pad1;
	volatile u8  camera_writethru;         /*  0x84   w */	u8 pad1a; u16 pad2;
	volatile u32 camera_fifo_data;         /*  0x88 r/w */
	volatile u16 camera_live_mode;         /*  0x8c r/w */   	u16 pad4;
	volatile u16 camera_integration_time;  /*  0x90 r/w */  	u16 pad5;
	volatile u16 camera_clock_divisor;     /*  0x94 r/w */  	u16 pad6;
	volatile u16 camera_fifo_info;         /*  0x98 r   */   	u16 pad7;
	volatile u16 camera_fifo_data_avail;   /*  0x9c r   */  	u16 pad8;
	volatile u16 camera_fifo_status;       /*  0xa0 r   */  	u16 pad9;
	volatile u16 camera_rowcount;          /*  0xa4 r   */  	u16 pad10;
	volatile  u8 fpga_reset;               /*  0xa8   w */  u8 pad11a; u16 pad11b;
	volatile u16 camera_integration_cmd;    /*  0xac r/w */  	u16 pad12;
	volatile u16 camera_interrupt_fifo;     /*  0xb0 r/w */ 	u16 pad13;
	volatile u16 fpga_interrupt_mask;       /*  0xb4 r/w */ 	u16 pad14;
	volatile u16 fpga_interrupt_status;     /*  0xb8 r   */ 	u16 pad15;
	volatile u32 fpga_test32addr;           /*  0xbc   w */ 
	volatile u16 fpga_accelx_t1;            /*  0xc0 r   */         u16 pad17;
	volatile u16 fpga_accelx_t2;            /*  0xc4 r   */         u16 pad18;
	volatile u16 fpga_accely_t1;            /*  0xc8 r   */         u16 pad19;
	volatile u16 fpga_accely_t2;            /*  0xcc r   */         u16 pad20;
        volatile u16 gasgauge_cmd;              /*  0xd0 r/w */         u16 pad22;
        volatile u16 gasgauge_result;           /*  0xd4 r   */         u16 pad23;
        volatile u16 gasgauge_wstate;           /*  0xd8 r   */         u16 pad24;
        volatile u16 gasgauge_rstate;           /*  0xdc r   */         u16 pad25;
	u32 pad26; /* e0 */
	volatile u16 camera_decimation_control; /*  0xe4 r/w */         u16 pad27;
	/* u32 pada[198]; */   /* 0xec */
	/* volatile u32 camera_fifo_read; */         /* 0x404 r   */
};

#define BACKPAQ_SYSCTL_BASE 0xf3000000 /* static memory bank 2 + 32MB */ 
#define BACKPAQ_SOCKET_STATUS_BASE 0xf3800000  /* static memory bank 4 + 0x30 */
#define BACKPAQ_FPGA_BASE 0xf3800000 /* static memory bacnk 4 */

/* Hardware versions */
/* ...This needs to be changed to have some useful bit fields */
/* #define BACKPAQ_HARDWARE_VERSION_1 0x00bc *//* Only run if we see this hardware version */

/* FPGA control (sysctl->fpga_ctl) 0x08 */
#define BACKPAQ_FPGACTL_M0      (1 << 0)
#define BACKPAQ_FPGACTL_M1      (1 << 1)
#define BACKPAQ_FPGACTL_M2      (1 << 2)
#define BACKPAQ_FPGACTL_PROGRAM (1 << 3)
/* #define BACKPAQ_FPGACTL_RESET   (1 << 4) */

/* FPGA Status [read-only] (systctl->fpgadout) 0x0c */
#define BACKPAQ_FPGASTATUS_DONE   (1 << 0)
#define BACKPAQ_FPGASTATUS_INITL  (1 << 1)
#define BACKPAQ_FPGASTATUS_EMPTY  (1 << 2)

#define BACKPAQ_OPTION_OPT_ON (1 << 0)
#define BACKPAQ_OPTION_DC_IN  (1 << 1)
#define BACKPAQ_OPTION_MCHG   (1 << 2)
#define BACKPAQ_OPTION_EBAT   (1 << 3)
#define BACKPAQ_OPTION_ETMI   (1 << 4)
#define BACKPAQ_OPTION_ETMO   (1 << 5)

#define BACKPAQ_GASGAUGE_BDIR (1 << 0)
#define BACKPAQ_GASGAUGE_BSCL (1 << 1)
#define BACKPAQ_GASGAUGE_BSDA (1 << 2)

/* Power control (sysctl->pcmcia_power) 0x14 */
#define BACKPAQ_PCMCIA_AEN0   (1 << 0)
#define BACKPAQ_PCMCIA_AEN1   (1 << 1)
#define BACKPAQ_PCMCIA_A5VEN  (1 << 2)
#define BACKPAQ_PCMCIA_A3VEN  (1 << 3)
#define BACKPAQ_PCMCIA_BEN0   (1 << 4)
#define BACKPAQ_PCMCIA_BEN1   (1 << 5)
#define BACKPAQ_PCMCIA_B5VEN  (1 << 6)
#define BACKPAQ_PCMCIA_B3VEN  (1 << 7)
#define BACKPAQ_PWRCTL_50VEN  (1 << 8)
#define BACKPAQ_PWRCTL_25VEN  (1 << 9)
#define BACKPAQ_PWRCTL_18VEN  (1 << 10)

/* Flash control (sysctl->flash_control) 0x18 */
#define BACKPAQ_REGC_FLASH_STATUS0 (1 << 0)
#define BACKPAQ_REGC_FLASH_STATUS1 (1 << 1)
#define BACKPAQ_REGC_ACCEL_X       (1 << 2)
#define BACKPAQ_REGC_ACCEL_Y       (1 << 3)
#define BACKPAQ_REGC_AUD_AGC       (1 << 4)
#define BACKPAQ_REGC_BATT_SDA      (1 << 5)
#define BACKPAQ_REGC_BATT_SDA_DIR  (1 << 6)

#define BACKPAQ_FLASH_ENABLE0 (1 << 4)
#define BACKPAQ_FLASH_ENABLE1 (1 << 5)
#define BACKPAQ_FLASH_VPPEN   (1 << 8)

/* General control register (sysctl->gencontrol) 0x1c */
#define BACKPAQ_REGC_CAM_CLKEN  (1 << 9)
#define BACKPAQ_REGC_AUDIO_CLKEN (1 << 10)
#define BACKPAQ_REGC_AUDIO_AMPEN (1 << 11)

/* Socket register information */
/* Interrupt bits */

#define BACKPAQ_SOCKET_INT_FIFO    (1 << 0)
#define BACKPAQ_SOCKET_INT_VBLANK  (1 << 1)
#define BACKPAQ_SOCKET_INT_WTBUSY  (1 << 2)  /* Write through register is tied up */

#define BACKPAQ_SOCKET_VS1    (1 << 0)
#define BACKPAQ_SOCKET_VS2    (1 << 1)
#define BACKPAQ_SOCKET_STSCHG (1 << 2)
#define BACKPAQ_SOCKET_BVD1   (1 << 2) /* same pin as STSCHG */
#define BACKPAQ_SOCKET_INPACK (1 << 3)
#define BACKPAQ_SOCKET_SPKR   (1 << 4)
#define BACKPAQ_SOCKET_BVD2   (1 << 4) /* same pin as SPKR */
#define BACKPAQ_SOCKET_SKTSEL (1 << 5)
#define BACKPAQ_SOCKET_CD0    (1 << 6)
#define BACKPAQ_SOCKET_CD1    (1 << 7)

#define BACKPAQ_SOCKET_FPGA_RESET (1 << 0)

#define BACKPAQ_GASGAUGE_READ  0x100 
#define BACKPAQ_GASGAUGE_WRITE 0x080 

#define BACKPAQ_GASGAUGE_RSTATE_IDLE 1
#define BACKPAQ_GASGAUGE_WSTATE_IDLE 1

/* Lower 8 bits is the bus width (16 or 32) */
/* Upper 8 bits is the number of bytes to read per read count */ 
#define BACKPAQ_CAMERA_FIFO_INFO_WIDTH 0x00ff   
#define BACKPAQ_CAMERA_FIFO_INFO_BPC   0xff00   



struct backpaq_audio {
   short reg40;
   short pad1;
   short reg44;
   short pad2;
   short reg48;
   short pad3;
   short reg4c;
   short pad4;
   short reg50;
   short pad5;
};

extern struct backpaq_audio *backpaq_audio;

#define BACKPAQ_REG44_L3_DATA_IN (1 << 0)
#define BACKPAQ_REG44_L3_AGCSTAT (1 << 1)
#define BACKPAQ_REG44_L3_OVERFL (1 << 2)
#define BACKPAQ_REG44_RESET     (1 << 3)
#define BACKPAQ_REG44_L3_MODE   (1 << 4)
#define BACKPAQ_REG44_L3_CLOCK  (1 << 5)
#define BACKPAQ_REG44_L3_DIR    (1 << 6)
#define BACKPAQ_REG44_L3_DATA_OUT (1 << 7)
#define BACKPAQ_REG44_L3_BCLK_EN (1 << 8)
#define BACKPAQ_REG44_L3_QMUTE  (1 << 9)

#define BACKPAQ_REG4C_TXD_FIFO_BUSY (1 << 0)
#define BACKPAQ_REG4C_TXD_FIFO_FULL (1 << 0)

#endif
