/*
* Driver interface to the ASIC Complasion chip on the iPAQ H3800
*
* Copyright 2001 Compaq Computer Corporation.
*
* Use consistent with the GNU GPL is permitted,
* provided that this copyright notice is
* preserved in its entirety in all copies and derived works.
*
* COMPAQ COMPUTER CORPORATION MAKES NO WARRANTIES, EXPRESSED OR IMPLIED,
* AS TO THE USEFULNESS OR CORRECTNESS OF THIS CODE OR ITS
* FITNESS FOR ANY PARTICULAR PURPOSE.
*
* Author:  Andrew Christian
*          <Andrew.Christian@compaq.com>
*          October 2001
*/

#include <linux/module.h>
#include <linux/version.h>

#include <linux/init.h>
#include <linux/fs.h>
#include <linux/interrupt.h>
#include <linux/sched.h>
#include <linux/pm.h>
#include <linux/proc_fs.h>

#include <asm/arch/hardware.h>
#include <asm/arch/irqs.h>
#include <asm/arch/h3600_hal.h>

#define H3600_ASIC_PROC_DIR     "asic"
#define H3600_ASIC_PROC_STATS   "stats"

static int g_suspended = 0;

/***********************************************************************************/
/*      Special button IRQs                                                        */
/***********************************************************************************/

#define MAKEKEY(index, down)  ((down) ? (index) : ((index) | 0x80))

static void h3600_asic_action_isr(int irq, void *dev_id, struct pt_regs *regs)
{
        int down = (GPLR & GPIO_H3600_ACTION_BUTTON) ? 0 : 1;
        if (dev_id != h3600_asic_action_isr)
                return;
           
	h3600_hal_keypress( MAKEKEY( 10, down ) );
}

static void h3600_asic_power_isr(int irq, void *dev_id, struct pt_regs *regs)
{
        int down = (GPLR & GPIO_H3600_NPOWER_BUTTON) ? 0 : 1;
        if (dev_id != h3600_asic_power_isr)
                return;

	h3600_hal_keypress( MAKEKEY( 11, down ) );
}


/***********************************************************************************/
/*      Standard entry points for HAL requests                                     */
/***********************************************************************************/

int h3600_asic_version( struct h3600_ts_version *result )
{
	return -EIO;
}

int h3600_asic_eeprom_read( unsigned short address, unsigned char *data, unsigned short len )
{
	if (0) printk(__FUNCTION__ ": address=%d len=%d\n", address, len);
        return -EIO;
}

int h3600_asic_eeprom_write( unsigned short address, unsigned char *data, unsigned short len )
{
        return -EIO;
}

int h3600_asic_thermal_sensor( unsigned short *result )
{
	return -EIO;
}

int h3600_asic_notify_led( unsigned char mode, unsigned char duration, 
			    unsigned char ontime, unsigned char offtime )
{
	return -EIO;
}

int h3600_asic_battery( struct h3600_battery *result )
{
	return -EIO;
}

int h3600_asic_spi_read(unsigned short address, unsigned char *data, unsigned short len)
{
        return -EIO;
}

int h3600_asic_spi_write(unsigned short address, unsigned char *data, unsigned short len)
{
	return -EIO;
}

int h3600_asic_read_light_sensor( unsigned char *result )
{
	return -EIO;
}

int h3600_asic_backlight_control( enum flite_pwr power, unsigned char level )
{
	return -EIO;
}


/***********************************************************************************/
/*   Power management                                                              */
/***********************************************************************************/

static int h3600_asic_pm_callback(struct pm_dev *pm_dev, pm_request_t req, void *data)
{
        if (0) printk(__FUNCTION__ ": req=%d suspended=%d\n", req, g_suspended);
	switch (req) {
	case PM_SUSPEND: /* enter D1-D3 */
                g_suspended = 1;
                break;
	case PM_BLANK:
		break;
	case PM_RESUME:  /* enter D0 - same as unblank */
	case PM_UNBLANK:
                if (g_suspended) {
                        g_suspended = 0;
                }
                break;
	}
	return 0;
}


/***********************************************************************************/
/*   Initialization code                                                           */
/***********************************************************************************/

static struct h3600_hal_ops h3800_asic_ops = {
	get_version         : h3600_asic_version,
	eeprom_read         : h3600_asic_eeprom_read,
	eeprom_write        : h3600_asic_eeprom_write,
	get_thermal_sensor  : h3600_asic_thermal_sensor,
	set_notify_led      : h3600_asic_notify_led,
	read_light_sensor   : h3600_asic_read_light_sensor,
	get_battery         : h3600_asic_battery,
	spi_read            : h3600_asic_spi_read,
	spi_write           : h3600_asic_spi_write,
	backlight_control   : h3600_asic_backlight_control,
};

static int h3600_asic_setup( void )
{
	int result;

        printk(__FUNCTION__ ": setting up asiccontroller interface\n");

	if ( ipaq_info.model != IPAQ_H3800 ) {
		printk(__FUNCTION__ ": unknown iPAQ model %d\n", ipaq_info.model );
		return -ENODEV;
	}

        set_GPIO_IRQ_edge( GPIO_H3600_ACTION_BUTTON, GPIO_BOTH_EDGES );
	result = request_irq(IRQ_GPIO_H3600_ACTION_BUTTON, 
			     h3600_asic_action_isr, 
			     SA_SHIRQ | SA_INTERRUPT | SA_SAMPLE_RANDOM,
			     "h3600_action", h3600_asic_action_isr);

	if ( result ) {
		printk(KERN_CRIT __FUNCTION__ ": unable to grab action button IRQ\n");
		return result;
	}

        set_GPIO_IRQ_edge( GPIO_H3600_NPOWER_BUTTON, GPIO_BOTH_EDGES );
	result = request_irq(IRQ_GPIO_H3600_NPOWER_BUTTON, 
			     h3600_asic_power_isr, 
			     SA_SHIRQ | SA_INTERRUPT | SA_SAMPLE_RANDOM,
			     "h3600_suspend", h3600_asic_power_isr);

	if ( result ) {
		printk(KERN_CRIT __FUNCTION__ ": unable to grab power button IRQ\n");
		free_irq( IRQ_GPIO_H3600_ACTION_BUTTON, h3600_asic_action_isr );
		return result;
	}

	/* Register with power management */
	pm_register(PM_SYS_DEV, PM_SYS_COM, h3600_asic_pm_callback);

	return 0;
}


int __init h3600_asic_init( void )
{
	int result;

	result = h3600_asic_setup();
	if ( !result )
		result = h3600_hal_register_interface( &h3800_asic_ops );
	return result;
}

void __exit h3600_asic_cleanup( void )
{
	h3600_hal_unregister_interface( &h3800_asic_ops );

	free_irq(IRQ_GPIO_H3600_ACTION_BUTTON, h3600_asic_action_isr);
	free_irq(IRQ_GPIO_H3600_NPOWER_BUTTON, h3600_asic_power_isr);

	pm_unregister_all( h3600_asic_pm_callback );
}


module_init(h3600_asic_init)
module_exit(h3600_asic_cleanup)
