/*
 * drivers/pcmcia/sa1100_bitsy.c
 *
 * PCMCIA implementation routines for Bitsy
 * All true functionality is shuttled off to the
 * pcmcia implementation for the current sleeve
 */
#include <linux/config.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/i2c.h>

#include <asm/hardware.h>
#include <asm/irq.h>
#include <asm/io.h>
#include <asm/arch/pcmcia.h>
#include <asm/arch/h3600-sleeve.h>

static struct pcmcia_init  sa1100_bitsy_pcmcia_init;       // Store the interrupt handler
struct pcmcia_low_level   *sa1100_bitsy_pcmcia_ops = NULL; // Initialize with no sleeve

/***********************************************/
/* Stub routines called by the PCMCIA device.  */
/***********************************************/

static int bitsy_pcmcia_init(struct pcmcia_init *init)
{
//	printk(__FUNCTION__ "\n");
	sa1100_bitsy_pcmcia_init = *init;

	if ( sa1100_bitsy_pcmcia_ops != NULL
	     && sa1100_bitsy_pcmcia_ops->init != NULL
	     && init->handler != NULL )
		return sa1100_bitsy_pcmcia_ops->init( init );

	return 2;   /* Return by default */
}

static int bitsy_pcmcia_shutdown(void)
{
//	printk(__FUNCTION__ "\n");
	if ( sa1100_bitsy_pcmcia_ops != NULL 
	     && sa1100_bitsy_pcmcia_ops->shutdown != NULL )
		sa1100_bitsy_pcmcia_ops->shutdown();

	return 0;   /* Not examined */
}

static int bitsy_pcmcia_socket_state(struct pcmcia_state_array *state_array)
{
//	printk(__FUNCTION__ "\n");
	if ( sa1100_bitsy_pcmcia_ops != NULL 
	     && sa1100_bitsy_pcmcia_ops->socket_state != NULL )
		return sa1100_bitsy_pcmcia_ops->socket_state( state_array );

	/* Default actions */
        if ( state_array->size < 2 ) 
		return -1;

        memset(state_array->state, 0, (state_array->size)*sizeof(struct pcmcia_state));
	return 0;
}

static int bitsy_pcmcia_get_irq_info(struct pcmcia_irq_info *info)
{
//	printk(__FUNCTION__ "\n");
	if ( sa1100_bitsy_pcmcia_ops != NULL 
	     && sa1100_bitsy_pcmcia_ops->get_irq_info != NULL )
		return sa1100_bitsy_pcmcia_ops->get_irq_info( info );

	return
		sa1100_bitsy_common_pcmcia_get_irq_info( info );
}

static int bitsy_pcmcia_configure_socket(const struct pcmcia_configure *configure)
{
//	printk(__FUNCTION__ "\n");
	if ( sa1100_bitsy_pcmcia_ops != NULL 
	     && sa1100_bitsy_pcmcia_ops->configure_socket != NULL )
		return sa1100_bitsy_pcmcia_ops->configure_socket( configure );

        return 0;
}

struct pcmcia_low_level bitsy_pcmcia_ops = { 
        bitsy_pcmcia_init,
        bitsy_pcmcia_shutdown,
        bitsy_pcmcia_socket_state,
        bitsy_pcmcia_get_irq_info,
        bitsy_pcmcia_configure_socket
};

/****************************************************/
/*  Swapping functions for PCMCIA operations        */
/****************************************************/

void sa1100_bitsy_pcmcia_change_sleeves(struct pcmcia_low_level *ops)
{
	if ( ops != sa1100_bitsy_pcmcia_ops ) {
		bitsy_pcmcia_shutdown();
		sa1100_bitsy_pcmcia_ops = ops;
		bitsy_pcmcia_init( &sa1100_bitsy_pcmcia_init );
	}
	printk("****" __FUNCTION__ " changed\n");
}

void sa1100_bitsy_pcmcia_remove_sleeve()
{
	if ( sa1100_bitsy_pcmcia_ops != NULL ) {
		bitsy_pcmcia_shutdown();
		sa1100_bitsy_pcmcia_ops = NULL;
	}
	printk("****" __FUNCTION__ "\n");
}

EXPORT_SYMBOL(sa1100_bitsy_pcmcia_change_sleeves);
EXPORT_SYMBOL(sa1100_bitsy_pcmcia_remove_sleeve);


/****************************************************/
/*  Common functions used by the different sleeves */
/****************************************************/

int sa1100_bitsy_common_pcmcia_init( struct pcmcia_init *init )
{
        int irq, res;

//	printk(__FUNCTION__ "\n");

        /* Enable PCMCIA/CF bus: */
//        set_bitsy_egpio(EGPIO_BITSY_OPT_NVRAM_ON);  // Handled by sleeve driver
	set_bitsy_egpio(EGPIO_BITSY_OPT_ON);
        clr_bitsy_egpio(EGPIO_BITSY_OPT_RESET);
        clr_bitsy_egpio(EGPIO_BITSY_CARD_RESET);

        /* All those are inputs */
        GPDR &= ~(GPIO_BITSY_PCMCIA_CD0 | GPIO_BITSY_PCMCIA_CD1
		  | GPIO_BITSY_PCMCIA_IRQ0| GPIO_BITSY_PCMCIA_IRQ1);

        /* Set transition detect */
        set_GPIO_IRQ_edge( GPIO_BITSY_PCMCIA_CD0 | GPIO_BITSY_PCMCIA_CD1, GPIO_BOTH_EDGES );
        set_GPIO_IRQ_edge( GPIO_BITSY_PCMCIA_IRQ0| GPIO_BITSY_PCMCIA_IRQ1, GPIO_FALLING_EDGE );

        /* Register interrupts */
        irq = IRQ_GPIO_BITSY_PCMCIA_CD0;
        res = request_irq( irq, init->handler, SA_INTERRUPT, "PCMCIA_CD0", NULL );
        if( res < 0 ) { 
		printk( KERN_ERR __FUNCTION__ ": Request for IRQ %u failed\n", irq );
		return -1;
	}

        irq = IRQ_GPIO_BITSY_PCMCIA_CD1;
        res = request_irq( irq, init->handler, SA_INTERRUPT, "PCMCIA_CD1", NULL );
        if( res < 0 ) { 
		printk( KERN_ERR __FUNCTION__ ": Request for IRQ %u failed\n", irq );
		return -1;
	}

        return 2;  /* Always allow for two PCMCIA devices */
}

int sa1100_bitsy_common_pcmcia_shutdown( void )
{
//	printk(__FUNCTION__ "\n");

	/* disable IRQs */
	free_irq( IRQ_GPIO_BITSY_PCMCIA_CD0, NULL );
	free_irq( IRQ_GPIO_BITSY_PCMCIA_CD1, NULL );
  
	/* Disable CF bus: */
//	clr_bitsy_egpio(EGPIO_BITSY_OPT_NVRAM_ON|EGPIO_BITSY_OPT_ON);
	clr_bitsy_egpio(EGPIO_BITSY_OPT_ON);
	set_bitsy_egpio(EGPIO_BITSY_OPT_RESET);
	return 0;
}

int sa1100_bitsy_common_pcmcia_socket_state( struct pcmcia_state_array *state_array )
{
        unsigned long levels;

//	printk(__FUNCTION__ "\n");

        if(state_array->size<2) 
		return -1;

        memset(state_array->state, 0, (state_array->size)*sizeof(struct pcmcia_state));
	
	levels=GPLR;
	
	state_array->state[0].detect=((levels & GPIO_BITSY_PCMCIA_CD0)==0)?1:0;
	state_array->state[0].ready=(levels & GPIO_BITSY_PCMCIA_IRQ0)?1:0;
	state_array->state[1].detect=((levels & GPIO_BITSY_PCMCIA_CD1)==0)?1:0;
	state_array->state[1].ready=(levels & GPIO_BITSY_PCMCIA_IRQ1)?1:0;

	return 0;
}

int sa1100_bitsy_common_pcmcia_get_irq_info( struct pcmcia_irq_info *info )
{
	printk(__FUNCTION__ "\n");

        switch (info->sock) {
        case 0:
                info->irq=IRQ_GPIO_BITSY_PCMCIA_IRQ0;
                break;
        case 1:
                info->irq=IRQ_GPIO_BITSY_PCMCIA_IRQ1;
                break;
        default:
                return -1;
        }
        return 0;
}

EXPORT_SYMBOL(sa1100_bitsy_common_pcmcia_init);
EXPORT_SYMBOL(sa1100_bitsy_common_pcmcia_shutdown);
EXPORT_SYMBOL(sa1100_bitsy_common_pcmcia_socket_state);
EXPORT_SYMBOL(sa1100_bitsy_common_pcmcia_get_irq_info);


