/*
 *  $Id: longhaul.c,v 1.9 2001/09/15 02:51:52 davej Exp $
 *
 *	(C) 2001  Dave Jones
 *
 * 	Licensed under the terms of the GNU GPL License version 2.
 *  Based upon datasheets 'bios5_2.pdf' and 'longhaul_3.doc' provided by VIA.
 *
 *  VIA have currently 3 different versions of Longhaul.
 *  v1.0
 *  	Found on VIA C3 Samuel & Stepping 0 VIA C3 Samuel 2
 *  	Provides software controlled clock multipliers
 *  v2.0
 *  	Found on VIA C3 Samuel 2 (stepping 0-7) & VIA C3 Ezra
 *  	Software controlled VID pins
 *  v3.0
 * 		Found on VIA C5M
 *  	Software controller BSEL pins
 *
 *  BIG FAT DISCLAIMER: Work in progress code. Possibly *dangerous*
 */

#include <linux/kernel.h>
#include <linux/module.h> 
#include <linux/sched.h>
#include <linux/init.h>
#include <asm/msr.h>
#include <asm/timex.h>
#include <asm/io.h>
#include <linux/delay.h>
#include <linux/proc_fs.h>

#define __hlt()     __asm__ __volatile__("hlt": : :"memory")

/* VIA C3 Samuel 1  & Samuel 2 (stepping 0)*/
static int __initdata longhaul1_clock_ratio[16] = {
	-1, /* 0000 -> RESERVED */
	30, /* 0001 ->  3.0x */
	40, /* 0010 ->  4.0x */
	-1, /* 0011 -> RESERVED */
	-1, /* 0100 -> RESERVED */
	35, /* 0101 ->  3.5x */
	45, /* 0110 ->  4.5x */
	55, /* 0111 ->  5.5x */
	60, /* 1000 ->  6.0x */
	70, /* 1001 ->  7.0x */
	80, /* 1010 ->  8.0x */
	50, /* 1011 ->  5.0x */
	65, /* 1100 ->  6.5x */
	75, /* 1101 ->  7.5x */
	-1, /* 1110 -> RESERVED */
	-1, /* 1111 -> RESERVED */
};

static int __initdata samuel1_eblcr[16] = {
	50, /* 0000 -> RESERVED */
	30, /* 0001 ->  3.0x */
	40, /* 0010 ->  4.0x */
	-1, /* 0011 -> RESERVED */
	55, /* 0100 ->  5.5x */
	35, /* 0101 ->  3.5x */
	45, /* 0110 ->  4.5x */
	-1, /* 0111 -> RESERVED */
	-1, /* 1000 -> RESERVED */
	70, /* 1001 ->  7.0x */
	80, /* 1010 ->  8.0x */
	60, /* 1011 ->  6.0x */
	-1, /* 1100 -> RESERVED */
	75, /* 1101 ->  7.5x */
	-1, /* 1110 -> RESERVED */
	65, /* 1111 ->  6.5x */
};

/* VIA C3 Samuel2 Stepping 1->15 & VIA C3 Ezra */
static int __initdata longhaul2_clock_ratio[16] = {
	100, /* 0000 -> 10.0x */
	30,  /* 0001 ->  3.0x */
	40,  /* 0010 ->  4.0x */
	90,  /* 0011 ->  9.0x */
	95,  /* 0100 ->  9.5x */
	35,  /* 0101 ->  3.5x */
	45,  /* 0110 ->  4.5x */
	55,  /* 0111 ->  5.5x */
	60,  /* 1000 ->  6.0x */
	70,  /* 1001 ->  7.0x */
	80,  /* 1010 ->  8.0x */
	50,  /* 1011 ->  5.0x */
	65,  /* 1100 ->  6.5x */
	75,  /* 1101 ->  7.5x */
	85,  /* 1110 ->  8.5x */
	120, /* 1111 -> 12.0x */
};

static int __initdata samuel2_eblcr[16] = {
	90,  /* 0000 ->  9.0x */
	30,  /* 0001 ->  3.0x */
	40,  /* 0010 ->  4.0x */
	100, /* 0011 -> 10.0x */
	55,  /* 0100 ->  5.5x */
	35,  /* 0101 ->  3.5x */
	45,  /* 0110 ->  4.5x */
	110, /* 0111 -> 11.0x */
	90,  /* 1000 ->  9.0x */
	70,  /* 1001 ->  7.0x */
	80,  /* 1010 ->  8.0x */
	60,  /* 1011 ->  6.0x */
	120, /* 1100 -> 12.0x */
	75,  /* 1101 ->  7.5x */
	130, /* 1110 -> 13.0x */
	65,  /* 1111 ->  6.5x */
};

static int __initdata ezra_eblcr[16] = {
	90,  /* 0000 ->  9.0x */
	30,  /* 0001 ->  3.0x */
	40,  /* 0010 ->  4.0x */
	100, /* 0011 -> 10.0x */
	55,  /* 0100 ->  5.5x */
	35,  /* 0101 ->  3.5x */
	45,  /* 0110 ->  4.5x */
	95,  /* 0111 ->  9.5x */
	50,  /* 1000 ->  5.0x */
	70,  /* 1001 ->  7.0x */
	80,  /* 1010 ->  8.0x */
	60,  /* 1011 ->  6.0x */
	120, /* 1100 -> 12.0x */
	75,  /* 1101 ->  7.5x */
	85,  /* 1110 ->  8.5x */
	65,  /* 1111 ->  6.5x */
};

/* VIA C5M. */
static int __initdata longhaul3_clock_ratio[32] = {
	100, /* 0000 -> 10.0x */
	30,  /* 0001 ->  3.0x */
	40,  /* 0010 ->  4.0x */
	90,  /* 0011 ->  9.0x */
	95,  /* 0100 ->  9.5x */
	35,  /* 0101 ->  3.5x */
	45,  /* 0110 ->  4.5x */
	55,  /* 0111 ->  5.5x */
	60,  /* 1000 ->  6.0x */
	70,  /* 1001 ->  7.0x */
	80,  /* 1010 ->  8.0x */
	50,  /* 1011 ->  5.0x */
	65,  /* 1100 ->  6.5x */
	75,  /* 1101 ->  7.5x */
	85,  /* 1110 ->  8.5x */
	120, /* 1111 ->  12.0x */

	-1,  /* 0000 -> RESERVED (10.0x) */
	110, /* 0001 -> 11.0x */
	120, /* 0010 -> 12.0x */
	-1,  /* 0011 -> RESERVED (9.0x)*/
	105, /* 0100 -> 10.5x */
	115, /* 0101 -> 11.5x */
	125, /* 0110 -> 12.5x */
	135, /* 0111 -> 13.5x */
	140, /* 1000 -> 14.0x */
	150, /* 1001 -> 15.0x */
	160, /* 1010 -> 16.0x */
	130, /* 1011 -> 13.0x */
	145, /* 1100 -> 14.5x */
	155, /* 1101 -> 15.5x */
	-1,  /* 1110 -> RESERVED (13.0x) */
	-1,  /* 1111 -> RESERVED (12.0x) */
};

static int __initdata c5m_eblcr[32] = {
	90,  /* 0000 ->  9.0x */
	30,  /* 0001 ->  3.0x */
	40,  /* 0010 ->  4.0x */
	100, /* 0011 -> 10.0x */
	55,  /* 0100 ->  5.5x */
	35,  /* 0101 ->  3.5x */
	45,  /* 0110 ->  4.5x */
	95,  /* 0111 ->  9.5x */
	50,  /* 1000 ->  5.0x */
	70,  /* 1001 ->  7.0x */
	80,  /* 1010 ->  8.0x */
	60,  /* 1011 ->  6.0x */
	120, /* 1100 -> 12.0x */
	75,  /* 1101 ->  7.5x */
	85,  /* 1110 ->  8.5x */
	65,  /* 1111 ->  6.5x */

	-1,  /* 0000 -> RESERVED (9.0x) */
	110, /* 0001 -> 11.0x */
	120, /* 0010 -> 12.0x */
	-1,  /* 0011 -> RESERVED (10.0x)*/
	135, /* 0100 -> 13.5x */
	115, /* 0101 -> 11.5x */
	125, /* 0110 -> 12.5x */
	105, /* 0111 -> 10.5x */
	130, /* 1000 -> 13.0x */
	150, /* 1001 -> 15.0x */
	160, /* 1010 -> 16.0x */
	140, /* 1011 -> 14.0x */
	-1,  /* 1100 -> RESERVED (12.0x) */
	155, /* 1101 -> 15.5x */
	-1,  /* 1110 -> RESERVED (13.0x) */
	145, /* 1111 -> 14.5x */
};


/* Clock ratios multiplied by 10 */
static int clock_ratio[32];
static int eblcr_table[32];
static int highest_speed;
static int lowest_speed;
static int longhaul; /* version. */
static unsigned int clock_bogomips[32];
static unsigned int busfreq,cpufreq;


unsigned int longhaul_validatespeed (unsigned int freq)
{
	if (freq<lowest_speed)
		freq=lowest_speed;
	if (freq>highest_speed)
		freq=highest_speed;

	// TODO: scan available freqs/FSBs here.

	return freq;
}


void longhaul_set_cpu_frequency (unsigned int Mhz, int fsb)
{
	int i;
	unsigned int best=200; /* safe initial values */
	unsigned int besti=4;
	unsigned long lo, hi;
	int numscales=16;
	int bits=-1;

	if (longhaul==3)
		numscales=32;

	cpufreq = Mhz;
	/* Find out which bit-pattern we want */

	//FIXME: Adjust FSB if necessary, not just mult.

	for (i=0;i<numscales;i++) {
		unsigned int newclock;
		if (clock_ratio[i]==-1)	/* skip RESERVED entries */
			continue;

		bits = clock_ratio[i];
		newclock = (bits*fsb/10);
		if ((newclock > best) && (newclock <= (Mhz+1))) {
			/* +1 is for compensating rounding errors */
			best = newclock;
			besti = i;
		}
	}
	if (bits==-1)
		return;

	/* "besti" now contains the bitpattern of the new multiplier.
	   we now need to transform it to the desired format. */

	cli();

	switch (longhaul) {
	case 1:
		rdmsr (0x1147, lo, hi);
		lo &= ~(1<<23|1<<24|1<<25|1<<26);
		lo |= (1<<19);		/* Enable software clock multiplier */
		lo |= bits<<23;		/* desired multiplier */
		wrmsr (0x1147, lo, hi);

		__hlt();

		/* Disable software clock multiplier */
		rdmsr (0x1147, lo, hi);
		lo &= ~(1<<19);
		wrmsr (0x1147, lo, hi);
		break;

	case 2:
		rdmsr (0x110a, lo, hi);
		lo &= ~(1<<16|1<<17|1<<18|1<<19);
		lo |= bits<<16;
		lo |= (1<<8);	/* EnableSoftBusRatio */
//		wrmsr (0x110a, lo, hi);

		__hlt();

		rdmsr (0x110a, lo, hi);
		lo &= ~(1<<8);
//		wrmsr (0x110a, lo, hi);
		break;

	case 3:
		rdmsr (0x110a, lo, hi);
		lo &= ~(1<<16|1<<17|1<<18|1<<19);
		lo |= bits<<16;
		lo |= (1<<8);	/* EnableSoftBusRatio */
		if (i>15) 	/* To access upper 16 scales, we must set bit 14 of 0x110a */
			lo |= 1<<14;

		wrmsr (0x110a, lo, hi);

		__hlt();

		rdmsr (0x110a, lo, hi);
		lo &= ~(1<<8);
		wrmsr (0x110a, lo, hi);
		break;
	}

	sti();

	/* now adjust bogomips */
	if (!clock_bogomips[besti]) {
		/*scale_bogomips(clock_ratio[besti]);*/
		clock_bogomips[besti] = loops_per_jiffy;
	} else {
		loops_per_jiffy = clock_bogomips[besti];
	}
}


static int longhaul_get_cpu_fsb (void)
{
	unsigned long invalue=0,lo, hi;
	unsigned int eblcr_fsb_table[4] = { 66, 133, 100, -1 };

	rdmsr (0x2a, lo, hi);
	invalue = (lo & (1<<19|1<<18)) >>18;
	return eblcr_fsb_table[invalue];
}


static int longhaul_get_cpu_mult (void)
{
	unsigned long invalue=0,lo, hi;

	rdmsr (0x2a, lo, hi);
	invalue = (lo & (1<<25|1<<24|1<<23|1<<22)) >>22;
	if (longhaul==3) {
		if (lo & (1<<27))
			invalue+=16;
	}
	return eblcr_table[invalue];
}


static void longhaul_get_ranges (void)
{
	unsigned long lo, hi, invalue;
	unsigned int minmult=0, maxmult=0, minfsb=0, maxfsb=0;
	unsigned int multipliers[32]= {
		50,30,40,100,55,35,45,95,90,70,80,60,120,75,85,65,
		-1,110,120,-1,135,115,125,105,130,150,160,140,-1,155,-1,145 };
	unsigned int fsb_table[4] = { 133, 100, -1, 66 };

	switch (longhaul) {
	case 1:
		//FIXME: fill in from datasheet
		break;

	case 2 ... 3:
		rdmsr (0x110a, lo, hi);
		invalue = (hi & (1<<0|1<<1|1<<2|1<<3));
		if (lo & (1<<12))
			invalue += 16;
		maxmult=multipliers[invalue];

		invalue = (hi & (1<<16|1<<17|1<<18|1<<19)) >> 16;
		minmult = multipliers[invalue];

		invalue = (hi & (1<<9|1<<10)) >> 9;
		maxfsb = fsb_table[invalue];

		invalue = (hi & (1<<25|1<<26)) >> 25;
		minfsb = fsb_table[invalue];
		break;
	}

	highest_speed = (maxmult/10) * maxfsb;
	if (((maxmult/10)*10)!=maxmult)
		highest_speed += maxfsb/2;

	lowest_speed = (minmult/10) * minfsb;
	if (((minmult/10)*10)!=minmult)
		lowest_speed += minfsb/2;

	printk ("Max FSB=%d MaxMult=%d Highestspeed=%d\n",
		maxfsb, maxmult, highest_speed);
	printk ("Min FSB=%d MinMult=%d Lowestspeed=%d\n",
		minfsb, minmult, lowest_speed);
}


int longhaul_init (void)
{
	struct cpuinfo_x86 *c = cpu_data;

	switch (c->x86_model) {
	case 6:		/* VIA C3 Samuel C5A */
		longhaul=1;
		memcpy (clock_ratio, longhaul1_clock_ratio, sizeof(longhaul1_clock_ratio));
		memcpy (eblcr_table, samuel1_eblcr, sizeof(samuel1_eblcr));
		break;

	case 7:		/* C5B / C5C */
		switch (c->x86_mask) {
		case 0:
			longhaul=1;
			memcpy (clock_ratio, longhaul1_clock_ratio, sizeof(longhaul1_clock_ratio));
			memcpy (eblcr_table, samuel2_eblcr, sizeof(samuel2_eblcr));
			break;
		case 1 ... 15:
			longhaul=2;
			memcpy (clock_ratio, longhaul2_clock_ratio, sizeof(longhaul2_clock_ratio));
			memcpy (eblcr_table, ezra_eblcr, sizeof(ezra_eblcr));
			break;
		}
		break;

	case 8:		/* C5M */
		longhaul=3;
		memcpy (clock_ratio, longhaul3_clock_ratio, sizeof(longhaul3_clock_ratio));
		memcpy (eblcr_table, c5m_eblcr, sizeof(c5m_eblcr));
		break;

	default:
		printk (KERN_INFO "Unknown VIA CPU. Contact davej@suse.de\n");
		return -1;
	}

	printk ("VIA CPU detected. Longhaul version %d supported\n", longhaul);
	printk ("CPU currently at %dMHz (%d x %d.%d)\n",
		(longhaul_get_cpu_fsb() * longhaul_get_cpu_mult()) / 10,
		longhaul_get_cpu_fsb(), longhaul_get_cpu_mult()/10,
		longhaul_get_cpu_mult()-((longhaul_get_cpu_mult()/10)*10));

	busfreq = longhaul_get_cpu_fsb();

	longhaul_get_ranges();

	//cpufreq_setfunctions (&longhaul_validatespeed, &longhaul_set_cpu_frequency) 
	return 0;
}

void longhaul_exit (void)
{
}

module_init(longhaul_init);
module_exit(longhaul_exit);
