/*
 * Driver for the Compaq iPAQ Mercury Backpaq camera
 * Video4Linux interface
 *
 * Copyright 2001 Compaq Computer Corporation.
 *
 * Use consistent with the GNU GPL is permitted,
 * provided that this copyright notice is
 * preserved in its entirety in all copies and derived works.
 *
 * COMPAQ COMPUTER CORPORATION MAKES NO WARRANTIES, EXPRESSED OR IMPLIED,
 * AS TO THE USEFULNESS OR CORRECTNESS OF THIS CODE OR ITS
 * FITNESS FOR ANY PARTICULAR PURPOSE.
 *
 * Author: Andrew Christian 
 *         <andyc@handhelds.org>
 *         4 May 2001
 *
 * Driver for Mercury BackPAQ camera
 *
 * Issues to be addressed:
 *    1. Writing to the FPGA when we need to do a functionality change
 *    2. Sampling the pixels correctly and building a pixel array
 *    3. Handling different pixel formats correctly
 *    4. Changing the contrast, brightness, white balance, and so forth.
 *    5. Specifying a subregion (i.e., setting "top, left" and SUBCAPTURE)
 */

#include <linux/module.h>
#include <linux/version.h>

#include <linux/init.h>
#include <asm/uaccess.h>         /* get_user,copy_to_user*/

#include <linux/videodev.h>
#include <linux/vmalloc.h>
#include <linux/proc_fs.h>

#include <asm/arch/hardware.h>
#include <asm/arch/backpaq.h>
#include <asm/irq.h>

#include "h3600_backpaq_camera.h"

/* Actual chip specifications */
#define HC_TRUE_WIDTH           644    /* Last two columns are black */
#define HC_TRUE_HEIGHT          482
#define HC_FRAME_BUFFER_SIZE    ((u32) HC_TRUE_WIDTH * (u32) HC_TRUE_HEIGHT )

#define HC_MAX_ALLOWED_WIDTH    640
#define HC_MAX_ALLOWED_HEIGHT   480

#define HC_MIN_ALLOWED_WIDTH    160
#define HC_MIN_ALLOWED_HEIGHT   120

/* #define BACKPAQ_CAMERA_DEBUG  1*/
#undef BACKPAQ_CAMERA_DEBUG


/* Camera modes */
#define HC_MODE_RAW             0
#define HC_MODE_DECIMATION_1	1
#define HC_MODE_DECIMATION_2	2
#define HC_MODE_DECIMATION_4	3

struct h3600_size_mode {
	int width;
	int height;
	int mode;
};

static struct h3600_size_mode hc_default_resolutions[] = {
	{ 160, 120, HC_MODE_DECIMATION_4 },
	{ 320, 240, HC_MODE_DECIMATION_2 },
	{ 640, 480, HC_MODE_DECIMATION_1 },
	{ 0, 0, 0 }
};

/* Core states */
#define HC_V4L_IDLE              0
#define HC_V4L_GRABBING          1
#define HC_V4L_STREAMING         2
#define HC_V4L_STREAMING_PAUSED  3

/* HC_V4L_GRABBING : substate */
#define GRAB_WAIT_FOR_REFRESH    0
#define GRAB_ACTIVE              1
#define GRAB_DONE                2
#define GRAB_ERROR               3

/* A single frame of raw data from the camera       */
/* We can always tell the active mode from the size */
struct capture_frame {
	unsigned char *data;
	int            bytes_read;     /* How full it is */
	int            bytes_to_read;  /* How much to get */
	int            width;
	int            height;
	int            row_width;
	volatile int   state;  /* idle, active, done */
};

#define FRAME_NUM	2	/* double buffering for now */
  
struct h3600_camera_struct {
	struct video_device     vdev;   /* This must be first */
	struct video_picture    vpic;   /* v4l camera settings */
	struct video_window     vwin;   /* v4l capture area */
	struct h3600_backpaq_camera_params params;   /* Our special settings */

	volatile int            state;	/* v4l current state */
	struct capture_frame    frame;

	int mode;                  /* Mode index selection                   */

	struct semaphore        lock;          /* Force single access */
	wait_queue_head_t       capq;          /* Wait queue */
	int                     usage_count;   /* How many are open */
	int                     fifo_high;     /* Tracks how deep the fifo goes */

	/* Count interrupts */
	int                     vblank_count;
	int                     fifo_count;

	/* mmap interface */
/*	int            curframe; */      /* the current frame to grab into */
/*	unsigned char *frame_buf;*/      /* frame buffer data */
/*        struct h3600_camera_frame frame[FRAME_NUM]; */
				/* FRAME_NUM-buffering, so we need a array */
};

/* Global variables */

static struct h3600_camera_struct hc_camera;     /* We only have a single camera */
static struct proc_dir_entry     *proc_camera = NULL;

static volatile struct mercury_backpaq_socket *mercury_backpaq_socket 
     = ((struct mercury_backpaq_socket *)BACKPAQ_SOCKET_STATUS_BASE);

static volatile struct mercury_backpaq_sysctl *mercury_backpaq_sysctl 
     = ((struct mercury_backpaq_sysctl *)BACKPAQ_SYSCTL_BASE);

unsigned int h3600_camera_debug = 0;

/* insmod options */

MODULE_PARM(h3600_camera_debug,"i");
MODULE_PARM_DESC(h3600_camera_debug,"debug messages, default is 0 (no)");

#define BANNER "Compaq iPAQ H3600 Mercury BackPAQ Camera for Video4Linux"

MODULE_DESCRIPTION(BANNER);
MODULE_AUTHOR("Andrew Christian <andyc@handhelds.org>");

/*******************************
 *  Utility routines 
 *******************************/

static void set_camera_resolution( struct h3600_camera_struct *cam,
				   int width, int height )
{
	struct h3600_size_mode *m = hc_default_resolutions;

	if ( cam->vpic.palette == VIDEO_PALETTE_RAW ) {
		cam->vwin.width  = HC_TRUE_WIDTH;
		cam->vwin.height = HC_TRUE_HEIGHT;
		cam->mode   = HC_MODE_RAW;
		return;
	}

	do {
		cam->vwin.width = m->width;
		cam->vwin.height = m->height;
		cam->mode = m->mode;
		m++;
	} while ( m->width > 0 && width >= m->width && height >= m->height );
}

static unsigned long required_buf_size( struct h3600_camera_struct *cam )
{
	return ((unsigned long) cam->vwin.width) * cam->vwin.height 
		* ( cam->vpic.palette == VIDEO_PALETTE_RGB24 ? 3 : 1 );
}

/*
static int allocate_frame_buf(struct h3600_camera_struct *cam)
{
	int i;

	cam->frame_buf = rvmalloc(FRAME_NUM * CPIA_MAX_FRAME_SIZE);
	if (!cam->frame_buf)
		return -ENOBUFS;

	for (i = 0; i < FRAME_NUM; i++)
		cam->frame[i].data = cam->frame_buf + i * CPIA_MAX_FRAME_SIZE;

	return 0;
}

static int free_frame_buf(struct h3600_camera_struct *cam)
{
	int i;
	
	rvfree(cam->frame_buf, FRAME_NUM*CPIA_MAX_FRAME_SIZE);
	cam->frame_buf = 0;
	for (i=0; i < FRAME_NUM; i++)
		cam->frame[i].data = NULL;

	return 0;
}


static void inline free_frames(struct cpia_frame frame[FRAME_NUM])
{
	int i;

	for (i=0; i < FRAME_NUM; i++)
		frame[i].state = FRAME_UNUSED;
	return;
}
*/


/* 
   Decimate in place, assuming a standard Bayer pattern starting at 0,0
   The subsampled image starts at offset x,y.
   We update "columns" and "rows" to return the new number of columns 
   and rows in the image.

   Bayer pattern is assumed to be:

       R G 
       G B

   starting at the top left corner.
   "Width" and "Height" should be multiples of two with x + width <= *columns, etc.

   We make _tons_ of assumptions

   Assumptions
   1. Error checking of bounds has been done.
   2. Width and height are multiples of 4
*/

static void decimate( struct capture_frame *frame )
{
	/* Pick results that make sense */
	int width  = (frame->width / 2) & 0xfffc;
	int height = (frame->height / 2) & 0xfffc;

	unsigned char *p    = frame->data;   /* We'll store data at here */
	unsigned char *colp = frame->data;

	unsigned char *q;   /* We'll read data from here */
	int onecol = frame->width;
	int twocol = frame->width * 2;
	int i,j;

	for ( j = 0 ; j < height ; j+= 2) {
		q = colp;
		for ( i = 0 ; i < width ; i+=2 ) {
			/* Do red */
			*p++ = (((unsigned short) *q) * 2 
				+ (unsigned short) *(q + 2) 
				+ (unsigned short) *(q + twocol)) / 4;
			q+=2;
			/* Do green */
			*p++ = (((unsigned short) *(q + 1)
				 + (unsigned short) *(q + onecol)) / 2 );
			q+=2;
		}
		colp += twocol;
		q = colp;
		for ( i = 0 ; i < width ; i+=2 ) {
			/* Do green */
			*p++ = (((unsigned short) *(q + 1)
				 + (unsigned short) *(q + onecol)) / 2 );
			q+=2;
			/* Do blue */
			*p++ = (((unsigned short) *(q + onecol + 1)) * 2 
				+ (unsigned short) *(q + onecol - 1) 
				+ (unsigned short) *(q - onecol + 1)) / 4;
			q+=2;
		}
		colp += twocol;
	}
	frame->width = width;
	frame->height = height;
	frame->row_width = width;
}

#define DO_RED_PIXEL \
    *to++ = *from;                                                   \
    *to++ = ((ushort)*(from-width) + (ushort)*(from+width) +         \
             (ushort)*(from-1) + (ushort)*(from+1)) / 4;             \
    *to++ = ((ushort)*(from-width-1) + (ushort)*(from-width+1) +     \
             (ushort)*(from+width-1) + (ushort)*(from+width+1)) / 4; \
     from++

#define DO_RED_TOP_PIXEL \
    *to++ = *from;                                                   \
    *to++ = ((ushort)*(from+width) * 2 +                             \
             (ushort)*(from-1) + (ushort)*(from+1)) / 4;             \
    *to++ = ((ushort)*(from+width-1) + (ushort)*(from+width+1)) / 2; \
     from++

#define DO_RED_LEFT_PIXEL \
    *to++ = *from;                                                   \
    *to++ = ((ushort)*(from-width) + (ushort)*(from+width) +         \
             + 2 * (ushort)*(from+1)) / 4;                           \
    *to++ = ((ushort)*(from-width+1) + (ushort)*(from+width+1)) / 2; \
     from++

#define DO_RED_TOP_LEFT_PIXEL \
    *to++ = *from;                                                   \
    *to++ = ((ushort)*(from+width) + (ushort)*(from+1)) / 2;         \
    *to++ = *(from+width+1);                                         \
     from++

#define DO_BLUE_PIXEL \
    *to++ = ((ushort)*(from-width-1) + (ushort)*(from-width+1) +     \
             (ushort)*(from+width-1) + (ushort)*(from+width+1)) / 4; \
    *to++ = ((ushort)*(from-width) + (ushort)*(from+width) +         \
             (ushort)*(from-1) + (ushort)*(from+1)) / 4;             \
    *to++ = *from++;  

#define DO_BLUE_RIGHT_PIXEL \
    *to++ = ((ushort)*(from-width-1) + (ushort)*(from+width-1)) / 2; \
    *to++ = ((ushort)*(from-width) + (ushort)*(from+width) +         \
             (ushort)*(from-1) * 2) / 4;                             \
    *to++ = *from++

#define DO_BLUE_BOTTOM_PIXEL \
    *to++ = ((ushort)*(from-width-1) + (ushort)*(from-width+1)) / 2; \
    *to++ = ((ushort)*(from-width) * 2 +                             \
             (ushort)*(from-1) + (ushort)*(from+1)) / 4;             \
    *to++ = *from++

#define DO_BLUE_BOTTOM_RIGHT_PIXEL \
    *to++ = *(from-width-1);                                         \
    *to++ = ((ushort)*(from-width) + (ushort)*(from-1)) / 2;         \
    *to++ = *from++

#define DO_GREEN_ODD_PIXEL \
    *to++ = ((ushort)*(from-1) + (ushort)*(from+1)) / 2;         \
    *to++ = *from;                                               \
    *to++ = ((ushort)*(from-width) + (ushort)*(from+width)) / 2; \
     from++

#define DO_GREEN_EVEN_PIXEL \
    *to++ = ((ushort)*(from-width) + (ushort)*(from+width)) / 2; \
    *to++ = *from;                                               \
    *to++ = ((ushort)*(from-1) + (ushort)*(from+1)) / 2; \
     from++

#define DO_GREEN_TOP_PIXEL \
    *to++ = ((ushort)*(from-1) + (ushort)*(from+1)) / 2;         \
    *to++ = *from;                                               \
    *to++ = *(from+width); \
     from++

#define DO_GREEN_TOP_RIGHT_PIXEL \
    *to++ = *(from-1);      \
    *to++ = *from;          \
    *to++ = *(from+width); \
     from++

#define DO_GREEN_LEFT_PIXEL \
    *to++ = ((ushort)*(from-width) + (ushort)*(from+width)) / 2; \
    *to++ = *from;                                               \
    *to++ = *(from+1);     \
     from++

#define DO_GREEN_BOTTOM_LEFT_PIXEL \
    *to++ = *(from-width);           \
    *to++ = *from;                   \
    *to++ = *(from+1);               \
     from++

#define DO_GREEN_RIGHT_PIXEL \
    *to++ = *(from-1);                                           \
    *to++ = *from;                                               \
    *to++ = ((ushort)*(from-width) + (ushort)*(from+width)) / 2; \
     from++

#define DO_GREEN_BOTTOM_PIXEL \
    *to++ = *(from-width);                                       \
    *to++ = *from;                                               \
    *to++ = ((ushort)*(from-1) + (ushort)*(from+1)) / 2;         \
     from++

#define FLUSH_PIXELS \
    __copy_to_user( buf, rgb_buf, width * 3 );     \
    buf += width * 3; \
    result += width * 3

/* This function assumes that we have a completely full buffer */
static unsigned long write_full_rgb( struct capture_frame *frame, char *buf )
{
	unsigned char *from = frame->data;
	int width = frame->width;
	int height = frame->height;
	unsigned char rgb_buf[644 * 3];
	unsigned char *to;
	int i, j;
	unsigned long result = 0;

	/* Write the top row */
	to = rgb_buf;
	DO_RED_TOP_LEFT_PIXEL;
	for ( i = 1 ; i < width - 1 ; i+=2 ) {
		DO_GREEN_TOP_PIXEL;
		DO_RED_TOP_PIXEL;
	}
	DO_GREEN_TOP_RIGHT_PIXEL;
	FLUSH_PIXELS;

	for ( j = 1 ; j < height - 1 ; j += 2 ) {
		/* Write an even row */
		to = rgb_buf;
		DO_GREEN_LEFT_PIXEL;
		for ( i = 1 ; i < width - 1 ; i+= 2 ) {
			DO_BLUE_PIXEL;
			DO_GREEN_EVEN_PIXEL;
		}
		DO_BLUE_RIGHT_PIXEL;
		FLUSH_PIXELS;

		/* Write an odd row */
		to = rgb_buf;
		DO_RED_LEFT_PIXEL;
		for ( i = 1 ; i < width - 1 ; i+= 2 ) {
			DO_GREEN_ODD_PIXEL;
			DO_RED_PIXEL;
		}
		DO_GREEN_RIGHT_PIXEL;
		FLUSH_PIXELS;
	}
	
	/* Write the bottom row */

	to = rgb_buf;
	DO_GREEN_BOTTOM_LEFT_PIXEL;
	for ( i = 1 ; i < width - 1 ; i+= 2 ) {
		DO_BLUE_BOTTOM_PIXEL;
		DO_GREEN_BOTTOM_PIXEL;
	}
	DO_BLUE_BOTTOM_RIGHT_PIXEL;
	FLUSH_PIXELS;
	
	return result;
}

/* This function assumes that we're clipped in by one pixel on top and left */
static unsigned long write_inset_rgb( struct capture_frame *frame, char *buf )
{
	unsigned char *source = frame->data;
	int width = frame->width;
	int height = frame->height;
	int row_width = frame->row_width;

	unsigned char rgb_buf[644 * 3];
	unsigned char *to;
	unsigned char *from;
	unsigned long result = 0;
	int i, j;

	for ( j = 0 ; j < height ; ) {
		/* Write an even row */
		from = source + (++j) * row_width + 1;
		to = rgb_buf;
		for ( i = 0 ; i < width ; i += 2 ) {
			DO_BLUE_PIXEL;
			DO_GREEN_EVEN_PIXEL;
		}
		FLUSH_PIXELS;

		/* Write an odd row */
		from = source + (++j) * row_width + 1;
		to = rgb_buf;
		for ( i = 0 ; i < width ; i += 2 ) {
			DO_GREEN_ODD_PIXEL;
			DO_RED_PIXEL;
		}
		FLUSH_PIXELS;
	}
	return result;
}

static unsigned long write_inset_grey( struct capture_frame *frame, char *to )
{
	int row_width = frame->row_width;
	unsigned char *p = frame->data + 1 + row_width;
	int height = frame->height;
	int width = frame->width;
	int row;
	unsigned long result = 0;

	for ( row = 0 ; row < height ; row++ ) {
		__copy_to_user( to, p, width );
		p += row_width;
		result += width;
		to += width;
	}

	return result;
}

/* Write the state out into a buffer */
/* We've already checked that the buffer is of the required size and has been verified */

static long process_frame( struct h3600_camera_struct *cam, unsigned char *buf )
{
	struct capture_frame *frame = &cam->frame;
	unsigned long result = 0;

	switch (cam->mode) {
	case HC_MODE_RAW:
		__copy_to_user( buf, frame->data, HC_FRAME_BUFFER_SIZE );
		result = HC_FRAME_BUFFER_SIZE; 
		break;

	case HC_MODE_DECIMATION_1:
		frame->width = cam->vwin.width;
		frame->height = cam->vwin.height;
		if ( cam->vpic.palette == VIDEO_PALETTE_GREY )
			result = write_inset_grey(frame,buf);
		else
			result = write_inset_rgb(frame,buf);
		break;

	case HC_MODE_DECIMATION_4:
		decimate( frame );
		/* Intentional fall through */
	case HC_MODE_DECIMATION_2:
		decimate( frame );
		if ( cam->vpic.palette == VIDEO_PALETTE_GREY ) {
			result = frame->width * frame->height;
			__copy_to_user(buf, frame->data, result);
		}
		else
			result = write_full_rgb(frame,buf);
		break;
	}
	return result;
}


#define ENABLE_OPT_INT(x) \
		mercury_backpaq_socket->fpga_interrupt_mask &= ~(x)
#define DISABLE_OPT_INT(x) \
		mercury_backpaq_socket->fpga_interrupt_mask |= (x)
#define READ_OPT_INT(x) \
		mercury_backpaq_socket->fpga_interrupt_status & (x) \
                & ~mercury_backpaq_socket->fpga_interrupt_mask

#define ENABLE_VBLANK_INT   ENABLE_OPT_INT(BACKPAQ_SOCKET_INT_VBLANK)
#define DISABLE_VBLANK_INT  DISABLE_OPT_INT(BACKPAQ_SOCKET_INT_VBLANK)
#define READ_VBLANK_INT     READ_OPT_INT(BACKPAQ_SOCKET_INT_VBLANK)

#define ENABLE_FIFO_INT     ENABLE_OPT_INT(BACKPAQ_SOCKET_INT_FIFO)
#define DISABLE_FIFO_INT    DISABLE_OPT_INT(BACKPAQ_SOCKET_INT_FIFO)
#define READ_FIFO_INT       READ_OPT_INT(BACKPAQ_SOCKET_INT_FIFO)

#define ENABLE_BOTH_INT     ENABLE_OPT_INT(BACKPAQ_SOCKET_INT_FIFO | BACKPAQ_SOCKET_INT_VBLANK)
#define DISABLE_BOTH_INT    DISABLE_OPT_INT(BACKPAQ_SOCKET_INT_FIFO | BACKPAQ_SOCKET_INT_VBLANK)
#define READ_BOTH_INT       READ_OPT_INT(BACKPAQ_SOCKET_INT_FIFO | BACKPAQ_SOCKET_INT_VBLANK)

#define IMAGER_CTL_GAIN_FORMAT         0x70
#define IMAGER_CTL_POWER_SETTING       0x80

/* Wait until the interrupts have filled the frame */

static long grab_frame( struct h3600_camera_struct *cam )
{
	int retval;

	/* We set up for grabbing at true width and height */
	cam->frame.state         = GRAB_WAIT_FOR_REFRESH;  /* Waiting for top of frame */
	cam->frame.bytes_to_read = HC_FRAME_BUFFER_SIZE;
	cam->frame.width         = HC_TRUE_WIDTH;          
	cam->frame.height        = HC_TRUE_HEIGHT;
	cam->frame.row_width     = HC_TRUE_WIDTH;
	cam->frame.bytes_read    = 0;                      /* No data grabbed yet */

	/* Set up the camera for doing stuff */
//	mercury_backpaq_socket->camera_clock_divisor = 0x0100;  /* Clock to 5 fps      */
//	mercury_backpaq_socket->camera_writethru     = 0x8c;    /* Power level to 12   */
//	mercury_backpaq_socket->camera_writethru     = 0x70;    /* Sets 8 bpp gain = 0 */
//	mercury_backpaq_socket->camera_live_mode     = 0x01;    /* Turn on live video mode */
//	mercury_backpaq_socket->camera_interrupt_fifo = 0x20;   /* Interrupt with 32x8 bytes */

	mercury_backpaq_socket->camera_clock_divisor = cam->params.clock_divisor; 
	mercury_backpaq_socket->camera_writethru     
		= IMAGER_CTL_POWER_SETTING | cam->params.power_setting;
	mercury_backpaq_socket->camera_writethru     
		= IMAGER_CTL_GAIN_FORMAT | cam->params.gain_format;
	mercury_backpaq_socket->camera_interrupt_fifo = cam->params.interrupt_fifo;
	mercury_backpaq_socket->camera_integration_time = 512 - (cam->vpic.brightness / 128);
	mercury_backpaq_socket->camera_live_mode     = 0x01;    /* Turn on live video mode */

	/* Set up the grabbing state */
	cam->state = HC_V4L_GRABBING; 
	cam->fifo_high = 0;

/*	mercury_backpaq_socket->fpga_interrupt_clear = BACKPAQ_SOCKET_INT_VBLANK;*/
	/* Make sure FIFO interrupt is off */
	DISABLE_FIFO_INT;
	/* Turn on just the vertical retrace interrupt */
	ENABLE_VBLANK_INT;

	/* Sit on a wait queue until the GRAB is done */
	/* Wait until this is true */
	retval = wait_event_interruptible(cam->capq,
					  cam->state != HC_V4L_GRABBING 
					  || cam->frame.state == GRAB_DONE
		                          || cam->frame.state == GRAB_ERROR);

	/* Turn off all interrupts */
	DISABLE_BOTH_INT;
	cam->state = HC_V4L_IDLE;

	if ( retval < 0 )
		return retval;

/*	printk(KERN_ALERT __FILE__ ": Got image. Fifo maximum %d\n", cam->fifo_high ); */

	return cam->frame.bytes_read;
}

/******************************************************************************
 *
 * Interrupt routines
 *
 ******************************************************************************/

static void get_fpga_data( struct h3600_camera_struct *cam )
{
	struct capture_frame *frame = &cam->frame;
	unsigned short       *lbuf = (unsigned short *) (frame->data + frame->bytes_read);
	unsigned short        data_avail;
	int i, count;

	while ( (data_avail = mercury_backpaq_socket->camera_fifo_data_avail) > 0
		&& ( frame->bytes_read < frame->bytes_to_read )) {

		if ( data_avail > cam->fifo_high )
			cam->fifo_high = data_avail;

		count = data_avail * 8;
		if ( count + frame->bytes_read > frame->bytes_to_read )
			count = frame->bytes_to_read - frame->bytes_read;
		
		for ( i = 0 ; i < count ; i+=2 )
			*lbuf++ = mercury_backpaq_socket->camera_fifo_read;
		
		frame->bytes_read += i;
	}

	if ( frame->bytes_read == frame->bytes_to_read ) {
		frame->state = GRAB_DONE;
		wake_up_interruptible(&cam->capq);
	}
}

static void fpga_fifo_interrupt( struct h3600_camera_struct *cam )
{
	struct capture_frame *frame = &cam->frame;
	cam->fifo_count++;

	if ( frame->state != GRAB_ACTIVE ) {
/*		printk(KERN_ALERT __FILE__ ": fifo int on non-active %d\n", frame->state);
		printk(KERN_ALERT __FILE__ ": fifo mask = %x\n", 
		       mercury_backpaq_socket->fpga_interrupt_mask);
		printk(KERN_ALERT __FILE__ ": fifo status = %x\n", 
		       mercury_backpaq_socket->fpga_interrupt_status);
		       printk(KERN_ALERT __FILE__ ": read both = %x\n", READ_BOTH_INT ); */
		return; 
	}

	get_fpga_data( cam );
}

static void fpga_vblank_interrupt( struct h3600_camera_struct *cam )
{
	struct capture_frame *frame = &cam->frame;
	cam->vblank_count++;

	switch (frame->state) {
	case GRAB_WAIT_FOR_REFRESH:
/*		printk(KERN_ALERT __FILE__ ": Refresh event\n"); */
		frame->state = GRAB_ACTIVE;
		break;

	case GRAB_ACTIVE:  /* Suck in the last bit of data */
		get_fpga_data(cam);
		if ( frame->state != GRAB_DONE ) {
			printk(KERN_ALERT __FILE__ ": vblank byte count err\n");
			printk(KERN_ALERT __FILE__ ": Bytes received %d\n", frame->bytes_read);
			frame->state = GRAB_ERROR;
			wake_up_interruptible(&cam->capq);
		}
		break;

	case GRAB_DONE: /* Ignore these */
	case GRAB_ERROR:
		printk(KERN_ALERT __FILE__ ": vblank on done or error %d\n", frame->state);
		break;
	}
}

static void fpga_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	unsigned short irq_value = READ_BOTH_INT;
	struct h3600_camera_struct *cam = (struct h3600_camera_struct *) dev_id;
	
	if (!cam ) {
		printk(KERN_ALERT __FILE__ ": interrupt %d without valid cam\n",irq);
		return;
	}
	
	if ( irq != IRQ_GPIO_BITSY_OPT_IRQ ) {
		printk(KERN_ALERT __FILE__ ": Unknown IRQ %d\n",irq);
		return;
	}

	/* Read just the interrupts we care about */

/*	if ( irq_value & BACKPAQ_SOCKET_INT_FIFO ) { */
		DISABLE_FIFO_INT;
		fpga_fifo_interrupt(cam);
		ENABLE_FIFO_INT;
/*	} */

	if ( irq_value & BACKPAQ_SOCKET_INT_VBLANK )
		fpga_vblank_interrupt(cam);

	GEDR = GPIO_BITSY_OPT_IRQ; /* Clear the interrupt */
}


/******************************************************************************
 *
 * Video 4 Linux interface
 *
 ******************************************************************************/

static long h3600_camera_read( struct video_device *dev, char *buf, 
			       unsigned long count, int noblock )
{
	struct h3600_camera_struct *cam = (struct h3600_camera_struct *) dev;
	long retval;

#ifdef BACKPAQ_CAMERA_DEBUG
	printk( KERN_ALERT __FILE__ ": read called with len = %ld (%ld)\n", count, required_buf_size(cam) );
#endif
	/* We're ignoring "noblock" */
	/* Check for adequate buffer size */
	if ( count < required_buf_size( cam )) 
		return -EIO;

	if ( verify_area(VERIFY_WRITE, buf, count))
		return -EFAULT;

	/* Lock the camera - no one else can capture while I have this */
	down(&cam->lock);
	
	/* Grab the next frame and stash in the local buffer */
	retval = grab_frame( cam );
	
	/* Do color space processing and write it out */
	if ( retval > 0 ) 
		retval = process_frame( cam, buf );

	/* Release the mutex */
	up(&cam->lock);

	return retval;
}


static int h3600_camera_ioctl( struct video_device *dev, unsigned int cmd, void *arg )
{
	struct h3600_camera_struct *cam = (struct h3600_camera_struct *) dev;
	int retval = 0;

	switch(cmd) {
	case VIDIOCGCAP:     /* Get core camera capabilities */
	{
		struct video_capability b;
		strcpy(b.name, "iPAQ H3600 Mercury BackPAQ");
		b.type      = VID_TYPE_CAPTURE;
		b.channels  = 1;
		b.audios    = 0;
		b.maxwidth  = HC_TRUE_WIDTH;
		b.maxheight = HC_TRUE_HEIGHT;
		b.minwidth  = HC_MIN_ALLOWED_WIDTH;
		b.minheight = HC_MIN_ALLOWED_HEIGHT;
		if (copy_to_user(arg, &b,sizeof(b)))
			retval = -EFAULT;
		break;
	}
	case VIDIOCGCHAN:     /* Get channel info (sources) - We have just one channel */
	{
		struct video_channel v;
		if (copy_from_user(&v, arg, sizeof(v)))
			return -EFAULT;
		if (v.channel !=0)       /* We only have a single channel */
			return -EINVAL;
		v.tuners = 0;
		v.flags  = 0;
		v.type   = VIDEO_TYPE_CAMERA;
		v.norm   = 0;              /* What is this? */
		strcpy(v.name, "Camera");  /* Generic camera */
		if (copy_to_user(arg, &v, sizeof(v)))
			retval = -EFAULT;
		break;
	}
	case VIDIOCSCHAN:     /* Set channel - must be 0 */
	{
		int v;
		if (copy_from_user(&v, arg,sizeof(v)))
			retval = -EFAULT;
		else if (v!=0)
			retval = -EINVAL;
		break;
	}
	case VIDIOCGPICT:     /* Get picture properties */
	{
		if (copy_to_user(arg, &cam->vpic, sizeof(struct video_picture)))
			retval = -EFAULT;
		break;
	}
	case VIDIOCSPICT:      /* Set picture properties */
	{
		struct video_picture p;
		if (copy_from_user(&p, arg, sizeof(p))) {
			retval = -EFAULT;
			break;
		}

		/* For the moment, we force the following defaults */
		if (! ((p.depth == 24 && p.palette == VIDEO_PALETTE_RGB24)
		       || (p.depth == 8 && p.palette == VIDEO_PALETTE_GREY)
		       || (p.depth == 8 && p.palette == VIDEO_PALETTE_RAW ))) {
			retval = -EINVAL;
			break;
		}
			
		/* Load the camera */
		down(&cam->lock);			
		cam->vpic.brightness = p.brightness;
		cam->vpic.depth = p.depth;
		cam->vpic.palette = p.palette;
		/* Fix camera resolution */
		set_camera_resolution( cam, cam->vwin.width, cam->vwin.height ); 
		up(&cam->lock);
		break;
	}
	case VIDIOCGWIN:       /* Get the video capture window */
	{
		if (copy_to_user(arg, &cam->vwin, sizeof(struct video_window)))
			retval = -EFAULT;
		break;
	}
	case VIDIOCSWIN:       /* Set the capture area */
	{
		struct video_window vw;

		if (copy_from_user(&vw, arg,sizeof(vw))) {
			retval = -EFAULT;
			break;
		}

		if (vw.clipcount != 0) {    /* clipping not supported */
			retval = -EINVAL;
			break;
		}

		if (vw.clips != NULL) {     /* clipping not supported */
			retval = -EINVAL;
			break;
		}

		/* In raw mode, you can't change the resolution */
		if ( cam->vpic.palette == VIDEO_PALETTE_RAW )
			break;

		if (vw.height < HC_MIN_ALLOWED_HEIGHT 
		    || vw.height > HC_TRUE_HEIGHT
		    || vw.width < HC_MIN_ALLOWED_WIDTH 
		    || vw.width > HC_TRUE_WIDTH) {
			retval = -EINVAL;
			break;
		}
				
		/* Fix the camera resolution */
		down(&cam->lock);
		set_camera_resolution( cam, vw.width, vw.height );
		up(&cam->lock);
		break;
	}
	/* Private interface */
	case H3600CAM_G_PARAMS:
	{
		if (copy_to_user(arg, &cam->params, sizeof(struct h3600_backpaq_camera_params)))
			retval = -EFAULT;
		break;
	}
	case H3600CAM_S_PARAMS:
	{ 
		struct h3600_backpaq_camera_params params;
		if (copy_from_user(&params, arg, sizeof(params))) {
			retval = -EFAULT;
			break;
		}
		/* Some sanity checking */
		if (params.clock_divisor < 16   /* About 160 Hz */ 
			|| params.interrupt_fifo > 255
			|| params.power_setting > 15
			|| params.gain_format > 5 ) {
			retval = -EINVAL; 
			break;
		}
		down(&cam->lock);
		cam->params.clock_divisor = params.clock_divisor & 0xfffe; /* make even */
		cam->params.interrupt_fifo = params.interrupt_fifo;
		cam->params.power_setting = params.power_setting;
		cam->params.gain_format = params.gain_format;
		up(&cam->lock);
		break;
	}
		
	/* mmap interface */
	case VIDIOCGMBUF:
	{
		struct video_mbuf vm;

		memset(&vm, 0, sizeof(vm));
/*		vm.size = CPIA_MAX_FRAME_SIZE*FRAME_NUM; */
/*		vm.frames = FRAME_NUM;*/
/*		for (i = 0; i < FRAME_NUM; i++) */
/*			vm.offsets[i] = CPIA_MAX_FRAME_SIZE * i;*/

		if (copy_to_user((void *)arg, (void *)&vm, sizeof(vm)))
			retval = -EFAULT;
		break;
	}

	case VIDIOCMCAPTURE:
	{
		struct video_mmap vm;
/*		int video_size; */

		if (copy_from_user((void *)&vm, (void *)arg, sizeof(vm))) {
			retval = -EFAULT;
			break;
		}
/*		if (vm.frame<0||vm.frame>FRAME_NUM) { */
/*			retval = -EINVAL;*/
/*			break;*/
/*		} */

		/* We should set all of the properties and begin capturing */
		break;
	}
	
	case VIDIOCSYNC:
	{
		int frame;

		if (copy_from_user((void *)&frame, arg, sizeof(int))) {
			retval = -EFAULT;
			break;
		}
/*
		if (frame<0 || frame >= FRAME_NUM) {
			retval = -EINVAL;
			break;
		}
*/
		break;
	}
	/* We don't implement overlay with this camera */
	case VIDIOCCAPTURE:
		retval = -EINVAL;
		break;
	case VIDIOCGFBUF:
		retval = -EINVAL;
		break;
	case VIDIOCSFBUF:
		retval = -EINVAL;
		break;
	case VIDIOCKEY:
		retval = -EINVAL;
		break;

		/* We have no tuner interface */
	case VIDIOCGTUNER:
		retval = -EINVAL;
		break;
	case VIDIOCSTUNER:
		retval = -EINVAL;
		break;
	case VIDIOCGFREQ:
		retval = -EINVAL;
		break;
	case VIDIOCSFREQ:
		retval = -EINVAL;
		break;

		/* We have no audio interface */
	case VIDIOCGAUDIO:
		retval = -EINVAL;
		break;
	case VIDIOCSAUDIO:
		retval = -EINVAL;
		break;
	default:
		retval = -ENOIOCTLCMD;
		break;
	}

	return retval;
}

static int h3600_camera_mmap(struct video_device *dev, const char *adr,
			     unsigned long size)
{
	return -ENODEV;
}

static int h3600_camera_open( struct video_device *dev, int flags )
{
	struct h3600_camera_struct *cam = (struct h3600_camera_struct *) dev;
	int retval = 0;

	if ( mercury_backpaq_sysctl->hardware_version != BACKPAQ_HARDWARE_VERSION_1 ) {
		printk(KERN_ALERT __FILE__ ": Unable to find backpaq\n");
		return -ENXIO;
	}

	if ( !(mercury_backpaq_sysctl->fpga_status & BACKPAQ_FPGASTATUS_DONE )) {
		printk(KERN_ALERT __FILE__ ": Backpaq FPGA not programmed\n");
		return -EIO;
	}

	down(&cam->lock);
	if ( cam->usage_count++ == 0 ) {    /* Set up interrupts */
		unsigned long flags;
		save_flags_cli(flags);

		GPDR &= ~GPIO_BITSY_OPT_IRQ;    /* GPIO line as input */
		set_GPIO_IRQ_edge( GPIO_BITSY_OPT_IRQ, GPIO_RISING_EDGE );  /* Rising edge */

		retval = request_irq(IRQ_GPIO_BITSY_OPT_IRQ,
				     fpga_interrupt,
				     SA_SHIRQ | SA_INTERRUPT | SA_SAMPLE_RANDOM,
				     "Backpaq FPGA", (void *)dev);
		restore_flags(flags);
	}
	up(&cam->lock);

	MOD_INC_USE_COUNT;
	return retval;
}

static void h3600_camera_close( struct video_device *dev )
{
	struct h3600_camera_struct *cam = (struct h3600_camera_struct *) dev;

	down(&cam->lock);
	if ( --cam->usage_count == 0 ) {
		unsigned long flags;
		save_flags_cli(flags);
		free_irq(IRQ_GPIO_BITSY_OPT_IRQ, (void *)dev);
		restore_flags(flags);
	}
	up(&cam->lock);
	MOD_DEC_USE_COUNT;
}


int h3600_camera_video_init(struct video_device *vdev)
{
#ifdef CONFIG_PROC_FS
/*	create_proc_cpia_cam(vdev->priv);*/
#endif
	return 0;
}

static struct video_device h3600_camera_template =
{
	owner:		THIS_MODULE,
	name:		"iPAQ H3600 Mercury BackPAQ",
	type:		VID_TYPE_CAPTURE,
	hardware:	VID_HARDWARE_H3600_BACKPAQ,
	open:		h3600_camera_open,
	close:		h3600_camera_close,
	read:		h3600_camera_read,
	ioctl:		h3600_camera_ioctl,
	mmap:		h3600_camera_mmap, 
	initialize:	h3600_camera_video_init,
};

/******************************************************************************
 *
 * Standard initialization
 *
 * We should add a check to see if the camera responds and if it is B/W or color
 *
 ******************************************************************************/

static int h3600_camera_startup( struct h3600_camera_struct *cam )
{
	memset(cam, 0, sizeof(struct h3600_camera_struct));
	memcpy(&cam->vdev, &h3600_camera_template, sizeof(h3600_camera_template));

	cam->frame.data = vmalloc( HC_FRAME_BUFFER_SIZE );
	if ( !cam->frame.data ) {
		printk(KERN_ALERT __FILE__ ": Unable to allocate frame buffer\n");
		return -ENOBUFS;
	}

 	init_MUTEX(&cam->lock);
        init_waitqueue_head(&cam->capq);

	/* Set up some plausible defaults */
	cam->state           = HC_V4L_IDLE;
	cam->vpic.palette    = VIDEO_PALETTE_RGB24;
	cam->vpic.depth      = 24;
	cam->vpic.brightness = 0x8000;
	set_camera_resolution( cam, 10000, 10000 );   /* Will set to 640 x 480 */

	cam->params.clock_divisor  = 0x100;   /* 5 fps default */
	cam->params.power_setting  = 0xc;     /* Normally "12" */
	cam->params.gain_format    = 0x0;     /* 8 bpp, gain = 0 */
	cam->params.interrupt_fifo = 0x20;    /* 32 deep (32x8 bytes) */

	return 0;
}

static int h3600_camera_shutdown( struct h3600_camera_struct *cam )
{
	/* Toss the interrupt routine if we've messed up a shutdown somewhere */
	if ( cam->usage_count > 0 ) {
		unsigned long flags;
		save_flags_cli(flags);
		free_irq(IRQ_GPIO_BITSY_COM_DCD, (void *)cam);
		restore_flags(flags);
	}

	/* Should we kill the wait queue? */

	if ( cam->frame.data)
		vfree( cam->frame.data );

	return 0;
}



/******************************************************************************
 *
 * /proc interface
 *
 ******************************************************************************/

#ifdef CONFIG_PROC_FS

static char *palette_names[] = {
	"","Grey","HI240","RGB565",
	"RGB24","RGB32","RGB444","YUV422",
	"YUYV","UYVY","YUV420","YUV411",
	"RAW","YUV422P","YUV411P","YUV420P",
	"YUV410P"
};

static int h3600_camera_read_proc( char *page, char **start, off_t off,
				   int count, int *eof, void *data )
{
	struct h3600_camera_struct *cam = &hc_camera;
	char *out = page;
	int len;

	out += sprintf(out, "%s\n", BANNER);
	out += sprintf(out, "CMOS Image Size %d %d\n", HC_TRUE_WIDTH, HC_TRUE_HEIGHT);
	out += sprintf(out, "Capture window\n");
	out += sprintf(out, "  x          : %d\n",cam->vwin.x);
	out += sprintf(out, "  y          : %d\n",cam->vwin.y);
	out += sprintf(out, "  width      : %d\n",cam->vwin.width);
	out += sprintf(out, "  height     : %d\n",cam->vwin.height);
	out += sprintf(out, "Image properties\n");
	out += sprintf(out, "  brightness : 0x%04x\n",cam->vpic.brightness);
	if ( cam->vpic.palette != VIDEO_PALETTE_GREY ) {
		out += sprintf(out, "  hue        : 0x%04x\n",cam->vpic.hue);
		out += sprintf(out, "  colour     : 0x%.4x\n",cam->vpic.colour);
	}
	out += sprintf(out, "  contrast   : 0x%04x\n",cam->vpic.contrast);
	if ( cam->vpic.palette == VIDEO_PALETTE_GREY ) {
		out += sprintf(out, "  whiteness  : 0x%04x\n",cam->vpic.whiteness);
	}
	out += sprintf(out, "  depth      : %d\n",cam->vpic.depth);
	out += sprintf(out, "  palette    : %s\n",palette_names[cam->vpic.palette]);
	out += sprintf(out, "Imager params counters\n");
	out += sprintf(out, "  clk divide : %d (%d Hz)\n", cam->params.clock_divisor,
		       1280 / cam->params.clock_divisor);
	out += sprintf(out, "  intr fifo  : %d\n", cam->params.interrupt_fifo);
	out += sprintf(out, "  power set  : %d\n", cam->params.power_setting);
	out += sprintf(out, "  gain fmt   : %d\n", cam->params.gain_format);
	out += sprintf(out, "Internal counters\n");
	out += sprintf(out, "  vblank     : %d\n",cam->vblank_count);
	out += sprintf(out, "  fifo       : %d\n",cam->fifo_count);
		    
	len = out - page;
	len -= off;
	if (len < count) {
		*eof = 1;
		if (len <= 0) return 0;
	} else
		len = count;

	*start = page + off;
	return len;

}


#endif /* CONFIG_PROC_FS */

/******************************************************************************
 *
 * Module interface
 *
 ******************************************************************************/

#ifdef MODULE
static int __init h3600_camera_init( void )
{
	int retval = 0;

	printk(KERN_INFO BANNER "\n");

	retval = h3600_camera_startup( &hc_camera );
	if ( retval )
		return retval;

	if (video_register_device(&hc_camera.vdev, VFL_TYPE_GRABBER)) {
		printk(KERN_ERR __FILE__ "Unable to H3600 BackPAQ camera\n");
		return -ENODEV;
	}

	/* Initialize proc file system...this should be a subdirectory */
	proc_camera = create_proc_entry("backpaqcam", 0, 0);    
	if (proc_camera)    
		proc_camera->read_proc = h3600_camera_read_proc;    
	else 
		printk(KERN_ALERT __FILE__ ": unable to create proc entry\n");

	return 0;  /* No error */
}

static void __exit h3600_camera_cleanup( void )
{
	if (proc_camera)
		remove_proc_entry("backpaqcam", 0);

	h3600_camera_shutdown( &hc_camera );

	video_unregister_device(&hc_camera.vdev);
}

module_init(h3600_camera_init);
module_exit(h3600_camera_cleanup);

#endif /* MODULE */
