/*
 * Copyright (c) 2001 Silicon Graphics, Inc.  All Rights Reserved.
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it would be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * 
 * Further, this software is distributed without any warranty that it is
 * free of the rightful claim of any third person regarding infringement
 * or the like.  Any license provided herein, whether implied or
 * otherwise, applies only to this software file.  Patent licenses, if
 * any, provided herein do not apply to combinations of this program with
 * other software, or any other product whatsoever.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write the Free Software Foundation, Inc., 59
 * Temple Place - Suite 330, Boston MA 02111-1307, USA.
 * 
 * Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
 * Mountain View, CA  94043, or:
 * 
 * http://www.sgi.com 
 * 
 * For further information regarding this notice, see: 
 * 
 * http://oss.sgi.com/projects/GenInfo/SGIGPLNoticeExplan/
 */

#include <linux/mm.h>
#include <linux/errno.h>
#include <linux/file.h>
#include <linux/smp_lock.h>
#include <linux/slab.h>
#include <linux/attributes.h>
#include <asm/uaccess.h>

/*
 * Revalidate the inode. This is required for proper NFS attribute caching.
 * Blatently copied wholesale from fs/stat.c
 */
static __inline__ int
do_revalidate(struct dentry *dentry)
{
        struct inode * inode = dentry->d_inode;
        if (inode->i_op && inode->i_op->revalidate)
                return inode->i_op->revalidate(dentry);
        return 0;
}


asmlinkage long sys_attrctl(attr_obj_t obj, int type, attr_op_t *ops, int count)
{
	int		error = 0;
	struct inode	*inode;
	struct dentry	*dentry;
	struct file	*f = NULL;
	struct nameidata nd;

	if (count <= 0)
		return -EINVAL;

	lock_kernel();

	switch (type) {
	case ATTR_TYPE_FD:
		if (! (f = fget(obj.fd))) {
			error = -ENOENT;
			goto unlock;
		}
		dentry = f->f_dentry;
		break;

	case ATTR_TYPE_PATH:
		/* follow symlinks */
		error = user_path_walk(obj.path, &nd);
		if (error)
			goto unlock;
		dentry = nd.dentry;
		break;

	case ATTR_TYPE_LPATH:
		error = user_path_walk_link(obj.path, &nd);
		if (error)
			goto unlock;
		dentry = nd.dentry;
		break;

	case ATTR_TYPE_PID:
		error = -ENOSYS;
		goto unlock;

	default:
		error = -EINVAL;
		goto unlock;
	}

	inode = dentry->d_inode;
	error = -EINVAL;

	if (   inode->i_op
	    && inode->i_op->attrctl
	    && ! (error = do_revalidate(dentry)))
	{
		attr_op_t kop, *kops;

		if (count == 1) {
			kops = &kop;
		}
		else {
			kops = (attr_op_t *) kmalloc(count * sizeof(attr_op_t),
						     GFP_KERNEL);
			if (! kops) {
				error = -ENOMEM;
				goto release;
			}
		}

		if (copy_from_user(kops, ops, count * sizeof(attr_op_t)) != 0) {
			error = -EFAULT;
			goto free_mem;
		}
		
		UPDATE_ATIME(inode);
		error = inode->i_op->attrctl(inode, kops, count);

		if (copy_to_user(ops, kops, count * sizeof(attr_op_t)) != 0) {
			error = -EFAULT;
			goto free_mem;
		}

	free_mem:
		if (count > 1)
			kfree(kops);

	}

 release:
	(type == ATTR_TYPE_FD) ? fput(f) : path_release(&nd);
 unlock:
	unlock_kernel();
	return error;
}
