#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>
#include <fcntl.h>
#include <string.h>
#include <signal.h>
#include <pwd.h>
#include <grp.h>

#include <gtk/gtk.h>
#include <glade/glade.h>
#include <glib.h>

#include <gnome-xml/tree.h>
#include <gnome-xml/parser.h>

#include "globals.h"

extern struct _app_globals app_globals;

void cancel_op (GtkButton * button);	/* external prototype */
void display_directory ();	/* prototype */

void
new_prog_win ()
{
  GtkWidget *win;
  GtkWidget *box;
  GtkWidget *lbl;
  GtkWidget *temp;
  GtkWidget *btn;

  box = gtk_vbox_new (FALSE, 0);
  temp = gtk_label_new ("Please wait...");
  lbl = gtk_label_new (g_strdup (""));
  btn = gtk_button_new_with_label ("Cancel");

  gtk_signal_connect (GTK_OBJECT (btn), "clicked", cancel_op, NULL);

  gtk_box_pack_start (GTK_BOX (box), temp, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (box), lbl, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (box), btn, FALSE, TRUE, 0);

  win = gtk_window_new (GTK_WINDOW_POPUP);
  gtk_container_add (GTK_CONTAINER (win), box);

  app_globals.win_curfile = win;
  app_globals.label_curfile = lbl;

  gtk_widget_show_all (win);
  gtk_widget_hide (win);
}


gchar *pretty_print_path (gchar *path) {
	GtkStyle *style;
	gint twidth;
	gint i = 0;

	style = gtk_widget_get_style(app_globals.win_main);
	if (style == NULL) {
		printf("WARNING: unable to get current style.\n");
		return NULL;
	}

	twidth = gdk_text_width(style->font, &path[i], strlen(&path[i]));
	while (twidth > MAX_TITLE_WIDTH) { 
		i++;
		while (path[i] != '/' && path[i] != '\0') i++;
		twidth = gdk_text_width(style->font, &path[i], strlen(&path[i]));
	}

	if (i == 0) return g_strdup(path);
	
	return g_strconcat("..", &path[i], NULL);
}


void
change_directory (gchar * path)
{
  char buf[256];
	char *old;

  chdir (path);
	g_free(app_globals.c_dir);
  app_globals.c_dir = g_strdup (getcwd (buf, 255));

	old = app_globals.c_dir_title;
	app_globals.c_dir_title = pretty_print_path(app_globals.c_dir);
	
	gtk_window_set_title
					(GTK_WINDOW(app_globals.win_main), app_globals.c_dir_title);

	g_free(old);
}

void
save_selection_delete_helper (gpointer data, gpointer userdata)
{
  g_free (data);
}

void
save_selection_insert_helper (gpointer data, gpointer userdata)
{
  gchar *buf;
  gtk_clist_get_text (GTK_CLIST (app_globals.clist_files), (gint) data, 1,
		      &buf);
  if (strcmp (buf, "..") == 0)
    return;			/* skip the .. entry */
  app_globals.sel_list = g_list_append (app_globals.sel_list, g_strdup (buf));
}

void
save_selection_list ()
{
  /* remove old selection list */
  g_list_foreach (app_globals.sel_list, save_selection_delete_helper, NULL);
  g_list_free (app_globals.sel_list);
  app_globals.sel_list = NULL;

  /* save selection path */
  g_free (app_globals.sel_path);

  app_globals.sel_path = g_strdup (app_globals.c_dir);

  /* insert all (g_strdup them) */
  g_list_foreach (GTK_CLIST (app_globals.clist_files)->selection,
		  save_selection_insert_helper, NULL);
}

void
file_delete_helper (gpointer data, gpointer userdata)
{
  gchar *buf;
  gtk_clist_get_text (GTK_CLIST (app_globals.clist_files), (gint) data, 1,
		      &buf);
  if (buf == NULL)
    return;
  if (buf[0] == '.' && buf[1] == '.')
    return;
  printf ("RDEL %s\n", buf);
  rdel (buf);
}

/* called into by the delete prompt dialog */
void
real_handle_delete ()
{
  /* modal progress box */
  gtk_widget_show (app_globals.win_curfile);
  app_globals.cancel_op = FALSE;

  /* delete each selected file */
  g_list_foreach (GTK_CLIST (app_globals.clist_files)->selection,
		  file_delete_helper, NULL);

  display_directory ();

  gtk_widget_hide (app_globals.win_curfile);
}

void
handle_delete ()
{
  /* ask user if OK to delete files */
  gtk_widget_show (app_globals.win_promptdel);
}

void
paste_helper (gpointer data, gpointer userdata)
{
  gchar *src;
  gchar *dst;

  src = g_strconcat (app_globals.sel_path, "/", data, NULL);
  dst = g_strconcat (app_globals.c_dir, "/", data, NULL);

  if (app_globals.sel_op == OP_COPY) {
    printf ("RCOPY %s -> %s\n", src, dst);
    rcopy (src, dst);
  }

  if (app_globals.sel_op == OP_CUT) {
    printf ("RMOVE %s -> %s\n", src, dst);
    rmove (src, dst);
  }

  g_free (src);
  g_free (dst);
}

gchar *
get_ext (gchar * filename)
{
  gint i;
  gint last;
  last = i = strlen (filename) - 1;
  while (i > 0 && filename[i] != '.')
    i--;

  if (i == 0)
    return &(filename[last + 1]);

  i++;				/* skip the '.' */
  return &(filename[i]);
}

void
handle_paste ()
{
  /* bail if nothing is selected */
  if (app_globals.sel_list == NULL)
    return;

  gtk_widget_show (app_globals.win_curfile);
  app_globals.cancel_op = FALSE;

  /* for each item in selection list */
  g_list_foreach (app_globals.sel_list, paste_helper, NULL);

  /* remove old selection list */
  g_list_foreach (app_globals.sel_list, save_selection_delete_helper, NULL);
  g_list_free (app_globals.sel_list);
  app_globals.sel_list = NULL;

  display_directory ();

  gtk_widget_hide (app_globals.win_curfile);
}

/* param can be NULL. otherwise it replaces the %f in the cmd */ 
void
run_cmd (gchar *cmd, gchar *param) {
	gchar **arglist;
	gchar **temp;
	gint i;

		temp = g_strsplit(cmd, " ", 0);	
		
		/* count arguments */
		i = 0;
		while (temp[i] != NULL) i++;
		arglist = g_malloc(i * sizeof(gchar*));

		i = 0;
		while (temp[i] != NULL) {
			if (temp[i][0] == '%' && temp[i][1] == 'f' && param != NULL) {
				arglist[i] = param;	
			} else {
				arglist[i] = temp[i];
			}
			i++;
		}
	
		gnome_execute_async_fds(NULL, i, arglist, TRUE);
		
		g_strfreev(temp);
		g_free(arglist);	
}

/* force_cmd is set when the normal ext is being overridden */
/* only with open_with for the moment */
/* FIXME, change this to allow filename vectors? */
void
handle_file_verb (gchar * filename, gchar * force_cmd)
{
  struct stat statbuf;
  gchar *appname;
  gint err;

  /* stat the file, is it a directory or regular file? */
  err = stat (filename, &statbuf);

  if (err != 0)
    return;

  if (S_ISDIR (statbuf.st_mode)) {
    change_directory (filename);
    display_directory ();
    return;
  }

  if (S_ISREG (statbuf.st_mode)) {
    if (force_cmd == NULL)
      appname = g_hash_table_lookup (app_globals.ext_map, get_ext (filename));
    else
      appname = force_cmd;

    /* FIXME, alert the user, if their file type isn't runnable */
    if (appname == NULL)
      return;

     run_cmd(appname,filename); 
  }
}

void
shutdown_app ()
{
  gtk_main_quit ();
}

void
show_creation_dialog (int type)
{
  app_globals.newfile_state = type;
  if (type == NEW_FILE)
    gtk_label_set_text (GTK_LABEL (app_globals.label_newfile),
			"Choose a name for your new file");
  if (type == NEW_DIR)
    gtk_label_set_text (GTK_LABEL (app_globals.label_newfile),
			"Choose a name for your new directory");

  gtk_widget_show (app_globals.win_newfile);
}

void
select_for_copy ()
{
  app_globals.sel_op = OP_COPY;
  save_selection_list ();
}

void
select_for_cut ()
{
  app_globals.sel_op = OP_CUT;
  save_selection_list ();
}

void
hide_creation_dialog (int type)
{
  int err;
  gchar *temp = gtk_entry_get_text (GTK_ENTRY (app_globals.entry_newfile));
  if (type == NEW_FILE) {
    err = creat (temp, 0660);
    if (err == -1) {
      /* FIXME, proper error reporting here */
      perror ("unable to creat");
    }
  }

  if (type == NEW_DIR) {
    err = mkdir (temp, 0770);
    if (err == -1) {
      /* FIXME - proper error handling */
      perror ("unable to mkdir");
    }
  }

  display_directory ();
  gtk_widget_hide (app_globals.win_newfile);
}


int
directory_filter (const struct dirent *d)
{
  /* filter out the current directory */
  if (d->d_name[0] == '.' && d->d_name[1] == '\0')
    return 0;

  return 1;			/* no filters for now */
}

void
add_dir_entry (gchar * name, gchar * misc, GdkPixmap * img, GdkBitmap * mask)
{
  gchar *list_row[3];
  gint pos;
  GtkCList *clist;

  clist = GTK_CLIST (app_globals.clist_files);

  list_row[0] = "";
  list_row[1] = name;
  list_row[2] = misc;

  gtk_clist_append (clist, list_row);
  pos = clist->rows - 1;
  gtk_clist_set_pixmap (clist, pos, 0, img, mask);
}

void
display_directory ()
{
  struct dirent **namelist;
  long count, i, pos;
  int err;
  off_t filesize;
  gchar *tempbuf;
  gchar sizebuf[256];
  struct stat statbuf;

  /* remove old selection state */
  app_globals.sel_all = FALSE;

  gtk_clist_freeze (GTK_CLIST (app_globals.clist_files));
  gtk_clist_clear (GTK_CLIST (app_globals.clist_files));

  /* note: we always scan the working directory "." */
  count = scandir (".", &namelist, directory_filter, alphasort);

  /* pass 1 - directories */
  pos = -1;			/* current row of clist */
  for (i = 0; i < count; i++) {
    err = lstat (namelist[i]->d_name, &statbuf);
    if (err != 0 || !S_ISDIR (statbuf.st_mode))
      continue;

    add_dir_entry (namelist[i]->d_name, "",
		   app_globals.img_dir, app_globals.img_dir_mask);
  }

  /* pass 2 - other files */
  for (i = 0; i < count; i++) {
    err = lstat (namelist[i]->d_name, &statbuf);
    if (err == 0) {
      if (S_ISREG (statbuf.st_mode)) {
	filesize = statbuf.st_size;
	snprintf (sizebuf, 19, "%d", filesize);
	add_dir_entry (namelist[i]->d_name, sizebuf,
		       app_globals.img_file, app_globals.img_file_mask);
      }

      if (S_ISFIFO (statbuf.st_mode)) {
	add_dir_entry (namelist[i]->d_name, "",
		       app_globals.img_fifo, app_globals.img_fifo_mask);
      }

      if (S_ISBLK (statbuf.st_mode) || S_ISCHR (statbuf.st_mode)
	  || S_ISSOCK (statbuf.st_mode)) {
	add_dir_entry (namelist[i]->d_name, "",
		       app_globals.img_dev, app_globals.img_dev_mask);
      }

      if (S_ISLNK (statbuf.st_mode)) {
	err = readlink (namelist[i]->d_name, sizebuf, 256);
	if (err == -1)
	  sizebuf[0] = '\0';
	else
	  sizebuf[err] = '\0';

	tempbuf = g_strconcat ("->", sizebuf, NULL);
	add_dir_entry (namelist[i]->d_name, tempbuf,
		       app_globals.img_lnk, app_globals.img_lnk_mask);
	g_free (tempbuf);
      }

    }

    g_free (namelist[i]);
  }

  free (namelist);

  /* do not allow the user to select the ".." entry */
  gtk_clist_set_selectable (GTK_CLIST (app_globals.clist_files), 0, FALSE);

  gtk_clist_unselect_all (GTK_CLIST (app_globals.clist_files));
  gtk_clist_columns_autosize (GTK_CLIST (app_globals.clist_files));
  gtk_clist_thaw (GTK_CLIST (app_globals.clist_files));
}

void
load_ext_map (gchar * file, GHashTable * type_map, GHashTable * ext_map)
{
  xmlDocPtr doc;
  xmlNodePtr node;
  gchar *ext;
  gchar *mimetype;
  gchar *cmd;
  gpointer oldkey;
  gpointer oldval;

  doc = xmlParseFile (file);
  if (doc == NULL)
    return;

  for (node = doc->root->childs; node != NULL; node = node->next) {
    oldkey = NULL;
    oldval = NULL;
    ext = g_strdup (xmlGetProp (node, "ext"));
    mimetype = xmlGetProp (node, "type");
    cmd = g_hash_table_lookup (type_map, mimetype);

    /* free old key/val pairs so we don't leak */
    g_hash_table_lookup_extended (ext_map, ext, &oldkey, &oldval);
    if (oldkey != NULL)
      g_hash_table_remove (ext_map, oldkey);
    if (oldkey != NULL)
      g_free (oldkey);
    if (oldval != NULL)
      g_free (oldval);

    g_hash_table_insert (ext_map, g_strdup (ext), g_strdup (cmd));
  }

  xmlFreeDoc (doc);
}

void
load_app_map (gchar * file, GHashTable * type_map, GHashTable * app_map)
{
  xmlDocPtr doc;
  xmlNodePtr node;
  gchar *mimetype;
  gchar *app;
  gchar *cmd;
  gpointer oldkey;
  gpointer oldval;

  doc = xmlParseFile (file);
  if (doc == NULL)
    return;

  for (node = doc->root->childs; node != NULL; node = node->next) {
    oldkey = NULL;
    oldval = NULL;
    mimetype = xmlGetProp (node, "type");
    app = xmlGetProp (node, "name");
    cmd = xmlGetProp (node, "cmd");

    /* so we don't leak */
    g_hash_table_lookup_extended (type_map, mimetype, &oldkey, &oldval);
    if (oldkey != NULL)
      g_hash_table_remove (type_map, oldkey);
    if (oldkey != NULL)
      g_free (oldkey);
    if (oldval != NULL)
      g_free (oldval);

    /* insert type -> cmd */
    g_hash_table_insert (type_map, g_strdup (mimetype), g_strdup (cmd));

    /* so we don't leak */
    g_hash_table_lookup_extended (app_map, app, &oldkey, &oldval);
    if (oldkey != NULL)
      g_hash_table_remove (app_map, oldkey);
    if (oldkey != NULL)
      g_free (oldkey);
    if (oldval != NULL)
      g_free (oldval);

    /* insert app -> cmd */
    g_hash_table_insert (app_map, g_strdup (app), g_strdup (cmd));
  }

  xmlFreeDoc (doc);
}


void
load_mime_mappings ()
{
  GHashTable *type_to_cmd;	/* temporary table */
  GHashTable *app_to_cmd;
  GHashTable *ext_to_cmd;
  gchar *homepath;
  gchar *override;

  homepath = getenv ("HOME");
  if (homepath == NULL)
    homepath = "/root";

  type_to_cmd = g_hash_table_new (g_str_hash, g_str_equal);
  app_to_cmd = g_hash_table_new (g_str_hash, g_str_equal);
  load_app_map (APP_MAP_PATH, type_to_cmd, app_to_cmd);

  override = g_strconcat (homepath, "/.app_map.xml", NULL);
  load_app_map (override, type_to_cmd, app_to_cmd);
  g_free (override);

  ext_to_cmd = g_hash_table_new (g_str_hash, g_str_equal);
  load_ext_map (EXT_MAP_PATH, type_to_cmd, ext_to_cmd);

  override = g_strconcat (homepath, "/.ext_map.xml", NULL);
  load_ext_map (override, type_to_cmd, ext_to_cmd);
  g_free (override);

  app_globals.ext_map = ext_to_cmd;
  app_globals.app_map = app_to_cmd;
}

void
open_with_helper (gpointer key, gpointer value, gpointer userdata)
{
  gchar *rows[2];
  gint pos = GTK_CLIST (app_globals.clist_runwith)->rows;
  GdkBitmap *mask = app_globals.img_dir_mask; /* FIXME */ 

  rows[0] = "";
  rows[1] = key;

  /* FIXME - use associated application image instead */

  gtk_clist_append (GTK_CLIST (app_globals.clist_runwith), rows);
  gtk_clist_set_pixmap (GTK_CLIST (app_globals.clist_runwith),
			pos, 0, app_globals.img_dir, mask);
}

void
real_handle_open_with ()
{
  gint row;
  gchar *file;
  gchar *app;
  gchar *cmd;

	/* get selected file */
  row = (gint)GTK_CLIST (app_globals.clist_files)->selection->data;
  gtk_clist_get_text (GTK_CLIST (app_globals.clist_files), row, 1, &file);

	/* get selected application */
  row = (gint)GTK_CLIST (app_globals.clist_runwith)->selection->data;
  gtk_clist_get_text (GTK_CLIST (app_globals.clist_runwith), row, 1, &app);
	cmd = g_hash_table_lookup(app_globals.app_map, app);
	
	printf("User selected: open %s with %s (%s)\n", file, app, cmd);
  
  handle_file_verb (file, cmd);
  gtk_widget_hide (app_globals.win_open_with);
}

void
handle_open_with ()
{
  /* ensure that exactly one file is selected */
  if (GTK_CLIST (app_globals.clist_files)->selection == NULL)
    return;
  if (GTK_CLIST (app_globals.clist_files)->selection->next != NULL)
    return;

  /* populate clist with known applications */
  gtk_clist_freeze (GTK_CLIST (app_globals.clist_runwith));
  gtk_clist_clear (GTK_CLIST (app_globals.clist_runwith));
  g_hash_table_foreach (app_globals.app_map, open_with_helper, NULL);
  gtk_clist_columns_autosize (GTK_CLIST (app_globals.clist_runwith));
  gtk_clist_thaw (GTK_CLIST (app_globals.clist_runwith));
  gtk_widget_show (app_globals.win_open_with);
}

void populate_uid_gid() {
	struct group *grp;
	struct passwd *pwd;
	GList *groups = NULL;
	GList *users = NULL;

	while (grp = getgrent())
		groups = g_list_prepend(groups, g_strdup(grp->gr_name));

	while (pwd = getpwent())
		users = g_list_prepend(users, g_strdup(pwd->pw_name));

	gtk_combo_set_popdown_strings(GTK_COMBO(app_globals.combo_user), users);
	gtk_combo_set_popdown_strings(GTK_COMBO(app_globals.combo_group), groups);
}

void reset_property_page() {
	gint i;
  GtkWidget **perms = app_globals.perms;

	gtk_label_set_text (GTK_LABEL(app_globals.label_prop_attrib), "");
	for (i = 0; i < 12; i++)
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(perms[i]), FALSE);
}

void setup_property_permissions (struct stat *statbuf) {
  struct passwd *pwd;
  struct group *grp;
  gint i;
  GtkWidget **perms = app_globals.perms;
  GtkWidget *w;
	gint err;

	/* set the apropos permission flags */
	for (i = 0; i < 12; i++)
		if (statbuf->st_mode & (1 << i))
			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(perms[i]), TRUE);

	/* select apropos owner and group */
	pwd = getpwuid(statbuf->st_uid);
	grp = getgrgid(statbuf->st_gid);

	w = GTK_COMBO(app_globals.combo_user)->entry;
	gtk_entry_set_text(GTK_ENTRY(w), pwd->pw_name);
	w = GTK_COMBO(app_globals.combo_group)->entry;
	gtk_entry_set_text(GTK_ENTRY(w), grp->gr_name);
}

void apply_property_dialog() {
	gint i;
	gint temp;
  GtkWidget **perms = app_globals.perms;
  gchar *file;
  gchar *file_new;
  gchar *gnam;
  gchar *unam;
  GtkWidget *w;
  struct passwd *pwd;
  struct group *grp;
  mode_t mode;
  struct stat statbuf;

  temp = (gint)GTK_CLIST (app_globals.clist_files)->selection->data;
  gtk_clist_get_text (GTK_CLIST (app_globals.clist_files), temp, 1, &file);

	/* lstat the file */
  temp = lstat (file, &statbuf);

  if (temp == -1) {
    printf ("init_property_dialog: error in lstat!\n");
    return;
  }

	w = GTK_COMBO(app_globals.combo_user)->entry;
	unam = gtk_entry_get_text(GTK_ENTRY(w));
	pwd = getpwnam (unam);

	w = GTK_COMBO(app_globals.combo_group)->entry;
	gnam = gtk_entry_get_text(GTK_ENTRY(w));
	grp = getgrnam (gnam);

	chown(file, pwd->pw_uid, grp->gr_gid);

	mode = 0;
	for (i = 0; i < 12; i++)
		if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(perms[i])))
			mode |= 1 << i;

	if (mode != (statbuf.st_mode & 0xFFF))
		chmod(file, mode);
		
	file_new = gtk_entry_get_text (GTK_ENTRY(app_globals.entry_prop_name));
	if (strcmp(file, file_new) != 0) {
		rename(file, file_new);
		display_directory();
	}
}

void
init_property_dialog ()
{
  gchar *file;
  gint temp;
  struct stat statbuf;
  gchar *tmp;
  GdkPixmap *img;
 	GdkBitmap *mask; 

  /* ensure that exactly one file is selected */
  /* and retrieve the selection */
  if (GTK_CLIST (app_globals.clist_files)->selection == NULL)
    return;
  if (GTK_CLIST (app_globals.clist_files)->selection->next != NULL)
    return;
  temp = (gint)GTK_CLIST (app_globals.clist_files)->selection->data;
  gtk_clist_get_text (GTK_CLIST (app_globals.clist_files), temp, 1, &file);

  /* figure out the file's type */
  temp = lstat (file, &statbuf);

  if (temp == -1) {
    printf ("init_property_dialog: error in lstat!\n");
    return;
  }

	reset_property_page();
	setup_property_permissions(&statbuf);
	gtk_entry_set_text (GTK_ENTRY(app_globals.entry_prop_name), file);

	if (S_ISDIR (statbuf.st_mode)) {
	  img = app_globals.img_dir;
	  mask = app_globals.img_dir_mask;
	}
	
 	if (S_ISREG (statbuf.st_mode)) {
 		/* FIXME - add support for type-specific icons */
		img = app_globals.img_file;
		mask = app_globals.img_file_mask;
		/* set file size, for files that support it */
		tmp = g_strdup_printf(" (%lu)",statbuf.st_size); 	
		gtk_label_set_text (GTK_LABEL(app_globals.label_prop_attrib), tmp);
		g_free(tmp);
 	}

  if (S_ISFIFO (statbuf.st_mode)) {
		img = app_globals.img_fifo;
		mask = app_globals.img_fifo_mask;
	}
  
  if (S_ISBLK (statbuf.st_mode) || 
      S_ISCHR (statbuf.st_mode) ||
      S_ISSOCK (statbuf.st_mode)) {
		img = app_globals.img_dev;
		mask = app_globals.img_dev_mask;
	}

  if (S_ISLNK (statbuf.st_mode)) {
  	/* FIXME - add relink support? */
		img = app_globals.img_lnk;
		mask = app_globals.img_lnk_mask;
	}

	gtk_pixmap_set(GTK_PIXMAP(app_globals.img_prop_type), img, mask);
	gtk_widget_show(app_globals.win_properties);
}
