/* copy_regular_file.c: Copy a regular file
 *
 * Stripped down from copy_file.c
 * Mini copy_file implementation for busybox
 *
 *
 * Copyright (C) 2001 by Matt Kraai <kraai@alumni.carnegiemellon.edu>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <utime.h>
#include <errno.h>
#include <dirent.h>
#include <stdlib.h>

#include "libbb-extract.h"

/* copy_regular_file
 *
 * Copy source to dest preserving time and ownership. If dest exists,
 * it will be truncated, (or, if necessary, unlinked first).
 */
int copy_regular_file(const char *source, const char *dest)
{
    FILE *sfp, *dfp;
    struct stat source_stat;
    struct stat dest_stat;
    int dest_exists = 1;
    int status = 0;

    if (stat(dest, &dest_stat) < 0) {
	if (errno != ENOENT) {
	    perror_msg("unable to stat `%s'", dest);
	    return -1;
	}
	dest_exists = 0;
    }

    if (dest_exists && source_stat.st_rdev == dest_stat.st_rdev &&
	source_stat.st_ino == dest_stat.st_ino) {
	error_msg("`%s' and `%s' are the same file", source, dest);
	return -1;
    }

    if (S_ISDIR(source_stat.st_mode)) {
	error_msg("%s: omitting directory", source);
	return -1;
    }

    if (! S_ISREG(source_stat.st_mode)) {
	error_msg("internal error: not a plain file");
	return -1;
    }

    if (dest_exists) {
	if ((dfp = fopen(dest, "w")) == NULL) {
	    /*
	    if (!(flags & FILEUTILS_FORCE)) {
		perror_msg("unable to open `%s'", dest);
		return -1;
	    }
	    */
		    
	    if (unlink(dest) < 0) {
		perror_msg("unable to remove `%s'", dest);
		return -1;
	    }
	    
	    dest_exists = 0;
	}
    }
    
    if (!dest_exists) {
	int fd;
	
	if ((fd = open(dest, O_WRONLY|O_CREAT, source_stat.st_mode)) < 0 ||
	    (dfp = fdopen(fd, "w")) == NULL) {
	    if (fd >= 0)
		close(fd);
	    perror_msg("unable to open `%s'", dest);
	    return -1;
	}
    }
    
    if ((sfp = fopen(source, "r")) == NULL) {
	fclose(dfp);
	perror_msg("unable to open `%s'", source);
	    status = -1;
	    goto end;
    }
    
    copy_file_chunk(sfp, dfp, source_stat.st_size);
    
    if (fclose(dfp) < 0) {
	perror_msg("unable to close `%s'", dest);
	status = -1;
    }
    
    if (fclose(sfp) < 0) {
	perror_msg("unable to close `%s'", source);
	status = -1;
    }
	
end:
	
    if (1) { /* (flags & FILEUTILS_PRESERVE_STATUS) { */
	struct utimbuf times;
	    
	times.actime = source_stat.st_atime;
	times.modtime = source_stat.st_mtime;
	if (utime(dest, &times) < 0)
	    perror_msg("unable to preserve times of `%s'", dest);
	if (chown(dest, source_stat.st_uid, source_stat.st_gid) < 0) {
	    source_stat.st_mode &= ~(S_ISUID | S_ISGID);
	    perror_msg("unable to preserve ownership of `%s'", dest);
	}
	if (chmod(dest, source_stat.st_mode) < 0)
	    perror_msg("unable to preserve permissions of `%s'", dest);
    }
	
    return status;
}
