/* mkdtemp.c - Create a unique temporary directory given a template

   Copyright (C) 2001 Carl D. Worth

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2, or (at
   your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
*/

#ifndef HAVE_MKDTEMP

#include <stdlib.h>
#include <errno.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#if HAVE_SYS_STAT_H
# include <sys/stat.h>
#endif

#if HAVE_SYS_WAIT_H
# include <sys/wait.h>
#endif

#if HAVE_UNISTD_H
# include <sys/types.h>
# include <unistd.h>
#endif

/* DESCRIPTION

   The mkdtemp() function creates a unique temporary directory based
   on the template. The template is modified using mktemp(3) and
   should meet the restrictions of that function.

   RETURN VALUE

   If succesful, mkdtemp returns the name of a directory created by
   mkdtemp, (the directory did not exist before). The directory is
   created with mode 0700. On failure, mkdtemp returns NULL;
*/
char *mkdtemp(char *template)
{
    int err;
    char *path;

    while(1) {
	path = mktemp(template);

	if (path == NULL || path[0] == '\0') {
	    return NULL;
	}

	err = mkdir(path, 0700);
	if (err) {
	    if (errno == EEXIST) {
		continue;
	    }
	    return NULL;
	}
	return path;
    }
}

#endif
