/*
linphone
Copyright (C) 2000  Simon MORLAT (simon.morlat@free.fr)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "linphonecore.h"
#include "gnome-config.h"
#include "misc.h"

#include <telephonyevents.h>
#include <utils.h>
#include "osipuacb.h"

#include <net/if.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>


#define MAX_IF 20
const double bandwidths[5]=
{
	28800.0,	/*28k modem*/
	28800.0, /*56k modem*/
	64000.0, /*64k modem (numeris)*/
	128000.0,	/*adsl or cable*/
	10e6, /* ethernet */
};/* carefull: these are the upstream bandwidth !!!*/

/* relative path where is stored local ring*/
#define LOCAL_RING "ring.wav"
/* same for remote ring (ringback)*/
#define REMOTE_RING "ringback.wav"


/* these are useful messages for the status */
gchar *ready;
gchar *end;
gchar *contacting;
gchar *connected;
gchar *cancel;
gchar *contacted;


/* this is the retrieve_if function for FreeBSD. It uses getifaddrs system call instead of ioctl in order to get interface names and address.*/
/* Thanks 	to Wolfgang for this .*/
#ifdef HAVE_GETIFADDRS
#include <sys/types.h>
#include <sys/socket.h>
#include <ifaddrs.h>
GList *
retrieve_if ()
{
	int i;
	struct ifaddrs *ifp;
	struct ifaddrs *ifpstart;
	GList *interfaces=NULL;

	if (getifaddrs (&ifpstart) < 0)
	{
		return NULL;
	}
	ifp=ifpstart;
	for (i = 0; ifp != NULL; ifp = ifp->ifa_next)
	{
		interface_t *netif;
		if (ifp->ifa_addr->sa_family != AF_INET)
			continue;

		/*printf ("ifc %s %s\n",
			ifp->ifa_name,
			inet_ntoa (((struct sockaddr_in *) ifp->ifa_addr)->
				   sin_addr));*/
			
		netif=g_new0(interface_t,1);
		netif->index=i;
		strncpy (netif->name, ifp->ifa_name,20);
		strncpy (netif->ip4addr, inet_ntoa (((struct sockaddr_in *) ifp->ifa_addr)->sin_addr),20);
		interfaces=g_list_append(interfaces,netif);
		i++;
	}

	freeifaddrs (ifpstart);

	return interfaces;
}
#else

GList * retrieve_if()
{
	int sock,err,if_count,i;
	struct ifconf netconf;
	char buffer[sizeof(struct ifreq)*MAX_IF];
	GList *interfaces=NULL;
	
	netconf.ifc_len=sizeof(struct ifreq)*MAX_IF;
	netconf.ifc_buf=buffer;
	sock=socket(PF_INET, SOCK_DGRAM,0);
	err=ioctl(sock,SIOCGIFCONF,&netconf);
	if (err<0) printf("Error in ioctl: %i.\n",errno);
	close(sock);
	
	if_count=netconf.ifc_len/sizeof(struct ifreq);
	g_message("Found %i interfaces.\n",if_count);
	for (i=0;i<if_count;i++)
	{
		interface_t *netif=g_new0(interface_t,1);
		strncpy(netif->name,netconf.ifc_req[i].ifr_name,20);
		strncpy(netif->ip4addr,inet_ntoa(((struct sockaddr_in*)(&netconf.ifc_req[i].ifr_addr))->sin_addr),20);
		netif->index=i;
		interfaces=g_list_append(interfaces,netif);
	}
	return interfaces;
}
#endif

LinphoneDialogParams * linphone_dialog_params_new()
{
	return g_new0(LinphoneDialogParams,1);
}

void linphone_dialog_params_destroy(LinphoneDialogParams *obj)
{
	if (obj->profile!=&av_profile) rtp_profile_destroy(obj->profile);
	g_free(obj);
}

gboolean find_interface(net_config_t *config, char *ifname)
{
	GList *interfaces=config->interfaces;
	interface_t *tmpif;
	
	tmpif=(interface_t*) interfaces->data;
	while (interfaces!=NULL){
		tmpif=(interface_t*) interfaces->data;
		if (ifname!=NULL){
			if (strcmp(ifname,tmpif->name)==0){
				config->sel_if=tmpif;
				return 1;
			}
		}
		interfaces=g_list_next(interfaces);
	}
	return 0;
}

void find_best_interface(net_config_t *config)
{
	interface_t *tmpif;
	tmpif=(interface_t*) config->interfaces->data;
	if (!find_interface(config,"ppp0")){
		/* if ppp0 is not found, then default to eth0 */
		if (!find_interface(config,"eth0")){
			/* if eth0 is not found, try ed0 (used on FreeBsd) */
			if (!find_interface(config,"ed0")){
				/* if eth0 is not found, then default to lo */
				if (!find_interface(config,"lo")){
					/* if lo is not found, default to first interface in the list */
					config->sel_if=tmpif;
				}
			}
		}
	}
}

void
net_config_init (net_config_t * config)
{
	gchar *ifname;
	gboolean def=FALSE;
	
	config->interfaces=retrieve_if();
	/* select the best interface, not lo */
	if (config->interfaces==NULL) {
		g_error("Could not get network interfaces !!");
		return;
	}
	ifname=gnome_config_get_string("net/if_name");
	if ((ifname!=NULL) && (strlen(ifname)!=0)){
		/* try to find it */
		if (!find_interface(config,ifname)){
			g_warning("Could not find network interface %s.",ifname);
			find_best_interface(config);
		}
	}else find_best_interface(config);
	
	config->con_type=gnome_config_get_int_with_default("net/con_type",&def);
	if (def) config->con_type=CONNECTION_TYPE_ETHERNET;
	config->bandwidth=bandwidths[config->con_type];
	
	config->nat_address=gnome_config_get_string_with_default("net/nat_address",&def);
	if (def) config->nat_address=NULL;
	if (config->nat_address!=NULL && (strlen(config->nat_address)<1)) config->nat_address=NULL;
	config->use_nat=gnome_config_get_int_with_default("net/use_nat",&def);
	if (def) config->use_nat=0;	
}

void sound_config_init(sound_config_t *config)
{
	gboolean def;
	int tmp;
	char *tmpbuf;
	SndCard *sndcard;
	config->dev_id=gnome_config_get_int_with_default("sound/dev_id",&def);
	if (def) config->dev_id=0;
	config->play_lev=gnome_config_get_int_with_default("sound/play_lev",&def);
	if (def) config->play_lev=80;
	config->rec_lev=gnome_config_get_int_with_default("sound/rec_lev",&def);
	if (def) config->rec_lev=80;
	tmpbuf=gnome_config_get_string_with_default("sound/source",&def);
	if (def) config->source='m';
	else {
		config->source=tmpbuf[0];
	}
	config->local_ring=gnome_config_get_string_with_default("sound/local_ring",&def);
	if (def) config->local_ring=PACKAGE_SOUND_DIR "/" LOCAL_RING;
	else {
		if (!g_file_test(config->local_ring,G_FILE_TEST_EXISTS)){
			config->local_ring=PACKAGE_SOUND_DIR "/" LOCAL_RING;
		}
		if (strstr(config->local_ring,".wav")==NULL){
			/* it currently uses old sound files, so replace them */
			config->local_ring=PACKAGE_SOUND_DIR "/" LOCAL_RING;
		}
	}
	config->remote_ring=gnome_config_get_string_with_default("sound/remote_ring",&def);
	if (def) config->remote_ring=PACKAGE_SOUND_DIR "/" REMOTE_RING;
	else {
		if (!g_file_test(config->remote_ring,G_FILE_TEST_EXISTS)){
			config->remote_ring=PACKAGE_SOUND_DIR "/" REMOTE_RING;
		}
		if (strstr(config->remote_ring,".wav")==NULL){
			/* it currently uses old sound files, so replace them */
			config->remote_ring=PACKAGE_SOUND_DIR "/" REMOTE_RING;
		}
	}
	config->autokill=1;
	
	sndcard=snd_card_manager_get_card(snd_card_manager,config->dev_id);
	if (sndcard==NULL){
		g_warning("Sound card with index %i does not exist. using 0 instead.",config->dev_id);
		config->dev_id=0;
		sndcard=snd_card_manager_get_card(snd_card_manager,config->dev_id);
		
	}
	config->sndcard=sndcard;
	g_return_if_fail(sndcard!=NULL);
	snd_card_set_rec_source(sndcard,config->source);	
	snd_card_set_level(sndcard,SND_CARD_LEVEL_OUTPUT,config->play_lev);
	snd_card_set_level(sndcard,SND_CARD_LEVEL_INPUT,config->rec_lev);
	tmp=test_audio_dev(config->dev_id);
	if (tmp>0) config->latency=tmp;
		
}

void registrar_config_init(registrar_config_t *config)
{
	gboolean def;
	config->use_registrar=gnome_config_get_int_with_default("sip/use_registrar",&def);
	if (def) config->use_registrar=0;
	config->as_proxy=gnome_config_get_int_with_default("sip/as_proxy",&def);
	if (def) config->as_proxy=0;
	config->expires=gnome_config_get_int_with_default("sip/expires",&def);
	if (def) config->expires=15*60;
	config->registrar=gnome_config_get_string_with_default("sip/registrar",&def);
	if (def) config->registrar=NULL;
	config->passwd=gnome_config_get_string_with_default("sip/passwd",&def);
	if (def) config->passwd=NULL;
	config->addr_of_rec=gnome_config_get_string_with_default("sip/addr_of_rec",&def);
	if (def) config->addr_of_rec=NULL;
	config->timer=g_timer_new();
	g_timer_start(config->timer);
}

void sip_config_init(sip_config_t *config)
{
	gboolean def;
	config->username=gnome_config_get_string_with_default("sip/username",&def);
	if (def) {
		gchar *logname=getenv("LOGNAME");
		if (logname!=NULL)
			config->username=g_strdup(logname);
		else config->username=g_strdup("toto");
	}
	config->hostname=gnome_config_get_string_with_default("sip/hostname",&def);
	if (def) {
		gchar *hostname=getenv("HOST");
		if (hostname!=NULL)
			config->hostname=g_strdup(hostname);
		else config->hostname=g_strdup("unknown-host");
	}
	config->sip_port=gnome_config_get_int_with_default("sip/sip_port",&def);
	if (def){
		config->sip_port=5060;
	}
	registrar_config_init(&config->reg_conf);
}

void rtp_config_init(rtp_config_t *config)
{
	gboolean def;
	config->audio_rtp_port=gnome_config_get_int_with_default("rtp/audio_rtp_port",&def);
	if (def) config->audio_rtp_port=7078;
	config->video_rtp_port=gnome_config_get_int_with_default("rtp/video_rtp_port",&def);
	if (def) config->video_rtp_port=9078;
	config->audio_jitt_comp=gnome_config_get_int_with_default("rtp/audio_jitt_comp",&def);
	if (def) config->audio_jitt_comp=60;	
	config->video_jitt_comp=gnome_config_get_int_with_default("rtp/video_jitt_comp",&def);
	if (def) config->video_jitt_comp=60;	
	
}

GList * parse_audio_codec_list()
{
	gboolean def;
	char **vector;
	int len,i;
	GList *list=NULL;
	MSCodecInfo *codec;
	/* get codecs config*/
	gnome_config_get_vector_with_default("codecs/audio_codecs",&len,&vector,&def);
	if (def) {
		g_warning("No audio codec list defined in config file.");
		return NULL;
	}
	for (i=0;i<len;i++)
	{
		/* vector is for example:  +gsm -lpc10-15 -pcma +pcmu 
		where +/- tells if the codec is enabled */
		//g_message("codec: %s",vector[i]);
		if (strlen(vector[i])<2) continue;
		codec=ms_audio_codec_info_get(vector[i]+1); 
		if (codec!=NULL)
		{
			list=g_list_append(list,codec);
			if (vector[i][0]=='+')
				codec->is_selected=1;
			else if (vector[i][0]=='-')
				codec->is_selected=0;
			else g_warning("Invalid enabling value for codec %s",vector[i]+1);
		}
		else {
			g_warning("Could not find codec %s.",vector[i]+1);
		}
	}
	return list;
}

GList * parse_video_codec_list()
{
	gboolean def;
	char **vector;
	int len,i;
	GList *list=NULL;
	MSCodecInfo *codec;
	/* get codecs config*/
	gnome_config_get_vector_with_default("codecs/video_codecs",&len,&vector,&def);
	if (def) {
		return NULL;
	}
	for (i=0;i<len;i++)
	{
		/* vector is for example:  +gsm -lpc10-15 -pcma +pcmu 
		where +/- tells if the codec is enabled */
		if (strlen(vector[i])<2) continue;
		codec=ms_video_codec_info_get(vector[i]+1); 
		if (codec!=NULL)
		{
			list=g_list_append(list,codec);
			if (vector[i][0]=='+')
				codec->is_selected=1;
			else if (vector[i][0]=='-')
				codec->is_selected=0;
			else g_warning("Invalid enabling value for codec %s",vector[i]+1);
		}
		else {
			g_warning("Could not find codec %s.",vector[i]+1);
		}
	}
	return list;
}


void codecs_config_init(codecs_config_t *config)
{
	
	GList *all_audio_codecs=ms_codec_get_all_audio();
	GList *all_video_codecs=ms_codec_get_all_video();
	
	config->audio_codecs=parse_audio_codec_list();
	if (config->audio_codecs==NULL){
		config->audio_codecs=all_audio_codecs;
	}else if (g_list_length(all_audio_codecs)!=g_list_length(config->audio_codecs)){
		g_warning("Not all codecs are listed in config file. Using generic list instead.");
		g_list_free(config->audio_codecs);
		config->audio_codecs=all_audio_codecs;
	}else g_list_free(all_audio_codecs);
	
	config->video_codecs=parse_video_codec_list();
	if (config->video_codecs==NULL){
		config->video_codecs=all_video_codecs;
	}else if (g_list_length(all_video_codecs)!=g_list_length(config->video_codecs)){
		g_list_free(config->video_codecs);
		config->video_codecs=all_video_codecs;
	}else g_list_free(all_video_codecs);
	
}

void video_config_init(video_config_t *config)
{
	gboolean def;
	config->enabled=gnome_config_get_int_with_default("video/enabled",&def);
	if (def) config->enabled=0;
	config->show_local=gnome_config_get_int_with_default("video/show_local",&def);
	if (def) config->show_local=0;
}

void ui_config_init(ui_config_t *config)
{
	int len,tmp;
	char *ab_entry;
	/* get address book entries*/
	len= gnome_config_get_int("address_book/entry_count");
	if (len!=-1)
	{
		for (tmp=0;tmp<len;tmp++)
		{
			ab_entry=g_strdup_printf("address_book/entry%i",tmp);
			config->address_list=g_list_append(config->address_list,(gpointer)gnome_config_get_string(ab_entry));
			g_free(ab_entry);
		}
	}
}

void autoreplier_config_init(autoreplier_config_t *config)
{
	gboolean def;
	config->enabled=gnome_config_get_int_with_default("autoreplier/enabled",&def);
	if (def) config->enabled=0;
	config->after_seconds=gnome_config_get_int_with_default("autoreplier/after_seconds",&def);
	if (def) config->after_seconds=6;
	config->max_users=gnome_config_get_int_with_default("autoreplier/max_users",&def);
	if (def) config->max_users=1;
	config->max_rec_time=gnome_config_get_int_with_default("autoreplier/max_rec_time",&def);
	if (def) config->max_rec_time=0;
	config->max_rec_msg=gnome_config_get_int_with_default("autoreplier/max_rec_msg",&def);
	if (def) config->max_rec_msg=0;
	config->message=gnome_config_get_string_with_default("autoreplier/message",&def);
	if (def) config->message=NULL;
}

void
linphone_core_init (LinphoneCore * lc,LinphoneCoreVTable *vtable, gchar *config_path, gpointer userdata)
{
	int len,err;
	char *prim_contact;
	
	//TRACE_INITIALIZE(TRACE_LEVEL5,stdout);
	osipua_init();
	ortp_init();
	ortp_set_debug_file("oRTP",NULL);
	rtp_profile_set_payload(&av_profile,115,&lpc1015);
	rtp_profile_set_payload(&av_profile,110,&speex_nb);
	rtp_profile_set_payload(&av_profile,101,&telephone_event);
	ms_init();
	ms_speex_codec_init();
	
	memset (lc, 0, sizeof (LinphoneCore));
	lc->data=userdata;
	ready=_("Ready.");
  	end=_("Communication ended.");
	contacting= _("Contacting ");
	contacted=_("is calling you.");
	connected=_("Connected.");
	cancel=_("Call cancelled.");
	
	memcpy(&lc->vtable,vtable,sizeof(LinphoneCoreVTable));

	if (config_path!=NULL){
		gnome_config_push_prefix(g_strdup_printf("=%s=",config_path));
	}else gnome_config_push_prefix("/linphone/");
	
	net_config_init(&lc->net_conf);
	sound_config_init(&lc->sound_conf);
	check_sound_device(lc);
	sip_config_init(&lc->sip_conf);
	rtp_config_init(&lc->rtp_conf);
	codecs_config_init(&lc->codecs_conf);
	video_config_init(&lc->video_conf);
	autoreplier_config_init(&lc->autoreplier_conf);
	ui_config_init(&lc->ui_conf);
	
	
	lc->ua=osip_ua_new();
	linphone_core_update_contact_info(lc);
  	err=osip_ua_signal_connect(lc->ua,"INVITE_ACCEPTED",invite_accepted_cb);
  	err=osip_ua_signal_connect(lc->ua,"BYE",bye_cb);
  	err=osip_ua_signal_connect(lc->ua,"FAILLURE",faillure_cb);
  	err=osip_ua_signal_connect(lc->ua,"INVITE",invite_cb);
  	err=osip_ua_signal_connect(lc->ua,"INFORMATIVE",informative_cb);
	/* LIK: Transfer call related functionality */
	err=osip_ua_signal_connect(lc->ua, "MUTE", mute_func_cb);
	err=osip_ua_signal_connect(lc->ua, "BYETRANSFER", byetransfer_func_cb);
	err=osip_ua_signal_connect(lc->ua, "REFERTRANSFER", refertransfer_func_cb);
	err=osip_ua_signal_connect(lc->ua, "NOTIFYTRANSFER", notifytransfer_func_cb);
  	lc->ua->data=(void*)lc;
	
	/* create a sdp handler */
	lc->sdph=sdp_handler_new();
	sdp_handler_set_write_offer_fcn(SDP_HANDLER(lc->sdph),set_audio_offer,set_video_offer);
	sdp_handler_set_accept_offer_fcn(SDP_HANDLER(lc->sdph),accept_audio_offer,accept_video_offer);
	sdp_handler_set_read_answer_fcn(SDP_HANDLER(lc->sdph),read_audio_answer,read_video_answer);
	
	/* add the sdp handler to the ua */
	osip_ua_add_body_handler(lc->ua,lc->sdph);
	
	linphone_core_set_registrar_usage(lc,lc->sip_conf.reg_conf.use_registrar,
											lc->sip_conf.reg_conf.as_proxy);
	linphone_core_set_connection_type(lc,lc->net_conf.con_type);
	linphone_core_setup_local_rtp_profile(lc);
	lc->in_main_thread=1;
	lc->lock=g_mutex_new();
}

GList *linphone_core_get_audio_codecs(LinphoneCore *lc)
{
	return g_list_copy(lc->codecs_conf.audio_codecs);
}

GList *linphone_core_get_video_codecs(LinphoneCore *lc)
{
	return g_list_copy(lc->codecs_conf.video_codecs);
}

int linphone_core_set_primary_contact(LinphoneCore *lc, gchar *contact)
{
	int error=0;
	from_t *ct;
	from_init(&ct);
	if ( from_parse(ct,contact) <0){
		g_warning("Bad contact url: %s", contact);
		error=1;
		goto end;
	}
	lc->sip_conf.username=g_strdup(ct->url->username);
	lc->sip_conf.hostname=g_strdup(ct->url->host);
	linphone_core_update_contact_info(lc);
	end:
		from_free(ct);
		sfree(ct);
	return error;
}

gchar *linphone_core_get_primary_contact(LinphoneCore *lc)
{
	return g_strdup(lc->sip_conf.contact);
}

from_t *linphone_core_get_primary_contact_parsed(LinphoneCore *lc)
{
	int err;
	from_t *contact;
	from_init(&contact);
	err=from_parse(contact,lc->sip_conf.contact);
	if (err<0) {
		from_free(contact);
		sfree(contact);
		return NULL;
	}
	return contact;
}

int linphone_core_set_audio_codecs(LinphoneCore *lc, GList *codecs)
{
	if (lc->codecs_conf.audio_codecs!=NULL) g_list_free(lc->codecs_conf.audio_codecs);
	lc->codecs_conf.audio_codecs=g_list_copy(codecs);
	return 0;
}

int linphone_core_set_video_codecs(LinphoneCore *lc, GList *codecs)
{
	if (lc->codecs_conf.video_codecs!=NULL) g_list_free(lc->codecs_conf.video_codecs);
	lc->codecs_conf.video_codecs=g_list_copy(codecs);
	return 0;
}

GList * linphone_core_get_address_list(LinphoneCore *lc)
{
	return g_list_copy(lc->ui_conf.address_list);
}

int linphone_core_set_address_list(LinphoneCore *lc, GList *addresses)
{
	if (lc->ui_conf.address_list!=NULL) g_list_free(lc->ui_conf.address_list);
	lc->ui_conf.address_list=g_list_copy(addresses);
	return 0;
}

char *linphone_core_get_registrar_addr(LinphoneCore *lc)
{
	return g_strdup(lc->sip_conf.reg_conf.registrar);
}

int linphone_core_set_registrar_addr(LinphoneCore *lc,char *url,char *addr_of_rec)
{
	url_t *surl=NULL,*surl2=NULL;
	int err=0;
	url_init(&surl);
	err=url_parse(surl,url);
	if (err<0){
		g_warning("Bad proxy or registrar url:%s",url);
		goto end;
	}
	if (addr_of_rec!=NULL){
		url_init(&surl2);
		err=url_parse(surl2,addr_of_rec);
		if (err<0){
			g_warning("Bad address of record:%s",url);
			goto end;
		}
	}
	if (lc->sip_conf.reg_conf.registrar!=NULL){
		g_free(lc->sip_conf.reg_conf.registrar);
	}
	lc->sip_conf.reg_conf.registrar=g_strdup(url);
	
	if (addr_of_rec!=NULL){
		if (lc->sip_conf.reg_conf.addr_of_rec!=NULL){
			g_free(lc->sip_conf.reg_conf.addr_of_rec);
		}
		lc->sip_conf.reg_conf.addr_of_rec=g_strdup(addr_of_rec);
	}else{
		/* guess a valid address of record */
		if (surl->port!=NULL)
			lc->sip_conf.reg_conf.addr_of_rec=g_strdup_printf("sip:%s@%s:%s",
				lc->sip_conf.username,surl->host,surl->port);
		else lc->sip_conf.reg_conf.addr_of_rec=g_strdup_printf("sip:%s@%s",
				lc->sip_conf.username,surl->host);
	}
	end:
		url_free(surl);
		sfree(surl);
		if (surl2!=NULL){
			url_free(surl2);
			sfree(surl2);
		}
	return err;
}

char *linphone_core_get_addr_of_rec(LinphoneCore *lc)
{
	return g_strdup(lc->sip_conf.reg_conf.addr_of_rec);
}

char *linphone_core_get_registrar_passwd(LinphoneCore *lc)
{
	return g_strdup(lc->sip_conf.reg_conf.passwd);
}

void linphone_core_set_registrar_passwd(LinphoneCore *lc, gchar *passwd)
{
	gchar *tmp=NULL;
	if (passwd!=NULL){
		tmp=g_strdup(passwd);
	}
	if (lc->sip_conf.reg_conf.passwd!=NULL){
		g_free(lc->sip_conf.reg_conf.passwd);
	}
	lc->sip_conf.reg_conf.passwd=tmp;
}


int linphone_core_register(LinphoneCore *lc, char *url){
	if (url!=NULL){
		linphone_core_set_registrar_addr(lc,url,NULL);
	}
	do_registration(lc,1);
	return 0;
}

void linphone_core_get_registrar_usage(LinphoneCore *lc,gboolean *reg, gboolean *proxy)
{
	if (reg!=NULL) *reg= lc->sip_conf.reg_conf.use_registrar;
	if (proxy!=NULL) *proxy=lc->sip_conf.reg_conf.as_proxy;
}



void linphone_core_set_registrar_usage(LinphoneCore *lc, gboolean use, gboolean outbound_proxy)
{
	if ((lc->sip_conf.reg_conf.registrar!=NULL) && (strlen(lc->sip_conf.reg_conf.registrar)!=0)){
		if (use){
			if (outbound_proxy)
				osip_ua_set_outbound_proxy(lc->ua,lc->sip_conf.reg_conf.registrar,lc->sip_conf.reg_conf.passwd);
			else osip_ua_set_outbound_proxy(lc->ua,NULL,NULL);
			do_registration(lc,1);
			/* TODO: setup a timeout to re-register after expires */
		}else{
			osip_ua_set_outbound_proxy(lc->ua,NULL,NULL);
			if (lc->sip_conf.reg_conf.use_registrar){
				/* the proxy is no more used, we need to unregister */
				do_registration(lc,0);
			}
		}
	}
	lc->sip_conf.reg_conf.use_registrar=use;
	lc->sip_conf.reg_conf.as_proxy=outbound_proxy;
}

GList * linphone_core_get_net_interfaces(LinphoneCore *lc)
{
	return g_list_copy(lc->net_conf.interfaces);
}

interface_t * linphone_core_get_selected_interface(LinphoneCore *lc)
{
	return lc->net_conf.sel_if;
}

void linphone_core_select_net_interface(LinphoneCore *lc, gchar *ifname)
{
	GList *elem=lc->net_conf.interfaces;
	while(elem!=NULL){
		interface_t *iface=(interface_t *)elem->data;
		if (strcmp(ifname,iface->name)==0){
			lc->net_conf.sel_if=iface;
			linphone_core_update_contact_info(lc);
			return;
		}
		elem=g_list_next(elem);
	}
	g_warning("Could not select interface %s: non existent.",ifname);
}

void linphone_core_set_connection_type(LinphoneCore *lc, int type)
{
	lc->net_conf.con_type=type;
	lc->net_conf.bandwidth=bandwidths[type];
	linphone_core_check_codecs_for_bandwidth(lc);
}

int linphone_core_get_connection_type(LinphoneCore *lc)
{
	return lc->net_conf.con_type;
}

int linphone_core_get_audio_jittcomp(LinphoneCore *lc)
{
	return lc->rtp_conf.audio_jitt_comp;
}

int linphone_core_get_audio_port(LinphoneCore *lc)
{
	return lc->rtp_conf.audio_rtp_port;
}

void linphone_core_set_audio_jittcomp(LinphoneCore *lc, int value)
{
	lc->rtp_conf.audio_jitt_comp=value;
}

void linphone_core_set_audio_port(LinphoneCore *lc, int port)
{
	lc->rtp_conf.audio_rtp_port=port;
}

int linphone_core_get_sip_port(LinphoneCore *lc)
{
	return lc->sip_conf.sip_port;
}

void linphone_core_set_sip_port(LinphoneCore *lc,int port)
{
	lc->sip_conf.sip_port=port;
	linphone_core_update_contact_info(lc);
}

void linphone_core_refresh(LinphoneCore *lc)
{
	check_for_registration(lc);
}

gboolean sipurl_check(gchar *url)
{
	from_t *surl;
	int err,ret;
	if (url==NULL) return FALSE;
	
	from_init(&surl);
	err=from_parse(surl,url);
	if (err<0) ret=FALSE;
	else ret=TRUE;
	from_free(surl);
	sfree(surl);
	return ret;
}

/* LIK: Transfer related functionality */
int linphone_core_transfer(LinphoneCore *lc, char *url)
{
	gchar *barmsg;
	gint err;
	char muteaddr[] = "0.0.0.0";

	linphone_core_lock(lc);
	if (!sipurl_check(url) ){
		lc->vtable.display_warning(lc,_("Bad formuled sip address. A sip address looks like <sip:username@domainname>"));
		linphone_core_unlock(lc);
		return -1;
	}
	barmsg=g_malloc(strlen(url)+strlen(contacting)+2);
	sprintf(barmsg,"%s %s",contacting,url);
	lc->vtable.display_status(lc,barmsg);

	if (lc->audiostream != NULL && lc->audiostream->timer != NULL) {
	    lc->audiostream->timer->mute = 1;
	}

	/* Put 0.0.0.0 in lc->ua->mute _ip4addr before sending out the
	   invite.  This is where the sdp handler gets the contact
	   information from */
	
	lc->ua->mute_ip4addr = muteaddr;
	osip_trace(8, ("About to transfer to %s", url));

	err=osip_dialog_transfer(lc->call,url,"application/sdp");

	lc->ua->mute_ip4addr = NULL;

	if (err!=0){
	    osip_trace(8, ("URL transfer to %s failed %d", url, err));
	    g_warning("error in invite: %s (%i)\n",strerror(err),err);
	    lc->vtable.display_status(lc,ready);
	}
	linphone_core_unlock(lc);
	return 0;    
}

int linphone_core_invite(LinphoneCore *lc, char *url)
{
	gchar *barmsg;
	gint err;
	
	linphone_core_lock(lc);
	if (!sipurl_check(url) ){
		lc->vtable.display_warning(lc,_("Bad formuled sip address. A sip address looks like <sip:username@domainname>"));
		linphone_core_unlock(lc);
		return -1;
	}
	barmsg=g_malloc(strlen(url)+strlen(contacting)+2);
	sprintf(barmsg,"%s %s",contacting,url);
	lc->vtable.display_status(lc,barmsg);
	
	lc->call=osip_dialog_new(lc->ua);
	err=osip_dialog_invite(lc->call,url,"application/sdp");
	if (err!=0){
		g_warning("error in invite: %s (%i)\n",strerror(err),err);
		lc->vtable.display_status(lc,ready);
	}
	linphone_core_unlock(lc);
	return 0;
}

int linphone_core_accept_dialog(LinphoneCore *lc, const char *url)
{
	LinphoneDialogParams *diaparams;
	StreamParams *audio_params;
	int pos;
	gint jitt_comp;
	
	if (lc->call==NULL){
		return -1;
	}
	linphone_core_lock(lc);
	if (lc->call->status!=DIALOG_INVITED){
		g_warning("No dialog to accept.");
		linphone_core_unlock(lc);
		return -1;
	}
	diaparams=(LinphoneDialogParams *)lc->call->data;
	audio_params=&diaparams->audio_params;
	
	/*stop ringing */
	if (lc->ringstream!=NULL) {
		ring_stop(lc->ringstream);
		lc->ringstream=NULL;
	}
	/* sends a 200 OK */
	osip_dialog_accept_invite(lc->call,lc->current_trn);
	lc->vtable.display_status(lc,connected);
	
	if (try_open_dsp(lc)){
		/* adjust rtp jitter compensation. It must be at least the latency of the sound card */
		jitt_comp=MAX(lc->sound_conf.latency,lc->rtp_conf.audio_jitt_comp);
		lc->audiostream=audio_stream_start_with_sndcard(diaparams->profile,audio_params->localport,
							audio_params->remoteaddr,audio_params->remoteport,
							audio_params->pt,jitt_comp,lc->sound_conf.sndcard);
	}
	linphone_core_unlock(lc);
	return 0;
}

int linphone_core_terminate_dialog(LinphoneCore *lc, const char *url)
{
	OsipDialog *call=lc->call;
	LinphoneDialogParams *diaparams;
	if (call==NULL){
		return -1;
	}
	
	linphone_core_lock(lc);
	diaparams=(LinphoneDialogParams*)call->data;
	lc->call=NULL;
	
	if (call->status==DIALOG_INVITED )
    {
       /*stop ringing*/
		if (lc->ringstream!=NULL) {
			ring_stop(lc->ringstream);
			lc->ringstream=NULL;
			restore_sound_daemons();
		}
		/* decline the invitation */
		osip_dialog_reject_invite(call);
		lc->vtable.display_status(lc,end);
    }
	else if (call->status==DIALOG_ESTABLISHED)
	{
    		 /* sends a bye, the call is terminated */
		if (lc->audiostream!=NULL) {
			audio_stream_stop(lc->audiostream);
			lc->audiostream=NULL;
			restore_sound_daemons();
		}
		osip_dialog_bye(call);
		lc->vtable.display_status(lc,end);
	}
	else if (call->status==DIALOG_INVITING)
	{
     /*sends a cancel or a bye ???*/
		if (lc->ringstream!=NULL) {
			ring_stop(lc->ringstream);
			lc->ringstream=NULL;
			restore_sound_daemons();
		}
		osip_dialog_bye(call);
		lc->vtable.display_status(lc,cancel);
	}
	if (diaparams!=NULL) {
		call->data=NULL;
		linphone_dialog_params_destroy(diaparams);
	}
	linphone_core_unlock(lc);
	return 0;
}

void linphone_core_set_presence_info(LinphoneCore *lc,gint minutes_away,
													gchar *contact,
													int presence_mode)
{
	if (minutes_away>0) osip_ua_set_presence_delay(lc->ua,minutes_away*60);
	if (contact!=NULL) {
		osip_ua_set_presence_contact_url(lc->ua,contact);
	}
	osip_ua_set_presence_mode(lc->ua,presence_mode);
}

/* sound functions */
gint linphone_core_get_play_level(LinphoneCore *lc)
{
	return lc->sound_conf.play_lev;
}
gint linphone_core_get_rec_level(LinphoneCore *lc){
	return lc->sound_conf.rec_lev;
}
void linphone_core_set_play_level(LinphoneCore *lc, gint level){
	lc->sound_conf.play_lev=level;
	g_return_if_fail(lc->sound_conf.sndcard!=NULL);
	snd_card_set_level(lc->sound_conf.sndcard,SND_CARD_LEVEL_OUTPUT,level);
}

void linphone_core_set_rec_level(LinphoneCore *lc, gint level)
{
	lc->sound_conf.rec_lev=level;
	g_return_if_fail(lc->sound_conf.sndcard!=NULL);
	snd_card_set_level(lc->sound_conf.sndcard,SND_CARD_LEVEL_INPUT,level);
}
gint linphone_core_set_sound_device(LinphoneCore *lc, gint devid)
{
	SndCard *sndcard;
	if (devid<0){
		g_warning("Bad devid value: %i",devid);
		return -1;
	}
	sndcard=snd_card_manager_get_card(snd_card_manager,devid);
	if (sndcard==NULL){
		g_warning("Sound card with index %i does not exist.");
		return -1;
	}
	lc->sound_conf.dev_id=devid;
	lc->sound_conf.sndcard=sndcard;
	return 0;
}
gint linphone_core_get_sound_device(LinphoneCore *lc)
{
	return lc->sound_conf.dev_id;
}

gchar linphone_core_get_sound_source(LinphoneCore *lc)
{
	return lc->sound_conf.source;
}
void linphone_core_set_sound_source(LinphoneCore *lc, gchar source)
{
	lc->sound_conf.source=source;
	g_return_if_fail(lc->sound_conf.sndcard!=NULL);
	snd_card_set_rec_source(lc->sound_conf.sndcard,source);
}

void linphone_core_send_dtmf(LinphoneCore *lc,gchar dtmf)
{
	if (lc->audiostream!=NULL){
		send_dtmf(lc->audiostream,dtmf);
	}
}

void linphone_core_set_nat_address(LinphoneCore *lc, gchar *addr, gboolean use)
{
	gchar *tmp=NULL;
	if (addr!=NULL && use){
		if (inet_addr(addr)!=-1){
			tmp=g_strdup(addr);
		}else {
			g_warning("Invalid nat address %s",addr);
			return;
		}
	}
	if (lc->net_conf.nat_address!=NULL){
		g_free(lc->net_conf.nat_address);
	}
	lc->net_conf.nat_address=tmp;
	if (use && (tmp!=NULL)){
		gchar *ct;
		/* change OsipUA object main contact */
		if (lc->sip_conf.sip_port==5060){
			ct=g_strdup_printf("sip:%s@%s",lc->sip_conf.username,tmp);
		}else{
			ct=g_strdup_printf("sip:%s@%s:%i",lc->sip_conf.username,tmp,
							lc->sip_conf.sip_port);
		}
		osip_ua_set_contact(lc->ua,ct);
		g_free(ct);
		lc->net_conf.use_nat=TRUE;
	}
	else {
		linphone_core_update_contact_info(lc);
		lc->net_conf.use_nat=FALSE;
	}
}

gchar *linphone_core_get_nat_address(LinphoneCore *lc, gboolean *use)
{
	if (use!=NULL) *use=lc->net_conf.use_nat;
	if (lc->net_conf.nat_address!=NULL) return g_strdup(lc->net_conf.nat_address);
	else return NULL;
}

void net_config_uninit(net_config_t *config)
{
	gnome_config_set_string("net/if_name",config->sel_if->name);
	gnome_config_set_int("net/con_type",config->con_type);
	g_list_foreach(config->interfaces,(GFunc)g_free,NULL);
	g_list_free(config->interfaces);
	gnome_config_set_int("net/use_nat",config->use_nat);
	gnome_config_set_string("net/nat_address",config->nat_address);
	config->interfaces=NULL;
}

void registrar_config_uninit(registrar_config_t *config)
{
	gnome_config_set_int("sip/use_registrar",config->use_registrar);
	gnome_config_set_int("sip/as_proxy",config->as_proxy);
	gnome_config_set_string("sip/registrar",config->registrar);
	gnome_config_set_string("sip/passwd",config->passwd);
	gnome_config_set_string("sip/addr_of_rec",config->addr_of_rec);
	gnome_config_set_int("sip/expires",config->expires);
	g_timer_destroy(config->timer);
}
void sip_config_uninit(sip_config_t *config)
{
	gnome_config_set_string("sip/username",config->username);
	gnome_config_set_string("sip/hostname", config->hostname);
	gnome_config_set_int("sip/sip_port",config->sip_port);
	registrar_config_uninit(&config->reg_conf);
}

void rtp_config_uninit(rtp_config_t *config)
{
	gnome_config_set_int("rtp/audio_rtp_port",config->audio_rtp_port);
	gnome_config_set_int("rtp/video_rtp_port",config->video_rtp_port);
	gnome_config_set_int("rtp/audio_jitt_comp",config->audio_jitt_comp);
	gnome_config_set_int("rtp/video_jitt_comp",config->audio_jitt_comp);
}

void sound_config_uninit(sound_config_t *config)
{
	char tmpbuf[2];
	gnome_config_set_int("sound/dev_id",config->dev_id);
	gnome_config_set_int("sound/rec_lev",config->rec_lev);
	gnome_config_set_int("sound/play_lev",config->play_lev);
	tmpbuf[0]=config->source;
	tmpbuf[1]='\0';
	gnome_config_set_string("sound/source",tmpbuf);
	gnome_config_set_string("sound/local_ring",config->local_ring);
	gnome_config_set_string("sound/remote_ring",config->remote_ring);
}

void video_config_uninit(video_config_t *config)
{
	gnome_config_set_int("video/enabled",config->enabled);
	gnome_config_set_int("video/show_local",config->show_local);
}

void codecs_config_uninit(codecs_config_t *config)
{
	int len,i;
	MSCodecInfo *codec;
	const gchar **vector;
	
	len=g_list_length(config->audio_codecs);
	vector=g_malloc(len*sizeof(char*));
	for (i=0;i<len;i++)
	{
		codec=(MSCodecInfo*)g_list_nth_data(config->audio_codecs,i);
		if (codec->is_selected) vector[i]=g_strdup_printf("+%s",codec->description);
		else vector[i]=g_strdup_printf("-%s",codec->description);
	}
	gnome_config_set_vector("codecs/audio_codecs",len,vector);
	/* REVISIT: do we need to free the vector ??? */
	len=g_list_length(config->video_codecs);
	vector=g_malloc(len*sizeof(char*));
	for (i=0;i<len;i++)
	{
		codec=(MSCodecInfo*)g_list_nth_data(config->video_codecs,i);
		if (codec->is_selected) vector[i]=g_strdup_printf("+%s",codec->description);
		else vector[i]=g_strdup_printf("-%s",codec->description);
	}
	gnome_config_set_vector("codecs/video_codecs",len,vector);
	/* REVISIT: do we need to free the vector ??? */
}

/* next generation codec config save, not yet activated */
void codecs_config_uninit2(codecs_config_t *config)
{
	PayloadType *pt;
	GList *node;
	gchar key[50];
	gint index;
	index=0;
	for(node=config->audio_codecs;node=g_list_next(node);node!=NULL){
		pt=(PayloadType*)(node->data);
		sprintf(key,"audio_codec_%i/mime",index);
		gnome_config_set_string(key,pt->mime_type);
		sprintf(key,"audio_codec_%i/rate",index);
		gnome_config_set_string(key,pt->clock_rate);
		sprintf(key,"audio_codec_%i/enabled",index);
		gnome_config_set_int(key,(pt->flags & PAYLOAD_TYPE_ENABLED));
		index++;
	}
	index=0;
	for(node=config->video_codecs;node=g_list_next(node);node!=NULL){
		pt=(PayloadType*)(node->data);
		sprintf(key,"video_codec_%i/mime",index);
		gnome_config_set_string(key,pt->mime_type);
		sprintf(key,"video_codec_%i/rate",index);
		gnome_config_set_string(key,pt->clock_rate);
		sprintf(key,"video_codec_%i/enabled",index);
		gnome_config_set_int(key,(pt->flags & PAYLOAD_TYPE_ENABLED));
		index++;
	}
}

void ui_config_uninit(ui_config_t *config)
{
	int len;
	GList *elem;
	gchar ab_entry[100];
	/* record address book entries*/
	len=0;
	elem=config->address_list;
	while(elem!=NULL)
	{
		snprintf(ab_entry,100,"address_book/entry%i",len);
		gnome_config_set_string(ab_entry,(gchar*)(elem->data));
		elem=g_list_next(elem);
		len++;
	}
	gnome_config_set_int("address_book/entry_count",len);
}

void linphone_core_uninit(LinphoneCore *lc)
{
	/* save all config */
	net_config_uninit(&lc->net_conf);
	sip_config_uninit(&lc->sip_conf);
	rtp_config_uninit(&lc->rtp_conf);
	sound_config_uninit(&lc->sound_conf);
	video_config_uninit(&lc->video_conf);
	codecs_config_uninit(&lc->codecs_conf);
	ui_config_uninit(&lc->ui_conf);
	gnome_config_sync();
	/* unregister if necessary*/
	if (lc->sip_conf.reg_conf.use_registrar){
		do_registration(lc,0);
		sleep(1);
	}
	ortp_exit();
	osipua_exit();
}

