
/*
linphone
Copyright (C) 2000  Simon MORLAT (simon.morlat@free.fr)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "linphonecore.h"
#include <regctxt.h>
#include <stdlib.h>
#include <stdio.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

static char lock_name[80];
static char lock_set=0;

/* put a lock file in /tmp. this is called when linphone runs as a daemon*/
int set_lock_file()
{
	FILE *lockfile;
	
	snprintf(lock_name,80,"/tmp/linphone.%i",getuid());
	lockfile=fopen(lock_name,"w");
	if (lockfile==NULL)
	{
		printf("Failed to create lock file.\n");
		return(-1);
	}
	fprintf(lockfile,"%i",getpid());
	fclose(lockfile);
	lock_set=1;
	return(0);
}

/* looks if there is a lock file. If presents return its content (the pid of the already running linphone), if not found, returns -1*/
int get_lock_file()
{
	
	int pid;
	FILE *lockfile;
	
	snprintf(lock_name,80,"/tmp/linphone.%i",getuid());
	lockfile=fopen(lock_name,"r");
	if (lockfile==NULL)
		return(-1);
	fscanf(lockfile,"%i",&pid);
	fclose(lockfile);
	return(pid);
}

/* remove the lock file if it was set*/
int remove_lock_file()
{
	int err=0;
	if (lock_set)
	{
		err=unlink(lock_name);
		lock_set=0;
	}
	return(err);
}

static int esd_killed=0;
static int artsd_killed=0;

int kill_sound_daemons()
{
	char *command;
	int err;
	FILE *out;
	pid_t pid;
	
	/* try to kill all artsd*/
	command=g_strdup_printf("ps -u %s |grep artsd",getenv("LOGNAME"));
	out=popen(command,"r");
	if (out!=NULL)
	{
		do{
			err=fscanf(out,"%i",&pid);
			if (err==1) {
				kill(pid,SIGINT);
				artsd_killed=1;
			}
		}while(err==1);
		pclose(out);
	}
	g_free(command);
	/* kill aplay, because for some reason on kde3 it is spawned when artsd dies*/
	command=g_strdup_printf("ps -u %s |grep aplay",getenv("LOGNAME"));
	out=popen(command,"r");
	if (out!=NULL)
	{
		do{
			err=fscanf(out,"%i",&pid);
			if (err==1) {
				kill(pid,SIGINT);
			}
		}while(err==1);
		pclose(out);
	}
	g_free(command);
	
	/* do the same with esd*/
	command=g_strdup_printf("ps -u %s |grep esd",getenv("LOGNAME"));
	out=popen(command,"r");
	if (out!=NULL)
	{
		do{
			err=fscanf(out,"%i",&pid);
			if (err==1) {
				kill(pid,SIGINT);
				esd_killed=1;
			}
		}while(err==1);
		pclose(out);
	}
	g_free(command);
	return(0);
}

void restore_sound_daemons()
{
	if (esd_killed){
		esd_killed=0;
		g_message("Restarting esd...");
		g_spawn_command_line_async("esd",NULL);
	}
	if (artsd_killed){
		artsd_killed=0;
		g_message("Restarting artsd...");
		g_spawn_command_line_async("artsd",NULL);
	}
}

int linphone_core_set_alias(LinphoneCore *lc)
{
	char *contact;
	char *user,*domain;
	
	if (lc->ua==NULL) return(-EFAULT);
	
	user=lc->sip_conf.username;
 	domain=lc->sip_conf.hostname;
 	
	if (lc->sip_conf.sip_port==5060)
		contact=g_strdup_printf("sip:%s@%s",user,domain);	
	else contact=g_strdup_printf("sip:%s@%s:%i",user,domain,lc->sip_conf.sip_port);
	//g_message("Adding %s to the list of alias.\n",contact);
	osip_ua_add_alias(lc->ua,contact);
	if (lc->sip_conf.contact!=NULL) g_free(lc->sip_conf.contact);
	lc->sip_conf.contact=contact;
	return 0;
}

int linphone_core_update_contact_info(LinphoneCore *lc)
{
	int len;
	char *prim_contact;
	
	len=strlen(lc->net_conf.sel_if->ip4addr)+12+strlen(lc->sip_conf.username);
	prim_contact=g_malloc(len);
	if (lc->sip_conf.sip_port==5060)
		snprintf(prim_contact,len,"sip:%s@%s",lc->sip_conf.username,lc->net_conf.sel_if->ip4addr);
	else 
		snprintf(prim_contact,len,"sip:%s@%s:%i",lc->sip_conf.username,
			lc->net_conf.sel_if->ip4addr,lc->sip_conf.sip_port);
	osip_ua_set_contact(lc->ua,prim_contact);
	g_free(prim_contact);
	linphone_core_set_alias(lc);
}

/* send a REGISTER request to the default registrar*/
/* if doit is 0, then unregistration is done (expires is set to zero) */
gint do_registration(LinphoneCore *lc,gboolean doit)
{
	static RegistrationCtxt *regctx=NULL;
	/* prevent from emmitting registration for address 127.0.0.1...*/
	if (lc->sip_conf.reg_conf.registrar==NULL) return;
	if (strcmp(lc->net_conf.sel_if->ip4addr,"127.0.0.1")!=0)
	{
		OsipDialog *call=osip_dialog_new(lc->ua);
		if (regctx!=NULL) registration_ctxt_destroy(regctx);
		regctx=registration_ctxt_new();		
		registration_ctxt_set_registrar(regctx,lc->sip_conf.reg_conf.registrar);
		registration_ctxt_set_password(regctx, lc->sip_conf.reg_conf.passwd);
		registration_ctxt_set_address_of_record(regctx, lc->sip_conf.reg_conf.addr_of_rec);
		if (doit) {
			lc->vtable.display_status(lc,_("Registering..."));
			registration_ctxt_set_expires(regctx,lc->sip_conf.reg_conf.expires);
		}else {
			registration_ctxt_set_expires(regctx,0);
			//lc->vtable.display_status(lc,_("Unregistering..."));
		}
		
		g_timer_reset(lc->sip_conf.reg_conf.timer);
		osip_dialog_register(call,regctx);
		/* we don't want more from the dialog:*/
		osip_dialog_release(call);
		g_message("Registration sent.\n");
	}
	else g_message("Registrations with local loopback address are forbidden in linphone.\n");
	return TRUE;
}

/*try to open dsp. If failed, kill sound daemons if allowed by user, and try agin. If fail again, display the error message. Close it in success*/
int try_open_dsp(LinphoneCore *lc)
{
	int err1,err2,retval=-1;
	
	err1=test_audio_dev(lc->sound_conf.dev_id);
	
	if (err1>=0){
		retval=err1;
		goto end;
	}
	
	switch (err1){
		case -ENODEV: /* no such device */
		case -ENOENT: /* no such file */
			return err1;
	}
	
	/* else it is probably EWOULDBLOCK, EAGAIN */	
	if (lc->sound_conf.autokill)
	{
		g_message("killing sound daemons.\n");
		kill_sound_daemons();
		sleep(1);   /* wait for sound daemons to be killed...*/
	}
	else
	{
		
		return(-1);
	}
	err2=test_audio_dev(lc->sound_conf.dev_id);
	if (err2<0)
	{
			
		lc->vtable.display_warning(lc,_("Linphone could not open the audio device. Check if your sound card is fully configured and working."));
		return(-1);
	}
	retval=err2;
	
	end:
	if (retval>0) lc->sound_conf.latency=retval/8;
	return retval;
}

void check_for_registration(LinphoneCore *lc)
{
	registrar_config_t *reg_conf;
	reg_conf=&lc->sip_conf.reg_conf;
	if (reg_conf->use_registrar){
		if (reg_conf->expires>0){
			gdouble time;
			time=g_timer_elapsed(reg_conf->timer,NULL);
			if ( ((int)time) > reg_conf->expires) do_registration(lc,1);
		}
	}
}

char *int2str(int number)
{
	gchar *numstr=g_malloc(10);
	snprintf(numstr,10,"%i",number);
	return numstr;
}



void check_sound_device(LinphoneCore *lc)
{
	int fd,len;
	gint a;
	struct stat statbuf;
	char *file=NULL;
	char *i810_audio=NULL;
	char *snd_pcm_oss=NULL;
	char *snd_mixer_oss=NULL;
	char *snd_pcm=NULL;
	
	fd=open("/proc/modules",O_RDONLY);
	if (fd>0){
		/* read the entire /proc/modules file and check if sound conf seems correct */
		/*a=fstat(fd,&statbuf);
		if (a<0) g_warning("Can't stat /proc/modules:%s.",strerror(errno));
		len=statbuf.st_size;
		if (len==0) g_warning("/proc/modules has zero size!");
		*/
		/***** fstat does not work on /proc/modules for unknown reason *****/
		len=6000;
		file=g_malloc(len+1);
		a=read(fd,file,len);
		if (a<len) file=g_realloc(file,a+1);
		file[a]='\0';
		i810_audio=strstr(file,"i810_audio");
		if (i810_audio!=NULL){
			lc->vtable.display_warning(lc,_("You are currently using the i810_audio driver.\nThis driver is buggy and so does not work with Linphone.\nWe suggest that you replace it by its equivalent ALSA driver,\neither with packages from your distribution, or by downloading\nALSA drivers at http://www.alsa-project.org."));
			goto end;
		}
		snd_pcm=strstr(file,"snd-pcm");
		if (snd_pcm!=NULL){
			snd_pcm_oss=strstr(file,"snd-pcm-oss");
			snd_mixer_oss=strstr(file,"snd-mixer-oss");
			if (snd_pcm_oss==NULL){
				lc->vtable.display_warning(lc,_("Your computer appears to be using ALSA sound drivers.\nThis is the best choice. However the pcm oss emulation module\nis missing and linphone needs it. Please execute\n'modprobe snd-pcm-oss' as root to load it."));
			}
			if (snd_mixer_oss==NULL){
				lc->vtable.display_warning(lc,_("Your computer appears to be using ALSA sound drivers.\nThis is the best choice. However the mixer oss emulation module\nis missing and linphone needs it. Please execute\n 'modprobe snd-mixer-oss' as root to load it."));
			}
		}
	}else {
#ifdef __LINUX__
		g_warning("Could not open /proc/modules.");
#endif
	}
	/* now check general volume. Some user forget to rise it and then complain that linphone is
	not working */
	if (lc->sound_conf.sndcard!=NULL){
		a=snd_card_get_level(lc->sound_conf.sndcard,SND_CARD_LEVEL_GENERAL);
	
		a=a>>8;
		if (a<50){
			g_warning("General level is quite low (%i). Linphone rises it up for you.",a);
			snd_card_set_level(lc->sound_conf.sndcard,SND_CARD_LEVEL_GENERAL,80);
		}
	}
	end:
	if (file!=NULL) g_free(file);
	if (fd>0) close(fd);
}

void linphone_core_check_codecs_for_bandwidth(LinphoneCore *lc)
{
	GList *elem;
	MSCodecInfo *codinfo;
	
	elem=lc->codecs_conf.audio_codecs;
	while(elem!=NULL)
	{
		codinfo=(MSCodecInfo*)elem->data;
		codinfo->is_usable=ms_codec_is_usable(codinfo,lc->net_conf.bandwidth);
		elem=g_list_next(elem);
	}
}

void linphone_core_setup_local_rtp_profile(LinphoneCore *lc)
{
	int i;
	GList *elem;
	MSCodecInfo *codec;
	PayloadType *payload;
	lc->local_profile=&av_profile;
	for (i=0;i<127;i++)
	{
		payload=rtp_profile_get_payload(lc->local_profile,i);
		if (payload!=NULL){
			/* find a mediastreamer codec for this payload type */
			switch (payload->type){
				case PAYLOAD_AUDIO_CONTINUOUS:
				case PAYLOAD_AUDIO_PACKETIZED:
					codec=ms_audio_codec_info_get(payload->mime_type);
					break;
				case PAYLOAD_VIDEO:
					codec=ms_video_codec_info_get(payload->mime_type);
					break;
				default:
					g_error("Unsupported rtp media type.");
			}
			if (codec==NULL){
				//g_warning("Cannot find a codec for rtp payload %s.",payload->mime_type);
			}else payload_type_set_user_data(payload,(gpointer)codec);
		}
	}
}

int from_2char_without_params(from_t *from,char **str)
{
	gchar *tmp=NULL;
	*str=NULL;
	if (from->displayname!=NULL){
		if (from->url->username!=NULL){
			tmp=g_strdup_printf("%s <sip:%s@%s>",from->displayname,from->url->username,
								from->url->host);
		}else{
			tmp=g_strdup_printf("%s <sip:%s>",from->displayname,
								from->url->host);
		}
	}else{
		if (from->url->username!=NULL){
			tmp=g_strdup_printf("sip:%s@%s",from->url->username,from->url->host);
		}else{
			tmp=g_strdup_printf("sip:%s",from->url->host);
		}
		
	}
	g_return_val_if_fail(tmp!=NULL,-1);
	*str=sgetcopy(tmp);
	g_free(tmp);
	return 0;
}

