/*
  The mediastreamer library aims at providing modular media processing and I/O
	for linphone, but also for any telephony application.
  Copyright (C) 2001  Simon MORLAT simon.morlat@linphone.org
  										
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_ALSA_ASOUNDLIB_H


#include "msossread.h"
#include "msosswrite.h"




int alsa_set_params(snd_pcm_t *pcm_handle, int bits, int stereo, int rate)
{
	snd_pcm_hw_params_t *hwparams;
	gint dir,exact_rate;
	gint channels;
	gint fsize;
	gint periods=2;
	gint periodsize=1024;
	
	/* Allocate the snd_pcm_hw_params_t structure on the stack. */
    snd_pcm_hw_params_alloca(&hwparams);
	
	/* Init hwparams with full configuration space */
    if (snd_pcm_hw_params_any(pcm_handle, hwparams) < 0) {
		g_warning("alsa_set_params: Cannot configure this PCM device.\n");
		return(-1);
    }
	
	if (snd_pcm_hw_params_set_access(pcm_handle, hwparams, SND_PCM_ACCESS_RW_INTERLEAVED) < 0) {
      g_warning("alsa_set_params: Error setting access.\n");
      return(-1);
    }
	/* Set sample format */
    if (snd_pcm_hw_params_set_format(pcm_handle, hwparams, SND_PCM_FORMAT_S16_LE) < 0) {
      g_warning("alsa_set_params: Error setting format.\n");
      return(-1);
    }
	/* Set sample rate. If the exact rate is not supported */
    /* by the hardware, use nearest possible rate.         */ 
    exact_rate = snd_pcm_hw_params_set_rate_near(pcm_handle, hwparams, rate, &dir);
    if (dir != 0) {
      g_warning("alsa_set_params: The rate %d Hz is not supported by your hardware.\n "
		"==> Using %d Hz instead.\n", rate, exact_rate);
    }
	/* Set number of channels */
	if (stereo) channels=2;
	else channels=1;
    if (snd_pcm_hw_params_set_channels(pcm_handle, hwparams, channels) < 0) {
      g_warning("alsa_set_params: Error setting channels.\n");
      return(-1);
    }
	/* Set number of periods. Periods used to be called fragments. */ 
    if (snd_pcm_hw_params_set_periods(pcm_handle, hwparams, periods, 0) < 0) {
      g_warning("alsa_set_params: Error setting periods.\n");
      return(-1);
    }
	/* Set buffer size (in frames). The resulting latency is given by */
    /* latency = periodsize * periods / (rate * bytes_per_frame)     */
    if ((fsize=snd_pcm_hw_params_set_buffer_size_near(pcm_handle, hwparams, (periodsize * periods)>>channels)) < 0) {
      g_warning("alsa_set_params: Error setting buffersize.\n");
      return(-1);
    }
	/* Apply HW parameter settings to */
    /* PCM device and prepare device  */
    if (snd_pcm_hw_params(pcm_handle, hwparams) < 0) {
      g_warning("alsa_set_params: Error setting HW params.\n");
      return(-1);
    }
	obj->frame_size=channels*(bits/8);
	SND_CARD(obj)->bsize=fsize*obj->frame_size;
	obj->frames=fsize;
	g_message("alsa_set_params: blocksize is %i.",SND_CARD(obj)->bsize);
	return bsize;	
}

int alsa_card_open_r(AlsaCard *obj,int bits,int stereo,int rate)
{
	int tmp;
	snd_pcm_t *pcm_handle;
	
	if (snd_pcm_open(&pcm_handle, obj->pcmdev,SND_PCM_STREAM_CAPTURE,SND_PCM_NONBLOCK) < 0) {
      g_warning("alsa_card_open_r: Error opening PCM device %s\n", );
      return(-1);
    }
	if ((tmp=alsa_set_params(pcm_handle,bits,stereo,rate))<0){
		snd_pcm_close(pcm_handle);
		return -1;
	}
	obj->read_handle=pcm_handle;
	SND_CARD(obj)->bsize=tmp;
	SND_CARD(obj)->flags|=SND_CARD_FLAGS_OPENED;
	return 0;
}

int alsa_card_open_w(AlsaCard *obj,int bits,int stereo,int rate)
{
	int tmp;
	snd_pcm_t *pcm_handle;
	
	if (snd_pcm_open(&pcm_handle, obj->pcmdev,SND_PCM_STREAM_PLAYBACK,SND_PCM_NONBLOCK) < 0) {
      g_warning("alsa_card_open_r: Error opening PCM device %s\n", );
      return(-1);
    }
	if ((tmp=alsa_set_params(pcm_handle,bits,stereo,rate))<0){
		snd_pcm_close(pcm_handle);
		return -1;
	}
	obj->write_handle=pcm_handle;
	SND_CARD(obj)->bsize=tmp;
	SND_CARD(obj)->flags|=SND_CARD_FLAGS_OPENED;
	return 0;
}



void alsa_card_close_r(AlsaCard *obj)
{
	if (obj->read_handle!=NULL){
		snd_pcm_close(obj->read_handle);
		obj->read_handle=NULL;
	}
}

void alsa_card_close_w(AlsaCard *obj)
{
	if (obj->write_handle!=NULL){
		snd_pcm_close(obj->write_handle);
		obj->write_handle=NULL;
	}
}

int alsa_card_probe(AlsaCard *obj,int bits,int stereo,int rate)
{
	int ret;
	ret=alsa_card_open_r(obj,bits,stereo,rate);
	alsa_card_close_r(obj);
	return ret;
}


void alsa_card_destroy(AlsaCard *obj)
{
	snd_card_uninit(SND_CARD(obj));
	g_free(obj->pcmdev);
}

gboolean alsa_card_can_read(AlsaCard *obj)
{
	g_return_val_if_fail(obj->read_handle!=NULL,0);
	if ( snd_pcm_avail_update(obj->read_handle)>=obj->frames) return 1;
	return 0;
}

int alsa_card_read(AlsaCard *obj,char *buf,int size)
{
	int err;
	
	err=snd_pcm_readi(obj->read_handle,buf,size/obj->frame_size);
	if (err<0) {
		g_warning("alsa_card_read: snd_pcm_readi failed:%s.",strerror(err));
	}
	return err;
}

int alsa_card_write(AlsaCard *obj,char *buf,int size)
{
	int err;
	err=snd_pcm_writei(obj->write_handle,obj->writebuf,size/obj->frame_size);
	if (err<0) g_warning("alsa_card_write: Error writing sound buffer.");
}

void alsa_card_set_level(AlsaCard *obj,gint way,gint a)
{
	
}

gint alsa_card_get_level(AlsaCard *obj,gint way)
{
	
	return 0;
}

void alsa_card_set_source(AlsaCard *obj,int source)
{
	
}

MSFilter *alsa_card_create_read_filter(AlsaCard *card)
{
	MSFilter *f=ms_alsa_oss_new();
	ms_oss_read_set_device(MS_OSS_READ(f),SND_CARD(card)->index);
	return f;
}

MSFilter *alsa_card_create_write_filter(AlsaCard *card)
{
	MSFilter *f=ms_oss_write_new();
	ms_oss_write_set_device(MS_OSS_WRITE(f),SND_CARD(card)->index);
	return f;
}


SndCard * alsa_card_new(gint devid)
{
	AlsaCard * obj;
	SndCard *base;
	int err;
	gchar *name=NULL;
	
	/* test if the card exists */
	/* carefull: this is an alsalib call despite its name! */
	err=snd_card_get_name(devid,&name);
	if (err<0) {
		return NULL;
	}
	obj= g_new0(AlsaCard,1);
	base= SND_CARD(obj);
	snd_card_init(base);
	
	base->card_name=name;
	base->_probe=(SndCardOpenFunc)alsa_card_probe;
	base->_open_r=(SndCardOpenFunc)alsa_card_open_r;
	base->_open_w=(SndCardOpenFunc)alsa_card_open_w;
	base->_can_read=(SndCardPollFunc)alsa_card_can_read;
	base->_read=(SndCardIOFunc)alsa_card_read;
	base->_write=(SndCardIOFunc)alsa_card_write;
	base->_close_r=(SndCardCloseFunc)alsa_card_close_r;
	base->_close_w=(SndCardCloseFunc)alsa_card_close_w;
	base->_set_rec_source=(SndCardMixerSetRecSourceFunc)alsa_card_set_source;
	base->_set_level=(SndCardMixerSetLevelFunc)alsa_card_set_level;
	base->_get_level=(SndCardMixerGetLevelFunc)alsa_card_get_level;
	base->_destroy=(SndCardDestroyFunc)alsa_card_destroy;
	base->_create_read_filter=(SndCardCreateFilterFunc)alsa_card_create_read_filter;
	base->_create_write_filter=(SndCardCreateFilterFunc)alsa_card_create_write_filter;
	
	obj->pcmdev=g_strdup_printf("hw:%i,0",devid);
	return base;
}



#endif


