/*
  The mediastreamer library aims at providing modular media processing and I/O
	for linphone, but also for any telephony application.
  Copyright (C) 2001  Simon MORLAT simon.morlat@linphone.org
  										
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include "msoss.h"
#include <string.h>
#include <errno.h>
#include <stdio.h>

#ifdef HAVE_SYS_SOUNDCARD_H
#include <sys/soundcard.h>
#endif 
#ifdef HAVE_SYS_AUDIO_H
#include <sys/audio.h>
#endif

#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <glib.h>

#define DEV_NAME_LEN 20
#define DEVICE_PREFIX "/dev/dsp"


#if GLIB_MAJOR_VERSION < 2
gint g_file_test(char *a,gint cond) 
{
	gint err;
	struct stat tmp;
	err=stat(a,&tmp);
	return err+1;
}
#define G_FILE_TEST_EXISTS 1
#endif


/* rewrite all this: create a SndCardManager containing all the SndCardManager
each SndCard can have its own get_buf() and put_buf() method.
The SndCard is a wrapper to the driver methods (oss,alsa,alsa-oss).
All oss cards should be accessed using _blocking_ i/o by a 
separate thread.*/


MSOss oss_mgr;

static gint oss_scan_dev(MSOss *oss, gchar *devfile);

static void snd_card_init(SndCard *card)
{
	memset(card,0,sizeof(SndCard));
}

/* scans all possible /dev/... that may be oss device file and configure the MSOss obj according
to the results*/
gint ms_oss_autoconf(MSOss *obj)
{
	gint found;
	found=oss_scan_dev(obj,"/dev/dsp");  /*on linux it should work every time*/
	if (found) return 0;
	found=oss_scan_dev(obj,"/dev/audio"); /* for solaris, hpux ...*/
	if (found) return 0;
	g_warning("No oss devices found.");
	return -1;
}

static char * oss_get_card_name(char *devfile)
{
	int fd;
	fd=open(devfile,O_RDONLY|O_NONBLOCK);
	/* need to find a way to get a string to identify the card..*/
	return NULL;
}

/*scans one type of device file */
static gint oss_scan_dev(MSOss *obj, char *devfile)
{
	char *name,*mixname;
	gint i,len=strlen(devfile)+2;

	name=g_malloc(len);
	strcpy(name,devfile);
	mixname=g_malloc(strlen("/dev/mixer")+2);
	strcpy(mixname,"/dev/mixer");
	for (i=0;i<MSOSS_MAX_DEVICES;i++)
	{
		
		if (g_file_test(name,G_FILE_TEST_EXISTS))
		{
			ms_oss_set_dev_name(obj,i,name);
			obj->wcards++;
			g_message("Detected %s - ",name);
		}
		else obj->card[i].dev_name=NULL;
		sprintf(name,"%s%i",devfile,i);
		
		if (g_file_test(mixname,G_FILE_TEST_EXISTS))
		{
			ms_oss_set_mixdev_name(obj,i,mixname);
		}
		else
		{
		   obj->card[i].mixdev_name=NULL;
		   //g_warning("Unable to find mixer for id %i.",i);
		}
		sprintf(mixname,"%s%i","/dev/mixer",i);
	}
	g_free(name);
	return obj->wcards;
}

void ms_oss_init(MSOss *klass)
{
	int i;

	for(i=0;i<MSOSS_MAX_DEVICES;i++)
	{
		snd_card_init(&klass->card[i]);
	}
	/* find devices */
	ms_oss_autoconf(klass);
	klass->defcard=0;
}

/* open a sound device for duplex */
int ms_oss_opendev(MSOss *obj,int devid, int bits, int stereo, int rate, int blocksize)
{
	int fd,p;
	gchar *buffer;
	SndCard *card=&obj->card[devid];
	gint min_size=0,cond,i;

	g_return_val_if_fail(devid<MSOSS_MAX_DEVICES,-EINVAL);
	if (card->fd!=0) return -EBUSY;
	g_return_val_if_fail(card->dev_name!=NULL,-EFAULT);

	if (blocksize<512){
		blocksize=512;
	}
	/* do a quick non blocking open to be sure that we are not going to be blocked here
		for the eternity */
	fd=open(card->dev_name,O_RDWR|O_NONBLOCK);
	if (fd<0) return -EWOULDBLOCK;
	close(fd);
	/* open the device */
	fd=open(card->dev_name,O_RDWR);

	g_return_val_if_fail(fd>0,-errno);
#ifdef HAVE_SYS_SOUNDCARD_H
	ioctl(fd, SNDCTL_DSP_RESET, 0);

	p =  bits;  /* 16 bits */
	ioctl(fd, SNDCTL_DSP_SAMPLESIZE, &p);

	p =  stereo;  /* number of channels */
	ioctl(fd, SNDCTL_DSP_CHANNELS, &p);

	p =  rate;  /* rate in khz*/
	ioctl(fd, SNDCTL_DSP_SPEED, &p);

	ioctl(fd, SNDCTL_DSP_GETBLKSIZE, &min_size);

	/* try to subdivide BLKSIZE to reach blocksize if necessary */
	if (min_size>blocksize)
  {
		cond=1;
    	p=min_size/blocksize;
    	while(cond)
    	{
			i=ioctl(fd, SNDCTL_DSP_SUBDIVIDE, &p);
			//printf("SUB_DIVIDE said error=%i,errno=%i\n",i,errno);
     	if ((i==0) || (p==1)) cond=0;
     	else p=p/2;
     }
	}
	ioctl(fd, SNDCTL_DSP_GETBLKSIZE, &min_size);
	if (min_size>blocksize)
	{
		g_warning("dsp block size set to %i.",min_size);
	}else{
		/* no need to access the card with less latency than needed*/
		min_size=blocksize;
	}
#endif
#ifdef HAVE_SYS_AUDIO_H
	ioctl(fd,AUDIO_RESET,0);
	ioctl(fd,AUDIO_SET_SAMPLE_RATE,rate);
	ioctl(fd,AUDIO_SET_CHANNELS,stereo);
	p=AUDIO_FORMAT_LINEAR16BIT;
	ioctl(fd,AUDIO_SET_DATA_FORMAT,p); 
	/* ioctl(fd,AUDIO_GET_RXBUFSIZE,&min_size); does not work ? */
	min_size=512;
#endif
	g_message("dsp blocksize is %i.",min_size);
	card->fd=fd;
	card->bits=bits;
	card->stereo=stereo;
	card->rate=rate;
	card->blocksize=min_size;
	buffer=g_malloc(min_size);
	/* make a dummy read to start I/O */
	read(fd,buffer,min_size);
	g_free(buffer);
	//ioctl(fd,SNDCTL_DSP_PROFILE,APF_CPUINTENS);
	
	return 0;
}

void ms_oss_set_default_card(MSOss *obj, int index)
{
	obj->defcard=index;
}

/* close a sound device */
void ms_oss_closedev(MSOss *obj,int devid)
{
	SndCard *card=&obj->card[devid];
	g_return_if_fail(devid<MSOSS_MAX_DEVICES);
	if (card->fd!=0) {
		close(card->fd);
		card->fd=0;
	}
}

/* set the path to device id */
void ms_oss_set_dev_name(MSOss *klass,int devid, char *path)
{
	SndCard *card;
	g_return_if_fail(devid<MSOSS_MAX_DEVICES);
	card=&klass->card[devid];
	if (card->dev_name!=NULL) g_free(card->dev_name);
	card->dev_name=g_strdup(path);;
}

/* set the path to mixer device id */
void ms_oss_set_mixdev_name(MSOss *klass,int devid, char *path)
{
	SndCard *card;
	g_return_if_fail(devid<MSOSS_MAX_DEVICES);
	card=&klass->card[devid];
	if (card->mixdev_name!=NULL) g_free(card->mixdev_name);
	card->mixdev_name=g_strdup(path);
}

/*set recording level */
void ms_oss_set_rlevel(MSOss *obj, gint devid, gint a)
{
   SndCard *card;
   int p,mix_fd;
	g_return_if_fail(devid<MSOSS_MAX_DEVICES);
	card=&obj->card[devid];
	g_return_if_fail(card->mixdev_name!=NULL);
#ifdef HAVE_SYS_SOUNDCARD_H
	p=(((int)a)<<8 | (int)a);
	mix_fd = open(card->mixdev_name, O_WRONLY);
	ioctl(mix_fd,MIXER_WRITE(SOUND_MIXER_RECLEV), &p);
	close(mix_fd);
#endif
}

/*set playback level */
void ms_oss_set_plevel(MSOss *obj, gint devid, gint a)
{
   SndCard *card;
   int p,mix_fd;
	g_return_if_fail(devid<MSOSS_MAX_DEVICES);
	card=&obj->card[devid];
	g_return_if_fail(card->mixdev_name!=NULL);
#ifdef HAVE_SYS_SOUNDCARD_H
	p=(((int)a)<<8 | (int)a);
	mix_fd = open(card->mixdev_name, O_WRONLY);
	ioctl(mix_fd,MIXER_WRITE(SOUND_MIXER_PCM), &p);
	close(mix_fd);
#endif
}

/* set general output level*/
void ms_oss_set_glevel(MSOss *obj,gint devid, gint a)
{
	SndCard *card;
	gint p,mix_fd;
	g_return_if_fail(devid<MSOSS_MAX_DEVICES);
	card=&obj->card[devid];
	g_return_if_fail(card->mixdev_name!=NULL);
#ifdef HAVE_SYS_SOUNDCARD_H
	p=(((int)a)<<8 | (int)a);
	mix_fd = open(card->mixdev_name, O_WRONLY);
	ioctl(mix_fd,MIXER_WRITE(SOUND_MIXER_VOLUME), &p);
	close(mix_fd);
#endif
	
}

/* set general output level*/
gint ms_oss_get_glevel(MSOss *obj,gint devid)
{
	SndCard *card;
	gint p=0,mix_fd;
	g_return_val_if_fail(devid<MSOSS_MAX_DEVICES,-1);
	card=&obj->card[devid];
	g_return_val_if_fail(card->mixdev_name!=NULL,-1);
#ifdef HAVE_SYS_SOUNDCARD_H
	mix_fd = open(card->mixdev_name, O_RDONLY);
	ioctl(mix_fd,MIXER_READ(SOUND_MIXER_VOLUME), &p);
	close(mix_fd);
#endif
	return p;
}
void ms_oss_set_source(MSOss *obj, gint devid, gchar source)
{
	int i, p, mix_fd;
	SndCard *card;
	g_return_if_fail(devid<MSOSS_MAX_DEVICES);
	card=&obj->card[devid];
	g_return_if_fail(card->mixdev_name!=NULL);
#ifdef HAVE_SYS_SOUNDCARD_H	
	if (source == 'c')
		p = 1 << SOUND_MIXER_CD;
	if (source == 'l')
		p = 1 << SOUND_MIXER_LINE;
	if (source == 'm')
		p = 1 << SOUND_MIXER_MIC;

	
	mix_fd = open(card->mixdev_name, O_WRONLY);
	i = ioctl(mix_fd, SOUND_MIXER_WRITE_RECSRC, &p);
	close(mix_fd);
#endif
}




