/*
  The mediastreamer library aims at providing modular media processing and I/O
	for linphone, but also for any telephony application.
  Copyright (C) 2001  Simon MORLAT simon.morlat@linphone.org
  										
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "msringplayer.h"
#include "mssync.h"
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <string.h>
#include <errno.h>

#define WAV_HEADER_OFFSET 48 /*should be set properly...*/

static MSRingPlayerClass *ms_ring_player_class=NULL;

/**
 * ms_ring_player_new:
 * @name:  The path to the 16-bit 8khz raw file to be played as a ring.
 * @seconds: The number of seconds that separates two rings.
 *
 * Allocates a new MSRingPlayer object.
 *
 *
 * Returns: a pointer the the object, NULL if name could not be open.
 */
MSFilter * ms_ring_player_new(char *name, gint seconds)
{
	MSRingPlayer *r;
	int fd=-1;
	
	if ((name!=NULL) && (strlen(name)!=0))
	{
		fd=open(name,O_RDONLY);
		if (fd<0) 
		{
			g_warning("ms_ring_player_new: failed to open %s.\n",name);
			return NULL;
		}
		
	}else {
		g_warning("ms_ring_player_new: Bad file name");
		return NULL;
	}
	
	r=g_new(MSRingPlayer,1);
	ms_ring_player_init(r);
	if (ms_ring_player_class==NULL)
	{
		ms_ring_player_class=g_new(MSRingPlayerClass,1);
		ms_ring_player_class_init(ms_ring_player_class);
	}
	MS_FILTER(r)->klass=MS_FILTER_CLASS(ms_ring_player_class);
	
	r->fd=fd;
	r->space=seconds*8000;
	if (strstr(name,".wav")!=NULL){
		/* skip the header */
		lseek(fd,WAV_HEADER_OFFSET,SEEK_SET);
#ifdef WORDS_BIGENDIAN
		r->need_swap=1;	
#else
		r->need_swap=0;
#endif
	}
	return(MS_FILTER(r));
}
	

/* FOR INTERNAL USE*/
void ms_ring_player_init(MSRingPlayer *r)
{
	ms_filter_init(MS_FILTER(r));
	MS_FILTER(r)->outfifos=r->foutputs;
	MS_FILTER(r)->outqueues=r->qoutputs;
	memset(r->foutputs,0,sizeof(MSFifo*)*MS_RING_PLAYER_MAX_OUTPUTS);
	memset(r->qoutputs,0,sizeof(MSQueue*)*MS_RING_PLAYER_MAX_OUTPUTS);
	r->fd=-1;
	r->current_pos=0;
	r->need_swap=0;
}

void ms_ring_player_class_init(MSRingPlayerClass *klass)
{
	ms_filter_class_init(MS_FILTER_CLASS(klass));
	ms_filter_class_set_name(MS_FILTER_CLASS(klass),"ringplay");
	ms_filter_class_set_attr(MS_FILTER_CLASS(klass),FILTER_IS_SOURCE);
	MS_FILTER_CLASS(klass)->max_foutputs=MS_RING_PLAYER_MAX_OUTPUTS;
	MS_FILTER_CLASS(klass)->max_qoutputs=MS_RING_PLAYER_MAX_OUTPUTS;
	MS_FILTER_CLASS(klass)->w_maxgran=MS_RING_PLAYER_DEF_GRAN;
	MS_FILTER_CLASS(klass)->setup=(MSFilterSetupFunc)ms_ring_player_setup;
	MS_FILTER_CLASS(klass)->destroy=(MSFilterDestroyFunc)ms_ring_player_destroy;
	MS_FILTER_CLASS(klass)->process=(MSFilterProcessFunc)ms_ring_player_process;
}
	
void ms_ring_player_process(MSRingPlayer *r)
{
	MSFifo *f;
	int err;
	gint gran=2*ms_sync_get_samples_per_tick(r->sync);
	void *p;
	
	g_return_if_fail(gran>0);
	/* process output fifos*/
	
	f=r->foutputs[0];
	if (f!=NULL)
	{
		err=ms_fifo_get_write_ptr(f,gran,&p);
		if (err>0)
		{
			//ms_trace("ms_ring_player_process: gran=%i",gran);
			if (r->current_pos==0)
			{
				err=read(r->fd,p,gran);
				//g_message("diskreader:  file.");
				if (err<0)
				{
					/* temp: */
					g_warning("ms_ring_player_process: failed to read: %s.\n",strerror(errno));
				}
				else if (err<gran)
				{
					r->current_pos=r->space;
					memset(((char*)p)+err,0,gran-err);
					//ms_trace("msringplayer: end of file.");
				}
				if (r->need_swap) swap_buffer(p,gran);
			}   /* copy out silence */
			else
			{
				memset(p,0,gran);
				r->current_pos-=gran;
				if (r->current_pos<=0)
				{
					r->current_pos=0;
					lseek(r->fd,WAV_HEADER_OFFSET,SEEK_SET);
				}
			}
		}
	}
	
}

/**
 * ms_ring_player_destroy:
 * @obj: A valid MSRingPlayer object.
 *
 * Destroy a MSRingPlayer object.
 *
 *
 */

void ms_ring_player_destroy( MSRingPlayer *obj)
{
	if (obj->fd!=0) close(obj->fd);
	g_free(obj);
}

void ms_ring_player_setup(MSRingPlayer *r,MSSync *sync)
{
	r->sync=sync;
}

