 /*
  The mediastreamer library aims at providing modular media processing and I/O
	for linphone, but also for any telephony application.
  Copyright (C) 2001  Simon MORLAT simon.morlat@linphone.org
  										
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include "mstimer.h"
#include <unistd.h>
#include <sys/types.h>
#include <sys/time.h>
#include <signal.h>

static MSTimerClass *ms_timer_class=NULL;

static void dummy_handler(int signum)
{
	//posix_timer_time+=POSIXTIMER_INTERVAL/1000;
	//alarm_received++;
	//printf("tick !!!\n");
}

static void posix_timer_init(MSTimer *posix_timer)
{
	struct itimerval timer;
	sigset_t set;
	/* block the SIGALRM signal 
	sigemptyset(&set);
	sigaddset(&set,SIGALRM);
	sigprocmask(SIG_BLOCK,&set,NULL);*/
	
	timer.it_value.tv_sec=posix_timer->interval.tv_sec;
	timer.it_value.tv_usec=posix_timer->interval.tv_usec;
	timer.it_interval.tv_sec=posix_timer->interval.tv_sec;
	timer.it_interval.tv_usec=posix_timer->interval.tv_usec;
	
	signal(SIGALRM,dummy_handler);
	setitimer(ITIMER_REAL,&timer,NULL);
	posix_timer->state=MS_TIMER_RUNNING;
	gettimeofday(&posix_timer->orig,NULL);
	posix_timer->late_ticks=0;
	posix_timer->time=0;
}



static void posix_timer_do(MSTimer *posix_timer)
{
	sigset_t set;
	guint32 time;
	gint32 diff;
	
	if (posix_timer->late_ticks>0){
		posix_timer->late_ticks--;
		posix_timer->time+=posix_timer->milisec;
		MS_SYNC(posix_timer)->time=posix_timer->time;
		return;
	}
	
	sigfillset(&set);
	sigdelset(&set,SIGALRM);
	
	//printf("time=%i\n",time);
	sigsuspend(&set);
		/* here we receive the signal */
	posix_timer->time+=posix_timer->milisec;
	MS_SYNC(posix_timer)->time=posix_timer->time;
	
	gettimeofday(&posix_timer->cur,NULL);
	time=((posix_timer->cur.tv_usec-posix_timer->orig.tv_usec)/1000 ) + ((posix_timer->cur.tv_sec-posix_timer->orig.tv_sec)*1000 );
	diff=time-posix_timer->time;
	
	if (diff> posix_timer->milisec){
		posix_timer->late_ticks=diff/posix_timer->milisec;
		if (posix_timer->late_ticks>5) g_warning("MSTimer: must catchup %i ticks.",posix_timer->late_ticks);
		return;
	}
}

static void posix_timer_uninit(MSTimer *posix_timer)
{
	struct itimerval timer;
	/* unset the timer */
	memset(&timer,0,sizeof(struct itimerval));
	setitimer(ITIMER_REAL,&timer,NULL);
	posix_timer->state=MS_TIMER_STOPPED;
}


void ms_timer_init(MSTimer *sync)
{
	ms_sync_init(MS_SYNC(sync));
	MS_SYNC(sync)->attached_filters=sync->filters;
	memset(sync->filters,0,MSTIMER_MAX_FILTERS*sizeof(MSFilter*));
	MS_SYNC(sync)->samples_per_tick=160;
	ms_timer_set_interval(sync,20);
	sync->state=MS_TIMER_STOPPED;
}

void ms_timer_class_init(MSTimerClass *klass)
{
	ms_sync_class_init(MS_SYNC_CLASS(klass));
	MS_SYNC_CLASS(klass)->max_filters=MSTIMER_MAX_FILTERS;
	MS_SYNC_CLASS(klass)->synchronize=(MSSyncSyncFunc)ms_timer_synchronize;
	MS_SYNC_CLASS(klass)->destroy=(MSSyncDestroyFunc)ms_timer_destroy;
	/* no need to overload these function*/
	MS_SYNC_CLASS(klass)->attach=ms_sync_attach_generic;
	MS_SYNC_CLASS(klass)->detach=ms_sync_detach_generic;
}

void ms_timer_destroy(MSTimer *timer)
{
	g_free(timer);
}


void ms_timer_synchronize(MSTimer *timer)
{
	//printf("ticks=%i \n",MS_SYNC(timer)->ticks);
	if (timer->state==MS_TIMER_STOPPED){
		posix_timer_init(timer);
		//printf("ms_timer_synchronize: starting timer.\n");
	}
	else posix_timer_do(timer);
	return;
}


MSSync *ms_timer_new()
{
	MSTimer *timer;
	
	timer=g_malloc(sizeof(MSTimer));
	ms_timer_init(timer);
	if (ms_timer_class==NULL)
	{
		ms_timer_class=g_new(MSTimerClass,1);
		ms_timer_class_init(ms_timer_class);
	}
	MS_SYNC(timer)->klass=MS_SYNC_CLASS(ms_timer_class);
	return(MS_SYNC(timer));
}

void ms_timer_set_interval(MSTimer *timer, int milisec)
{
	
	MS_SYNC(timer)->ticks=0;
	timer->interval.tv_sec=milisec/1000;
	timer->interval.tv_usec=(milisec % 1000)*1000;
	timer->milisec=milisec;
	
	
}

