 /*
  * The osipua library is a library based on oSIP that implements CallLeg and User Agent
  * level.
  * Copyright (C) 2001  Simon MORLAT simon.morlat@free.fr
  * Aymeric MOIZARD jack@atosc.org
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Lesser General Public
  * License as published by the Free Software Foundation; either
  * version 2.1 of the License, or (at your option) any later version.
  * 
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  * Lesser General Public License for more details.
  * 
  * You should have received a copy of the GNU Lesser General Public
  * License along with this library; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
  */

#include "sdphandler.h"
#include "utils.h"


int sdp_payload_init (SdpPayload * payload)
{
	memset(payload,0,sizeof(SdpPayload));
	return 0;
}

void sdp_payload_uninit(SdpPayload * payload)
{
	
}

static SdpHandlerClass *sdp_handler_class = NULL;

void
sdp_handler_init (SdpHandler * config)
{
	memset(config,0,sizeof(SdpHandler));
	body_handler_init (BODY_HANDLER (config));
}

static void stub(){
}

void
sdp_handler_class_init (SdpHandlerClass * klass)
{
	BodyHandlerClass *bklass = BODY_HANDLER_CLASS (klass);
	body_handler_class_init (bklass);
	bklass->mime_type = "application/sdp";
	bklass->_body_context_new = (BodyContextNewFunc) sdp_context_new;
	bklass->_init = (BodyHandlerFunc) stub;

}

BodyHandler *
sdp_handler_new ()
{
	SdpHandler *obj;
	if (sdp_handler_class == NULL)
	{
		sdp_handler_class = smalloc (sizeof (SdpHandlerClass));
		sdp_handler_class_init (sdp_handler_class);
	}
	obj = smalloc (sizeof (SdpHandler));
	sdp_handler_init (obj);
	BODY_HANDLER (obj)->klass = BODY_HANDLER_CLASS (sdp_handler_class);
	return BODY_HANDLER (obj);
}


void
sdp_handler_uninit (SdpHandler * config)
{	
	
}

void
sdp_handler_destroy (SdpHandler * obj)
{
	sdp_handler_uninit (obj);
	sfree (obj);
}

void sdp_handler_set_accept_offer_fcn(SdpHandler *sh,SdpHandlerReadCodecFunc audiofunc,
										SdpHandlerReadCodecFunc videofunc)
{
	sh->accept_audio_codecs=audiofunc;
	sh->accept_video_codecs=videofunc;
}

void sdp_handler_set_write_offer_fcn(SdpHandler *sh,SdpHandlerWriteCodecFunc audiofunc,
										SdpHandlerWriteCodecFunc videofunc)
{
	sh->set_audio_codecs=audiofunc;
	sh->set_video_codecs=videofunc;
}

void sdp_handler_set_read_answer_fcn(SdpHandler *sh,SdpHandlerReadCodecFunc audiofunc,
										SdpHandlerReadCodecFunc videofunc)
{
	sh->get_audio_codecs=audiofunc;
	sh->get_video_codecs=videofunc;
}


/* guess the sdp config from the ua config, and generate a template sdp */
sdp_t *
sdp_handler_generate_template(SdpHandler * obj)
{
	OsipUA *ua = BODY_HANDLER (obj)->ua;
	url_t *url = contact_geturl (ua->contact);
	sdp_t *local;
	int err;

	if (ua == NULL)
	{
		osip_trace (OSIP_ERROR, ("SdpHandler: ua is NULL"));
		return;
	}

	sdp_init (&local);

	sdp_v_version_set (local, sgetcopy ("0"));
	sdp_o_origin_set (local, sgetcopy (url_getusername (url)),
			  sgetcopy ("123456"), sgetcopy ("654321"),
			  sgetcopy ("IN"), sgetcopy ("IP4"),
			  sgetcopy (ua->ua_ip4addr));
	sdp_s_name_set (local, sgetcopy ("A conversation"));

	if (ua->mute_ip4addr != NULL)
	    sdp_c_connection_add (local, -1,
				  sgetcopy ("IN"), sgetcopy ("IP4"),
				  sgetcopy (ua->mute_ip4addr), NULL, NULL);
	else
	    sdp_c_connection_add (local, -1,
				  sgetcopy ("IN"), sgetcopy ("IP4"),
				  sgetcopy (ua->ua_ip4addr), NULL, NULL);
	    
	sdp_t_time_descr_add (local, sgetcopy ("0"), sgetcopy ("0"));

	return local;
}

/* to add payloads to the offer, must be called inside the write_offer callback */
void sdp_handler_add_payload(SdpHandler *sh,SdpContext *ctx,SdpPayload *payload,char *media)
{
	sdp_t *offer=ctx->offer;
	char *tmp;
	char *attr_field;
	if (!ctx->incb){
		osip_trace(OSIP_ERROR,("You must not call sdp_handler_add_*_payload outside the write_offer callback\n"));
		abort();
	}
	if (payload->proto==NULL) payload->proto="RTP/AVP";
	if (sdp_m_media_get(offer,payload->line)==NULL){
		/* need a new line */
		sdp_m_media_add(offer,sgetcopy(media),int_2char(payload->localport),
						NULL,sgetcopy(payload->proto));
	}
	sdp_m_payload_add(offer,payload->line,int_2char(payload->pt));
	if (payload->a_rtpmap!=NULL){
		attr_field=sstrdup_sprintf("%i %s",payload->pt,payload->a_rtpmap);
		sdp_a_attribute_add(offer,payload->line,sgetcopy("rtpmap"),attr_field);
	}
	if (payload->a_fmtp!=NULL){
		attr_field=sstrdup_sprintf("%i %s",payload->pt,payload->a_fmtp);
		sdp_a_attribute_add(offer,payload->line,sgetcopy("fmtp"),attr_field);
	}
	if (payload->b_as_bandwidth!=0){
		attr_field=sstrdup_sprintf("%i %i",payload->pt,payload->b_as_bandwidth);
		sdp_b_bandwidth_add (offer,payload->line,sgetcopy("AS"),attr_field);
	}
}

void sdp_handler_add_audio_payload(SdpHandler *sh,SdpContext *ctx,SdpPayload *payload)
{
	sdp_handler_add_payload(sh,ctx,payload,"audio");
}

void sdp_handler_add_video_payload(SdpHandler *sh,SdpContext *ctx,SdpPayload *payload)
{
	sdp_handler_add_payload(sh,ctx,payload,"video");
}

sdp_t * sdp_handler_generate_offer(SdpHandler *sdph,SdpContext *ctx)
{
	sdp_t *offer;
	char *tmp;
	char *attr_field;
	int ret;
	SdpPayload payload;
	int nline=0;
	
	offer=sdp_handler_generate_template(sdph);
	/* add audio codecs*/
	ctx->offer=offer;
	ctx->incb=1;
	if (sdph->set_audio_codecs!=NULL)
		sdph->set_audio_codecs(sdph,ctx);
	if (sdph->set_video_codecs!=NULL)
		sdph->set_video_codecs(sdph,ctx);
	ctx->incb=0;
	return offer;
}

sdp_t * sdp_handler_generate_answer(SdpHandler *sdph, struct _SdpContext *ctx)
{
	sdp_t *remote=ctx->remote;
	sdp_t *answer;
	char *mtype,*tmp;
	char *proto,*port,*pt;
	int i,j,ncodec,m_lines_accepted=0;
	int err;
	SdpPayload payload;
	
	
	answer=sdp_handler_generate_template(sdph);
	
	/* for each m= line */
	for (i=0;!sdp_endof_media(remote,i);i++){
		memset(&payload,0,sizeof(SdpPayload));
		mtype=sdp_m_media_get(remote,i);
		proto=sdp_m_proto_get(remote,i);
		port=sdp_m_port_get(remote,i);
		payload.remoteport=satoi(port);
		payload.proto=proto;
		payload.line=i;
		payload.c_addr=sdp_c_addr_get(remote,i,0);
		if (payload.c_addr==NULL) payload.c_addr=sdp_c_addr_get(remote,-1,0);
		if (keywordcmp("audio",mtype)==0){
			if (sdph->accept_audio_codecs!=NULL){
				ncodec=0;
				/* for each payload type */
				for(j=0;( (pt=sdp_m_payload_get(remote,i,j))!=NULL);j++){
					payload.pt=satoi(pt);
					/* get the rtpmap associated to this codec, if any */
					payload.a_rtpmap=sdp_a_attr_value_get_with_pt(remote,i,payload.pt,"rtpmap");
					/* get the fmtp, if any */
					payload.a_fmtp=sdp_a_attr_value_get_with_pt(remote,i,payload.pt,"fmtp");
					/* get application specific bandwidth, if any */
					payload.b_as_bandwidth=sdp_b_bandwidth_get_with_pt(remote,i,payload.pt);
					/* ask the application if this codec is supported */
					err=sdph->accept_audio_codecs(sdph,ctx,&payload);
					if (err==0 && payload.localport>0){
						ncodec++;
						/* codec accepted */
						if (ncodec==1){
							/* first codec accepted, setup the line  */
							sdp_m_media_add(answer,sgetcopy(mtype),int_2char(payload.localport),NULL,sgetcopy(proto));
						}
						/* add the payload, rtpmap, fmtp*/
						sdp_m_payload_add(answer,i,int_2char(payload.pt));
						if (payload.a_rtpmap!=NULL) {
							sdp_a_attribute_add(answer,i,
								sgetcopy("rtpmap"),
								sstrdup_sprintf("%i %s",payload.pt,payload.a_rtpmap));
						}
						if (payload.a_fmtp!=NULL) {
							sdp_a_attribute_add(answer,i,
								sgetcopy("fmtp"),
								sstrdup_sprintf("%i %s",payload.pt,payload.a_fmtp));
						}
						if (payload.b_as_bandwidth!=0) {
							sdp_b_bandwidth_add(answer,i,
								sgetcopy("AS"),
								sstrdup_sprintf("%i %i",payload.pt,payload.b_as_bandwidth));
						}
					}
				}
				if (ncodec==0){
					/* refuse the line */
					sdp_m_media_add(answer,sgetcopy(mtype),int_2char(0),NULL,sgetcopy(proto));
				}else m_lines_accepted++;
			}else{
				/* refuse this line (leave port to 0)*/
				sdp_m_media_add(answer,sgetcopy(mtype),int_2char(0),NULL,sgetcopy(proto));
			}
			
		}else if (keywordcmp("video",mtype)==0){
			if (sdph->accept_video_codecs!=NULL){
				ncodec=0;
				/* for each payload type */
				for(j=0;( (pt=sdp_m_payload_get(remote,i,j))!=NULL);j++){
					payload.pt=satoi(pt);
					/* get the rtpmap associated to this codec, if any */
					payload.a_rtpmap=sdp_a_attr_value_get_with_pt(remote,i,payload.pt,"rtpmap");
					/* get the fmtp, if any */
					payload.a_fmtp=sdp_a_attr_value_get_with_pt(remote,i,payload.pt,"fmtp");
					/* ask the application if this codec is supported */
					err=sdph->accept_video_codecs(sdph,ctx,&payload);
					if (err==0 && payload.localport>0){
						ncodec++;
						/* codec accepted */
						if (ncodec==1){
							/* first codec accepted, setup the line  */
							sdp_m_media_add(answer,sgetcopy(mtype),int_2char(0),NULL,sgetcopy(proto));
						}
						/* add the payload, rtpmap, fmtp*/
						sdp_m_payload_add(answer,i,int_2char(payload.pt));
						if (payload.a_rtpmap!=NULL) {
							sdp_a_attribute_add(answer,i,
								sgetcopy("rtpmap"),
								sstrdup_sprintf("%i %s",payload.pt,payload.a_rtpmap));
						}
						if (payload.a_fmtp!=NULL) {
							sdp_a_attribute_add(answer,i,
								sgetcopy("fmtp"),
								sstrdup_sprintf("%i %s",payload.pt,payload.a_fmtp));
						}
					}
				}
				if (ncodec==0){
					/* refuse the line */
					sdp_m_media_add(answer,sgetcopy(mtype),int_2char(0),NULL,sgetcopy(proto));
				}else m_lines_accepted++;
			}else{
				/* refuse this line (leave port to 0)*/
				sdp_m_media_add(answer,sgetcopy(mtype),int_2char(0),NULL,sgetcopy(proto));
			}
		}
	}
	ctx->answer=answer;
	if (m_lines_accepted>0) ctx->negoc_status=200;
	else ctx->negoc_status=415;
	return answer;
}
void sdp_handler_read_remote_answer(SdpHandler *sdph,struct _SdpContext *ctx)
{
	sdp_t *remote=ctx->remote;
	char *mtype,*tmp;
	char *proto,*port,*pt;
	int i,j;
	int err;
	SdpPayload payload;
	
	
	/* for each m= line */
	for (i=0;!sdp_endof_media(remote,i);i++){
		memset(&payload,0,sizeof(SdpPayload));
		mtype=sdp_m_media_get(remote,i);
		proto=sdp_m_proto_get(remote,i);
		port=sdp_m_port_get(remote,i);
		payload.remoteport=satoi(port);
		payload.localport=satoi(sdp_m_port_get(ctx->offer,i));
		payload.proto=proto;
		payload.line=i;
		payload.c_addr=sdp_c_addr_get(remote,i,0);
		if (payload.c_addr==NULL) payload.c_addr=sdp_c_addr_get(remote,-1,0);
		if (keywordcmp("audio",mtype)==0){
			if (sdph->get_audio_codecs!=NULL){
				/* for each payload type */
				for(j=0;( (pt=sdp_m_payload_get(remote,i,j))!=NULL);j++){
					payload.pt=satoi(pt);
					/* get the rtpmap associated to this codec, if any */
					payload.a_rtpmap=sdp_a_attr_value_get_with_pt(remote,i,payload.pt,"rtpmap");
					/* get the fmtp, if any */
					payload.a_fmtp=sdp_a_attr_value_get_with_pt(remote,i,payload.pt,"fmtp");
					/* ask the application if this codec is supported */
					err=sdph->get_audio_codecs(sdph,ctx,&payload);
				}
			}
		}else if (keywordcmp("video",mtype)==0){
			if (sdph->accept_video_codecs!=NULL){
				/* for each payload type */
				for(j=0;( (pt=sdp_m_payload_get(remote,i,j))!=NULL);j++){
					payload.pt=satoi(pt);
					/* get the rtpmap associated to this codec, if any */
					payload.a_rtpmap=sdp_a_attr_value_get_with_pt(remote,i,payload.pt,"rtpmap");
					/* get the fmtp, if any */
					payload.a_fmtp=sdp_a_attr_value_get_with_pt(remote,i,payload.pt,"fmtp");
					/* ask the application if this codec is supported */
					err=sdph->accept_video_codecs(sdph,ctx,&payload);
				}
			}
		}
	}
}


