/*
linphone
Copyright (C) 2000  Simon MORLAT (simon.morlat@free.fr)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "osipuacb.h"
#include "linphonecore.h"
#include "misc.h"

/* these are useful messages for the status */
extern gchar *ready;
extern gchar *end;
extern gchar *contacting;
extern gchar *connected;
extern gchar *cancel;
extern gchar *contacted;

int invite_accepted_cb(OsipDialog *dialog, transaction_t *trn, sip_t *sipmsg, void *data)
{
	OsipUA *ua=osip_dialog_get_ua(dialog);
	LinphoneCore *lc=(LinphoneCore*)ua->data;
	gint jitt_comp;
	gint err;
	LinphoneDialogParams *diaparams=(LinphoneDialogParams *)lc->call->data;
	StreamParams *audio_params=&diaparams->audio_params;

	linphone_core_thread_enter(lc);	
	lc->vtable.show(lc);
	lc->vtable.display_status(lc,connected);
	
	if (lc->ringstream!=NULL){
		ring_stop(lc->ringstream);
		lc->ringstream=NULL;
	}else err=try_open_dsp(lc);
	
	if (err>0){
		/* adjust rtp jitter compensation. It must be at least the latency of the sound card */
		jitt_comp=MAX(lc->sound_conf.latency,lc->rtp_conf.audio_jitt_comp);
		lc->audiostream=audio_stream_start_with_sndcard(diaparams->profile,audio_params->localport,
					audio_params->remoteaddr,audio_params->remoteport,
					audio_params->pt,jitt_comp,lc->sound_conf.sndcard);
	}
	
	linphone_core_thread_leave(lc);	
	return 0;
}


int bye_cb(OsipDialog *dialog, transaction_t *trn, sip_t *msg, void *p2)
{
	OsipUA *ua=osip_dialog_get_ua(dialog);
	LinphoneCore *lc=(LinphoneCore*)ua->data;
	LinphoneDialogParams *diaparams;
	
	linphone_core_thread_enter(lc);	
	linphone_core_lock(lc);
	
	diaparams=(LinphoneDialogParams*)dialog->data;
	
	/*stop ringing if necessary*/
	if (lc->ringstream!=NULL) {
		ring_stop(lc->ringstream);
		lc->ringstream=NULL;
	}
	if (lc->audiostream!=NULL) {
		audio_stream_stop(lc->audiostream);
		lc->audiostream=NULL;
	}
	lc->vtable.show(lc);
	lc->vtable.display_status(lc,end);
	
	lc->call=NULL;
	if (diaparams!=NULL) linphone_dialog_params_destroy(diaparams);
	restore_sound_daemons();
	linphone_core_thread_leave(lc);	
	linphone_core_unlock(lc);
	return 0;
}

int faillure_cb(OsipDialog *dialog, transaction_t *trn, sip_t *msg, void *p2)
{
	int err=((int*)p2)[0];
	char *reason;
	header_t *retry;
	char *umsg;
	char *msg486=_("User is busy.");
	char *msg480=_("User is temporarily unavaillable.");
	char *retrymsg=_("%s. Retry after %i minute(s).");
	char *msg600=_("User does not want to be disturbed.");
	char *msg603=_("Call declined.");
	char* tmpmsg=msg486;
	OsipUA *ua=osip_dialog_get_ua(dialog);
	LinphoneCore *lc=(LinphoneCore*)ua->data;
	LinphoneDialogParams *diaparams;
	linphone_core_lock(lc);
	diaparams=(LinphoneDialogParams*)dialog->data;
	linphone_core_thread_enter(lc);	
	lc->vtable.show(lc);
	switch(err)
	{
		case 404:
			lc->vtable.display_status(lc,_("User cannot be found at given address."));
		break;
		case 415:
			lc->vtable.display_status(lc,_("Remote user cannot support any of proposed codecs."));
		break;
		case 480:
			tmpmsg=msg480;
		case 486:
			msg_header_getbyname(msg,"retry-after",0,&retry);
			if (retry!=NULL)
			{
				umsg=g_malloc(strlen(tmpmsg)+strlen(retrymsg)+13);
				sprintf(umsg,retrymsg,tmpmsg,atoi(retry->hvalue)/60);
				lc->vtable.display_message(lc,umsg);
				g_free(umsg);
			}		
			lc->vtable.display_message(lc,tmpmsg);
		break;
		
		case 600:
			lc->vtable.display_message(lc,msg600);
		break;
		case 603:
			lc->vtable.display_status(lc,msg603);
		break;
		case -110:  /* time out, call leg is lost */
			lc->vtable.display_status(lc,_("Timeout."));
		break;
		case -111:
			lc->vtable.display_status(lc,_("Remote host was found but refused connection."));
		break;
		
		default:
			if (err>0)
			{
				reason=msg_getreason(err);
				lc->vtable.display_status(lc,reason);
				sfree(reason);
			}
			else g_message("faillure_cb: %s\n",strerror(-err));	
	}
	if (lc->ringstream!=NULL) {
		ring_stop(lc->ringstream);
		lc->ringstream=NULL;
		restore_sound_daemons();
	}
	if (diaparams!=NULL) linphone_dialog_params_destroy(diaparams);
	lc->call=NULL;
	linphone_core_thread_leave(lc);	
	linphone_core_unlock(lc);
	return 0;
}

int invite_cb(OsipDialog *dialog, transaction_t *trn, sip_t *sipmsg, void *p)
{
	from_t *from=sipmsg->from;
	char *barmesg,*rem_url;
	int addrlen;
	BodyContext *sdpctx;
	OsipUA *ua=osip_dialog_get_ua(dialog);
	LinphoneCore *lc=(LinphoneCore*)ua->data;
	linphone_core_lock(lc);
	linphone_core_thread_enter(lc);	
	/* first check if we can answer successfully to this invite */
	/* not busy */
	if (lc->call!=NULL){
		osip_dialog_respond(dialog,trn,486);
		goto end;
	}
	/* sdp params ok */
	sdpctx=osip_dialog_get_body_context(dialog,"application/sdp",0);
	/* get the result of the negociation */
	switch(SDP_CONTEXT(sdpctx)->negoc_status)
	{
		case -1:	
			g_warning("Error during sdp negociation. Cannot accept incoming call.\n");
			osip_dialog_respond(dialog,trn,606);
		break;
		case 200:
			from_2char_without_params(from,&rem_url);
			if (rem_url!=NULL){
				barmesg=g_strdup_printf("%s %s.",rem_url,contacted);
			}
			else barmesg=g_strdup_printf("Somebody %s.",contacted);
			
			lc->vtable.show(lc);
			lc->vtable.display_status(lc,barmesg);	
			lc->vtable.inv_recv(lc,rem_url);
			sfree(rem_url);
			lc->call=dialog;
				/* play the ring */
			if (try_open_dsp(lc)>0){
				g_message("Starting local ring...");
				lc->ringstream=ring_start(lc->sound_conf.local_ring,2,lc->sound_conf.sndcard);
			}
			lc->current_trn=trn;
			g_free(barmesg);
		break;
		default:
			osip_dialog_respond(dialog,trn,SDP_CONTEXT(sdpctx)->negoc_status);
	}
	end:
	
	linphone_core_unlock(lc);
	linphone_core_thread_leave(lc);	
	return 0;
}


int informative_cb(OsipDialog *dialog, transaction_t *trn, sip_t *msg, void *p2)
{
	int err=((int*)p2)[0];
	contact_t *contact;
	char *ctt;
	OsipUA *ua=osip_dialog_get_ua(dialog);
	LinphoneCore *lc=(LinphoneCore*)ua->data;
	linphone_core_thread_enter(lc);	
	linphone_core_lock(lc);
	switch(err)
	{
		case 100:
			break;
		case 180:
			//g_message("We are in 180 switch of informative_cb()");
			if (try_open_dsp(lc)>0)
				lc->ringstream=ring_start(lc->sound_conf.remote_ring,2,lc->sound_conf.sndcard);
			break;
 		case 183:
 			// this is used by cisco gateways
 			break;
		case 200:   /* for the moment only REGISTER request followed by 200 OK are said to the user*/
			 if (lc->sip_conf.reg_conf.expires) lc->vtable.display_status(lc,_("Registration successfull."));
			 else  lc->vtable.display_status(lc,_("Unregistration successfull."));
		break;
		case 380:
			msg_getcontact(msg,0,&contact);
			if (contact!=NULL)
			{			
				contact_2char(contact,&ctt);
			}else ctt=sgetcopy(_("(No contact information !)"));
			
			lc->vtable.display_url(lc,_("User is not reachable at the moment but he invites you\nto contact him using the following alternate resource:"),ctt);
			sfree(ctt);
			lc->call=NULL;
		break;			
		default:
			g_warning("unsupported informative message.\n");
	}
	linphone_core_thread_leave(lc);	
	linphone_core_unlock(lc);
	return 0;
}



/* these are the SdpHandler callbacks: we are called in to be aware of the content
of the SDP messages exchanged */

int set_audio_offer(SdpHandler *sdph,SdpContext *ctx)
{
	LinphoneCore *lc;
	OsipUA *ua;
	PayloadType *codec;
	GList *elem;
	SdpPayload payload;
	ua=body_handler_get_ua(BODY_HANDLER(sdph));
	lc=(LinphoneCore*)ua->data;
	
	/* set the nat address in the SDP, if given */
	if (lc->net_conf.nat_address!=NULL && lc->net_conf.use_nat){
		sdp_t *sdp=ctx->offer;
		sfree(sdp->c_connection->c_addr);
		sdp->c_connection->c_addr=sgetcopy(lc->net_conf.nat_address);
	}
	
	
	elem=lc->codecs_conf.audio_codecs;
	while(elem!=NULL){
		codec=(PayloadType*) elem->data;
		if (payload_type_usable(codec) && payload_type_enabled(codec)){
			
			sdp_payload_init(&payload);
			payload.a_rtpmap=g_strdup_printf("%s/%i/1",codec->mime_type,codec->clock_rate);
			payload.pt=rtp_profile_get_payload_number_from_rtpmap(lc->local_profile,payload.a_rtpmap);
			payload.localport=lc->rtp_conf.audio_rtp_port;
			/* when setting speex, add a b=<AS>: to choose bitrate */
			if (strcmp(codec->mime_type,"speex")==0){
				int ctype=linphone_core_get_connection_type(lc);
				if (ctype<CONNECTION_TYPE_NUMERIS){
					payload.b_as_bandwidth=8;
					codec->normal_bitrate=8000;
				}
				else {
					payload.b_as_bandwidth=20;
					codec->normal_bitrate=20000;
				}
			}
			sdp_handler_add_audio_payload(sdph,ctx,&payload);
			g_free(payload.a_rtpmap);
		}
		elem=g_list_next(elem);
	}
	/* add telephone-event payload*/
	sdp_payload_init(&payload);
	payload.pt=rtp_profile_get_payload_number_from_mime(lc->local_profile,"telephone-event");
	payload.a_rtpmap="telephone-event/8000";
	payload.a_fmtp="0-11";
	sdp_handler_add_audio_payload(sdph,ctx,&payload);
	return 0;
}


int set_video_offer(SdpHandler *sdph,SdpContext *ctx)
{
	LinphoneCore *lc;
	OsipUA *ua;
	PayloadType *codec;
	GList *elem;
	ua=body_handler_get_ua(BODY_HANDLER(sdph));
	lc=(LinphoneCore*)ua->data;
	elem=lc->codecs_conf.video_codecs;
	while(elem!=NULL){
		codec=(PayloadType*) elem->data;
		if (payload_type_usable(codec) && payload_type_enabled(codec)){
			SdpPayload payload;
			PayloadType *rtpp;
			sdp_payload_init(&payload);
			payload.a_rtpmap=g_strdup_printf("%s/%i",codec->mime_type,codec->clock_rate);
			payload.localport=lc->rtp_conf.video_rtp_port;
			sdp_handler_add_video_payload(sdph,ctx,&payload);
			g_free(payload.a_rtpmap);
		}
		elem=g_list_next(elem);
	}
	return 0;
}

int payload_is_supported(SdpPayload *payload,RtpProfile *local_profile,RtpProfile *dialog_profile)
{
	int localpt;
	if (payload->a_rtpmap!=NULL){
		localpt=rtp_profile_get_payload_number_from_rtpmap(local_profile,payload->a_rtpmap);
	}else{
		localpt=payload->pt;
		g_warning("payload has no rtpmap.");
	}
	
	if (localpt>=0){
		/* this payload is understood, but does the user want to use it ?? */
		PayloadType *rtppayload;
		rtppayload=rtp_profile_get_payload(local_profile,localpt);
		if (rtppayload==NULL) {
			g_warning("strange error !!");
			return 0;
		}
		if (strcmp(rtppayload->mime_type,"telephone-event")!=0){
			if (!(payload_type_usable(rtppayload) && payload_type_enabled(rtppayload))) {
				g_warning("payload %s is not usable or enabled.",rtppayload->mime_type);
				return 0;
			}
		}
		/* this payload is supported in our local rtp profile, so add it to the dialog rtp
		profile */
		rtp_profile_set_payload(dialog_profile,payload->pt,payload_type_clone(rtppayload));
		/* add to the rtp payload type some other parameters (bandwidth) */
		if (payload->b_as_bandwidth!=0) rtppayload->normal_bitrate=payload->b_as_bandwidth;
		return 1;
	}
	return 0;
}

int accept_audio_offer(SdpHandler *sdph,SdpContext *ctx,SdpPayload *payload)
{
	OsipDialog *dialog=BODY_CONTEXT(ctx)->dialog;
	LinphoneDialogParams *diaparams;
	LinphoneCore *lc=(LinphoneCore*) BODY_HANDLER(sdph)->ua->data;
	RtpProfile *remote_profile;
	StreamParams *params;
	int supported;
	if (dialog->data==NULL){
		diaparams=linphone_dialog_params_new();
		dialog->data=(void*) diaparams;
	}else diaparams=(LinphoneDialogParams*)dialog->data;
	if (diaparams->profile==NULL){
		/* create a remote user agent profile */
		diaparams->profile=remote_profile=rtp_profile_new("remote");
	}else remote_profile=diaparams->profile;
	/* see if this codec is supported in our local rtp profile*/
	supported=payload_is_supported(payload,lc->local_profile,remote_profile);
	if (!supported) {
		g_message("Refusing codec %i (%s)",payload->pt,payload->a_rtpmap);
		return -1;
	}
	params=&diaparams->audio_params;
	if (params->initialized==0){
		/* this is the first codec we may accept*/
		params->localport=payload->localport=lc->rtp_conf.audio_rtp_port;
		params->remoteport=payload->remoteport;
		params->line=payload->line;
		params->pt=payload->pt; /* remember the first payload accepted */
		params->remoteaddr=payload->c_addr;
		params->initialized=1;
	}else{
		/* refuse all other audio lines*/
		if(params->line!=payload->line) return -1;
	}
	return 0;
}

int accept_video_offer(SdpHandler *sdph,SdpContext *ctx,SdpPayload *payload)
{
	OsipDialog *dialog=BODY_CONTEXT(ctx)->dialog;
	LinphoneDialogParams *diaparams;
	LinphoneCore *lc=(LinphoneCore*) BODY_HANDLER(sdph)->ua->data;
	RtpProfile *remote_profile;
	StreamParams *params;
	int supported;
	if (dialog->data==NULL){
		diaparams=linphone_dialog_params_new();
		dialog->data=(void*) diaparams;
	}else diaparams=(LinphoneDialogParams*)dialog->data;
	if (diaparams->profile==NULL){
		/* create a remote user agent profile */
		diaparams->profile=remote_profile=rtp_profile_new("remote");
	}else remote_profile=diaparams->profile;
	/* see if this codec is supported in our local rtp profile*/
	supported=payload_is_supported(payload,lc->local_profile,remote_profile);
	if (!supported) return -1;
	params=&diaparams->video_params;
	if (params->initialized==0){
		/* this is the first codec we may accept*/
		params->localport=payload->localport=lc->rtp_conf.audio_rtp_port;
		params->remoteport=payload->remoteport;
		params->line=payload->line;
		params->pt=payload->pt; /* remember the first payload accepted */
		params->remoteaddr=payload->c_addr;
		params->initialized=1;
	}else{
		/* refuse all other audio lines*/
		if(params->line!=payload->line) return -1;
	}
	return 0;
}

int read_audio_answer(SdpHandler *sdph,SdpContext *ctx,SdpPayload *payload)
{
	OsipDialog *dialog=BODY_CONTEXT(ctx)->dialog;
	LinphoneDialogParams *diaparams;
	LinphoneCore *lc=(LinphoneCore*) BODY_HANDLER(sdph)->ua->data;
	StreamParams *params;
	int supported;
	if (dialog->data==NULL){
		diaparams=linphone_dialog_params_new();
		dialog->data=(void*) diaparams;
		diaparams->profile=rtp_profile_clone_full(lc->local_profile);
	}else diaparams=(LinphoneDialogParams*)dialog->data;
	/* paranoid check: see if this codec is supported in our local rtp profile*/
	supported=payload_is_supported(payload,lc->local_profile,diaparams->profile);
	if (!supported) {
		g_warning("this fucking remote sip phone did not answered properly to my sdp offer!");
		return 0;
	}
	params=&diaparams->audio_params;
	if (params->initialized==0){
		/* this is the first codec we may accept*/
		params->localport=lc->rtp_conf.audio_rtp_port;
		params->remoteport=payload->remoteport;
		params->line=payload->line;
		params->pt=payload->pt; /* remember the first payload accepted */
		params->remoteaddr=payload->c_addr;
		params->initialized=1;
	}
}

int read_video_answer(SdpHandler *sdph,SdpContext *ctx,SdpPayload *payload)
{
	OsipDialog *dialog=BODY_CONTEXT(ctx)->dialog;
	LinphoneDialogParams *diaparams;
	LinphoneCore *lc=(LinphoneCore*) BODY_HANDLER(sdph)->ua->data;
	StreamParams *params;
	int supported;
	if (dialog->data==NULL){
		diaparams=linphone_dialog_params_new();
		dialog->data=(void*) diaparams;
		diaparams->profile=rtp_profile_clone_full(lc->local_profile);
	}else diaparams=(LinphoneDialogParams*)dialog->data;
	/* paranoid check: see if this codec is supported in our local rtp profile*/
	supported=payload_is_supported(payload,lc->local_profile,diaparams->profile);
	if (!supported) {
		g_warning("this fucking remote sip phone did not answered properly to my sdp offer!");
		return 0;
	}
	params=&diaparams->video_params;
	if (params->initialized==0){
		/* this is the first codec we may accept*/
		params->localport=lc->rtp_conf.audio_rtp_port;
		params->remoteport=payload->remoteport;
		params->line=payload->line;
		params->pt=payload->pt; /* remember the first payload accepted */
		params->remoteaddr=payload->c_addr;
		params->initialized=1;
	}
}
