
/*
  The oRTP library is an RTP (Realtime Transport Protocol - rfc1889) stack.
  Copyright (C) 2001  Simon MORLAT simon.morlat@linphone.org

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* port_fct.h.  define methods to help for portability between unix and win32 */


#include "rtpsession.h"

#include "port_fct.h"



/**
 * this method is an utility method that calls fnctl() on UNIX or
 * ioctlsocket on Win32.
 * int retrun the result of the system method
 */
int set_non_blocking_socket (RtpSession *session)
{
	unsigned long nonBlock = 1;

#ifndef _WIN32
	return fcntl (session->rtp.socket, F_SETFL, O_NONBLOCK);
#else
	return ioctlsocket(session->rtp.socket, FIONBIO , &nonBlock);
#endif
}


/**
 * this method is an utility method that calls close() on UNIX or
 * closesocket on Win32.
 * int retrun the result of the system method
 */
#ifndef _WIN32
	int close_socket(gint sock)
	{
		return close (sock);
	}
#else
	int close_socket(SOCKET sock)
	{
		return closesocket(sock);
	}
#endif

	


/**
 * this method is an utility method that calls close() on UNIX or
 * closesocket on Win32.
 * it retrun 0 on success 1 on error, call GetLastError for more info...
 */
#ifndef _WIN32
	// use unix pthread ...
#else
	int pthread_create(pthread_t *thread,const void *attr,void *(__cdecl *start)(void *),void* arg)
	{
		unsigned long threadNum;
		*thread = CreateThread (NULL,0,(LPTHREAD_START_ROUTINE)start,	arg,0,&threadNum);

		if (*thread == NULL) 
			// Thread not created
			return 1;
		else
			return 0;
	}

#endif


/**
 * this method is an utility method that calls close() on UNIX or
 * closesocket on Win32.
 * int retrun the result of the system method
 */
#ifndef _WIN32
	// use unix pthread ...
#else
	int pthread_join(pthread_t thread,void **unused)
	{
		if (WaitForSingleObject (thread,INFINITE) == WAIT_OBJECT_0)
			return 0;
		else 
			return 1;
	}
#endif


/**
 * this method is an utility method that calls fnctl() on UNIX or
 * ioctlsocket on Win32.
 * int retrun the result of the system method
 */
int set_thread_priority ()
{
#ifndef _WIN32
	/*
	struct sched_param param;
	param.sched_priority=1;
	return sched_setscheduler(0,SCHED_RR,&param);
	*/
#else
	HANDLE thread = GetCurrentThread();
	return (SetThreadPriority(thread,THREAD_PRIORITY_ABOVE_NORMAL)==0)?-1:0;
#endif
}


/**
 * getSocketError() return a string describing the error
 */
#ifdef _WIN32
char *getSocketError() 
{
	int error = WSAGetLastError ();
	static char buf[80];

	switch (error)
	{
		case WSANOTINITIALISED: return "Windows sockets not initialized : call WSAStartup";
		case WSAEADDRINUSE:		return "Local Address already in use";
		case WSAEADDRNOTAVAIL:	return "The specified address is not a valid address for this machine";
//		case WSAEFAULT:			return "";
//		case WSAEINPROGRESS:	return "";
		case WSAEINVAL:			return "The socket is already bound to an address.";
		case WSAENOBUFS:		return "Not enough buffers available, too many connections.";
		case WSAENOTSOCK:		return "The descriptor is not a socket.";
		case WSAECONNRESET:		return "Connection reset by peer";
/*		
		
		case : return "";
		case : return "";
		case : return "";
		case : return "";
		case : return "";
		case : return "";
*/
		default :
			sprintf (buf,"Error code : %d", error);
			return buf;
		break;
	}

	return buf; 

}
#endif

#ifndef _WIN32
	// Use UNIX inet_aton method
#else
	int inet_aton (const char * cp, struct in_addr * addr)
	{
		unsigned long retval;
		
		retval = inet_addr (cp);

		if (retval == INADDR_NONE) 
		{
			return -1;
		}
		else
		{
			addr->S_un.S_addr = retval;
			return 1;
		}
	}
#endif


