 /*
  The oRTP library is an RTP (Realtime Transport Protocol - rfc1889) stack.
  Copyright (C) 2001  Simon MORLAT simon.morlat@linphone.org

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef RTPSESSION_H
#define RTPSESSION_H


#include "rtpport.h"
#include "rtp.h"
#include "payloadtype.h"
#include "sessionset.h"


#ifndef TARGET_IS_HPUXKERNEL

	#include <stdio.h>
	#include "str_utils.h"

	#ifndef _WIN32
		#include <sys/types.h>
    	#include <sys/socket.h>
    	#include <errno.h>
    	#include <netinet/in.h>
    	#include <unistd.h>
    	#include <sys/time.h>
	#else
		#include <winsock2.h>
#endif   /* _WIN32 */
#else
#endif   /* TARGET_IS_HPUXKERNEL */


typedef enum {
	RTP_SESSION_RECVONLY,
	RTP_SESSION_SENDONLY,
	RTP_SESSION_SENDRECV
} RtpSessionMode;



typedef enum {
	RTP_SESSION_RECV_SYNC=1,	/* the rtp session is synchronising in the incoming stream */
	RTP_SESSION_SEND_SYNC=1<<1, /* the rtp session is synchronising in the outgoing stream */
	RTP_SESSION_SCHEDULED=1<<2, /* the rtp session has to be scheduled */
	RTP_SESSION_BLOCKING_MODE=1<<3, /* in blocking mode */
	RTP_SESSION_RECV_NOT_STARTED=1<<4,	/* the application has not started to try to recv */
	RTP_SESSION_SEND_NOT_STARTED=1<<5,  /* the application has not started to send something */
	RTP_SESSION_IN_SCHEDULER=1<<6	/* the rtp session is in the scheduler list */
}RtpSessionFlags;

	
typedef struct _RtpStream
{
	gint socket;
	gint jitt_comp;   /* the user jitt_comp in miliseconds*/
	gint jitt_comp_time; /* the jitt_comp converted in rtp time (same unit as timestamp) */
	gint max_rq_size;
	struct timeval *timeout;
	struct timeval _timeout;
#ifndef TARGET_IS_HPUXKERNEL
	queue_t _rq;  /* should not be accessed direclty, use RtpStream::rq instead */
	queue_t _wq;  /* should not be accessed direclty, use RtpStream::wq instead */
#endif
	queue_t *rq;
	queue_t *wq;
#ifdef INET6
	struct sockaddr_storage loc_addr;
	struct sockaddr_storage rem_addr;
#else
	struct sockaddr_in loc_addr;
	struct sockaddr_in rem_addr;
#endif
	guint32 snd_time_offset;/*the scheduler time when the application send its first timestamp*/	
	guint32 snd_ts_offset;	/* the first application timestamp sent by the application */
	guint32 snd_rand_offset;	/* a random number added to the user offset to make the stream timestamp*/
	guint32 snd_last_ts;	/* the last stream timestamp sended */
	guint32 snd_ts;	 		/* to be unused */
	guint32 rcv_time_offset; /*the scheduler time when the application ask for its first timestamp*/
	guint32 rcv_ts_offset;  /* the first stream timestamp */
	guint32 rcv_query_ts_offset;	/* the first user timestamp asked by the application */
	guint32 rcv_app_ts_offset;  	/* the user timestamp that corresponds to the first timestamp of the stream*/
	guint32 rcv_diff_ts;	/* difference between the first user timestamp and first stream timestamp */
	guint32 rcv_ts;			/* to be unused */
	guint32 rcv_last_ts;	/* the last stream timestamp got by the application */
	guint32 rcv_last_app_ts; /* the last application timestamp asked by the application */	
	guint32 rcv_last_ret_ts; /* the timestamp of the last sample returned (only for continuous audio)*/
	guint16 rcv_seq;
	guint16 snd_seq;
	rtp_stats_t stats;
#ifdef BUILD_SCHEDULER
	GCond *wait_for_packet_to_be_sent_cond;
	GMutex *wait_for_packet_to_be_sent_mutex;
	GCond *wait_for_packet_to_be_recv_cond;
	GMutex *wait_for_packet_to_be_recv_mutex;
#endif
}RtpStream;

typedef struct _RtcpStream
{
	gint socket;
	gint pad;
#ifndef TARGET_IS_HPUXKERNEL
	queue_t _rq;  /* should not be accessed direclty, use RtpStream::rq instead */
	queue_t _wq;  /* should not be accessed direclty, use RtpStream::wq instead */
#endif
	queue_t *rq;
	queue_t *wq;
#ifdef INET6
	struct sockaddr_storage loc_addr;
	struct sockaddr_storage rem_addr;
#else
	struct sockaddr_in loc_addr;
	struct sockaddr_in rem_addr;
#endif
} RtcpStream;

typedef struct _RtpSession RtpSession;

#include "rtpsignaltable.h"

struct _RtpSession
{
	RtpSession *next;	/* next RtpSession, when the session are enqueued by the scheduler */
	RtpProfile *profile;
	GMutex *lock;
	guint32 ssrc;
	gint payload_type;
	gint max_buf_size;
#ifdef TARGET_IS_HPUXKERNEL
	mblk_t *dest_mproto; 	/* a M_PROTO that contains the destination address for outgoing packets*/
#endif
	RtpSignalTable on_ssrc_changed;
	RtpSignalTable on_payload_type_changed;
	RtpSignalTable on_telephone_event_packet;
	RtpSignalTable on_telephone_event;
	RtpSignalTable on_timestamp_jump;
	RtpStream rtp;
	RtcpStream rtcp;
	RtpSessionMode mode;
#ifdef BUILD_SCHEDULER
	struct _RtpScheduler *sched;
#endif
	guint32 flags;
	rtp_stats_t stats;
	gint mask_pos;	/* the position in the scheduler mask of RtpSession */
	gpointer user_data;
	
	/* telephony events extension */
	gint telephone_events_pt;	/* the payload type used for telephony events */
	mblk_t *current_tev;		/* the pending telephony events */
};
	



#ifdef __cplusplus
extern "C"
{
#endif


void rtp_session_init(RtpSession *session, gint mode);
/* for use in the HPUX kernel */
#define rtp_session_set_rq(s,q) (s)->rtp.rq=(q)
#define rtp_session_set_wq(s,q) (s)->rtp.wq=(q)

#define rtp_session_lock(session) 	g_mutex_lock(session->lock)
#define rtp_session_unlock(session) g_mutex_unlock(session->lock)


RtpSession *rtp_session_new(gint mode);

void rtp_session_set_scheduling_mode(RtpSession *session, gint yesno);
void rtp_session_set_blocking_mode(RtpSession *session, gint yesno);
void rtp_session_set_profile(RtpSession *session,RtpProfile *profile);
#define rtp_session_get_profile(session)	(session)->profile

#define rtp_session_set_flag(session,flag) (session)->flags|=(flag)
#define rtp_session_unset_flag(session,flag) (session)->flags&=~(flag)
int rtp_session_signal_connect(RtpSession *session,char *signal, RtpCallback cb, gpointer user_data);
int rtp_session_signal_disconnect_by_callback(RtpSession *session,char *signal, RtpCallback cb);
void rtp_session_set_ssrc(RtpSession *session, guint32 ssrc);
void rtp_session_set_jitter_compensation(RtpSession *session, int milisec);
#ifndef TARGET_IS_HPUXKERNEL
/* setting a local address is not provided for the HPUX kernel environement */
int rtp_session_set_local_addr(RtpSession *session,gchar *addr, gint port);
#endif

#ifdef TARGET_IS_HPUXKERNEL
gint rtp_session_set_remote_addr(RtpSession *session,struct sockaddr_in *dest);
#else
gint rtp_session_set_remote_addr(RtpSession *session,gchar *addr, gint port);
#endif

/* alternatively to the set_remote_addr() and set_local_addr(), an application can give
a valid socket to be used by the RtpSession */
void rtp_session_set_sockets(RtpSession *session, gint rtpfd, gint rtcpfd);

int rtp_session_set_payload_type(RtpSession *session, int paytype);
#define rtp_session_get_payload_type(session)	((session)->payload_type)

void rtp_session_set_timeout(RtpSession *session,guint timeout);

/*low level recv and send functions */
mblk_t * rtp_session_recvm_with_ts (RtpSession * session, guint32 user_ts);
mblk_t * rtp_session_create_packet(RtpSession *session,gint header_size, char *payload, gint payload_size);
gint rtp_session_sendm_with_ts (RtpSession * session, mblk_t *mp, guint32 userts);

/* high level recv and send functions */
gint rtp_session_recv_with_ts(RtpSession *session, gchar *buffer, gint len, guint32 time, gint *have_more);
gint rtp_session_send_with_ts(RtpSession *session, gchar *buffer, gint len, guint32 userts);


guint32 rtp_session_get_current_ts(RtpSession *session);
void rtp_session_reset(RtpSession *session);
void rtp_session_uninit(RtpSession *session);
void rtp_session_destroy(RtpSession *session);

#define rtp_session_get_stats(session) (&(session)->stats)
#define rtp_session_reset_stats(session)	memset(&(session)->stats,0,sizeof(rtp_stats_t))

#define rtp_session_set_data(session,data)	(session)->user_data=(data)
#define rtp_session_get_data(session,data)	((session)->user_data)

#define rtp_session_max_buf_size_set(session,bufsize)	(session)->max_buf_size=(bufsize)

void rtp_parse(RtpSession *session, mblk_t *mp);

/* in use with the scheduler to convert a timestamp in scheduler time unit (ms) */
guint32 rtp_session_ts_to_t(RtpSession *session,guint32 timestamp);


/* packet api */
/* the first argument is a mblk_t. The header is supposed to be not splitted  */
#define rtp_set_markbit(mp,value)		((rtp_header_t*)((mp)->b_rptr))->markbit=(value)
#define rtp_set_seqnumber(mp,seq)	((rtp_header_t*)((mp)->b_rptr))->seq_number=(seq)
#define rtp_set_timestamp(mp,ts)	((rtp_header_t*)((mp)->b_rptr))->timestamp=(ts)
#define rtp_set_ssrc(mp,_ssrc)		((rtp_header_t*)((mp)->b_rptr))->ssrc=(_ssrc)
void rtp_add_csrc(mblk_t *mp,guint32 csrc);
#define rtp_set_payload_type(mp,pt)	((rtp_header_t*)((mp)->b_rptr))->paytype=(pt)

#ifdef __cplusplus
}
#endif

#endif
