/*
  The osipua library is a library based on oSIP that implements CallLeg and User Agent
  level.
  Copyright (C) 2001  Simon MORLAT simon.morlat@free.fr
  											Aymeric MOIZARD jack@atosc.org
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include "osipmanager.h"
#include "osipua.h"
#include "utils.h"
#include "uatransaction.h"
#include "resolver.h"
#include "digcalc.h"

#ifdef INET6
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#endif


list_t callleg_list = { 0, NULL };


int osip_ua_has_specific_bind(OsipUA *ua){
	if (strncmp(ua->contact->url->host,"0.0.0.0",strlen("0.0.0.0"))==0) return 0;
	if (strncmp(ua->contact->url->host,"[::0]",strlen("[::0]"))==0) return 0;
	osip_trace(OSIP_INFO1,("ua needs specific binding on %s\n",ua->contact->url->host));
	return 1;
}

void
osip_dialog_init (OsipDialog * new_call, OsipUA * ua)
{
	contact_t *ctt;

	memset (new_call, 0, sizeof (OsipDialog));
	/*set status for this new call */
	new_call->status = DIALOG_NEW;
	/* init transaction list */
	/* AMD: list_init (&new_call->incoming_transactions); */
	/* list_init (&new_call->outgoing_transactions); */
	/* set the ua reference on the newly created call-leg */
	new_call->ua = ua;
	/* add the call-leg to the list of call-legs handled by the stack */
	list_add (&callleg_list, (void *) new_call, -1);
	/* add the call-leg to the list of call-legs managed by the ua */
	list_add (&ua->call_list, (void *) new_call, -1);
	ctt = (contact_t *) list_get (ua->alias, 0);

	if (ctt != NULL)
	{
		from_clone ((from_t *) ctt, &new_call->from);
	}
	else
	{
		from_clone ((from_t *) ua->contact, &new_call->from);
	}

	/* set default registrar if given */
	if (ua->registrar != NULL)
	{
		url_clone (ua->registrar, &new_call->registrar);
	}
	/* init the list of body handlers */
	list_init (&new_call->body_contexts);

	ua->dialog_count++;

}

/* this is the init function that initialize the call-leg if it is created
locally: callid, ceq... are choosed randomly */
void
osip_dialog_init2 (OsipDialog * new_call, OsipUA * ua)
{
}

/* internal constructor ! do not use !*/
OsipDialog *
osip_dialog_alloc (OsipUA * ua)
{
	OsipDialog *new_call;

	new_call = smalloc (sizeof (OsipDialog));
	osip_dialog_init (new_call, ua);
	return new_call;
}

/* this is called by the application to create a new call-leg */

/**
 * osip_dialog_new:
 * @ua: an existing user agent.
 *
 * Creates a new #OsipDialog object managed by the user agent @ua.
 *
 * Returns: a new call leg.
 */
OsipDialog *
osip_dialog_new (OsipUA * ua)
{
	OsipDialog *new_call;
	new_call = osip_dialog_alloc (ua);
	osip_dialog_init2 (new_call, ua);
	return (new_call);
}



/* create a new call-leg from an initial incoming request . The function
assumes that there is no existing call-leg that match the new request.
You should use osip_dialog_find() to check this */

/**
 * osip_dialog_new_from_incoming_trn:
 * @trn: an incoming transaction
 *
 * Create a new call-leg from an initial incoming request. The function
 * assumes that there is no existing call-leg that match the new request.
 * You should use #osip_dialog_find() to check this.
 *
 * Returns: a new call leg.
 */
OsipDialog *
osip_dialog_new_from_incoming_trn (transaction_t * trn)
{
	OsipDialog *call;
	OsipUA *ua;
	sip_t *sipmsg = trn->orig_request;
	contact_t *ctt = NULL;
	via_t *via=NULL;
	char *from = NULL;
	char *to = NULL;
	int i;

	call = ua_transaction_get_dialog (trn);
	if (call != NULL)
	{
		osip_trace (OSIP_INFO1,
			    ("A call-leg already exists for this transaction. (%x)",
			     trn));
		return NULL;
	}

	/* first find a ua for this call-leg */
	ua = osip_ua_find (trn->orig_request);

	i = from_2char (trn->orig_request->from, &from);
	if (i != 0)
		return NULL;
	osip_trace (OSIP_INFO1,
		    ("%s has called at %i.\n", from, time (NULL)));
	sfree (from);


	if (ua == NULL)
	{
		i = to_2char (trn->orig_request->to, &to);
		if (i != 0)
			return NULL;
		/* we should return a 404 Not Found */
		osip_trace (OSIP_INFO1,
			    ("error: Requested user (%s) does not exist.\n",
			     to));
		sfree (to);
		/* take the first user agent of the list to respond to this request  :  Maybe revisit */
		respond_to_request (def_manager->config, trn, 404);
		return NULL;
	}
	/* check if the user agent can answer to the call */
	if (ua->dialog_count >= ua->max_dialogs)
	{
		/*respond a 486 busy here */
		respond_to_request (ua->config, trn, 486);
		return NULL;
	}
	/* the transaction is accepted, it can open a new dialog */

	call = osip_dialog_alloc (ua);

	if (MSG_IS_INVITE (sipmsg))
	{
		osip_dialog_set_state (call, DIALOG_INVITED);
		ua_transaction_set_incoming_invite_tr (trn, call);
	}
	else if (MSG_IS_BYE (sipmsg))
	{
		osip_dialog_set_state (call, DIALOG_FAKE);
		ua_transaction_set_incoming_bye_tr (trn, call);
	}
	else
	{
		osip_trace (OSIP_WARNING,
			    ("Unsupported new incoming request."));
		respond_to_request (ua->config, trn, 501);
		return NULL;
	}
	via=list_get(trn->orig_request->vias,0);
	if (via!=NULL){
		generic_param_t *received=NULL;
		via_param_getbyname(via,"received",&received);
	if (received!=NULL && received->gvalue!=NULL){
			call->received=sgetcopy(received->gvalue);
		}else{
			call->received=sgetcopy(via->host);
		}
	}
	if (osip_ua_has_specific_bind(ua)){
		call->localip=sgetcopy(ua->contact->url->host);
	}else guess_local_address(call->received,&call->localip);
	return call;
}


char *
call_id_new_random ()
{
	char *tmp = (char *) smalloc (33);
	unsigned int number = new_random_number ();
	sprintf (tmp, "%u", number);
	return tmp;
}

char *
from_tag_new_random ()
{
	return call_id_new_random ();
}

char *
to_tag_new_random ()
{
	return call_id_new_random ();
}

unsigned int
via_branch_new_random ()
{
	return new_random_number ();
}

int
complete_answer_that_establish_a_dialog (OsipDialog * dialog,
					 sip_t * response, sip_t * request)
{
	int i;
	int pos = 0;
	contact_t *ctt;
	char *p;
	OsipUA *ua = osip_dialog_get_ua (dialog);
	/* 12.1.1:
	 * copy all record-route in response
	 * add a contact with global scope
	 */
	while (!list_eol (request->record_routes, pos))
	{
		record_route_t *rr;
		record_route_t *rr2;
		rr = list_get (request->record_routes, pos);
		i = record_route_clone (rr, &rr2);
		if (i != 0)
			return -1;
		list_add (response->record_routes, rr2, -1);
		pos++;
	}
	contact_clone(ua->contact,&ctt);
	/* modify the contact address so that it match the interface we're going to use
	to send the response */
	sfree(ctt->url->host);
	ctt->url->host=sgetcopy(dialog->localip);
	contact_2char (ctt, &p);
	contact_free(ctt);
	sfree(ctt);
	msg_setcontact (response, p);
	sfree (p);
	/* create a dialog_t if necessary */
	if (dialog->dialog == NULL)
	{
		i = dialog_init_as_uas (&(dialog->dialog), request, response);
		if (i != 0)
		{
			osip_trace (OSIP_WARNING,
				    ("error - could not create a new dialog_t."));
		}
	}
	return 0;
}

int
osip_dialog_generate_response_default (OsipDialog * call_leg,
				       int status, sip_t * request,
				       sip_t ** dest)
{
	generic_param_t *tag;
	sip_t *response;
	int pos;
	int i;
	char *tmp;
	BodyContext *bh;

	if (status < 299 && status > 100 && call_leg == NULL)
		return -1;	/* not allowed */
	if (status == 200 &&
	    (MSG_IS_INVITE (request) || MSG_IS_OPTIONS (request))
	    && call_leg == NULL)
		return -1;	/* not allowed */

	i = msg_init (&response);
	/* initialise sip_t structure */
	/* yet done... */

	response->strtline->statuscode = (char *) smalloc (5 * sizeof (char));
	sprintf (response->strtline->statuscode, "%i", status);

	msg_setmethod (response, NULL);
	tmp = msg_getreason (status);
	if (tmp == NULL)
		msg_setreasonphrase (response,
				     sgetcopy ("Unkown status code"));
	else
		msg_setreasonphrase (response, tmp);
	msg_setversion (response, sgetcopy ("SIP/2.0"));

	response->strtline->rquri = NULL;

	i = to_clone (request->to, &(response->to));
	if (i != 0)
		goto grd_error_1;


	i = to_get_tag (response->to, &tag);
	if (i != 0)
	{			/* we only add a tag if it does not already contains one! */
		if ((call_leg != NULL) && (call_leg->dialog != NULL)
		    && (call_leg->dialog->local_tag != NULL))
			/* it should contain the local TAG we created */
		{
			to_set_tag (response->to,
				    sgetcopy (call_leg->dialog->local_tag));
		}
		else
		{
			if (status != 100)
				to_set_tag (response->to,
					    to_tag_new_random ());
		}
	}

	i = from_clone (request->from, &(response->from));
	if (i != 0)
		goto grd_error_1;

	pos = 0;
	while (!list_eol (request->vias, pos))
	{
		via_t *via;
		via_t *via2;
		via = (via_t *) list_get (request->vias, pos);
		i = via_clone (via, &via2);
		if (i != -0)
			goto grd_error_1;
		list_add (response->vias, via2, -1);
		pos++;
	}

	i = call_id_clone (request->call_id, &(response->call_id));
	if (i != 0)
		goto grd_error_1;
	i = cseq_clone (request->cseq, &(response->cseq));
	if (i != 0)
		goto grd_error_1;

	if (status < 299 && status > 100 && MSG_IS_INVITE (request))
	{
		//printf("Completing response******************************\n");
		complete_answer_that_establish_a_dialog (call_leg, response,
							 request);
	}
	*dest = response;

	if ((status == 200) &&
	    ((MSG_IS_INVITE (request)) || (MSG_IS_OPTIONS (request))))
	{

		/* if we send a 200 Ok for INVITE or OPTIONS, then invoke the body handlers to add their bodies if they want */
		int pos = 0;
		while (!list_eol (&call_leg->body_contexts, pos))
		{
			bh = (BodyContext *) list_get (&call_leg->
						       body_contexts, pos);
			body_context_gen_out_response (bh, response);
			pos++;
		}
	}

	return 0;


      grd_error_1:
	msg_free (response);
	sfree (response);
	return -1;
}


/* accept an invitation for an existing call-leg*/
/**
 *osip_dialog_accept_invite:
 *@dialog: a dialog.
 *@trn:	   a transaction initiated by INVITE from a remote client.
 *
 * On reception of the 200 Ok for invite, then the osipua library calls
 * the "INVITE" signal handler. Then the user can choose to accept the call
 * using this function.
 * If the user wants to reject the call, then it should call
 * osip_dialog_bye().
 *
 */
void
osip_dialog_accept_invite (OsipDialog * dialog, transaction_t * trn)
{
	OsipUA *ua;

	/* check if we have good status for this call to answer 200OK */
	if (osip_dialog_exists (dialog) == 0)
	{
		osip_trace (OSIP_INFO1,
			    ("error: SIP dialog does not exist.\n"));
		return;
	}
	ua = osip_dialog_get_ua (dialog);


	if (trn == NULL)
	{
		osip_trace (OSIP_INFO1,
			    ("error: could not find last invite incoming transaction\n"));
		return;
	}
	osip_dialog_set_state (dialog, DIALOG_ESTABLISHED);
	osip_dialog_respond (dialog, trn, 200);
}

int
osip_dialog_reject_invite (OsipDialog * call_leg)
{
	sip_t *sipmesg = NULL;
	int i;
	char *p;
	transaction_t *trn = call_leg->inc_invite_tr;
	if (trn == NULL)
		return -1;
	if (call_leg->status != DIALOG_ESTABLISHED)
	{
		osip_dialog_respond (call_leg, trn, 603);
		osip_dialog_release (call_leg);
	}
	return (0);
}

/* register your identity to a registrar */

/**
 * osip_dialog_register:
 * @call_leg: a call-leg.
 * @context: a registration context.
 *
 * Sends a REGISTER request to a registrar.
 *
 * Returns: 0 if the REGISTER was sent, a negative value if not.
 */
int
osip_dialog_register (OsipDialog * call_leg, RegistrationCtxt * context)
{
	sip_t *sipmesg;
	char *tmp;
	url_t *rgt;
	from_t *addr_of_rec;
	int err;
	int i;

	OsipUA *ua = osip_dialog_get_ua (call_leg);

	if (context->registrar == NULL)
	{
		osip_trace (OSIP_INFO1,
			    ("error: You have to specify a registrar.\n"));
		return (-1);
	}

	url_init (&rgt);
	err = url_parse (rgt, context->registrar);
	if (err != 0)
	{
		osip_trace (OSIP_INFO1,
			    ("error: Bad registrar address (%s).\n",
			     context->registrar));
		url_free (rgt);
		sfree (rgt);
		return -EINVAL;
	}

	if (context->address_of_record != NULL)
	{
		from_init (&addr_of_rec);
		err = from_parse (addr_of_rec, context->address_of_record);
		if (err != 0)
		{
			osip_trace (OSIP_INFO1,
				    ("error: Bad address of record (%s).\n",
				     context->address_of_record));
			url_free (rgt);
			sfree (rgt);
			from_free (addr_of_rec);
			sfree (addr_of_rec);
			return -EINVAL;
		}
		/* set to=from= address_of_record */
		from_free (call_leg->from);
		sfree (call_leg->from);
		call_leg->from = addr_of_rec;
	}
	else
	{
		/* if no address of record is given in the context, then use the default from field of the call-leg */
		/* AMD: Not possible any more.
		 * to_2char ((from_t *) call_leg->from, &tmp);
		 * to_parse (call_leg->to, tmp);
		 * sfree (tmp);
		 */
	}
	call_leg->reg_context = context;
	context->ref_count++;	/* context is in use ! */
	i = generating_request_out_of_dialog (call_leg,
					      "REGISTER", NULL, &sipmesg);
	if (i != 0)
		return (-1);
	osip_dialog_send_request (call_leg, sipmesg);
	osip_dialog_set_state (call_leg, DIALOG_FAKE);
	return 0;
}

int
osip_dialog_register_with_authentication (OsipDialog * call_leg,
					  sip_t * previous_answer,
					  char *passwd)
{
	sip_t *sipmesg;
	authorization_t *aut = NULL;
	www_authenticate_t *wwwauth = NULL;
	proxy_authorization_t *proxy_aut = NULL;
	proxy_authenticate_t *proxyauth = NULL;
	int i;
	RegistrationCtxt *regctx;
	char *uri;
	char *pass;
	OsipUA *ua = osip_dialog_get_ua (call_leg);

	if (passwd == NULL)
	{
		/* then tries to find a password in the registration context */
		regctx = call_leg->reg_context;
		if (regctx != NULL)
		{
			pass = regctx->password;
		}
		else
		{
			osip_trace (OSIP_ERROR,
				    ("osip_dialog_register_with_authentification: no password, aborting"));
			return -1;
		}
	}
	else
		pass = passwd;
	msg_getwww_authenticate (previous_answer, 0, &wwwauth);
	msg_getproxy_authenticate (previous_answer, 0, &proxyauth);
	if ((wwwauth != NULL) && (proxyauth != NULL))
		return -1;

	call_leg->reg_context->cseq_number++;
	i = generating_request_out_of_dialog (call_leg,
					      "REGISTER", NULL, &sipmesg);
	if (i != 0)
		return (-1);

	/* ADD AUTHENTICATION STUFF.... */
	url_2char (sipmesg->strtline->rquri, &uri);

	if (proxyauth != NULL)
	{
		i = osip_create_proxy_authorization_header (previous_answer,
							    uri,
							    ua->contact->url->
							    username, pass,
							    &proxy_aut);
		if (i != 0)
		{
			osip_trace (OSIP_INFO1,
				    ("error: could not create authorization header.\n"));
			return -1;
		}
	}

	if (wwwauth != NULL)
	{
		i = osip_create_authorization_header (previous_answer,
						      uri,
						      ua->contact->url->
						      username, pass, &aut);
		if (i != 0)
		{
			osip_trace (OSIP_INFO1,
				    ("error: could not build the proxy_authorization header.\n"));
			return -1;
		}
	}
	/* TODO */
	/* here we should copy the proxy_authorization header
	 * from the previous_answer to the new request. This
	 * proxy_authorization is the response given by the
	 * first proxy made for the second proxy!
	 * oSIP must be fixed here to support multiple
	 * proxy_authorization, authorization, www_authenticate,
	 * and proxy_authenticate in one message.
	 * jack@atosc.org
	 */
	/* !TODO */

	list_add (sipmesg->authorizations, aut, -1);
	if (proxy_aut != NULL)
		list_add (sipmesg->proxy_authorizations, proxy_aut, -1);

	/* ADD AUTHENTICATION STUFF.... */

	osip_dialog_send_request (call_leg, sipmesg);
	osip_dialog_set_state (call_leg, DIALOG_FAKE);
	return 0;
}

/**
 * osip_dialog_unregister:
 * @call: a call-leg.
 * @ctxt: a registration context that has been used to make a previous registration.
 *
 * Sends a REGISTER request to a registrar with a expire=0, so that
 * the registration is cancelled.
 * Warning: the ctxt structure may be changed. Do not rely on it to make a new registration.
 *
 * Returns: 0 if the REGISTER was sent, a negative value if not.
 */
int
osip_dialog_unregister (OsipDialog * call, RegistrationCtxt * ctxt)
{
	int error;
	ctxt->expires = 0;
	error = osip_dialog_register (call, ctxt);
	return error;
}

/* invite somebody */
/**
 * osip_dialog_invite:
 * @call_leg: a call-leg.
 * @callee: the URI of the person to invite.
 *
 * Sends an invite to @callee.
 *
 * Returns: 0 if the invite was sent, a negative value if not.
 */
int
osip_dialog_invite (OsipDialog * call_leg, char *callee, char *body_mime)
{
	OsipUA *ua = osip_dialog_get_ua (call_leg);
	sip_t *sipmesg;
	int err;
	call_id_t *callid;
	to_t *ato;
	int i;
	if (call_leg->status != DIALOG_NEW)
		return (-ESRCH);

	/* set to */
	to_init (&ato);
	err = to_parse (ato, callee);
	to_free (ato);
	sfree (ato);
	if (err == SIP_SYNTAX_ERROR)
	{
		osip_trace (OSIP_ERROR, ("Invalid sip address.\n"));
		return -EINVAL;
	}
	/* compose an invite */

	i = generating_request_out_of_dialog (call_leg,
					      "INVITE", callee, &sipmesg);
	if (i != 0)
	{
		osip_trace (OSIP_ERROR,
			    ("error: could not build sip message.\n"));
		return (-1);
	}
	/* add the body */
	if (body_mime != NULL)
		osip_dialog_add_body_to_request (call_leg, sipmesg,
						 body_mime);
	/* and send it */
	osip_dialog_send_request (call_leg, sipmesg);
	osip_dialog_set_state (call_leg, DIALOG_INVITING);
	return (0);
}

/* invite somebody */
int
osip_dialog_reinvite_with_authentication (OsipDialog * call_leg,
					  sip_t * previous_answer,
					  char *password)
{
	OsipUA *ua = osip_dialog_get_ua (call_leg);
	sip_t *sipmesg;
	int cseq;
	char *tmp;
	char *uri;
	via_t *via;
	authorization_t *aut = NULL;
	proxy_authorization_t *proxy_aut = NULL;
	proxy_authenticate_t *proxyauth = NULL;
	www_authenticate_t *wwwauth = NULL;


	msg_clone (call_leg->out_invite_tr->orig_request, &sipmesg);
	/* increase cseq */
	cseq = atoi (call_leg->out_invite_tr->orig_request->cseq->number) + 1;
	tmp = smalloc (10);
	sprintf (tmp, "%d", cseq);
	sfree (sipmesg->cseq->number);
	cseq_setnumber (sipmesg->cseq, tmp);

	/* modify the Via branch so that it's a new transaction */
	via = list_get (sipmesg->vias, 0);
	list_remove (sipmesg->vias, 0);
	via_free (via);
	sfree (via);

#ifdef INET6
	switch (ua->ua_family)
	{
	case AF_INET:
	{
		char *tmp = (char *) smalloc (90 * sizeof (char));
		sprintf (tmp, "SIP/2.0/UDP %s:%i;branch=z9hG4bK%u",
			 ua->ua_ipaddr,
			 ua->ua_port, via_branch_new_random ());
		msg_setvia (sipmesg, tmp);
		sfree (tmp);
	}
		break;
	case AF_INET6:
	{
		char *tmp = (char *) smalloc (90 * sizeof (char));
		sprintf (tmp, "SIP/2.0/UDP [%s]:%i;branch=z9hG4bK%u",
			 ua->ua_ipaddr,
			 ua->ua_port, via_branch_new_random ());
		msg_setvia (sipmesg, tmp);
		sfree (tmp);
	}
		break;
	}
#else
	{
		char *tmp = (char *) smalloc (90 * sizeof (char));
		sprintf (tmp, "SIP/2.0/UDP %s:%i;branch=z9hG4bK%u",
			 ua->ua_ip4addr,
			 ua->ua_port, via_branch_new_random ());
		msg_setvia (sipmesg, tmp);
		sfree (tmp);
	}
#endif
	if (sipmesg == NULL)
	{
		osip_trace (OSIP_INFO1,
			    ("error: could not build sip message.\n"));
		return (-1);
	}


	/* ADD AUTHENTICATION STUFF.... */
	url_2char (sipmesg->strtline->rquri, &uri);
	/*     if (MSG_TEST_CODE(previous_answer,407)
	 * ||MSG_TEST_CODE(previous_answer,401))
	 * { */
	msg_getwww_authenticate (previous_answer, 0, &wwwauth);
	msg_getproxy_authenticate (previous_answer, 0, &proxyauth);
	if ((proxyauth != NULL) && (wwwauth != NULL))
		return -1;
	if (proxyauth != NULL)
	{
		int i = osip_create_proxy_authorization_header
			(previous_answer,
			 uri,
			 ua->contact->url->username,
			 password,
			 &proxy_aut);
		if (i != 0)
		{
			osip_trace (OSIP_INFO1,
				    ("error: could not build the authorization header.\n"));
			return -1;
		}
	}

	if (wwwauth != NULL)
	{
		int i = osip_create_authorization_header (previous_answer,
							  uri,
							  ua->contact->url->
							  username,
							  password, &aut);
		if (i != 0)
		{
			osip_trace (OSIP_INFO1,
				    ("error: could not build the proxy_authorization header.\n"));
			return -1;
		}
	}
	/* TODO */
	/* here we should copy the proxy_authorization header
	 * from the previous_answer to the new request. This
	 * proxy_authorization is the response given by the
	 * first proxy made for the second proxy!
	 * oSIP must be fixed here to support multiple
	 * proxy_authorization, authorization, www_authenticate,
	 * and proxy_authenticate in one message.
	 * jack@atosc.org
	 */
	/* !TODO */


	list_add (sipmesg->authorizations, aut, -1);
	if (proxy_aut != NULL)
		list_add (sipmesg->proxy_authorizations, proxy_aut, -1);
	/* !ADD AUTHENTICATION STUFF.... */

	/* and send it */
	osip_dialog_send_request (call_leg, sipmesg);
	osip_dialog_set_state (call_leg, DIALOG_INVITING);
	return (0);

}

sip_t *
make_cancel (sip_t * request)
{
	sip_t *sipmesg;
	int i = 0;
	body_t *body;
	msg_clone (request, &sipmesg);
	if (sipmesg == NULL)
	{
		osip_trace (OSIP_WARNING,
			    ("Could not msg_clone() %s\n", request));
		return NULL;
	}
	msg_setmethod (sipmesg, sgetcopy ("CANCEL"));
	/* we must remove the bodies */
	while ((body = list_get (sipmesg->bodies, 0)) != NULL)
	{
		list_remove (sipmesg->bodies, 0);
		body_free (body);
		sfree (body);
	}
	sfree (sipmesg->cseq->method);
	cseq_setmethod (sipmesg->cseq, sgetcopy ("CANCEL"));
	content_type_free (sipmesg->content_type);
	sfree (sipmesg->content_type);
	sipmesg->content_type = NULL;
	content_length_free (sipmesg->contentlength);
	sfree (sipmesg->contentlength);
	sipmesg->contentlength = NULL;
	msg_force_update (sipmesg);
	/*
	 * msg_2char(sipmesg,&p);
	 * //printf("Here is the cancel: %s\n",p);
	 * sfree(p);
	 * }else return -1;
	 */
	return sipmesg;

}


/* bye and destroy the call-leg*/

/**
 * osip_dialog_bye:
 * @call_leg: a call-leg.
 *
 *
 * Sends a bye and ends the call-leg.
 *
 * Returns: 0 if the bye was sent, a negative value if not.
 */
int
osip_dialog_bye (OsipDialog * call_leg)
{
	sip_t *sipmesg = NULL;
	int i;
	char *p;
	transaction_t *trn = call_leg->out_invite_tr;
	if (call_leg->status != DIALOG_ESTABLISHED)
	{
		if (trn != NULL)
		{
			if (trn->state == ICT_PRE_CALLING
			    || trn->state == ICT_CALLING)
			{
				/* rfc3261 do not allow to send CANCELs if
				 * no provisionnal response has been received. */
				return 0;
			}
			call_leg->status = DIALOG_CANCELLING;
			sipmesg = make_cancel (trn->orig_request);

		}
		else
		{
			osip_trace (OSIP_ERROR,
				    ("Can't cancel last transaction\n"));
		}
	}
	else
	{
		i = osip_dialog_generate_request_within_dialog (call_leg,
								"BYE",
								&sipmesg);
		if (i != 0)
		{
			osip_trace (OSIP_ERROR,
				    ("Could not generate BYE message.\n"));
			return (-1);	/* to be set... */
		}
	}
	if (sipmesg == NULL)
		return -1;
	osip_dialog_send_request (call_leg, sipmesg);
	osip_dialog_release (call_leg);
	return (0);
}

int
dialog_fill_route_set (dialog_t * dialog, sip_t * request)
{
	/* if the pre-existing route set contains a "lr" (compliance
	 * with bis-08) then the rquri should contains the remote target
	 * URI */
	int i;
	int pos = 0;
	url_param_t *lr_param;
	route_t *route;
	char *last_route;
	/* AMD bug: fixed 17/06/2002 */

	if (dialog->type == CALLER)
	{
		pos = list_size (dialog->route_set) - 1;
		route = (route_t *) list_get (dialog->route_set, pos);
	}
	else
		route = (route_t *) list_get (dialog->route_set, 0);

	url_uparam_getbyname (route->url, "lr", &lr_param);
	if (lr_param != NULL)	/* the remote target URI is the rquri! */
	{
		i = url_clone (dialog->remote_contact_uri->url,
			       &(request->strtline->rquri));
		if (i != 0)
			return -1;
		/* "[request] MUST includes a Route header field containing
		 * the route set values in order." */
		/* AMD bug: fixed 17/06/2002 */
		pos = 0;	/* first element is at index 0 */
		while (!list_eol (dialog->route_set, pos))
		{
			route_t *route2;
			route = list_get (dialog->route_set, pos);
			i = route_clone (route, &route2);
			if (i != 0)
				return -1;
			if (dialog->type == CALLER)
				list_add (request->routes, route2, 0);
			else
				list_add (request->routes, route2, -1);
			pos++;
		}
		return 0;
	}

	/* if the first URI of route set does not contain "lr", the rquri
	 * is set to the first uri of route set */


	i = url_clone (route->url, &(request->strtline->rquri));
	if (i != 0)
		return -1;
	/* add the route set */
	/* "The UAC MUST add a route header field containing
	 * the remainder of the route set values in order. */
	pos = 0;		/* yes it is */

	while (!list_eol (dialog->route_set, pos))	/* not the first one in the list */
	{
		route_t *route2;
		route = list_get (dialog->route_set, pos);
		i = route_clone (route, &route2);
		if (i != 0)
			return -1;
		if (dialog->type == CALLER)
		{
			if (pos != 0)
				list_add (request->routes, route2, 0);
		}
		else
		{
			if (!list_eol (dialog->route_set, pos + 1))
				list_add (request->routes, route2, -1);
		}
		pos++;
	}
	/* The UAC MUST then place the remote target URI into
	 * the route header field as the last value */
	i = url_2char (dialog->remote_contact_uri->url, &last_route);
	if (i != 0)
		return -1;
	i = msg_setroute (request, last_route);
	if (i != 0)
	{
		sfree (last_route);
		return -1;
	}


	/* route header and rquri set */
	return 0;
}

int
osip_dialog_generate_request_within_dialog (OsipDialog * call_leg,
					    char *method_name, sip_t ** dest)
{
	int i;
	sip_t *request;
	char *max_fwd;
	char *proxy_require;
	OsipUA *ua = osip_dialog_get_ua (call_leg);

	i = msg_init (&request);
	if (i != 0)
		return -1;

	if (call_leg->dialog->remote_contact_uri == NULL)
	{
		/* this dialog is probably not established! or the remote UA
		 * is not compliant with the latest RFC
		 */
		msg_free (request);
		sfree (request);
		return -1;
	}

	/* prepare the request-line */
	msg_setmethod (request, sgetcopy (method_name));
	msg_setstatuscode (request, NULL);
	msg_setreasonphrase (request, NULL);
	msg_setversion (request, sgetcopy ("SIP/2.0"));

	/* and the request uri???? */
	if (list_eol (call_leg->dialog->route_set, 0))
	{
		/* The UAC must put the remote target URI (to field) in the rquri */
		i = url_clone (call_leg->dialog->remote_contact_uri->url,
			       &(request->strtline->rquri));
		if (i != 0)
			goto grwd_error_1;
	}
	else
	{
		/* fill the request-uri, and the route headers. */
		dialog_fill_route_set (call_leg->dialog, request);
	}

	/* To and From already contains the proper tag! */
	i = to_clone (call_leg->dialog->remote_uri, &(request->to));
	if (i != 0)
		goto grwd_error_1;
	i = from_clone (call_leg->dialog->local_uri, &(request->from));
	if (i != 0)
		goto grwd_error_1;

	/* set the cseq and call_id header */
	msg_setcall_id (request, call_leg->dialog->call_id);

	if (0 == strcmp ("ACK", method_name))
	{
		cseq_t *cseq;
		char *tmp;
		i = cseq_init (&cseq);
		if (i != 0)
			goto grwd_error_1;
		tmp = smalloc (10);
		sprintf (tmp, "%i", call_leg->dialog->local_cseq);
		cseq_setnumber (cseq, tmp);
		cseq_setmethod (cseq, sgetcopy (method_name));
		request->cseq = cseq;
	}
	else
	{
		cseq_t *cseq;
		char *tmp;
		i = cseq_init (&cseq);
		if (i != 0)
			goto grwd_error_1;
		call_leg->dialog->local_cseq++;	/* we should we do that?? */
		tmp = smalloc (10);
		sprintf (tmp, "%i", call_leg->dialog->local_cseq);
		cseq_setnumber (cseq, tmp);
		cseq_setmethod (cseq, sgetcopy (method_name));
		request->cseq = cseq;
	}

	msg_setmax_forward (request, "10");	/* a UA should start a request with 70 */


	/* even for ACK for 2xx (ACK within a dialog), the branch ID MUST
	 * be a new ONE! */
#ifdef INET6
	{
		char *tmp = (char *) smalloc (90 * sizeof (char));
		switch (ua->ua_family)
		{
		case AF_INET:
			sprintf (tmp, "SIP/2.0/UDP %s:%i;branch=z9hG4bK%u",
				 call_leg->localip,
				 ua->ua_port, via_branch_new_random ());
			break;
		case AF_INET6:
			sprintf (tmp, "SIP/2.0/UDP [%s]:%i;branch=z9hG4bK%u",
				 call_leg->localip,
				 ua->ua_port, via_branch_new_random ());
			break;
		default:
			break;
		}
		msg_setvia (request, tmp);
		sfree (tmp);
	}

#else
	{
		char *tmp = (char *) smalloc (90 * sizeof (char));
		sprintf (tmp, "SIP/2.0/UDP %s:%i;branch=z9hG4bK%u",
			 ua->ua_ip4addr,
			 ua->ua_port, via_branch_new_random ());
		msg_setvia (request, tmp);
		sfree (tmp);
	}
#endif
	/* add specific headers for each kind of request... */
	if (0 == strcmp ("INVITE", method_name))
	{
		contact_t *ctt;
		contact_clone (ua->contact, &ctt);
		sfree(ctt->url->host);
		ctt->url->host=sgetcopy(call_leg->localip);
		list_add (request->contacts, ctt, 0);
	}
	else if (0 == strcmp ("INFO", method_name))
	{

	}
	else if (0 == strcmp ("OPTIONS", method_name))
	{
		msg_setaccept (request, "application/sdp");
	}
	else if (0 == strcmp ("ACK", method_name))
	{
		/* The ACK MUST contains the same credential than the INVITE!! */
		/* TODO... */
	}

	msg_setuser_agent (request, "oSIP/Linphone-" VERSION);
	/*  else if ... */
	*dest = request;
	return 0;

	/* grwd_error_2: */
	call_leg->dialog->local_cseq--;
      grwd_error_1:
	msg_free (request);
	sfree (request);
	*dest = NULL;
	return -1;
}


/* returns 1 if call leg exists*/
int
osip_dialog_exists (OsipDialog * call)
{
	int pos;
	OsipDialog *callleg;

	for (pos = 0; pos < callleg_list.nb_elt; pos++)
	{
		callleg = (OsipDialog *) list_get (&callleg_list, pos);
		if (callleg == call)
			return 1;
	}
	return 0;
}


/**
 * osip_dialog_find:
 * @sipmsg: an incoming request.
 * @dialog: the dialog as the return value.
 *
 * Tries to find a call-leg that may be concerned by the incoming request.
 *
 * Returns: 1 if a dialog was found, 0 if not found, -1 if the message has to be discarded
 *	and no dialog should be created after.
 */
int
osip_dialog_find (sip_t * sipmsg, OsipDialog ** ret_dialog)
{
	int pos = 0;
	int cond = 0;
	int error = 0;
	OsipDialog *dialog;

	*ret_dialog = NULL;

	/*find an existant call-leg that may match  this new invite */
	while (!list_eol (&callleg_list, pos) && (!cond))
	{
		dialog = (OsipDialog *) list_get (&callleg_list, pos);
		if (dialog->dialog == NULL)
		{
			/* this is not yet established dialog */
			pos++;
			printf ("not yet established !\n");
			continue;
		}
		/* there are much more things to check: callid, from tags, to etc... */
		if (dialog_match_as_uas (dialog->dialog, sipmsg) == 0)
		{
			cond = 1;
			if (MSG_IS_REQUEST (sipmsg))
			{
				if (MSG_IS_ACK (sipmsg)
				    || MSG_IS_CANCEL (sipmsg))
				{
					if (atoi (sipmsg->cseq->number) <
					    dialog->dialog->remote_cseq)
					{
						osip_trace (OSIP_INFO1,
							    ("Ack or Cancel cseq does not match !!!!!!!!!!!!!\n"));
						error = 1;
					}
				}
				else
				{
					if (atoi (sipmsg->cseq->number) <=
					    dialog->dialog->remote_cseq)
					{
						error = 1;
						osip_trace (OSIP_INFO1,
							    ("Request Cseq does not match !!!!!!!!!!!!!\n"));
					}
				}
			}
			else
			{
				if (atoi (sipmsg->cseq->number) <
				    dialog->dialog->local_cseq)
				{
					osip_trace (OSIP_INFO1,
						    ("Response Cseq does not match !!!!!!!!!!!!!\n"));
					error = 1;
				}
			}
		}
		pos++;
	}
	if (error)
		return -1;
	if (cond)
	{
		osip_trace (OSIP_INFO1, ("call-leg has been found!\n"));
		/* an existing call-leg was found */
		*ret_dialog = dialog;
		return 1;
	}
	else
		return 0;
}


/**
 * osip_dialog_send_response:
 * @call: a call-leg
 * @trn: a transaction
 * @resp: a sip response
 *
 *
 * Sends the response @resp that will be part of the transaction @trn,
 * trn being part of the call-leg @call.
 *
 */
void
osip_dialog_send_response (OsipDialog * call, transaction_t * trn,
			   sip_t * resp)
{
	osip_send_response (call->ua->manager, trn, resp);
}


/* respond to a request that is part from a call-leg : ex 200OK (INVITE)*/
/**
 * osip_dialog_respond:
 * @call: a call-leg.
 * @code: the code of the response.
 *
 *
 * This function creates a sip response in the context of the call-leg @call
 * for the last transaction and sends it automatically.
 * A sdp body can be optionnaly appended to the response.
 *
 */
void
osip_dialog_respond (OsipDialog * call, transaction_t * transaction, int code)
{
	sip_t *response;
	int i;
	i = osip_dialog_generate_response_default (call,
						   code,
						   transaction->orig_request,
						   &response);

	if (i != 0)
		return;

#ifdef OSIP_RETRANSMISSIONS
	if (call->dialog != NULL && code == 200
	    && MSG_IS_INVITE (transaction->orig_request))
	{
		osip_start_200ok_retransmissions (call->ua->config,
						  call->dialog, response,
						  transaction->out_socket);
	}
#endif
	osip_dialog_send_response (call, transaction, response);
}

/**
 * osip_dialog_ack:
 * @call: a call-leg.
 * @trn: a transaction part of the dialog.
 *
 *
 * This function generates and sends an ACK request that will end the
 * transaction @trn. Must never be called outside the osip callbacks.
 *
 */


void
osip_dialog_ack (OsipDialog * call, transaction_t * trn)
{
	sip_t *ack;
	char *dest;
	OsipUA *ua = osip_dialog_get_ua (call);
	int port = 0;
	int i;
	route_t *route;
#ifdef INET6
	struct sockaddr_storage ss;
	struct addrinfo hints, *res;
	int error;
#endif

	i = osip_dialog_generate_request_within_dialog (call, "ACK", &ack);
	if (i != 0)
		return;

	/** AMD: destination is not set automaticly by osip!
            It may be NULL or at least will contain the
            detination for the INVITE. But ACK must be sent
            to the contact from the remote 200 ok.
        */
	msg_getroute (ack, 0, &route);
	if (route != NULL)
	{
		int port = 5060;

		if (route->url->port != NULL)
			port = satoi (route->url->port);
		ict_set_destination (trn->ict_context,
				     sgetcopy (route->url->host), port);
	}
	else
	{
		int port = 5060;

		if (ack->strtline->rquri->port != NULL)
			port = satoi (ack->strtline->rquri->port);
		ict_set_destination (trn->ict_context,
				     sgetcopy (ack->strtline->rquri->host),
				     port);
	}
	ua_transaction_get_destination (trn, &dest, &port);

	/* see if we need a name resolution */
#ifdef INET6
	memset (&hints, 0, sizeof (hints));
	hints.ai_family = PF_UNSPEC;
	hints.ai_socktype = SOCK_DGRAM;
	hints.ai_flags = AI_NUMERICHOST;
	error = getaddrinfo (dest, NULL, &hints, &res);
	if (error)
#else
	if (inet_addr (dest) == -1)
#endif
	{
		async_resolv_and_send_ack (ua->manager, call->dialog, dest,
					   port, ack);
		return;
	}
#ifdef INET6
	else freeaddrinfo (res);
#endif
	/* ack for invite cannot be sent in the same way */
	udp_send (trn, ack, dest, port, trn->out_socket);
#ifdef OSIP_RETRANSMISSIONS
	osip_start_ack_retransmissions (ua->config, call->dialog, ack, dest,
					port, trn->out_socket);
#endif
}


/**
 * osip_dialog_compose_request:
 * @call_leg: a call-leg.
 * @msgtype: the request type in INVITE, BYE, REGISTER.
 *
 * Creates a sip request of type @msgtype according to the context of
 * the call-leg @call_leg. The request is not sent.
 *
 * Returns: a sip message.
 */
/* History:
   AMD: replace evrywhere occurence of
   >   msg_2char();
   >   msg_parse()
   with the equivalent but cleaner:
   >   msg_clone();
*/
int
generating_request_out_of_dialog (OsipDialog * call_leg,
				  char *method_name,
				  char *callee, sip_t ** dest)
{
	/* Section 8.1:
	 * A valid request contains at a minimum "To, From, Call-iD, Cseq,
	 * Max-Forwards and Via
	 */
	generic_param_t *tag;
	int i;
	char *max_fwd;
	char *proxy_require;
	sip_t *request;
	OsipUA *ua = (OsipUA *) call_leg->ua;
	RegistrationCtxt *context = call_leg->reg_context;
	char *localip=NULL;


	if (context == NULL && 0 == strcmp ("REGISTER", method_name))
		return -1;
	if (context != NULL && context->callid_number == NULL)
		context->callid_number = call_id_new_random ();

	i = msg_init (&request);
	if (i != 0)
		return -1;


	/* prepare the request-line */
	msg_setmethod (request, sgetcopy (method_name));
	msg_setstatuscode (request, NULL);
	msg_setreasonphrase (request, NULL);
	msg_setversion (request, sgetcopy ("SIP/2.0"));

	from_clone (call_leg->from, &(request->from));

	from_set_tag (request->from, from_tag_new_random ());

	if (0 == strcmp ("REGISTER", method_name))
	{
		url_init (&request->strtline->rquri);
		url_parse (request->strtline->rquri, context->registrar);
		from_clone (request->from, &(request->to));
	}
	else
	{
		/* in any cases except REGISTER: */
		i = msg_setto (request, callee);
		if (i != 0)
		{
			fprintf (stderr,
				 "ERROR: callee address does not seems to be a sipurl: %s\n",
				 callee);
			goto brood_error_1;
		}
		if ((ua->registrar != NULL)
		    && (strcmp (ua->registrar->host, request->to->url->host)))
		{		/* equal to a pre-existing route set */
			/* if the pre-existing route set contains a "lr" (compliance
			 * with bis-08) then the rquri should contains the remote target
			 * URI */
			url_param_t *lr_param;
			route_t *o_proxy;
			route_init (&o_proxy);
			url_clone (ua->registrar, &(o_proxy->url));
			url_uparam_getbyname (o_proxy->url, "lr", &lr_param);

			/* I always add it if not specified to mandate people
			 * to use a compliant (rfc3261) proxy.  :)
			 * try partysip... */
			if (lr_param == NULL)	/* to is the remote target URI in this case! */
			{
				url_uparam_add (o_proxy->url, sgetcopy ("lr"),
						NULL);
				url_uparam_getbyname (o_proxy->url, "lr",
						      &lr_param);
			}

			if (lr_param != NULL)	/* to is the remote target URI in this case! */
			{
				url_clone (request->to->url,
					   &(request->strtline->rquri));
				/* "[request] MUST includes a Route header field containing
				 * the route set values in order." */
				list_add (request->routes, o_proxy, 0);
			}
			else
				/* if the first URI of route set does not contain "lr", the rquri
				 * is set to the first uri of route set */
			{
				request->strtline->rquri = o_proxy->url;
				o_proxy->url = NULL;
				route_free (o_proxy);
				sfree (o_proxy);
				/* add the route set */
				/* "The UAC MUST add a route header field containing
				 * the remainder of the route set values in order.
				 * The UAC MUST then place the remote target URI into
				 * the route header field as the last value
				 */
				msg_setroute (request, callee);
			}
		}
		else		/* No route set (outbound proxy) is used */
		{
			/* The UAC must put the remote target URI (to field) in the rquri */
			i = url_clone (request->to->url,
				       &(request->strtline->rquri));
			if (i != 0)
				goto brood_error_1;
		}
	}
	if (osip_ua_has_specific_bind(ua)){
		call_leg->localip=sgetcopy(ua->contact->url->host);
	}else {
		i=guess_local_address(request->strtline->rquri->host,&call_leg->localip);
		if (i<0) return -1;
	}
	localip=call_leg->localip;

	/* set the cseq and call_id header */
	if (0 == strcmp ("REGISTER", method_name))
	{
		call_id_t *callid;
		cseq_t *cseq;
		char *num;

		/* call-id is always the same for REGISTRATIONS */
		i = call_id_init (&callid);
		if (i != 0)
			goto brood_error_1;
		call_id_setnumber (callid,
				   sgetcopy (call_leg->reg_context->
					     callid_number));
	
		call_id_sethost (callid, sgetcopy (localip));
		request->call_id = callid;

		i = cseq_init (&cseq);
		if (i != 0)
			goto brood_error_1;
		num = (char *) smalloc (10);	/* should never be more than 10 chars... */
		sprintf (num, "%i", call_leg->reg_context->cseq_number);
		cseq_setnumber (cseq, num);
		cseq_setmethod (cseq, sgetcopy (method_name));
		request->cseq = cseq;
	}
	else
	{
		/* set the call-id */
		call_id_t *callid;
		cseq_t *cseq;
		i = call_id_init (&callid);
		if (i != 0)
			goto brood_error_1;
		call_id_setnumber (callid, call_id_new_random ());

		call_id_sethost (callid, sgetcopy (localip));
		request->call_id = callid;

		i = cseq_init (&cseq);
		if (i != 0)
			goto brood_error_1;
		cseq_setnumber (cseq, sgetcopy ("20"));	/* always start with 20... :-> */
		cseq_setmethod (cseq, sgetcopy (method_name));
		request->cseq = cseq;
	}

	msg_setmax_forward (request, "10");	/* a UA should start a request with 70 */
	/* should be changed to:
	 * msg_setmax_forwards(request, "10");  */

#ifdef INET6
	{
		char *tmp = (char *) smalloc (90 * sizeof (char));
		switch (ua->ua_family)
		{
		case AF_INET:
			sprintf (tmp, "SIP/2.0/UDP %s:%i;branch=z9hG4bK%u",
				 localip,
				 ua->ua_port, via_branch_new_random ());
			break;
		case AF_INET6:
			sprintf (tmp, "SIP/2.0/UDP [%s]:%i;branch=z9hG4bK%u",
				 localip,
				 ua->ua_port, via_branch_new_random ());
			break;
		default:
			break;
		}
		msg_setvia (request, tmp);
		sfree (tmp);
	}
#else
	{
		char *tmp = (char *) smalloc (90 * sizeof (char));
		sprintf (tmp, "SIP/2.0/UDP %s:%i;branch=z9hG4bK%u",
			 localip,
			 ua->ua_port, via_branch_new_random ());
		msg_setvia (request, tmp);
		sfree (tmp);
	}
#endif
	/* add specific headers for each kind of request... */

	if (0 == strcmp ("INVITE", method_name))
	{
		/* add a Contact header for requests that establish a dialog:
		 * (only "INVITE") */
		/* this Contact is the global location where to send request
		 * outside of a dialog! like sip:jack@atosc.org? */
		contact_t *ctt;
		contact_clone (ua->contact, &ctt);
		sfree(ctt->url->host);
		ctt->url->host=sgetcopy(localip);
		list_add (request->contacts, ctt, 0);

		/* Here we'll add the supported header if it's needed! */
		/* the require header must be added by the upper layer if needed */
	}
	else if (0 == strcmp ("REGISTER", method_name))
	{
		contact_t *ctt;
		char exp[20];
		contact_clone (ua->contact, &ctt);
		sfree(ctt->url->host);
		ctt->url->host=sgetcopy(localip);
		list_add (request->contacts, ctt, 0);

		/* add the expire header == 0 or == expire */
		sprintf (exp, "%i", context->expires);
		msg_setexpires (request, exp);
	}
	else if (0 == strcmp ("INFO", method_name))
	{

	}
	else if (0 == strcmp ("OPTIONS", method_name))
	{

	}

	msg_setuser_agent (request, "oSIP/Linphone-" VERSION);
	/*  else if ... */
	*dest = request;
	return 0;

      brood_error_1:
	msg_free (request);
	sfree (request);
	*dest = NULL;
	return -1;
}

int
osip_dialog_add_body_to_request (OsipDialog * call, sip_t * msg,
				 char *body_mime)
{
	OsipUA *ua = osip_dialog_get_ua (call);
	BodyHandler *bh;
	BodyContext *bc;

	bc = osip_dialog_get_body_context (call, body_mime, 0);
	if (bc == NULL)
	{
		bh = osip_ua_find_handler (ua, body_mime);
		if (bh == NULL)
		{
			osip_trace (OSIP_WARNING,
				    ("There is no handler for %s\n",
				     body_mime));
			return -1;
		}
		bc = body_handler_create_context (bh, call);
		if (bc == NULL)
		{
			osip_trace (OSIP_ERROR,
				    ("The handler for %s could not create a new context\n",
				     body_mime));
			return -1;
		}
		osip_dialog_add_body_context (call, bc);
	}
	body_context_gen_out_request (bc, msg);
	return 0;
}

/**
 * osip_dialog_send_request:
 * @call_leg: a call-leg.
 * @sipmsg: a sip request.
 *
 *
 * Sends a sip request. The transaction is created to wrap the request and
 * to the list of transaction of call-leg @call_leg.
 *
 * Returns: 0 if successfull.
 */
int
osip_dialog_send_request (OsipDialog * call_leg, sip_t * sipmsg)
{
	sipevent_t *sipevent;
	transaction_t *transaction;
	char *dest;
	int port = 0;
	OsipUA *ua = (OsipUA *) call_leg->ua;
#ifdef INET6
	struct sockaddr_storage ss;
	struct addrinfo hints, *res = NULL;
	int error;
#endif

	if (MSG_IS_INVITE (sipmsg) || MSG_IS_REGISTER (sipmsg)
	    || MSG_IS_BYE (sipmsg))
	{
		sipevent = osip_new_outgoing_sipmessage (sipmsg);
		transaction = ua_transaction_new (call_leg, sipmsg);
	}
	else if (MSG_IS_CANCEL (sipmsg))
	{
		sipevent = osip_new_outgoing_sipmessage (sipmsg);
		//transaction=call_leg->out_invite_tr;  /*REVISIT: only cancel INVITE trn */
		transaction = ua_transaction_new (call_leg, sipmsg);
		if (transaction == NULL)
		{
			printf ("Could not create CANCEL transaction\n");
			return 0;
		}
	}
	else
	{
		osip_trace (OSIP_INFO1,
			    ("error: you must not use osip_dialog_send_request() to send acks."));
		/* "Acks must only be sent using osip_dialog_ack()" */
		exit (1);
	}
	sipevent->transactionid = transaction->transactionid;
	/* see if we need a name resolution */

	ua_transaction_get_destination (transaction, &dest, &port);
#ifdef INET6  
 	memset(&hints, 0, sizeof(hints));  
 	hints.ai_family = PF_UNSPEC;  
 	hints.ai_socktype = SOCK_DGRAM;  
 	hints.ai_flags = AI_NUMERICHOST;  
 	error = getaddrinfo(dest, NULL, &hints, &res);  
 	if (error)  
#else
  	if (inet_addr (dest) == -1)
#endif
		return (async_resolv_and_send
			(ua->manager, transaction, sipevent));
#ifdef INET6
	else freeaddrinfo(res);
#endif

	ua_transaction_execute (transaction, sipevent);
	return transaction->transactionid;
}

/**
 * osip_dialog_release:
 * @call: a call-leg.
 *
 *
 * Schedule the end of the call-leg. The call-leg object will be destroyed as
 * soon as it the osip stack will terminate its last transaction.
 *
 */


/* Destroy a call-leg and remove it from ua list and stack list*/

/**
 * osip_dialog_destroy:
 * @call: a call-leg.
 *
 *
 * Destroy a call-leg and all data associated with it.
 * BUT: this function should not be called directly. You should use
 * #osip_dialog_release instead to schedule the destruction of the call
 * leg.
 *
 * Returns 0.
 */

int
osip_dialog_destroy (OsipDialog * call)
{
	int err = 0;
	struct _OsipUA *ua = call->ua;

	err = list_remove_el (&callleg_list, call);
	if (err < 0)
		osip_trace (OSIP_ERROR,
			    ("Could not remove dialog from list.\n"));
	err = list_remove_el (&ua->call_list, call);

	if (err < 0)
	{
		osip_trace (OSIP_ERROR,
			    ("Could not remove dialog from ua list.\n"));
	}
	else
	{
		osip_trace (OSIP_INFO1,
			    ("Dialog is removed. It remains %i dialog(s) in the ua list.\n",
			     list_size (&ua->call_list)));
	}


	from_free (call->from);
	sfree (call->from);
	url_free (call->registrar);
	sfree (call->registrar);
#ifdef OSIP_RETRANSMISSIONS
	/* remove all retransmissions context associated with the dialog */
	osip_stop_retransmissions_from_dialog (ua->config, call->dialog);
#endif
	dialog_free (call->dialog);
	sfree (call->dialog);
	sfree(call->received);
	if (call->localip!=NULL) sfree(call->localip);
	sfree (call);
	return (err);
}



/**
 * osip_dialog_release:
 * @call: a dialog.
 *
 * Release a dialog. You can't use it anymore, and the object will be
 * destroyed by the stack when the last transaction of the call-leg
 * will terminate, either by successfull completion or timeout.
 *
 *
 */
void
osip_dialog_release (OsipDialog * call)
{
	OsipUA *ua = osip_dialog_get_ua (call);
	ua->dialog_count--;
	osip_dialog_set_state (call, DIALOG_TERMINATED);
}

void
osip_dialog_add_body_context (OsipDialog * call, BodyContext * handler)
{
	if (handler == NULL)
	{
		osip_trace (OSIP_INFO1,
			    ("Cannot add NULL body context to dialog.\n"));
		return;
	}
	list_add (&call->body_contexts, (void *) handler, 0);
}


BodyContext *
osip_dialog_get_body_context (OsipDialog * call, char *body_mime, int pos)
{
	BodyContext *context = NULL;
	char *tmp;
	int i = 0;
	int body_pos = 0;	/* the position in list of the body handler of the same type */
	//printf("entering osip_dialog_get_body_context list_size=%i\n", list_size(&call->body_contexts));
	while (!list_eol (&call->body_contexts, i))
	{
		context = (BodyContext *) list_get (&call->body_contexts, i);
		tmp = body_context_get_mime (context);
		//printf("Comparing %s <> %s , context=%x, handler=%x\n",tmp,body_mime,context,context->handler);
		if (strcmp (tmp, body_mime) == 0)
		{
			if ((body_pos == pos) || (pos == -1))
				return context;
			else
				body_pos++;
		}
		i++;
	}
	return NULL;
}
