/*
  The osipua library is a library based on oSIP that implements CallLeg and User Agent
  level.
  Copyright (C) 2001  Simon MORLAT simon.morlat@free.fr
  											Aymeric MOIZARD jack@atosc.org
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <sys/types.h>
#include <sys/socket.h>

#include <netinet/in.h>
#include <arpa/inet.h>
#include <unistd.h>
#include "udp.h"
#include "osipmanager.h"
#include "osipua.h"

#ifdef INET6  
#include <sys/types.h>  
#include <sys/socket.h>  
#include <netdb.h>  
#define STRN_EQ(x,y,l) (strncasecmp((x),(y),(l)) == 0)  
#endif

extern OsipManager *def_manager;

int udp_receive (OsipManager * manager);

void osipua_execute(OsipManager *manager)
{
	long int time_elapsed;
	transaction_t *trn;
	
	osip_ict_execute (manager->config);
	osip_nict_execute (manager->config);
	osip_ist_execute (manager->config);
	osip_nist_execute (manager->config);
	
	
	osip_timers_ict_execute (manager->config);
	osip_timers_ist_execute (manager->config);
	osip_timers_nict_execute (manager->config);
	osip_timers_nist_execute (manager->config);
#ifdef OSIP_RETRANSMISSIONS
	osip_retransmissions_execute(manager->config);
#endif
	/* free killed transaction */
	while ( (trn=fifo_tryget(&manager->garbage_trn))!=NULL){
		transaction_free(trn);
		sfree(trn);
	}
}

void osipua_distribute_event(OsipManager *m, sipevent_t *ev)
{
	osip_t *config=m->config;
	transaction_t *trn;
	int err;
	
	smutex_lock(m->mutex);
	err=osip_find_transaction_and_add_event(config,ev);
	if (err<0){
		/* check if it is not an ack for invite (not handled by osip)*/
		if (ev->type==RCV_REQACK){
			/* stop the retransmissions of the 200 Ok the ACK matches */
#ifdef OSIP_RETRANSMISSIONS	
			osip_stop_200ok_retransmissions(config,ev->sip);
#endif
			msg_free(ev->sip);
			sfree(ev->sip);
			sfree(ev);
		}
		else if (ev->type==RCV_STATUS_2XX){
			/* this a retransmission of 200 Ok */
			msg_free(ev->sip);
			sfree(ev->sip);
			sfree(ev);
		}
		else if (EVT_IS_INCOMINGREQ(ev)){
			/* not an ack, of course */
			trn=osip_create_transaction(config,ev);
			transaction_execute(trn,ev);
		}
	}else{
		/* an existing transaction was found */
		osipua_execute(m);
	}
	
	smutex_unlock(m->mutex);
}


void *
sipd_thread (void *managerp)
{
	OsipManager *manager = (OsipManager *) managerp;	
	int err;
	
	osip_trace (OSIP_INFO1, ("Entering osipua thread.\n"));
	manager->thread_pid=getpid();
	
	while (manager->udp_run_cond)
	{
		int i, j, k, nb;
		fd_set osipfdset;
		struct timeval timeout;
#ifdef __linux
		socklen_t slen;
#else
		int slen;
#endif
#ifdef INET6  
		struct sockaddr_storage sa;  
#else
		struct sockaddr_in sa;
#endif

		memcpy (&osipfdset, &manager->udpfdset, sizeof (fd_set));
		timeout.tv_sec = manager->recv_tout.tv_sec;	/* timeout can be modified by linux !! */
		timeout.tv_usec = manager->recv_tout.tv_usec;
		nb = select (manager->max_udpfd + 1, &osipfdset, NULL, NULL,&timeout);
		/* if something to read on the control file descriptor, then exit thread */
		if (nb > 0)
		{
			if (FD_ISSET (manager->udp_unblock_fd, &osipfdset))
			{
				//printf("Receiving something on unblock_fd\n");
				read (manager->udp_unblock_fd, manager->udp_buf,
			      SIP_MESSAGE_MAX_LENGTH);
				nb--;
			}
			//printf("Something to read !!!!!!!!!!!!!!!!!\n");
			for (j = 0, k = 0; (j < OSIP_MAX_UDP_PORTS) && (k < nb); j++)
			{
				if (FD_ISSET (manager->udpfds[j], &osipfdset))
				{
					k++;
					slen = sizeof (sa);
					i = recvfrom (manager->udpfds[j],
					  manager->udp_buf,
					  SIP_MESSAGE_MAX_LENGTH, 0,
					  (struct sockaddr *) &sa, &slen);
					if (i > 0)
					{
						/* Message might not end with a "\0" but we know the number of */
						/* char received! */
						sipevent_t *sipevent;
						unsigned short int port = 0;  
						char *name;
#ifdef INET6
						int error;  
						char namebuf[BUFSIZ];  
						char sbuf[NI_MAXSERV];
#endif
						manager->udp_buf[i] = '\0';
#ifdef INET6
						error = getnameinfo((struct sockaddr *)&sa, slen,
								    namebuf, sizeof(namebuf), NULL, 0, NI_NUMERICHOST);
						if (error) {
						  osip_trace (OSIP_INFO1, ("getnameinfo error\n"));
						}
						else
						{
					  	  name = namebuf;
						  if (STRN_EQ(name, "::ffff:", 7))
						      name += 7;
						  switch(((struct sockaddr *)&sa)->sa_family){
						  case AF_INET:
						    port = ntohs(((struct sockaddr_in*)&sa)->sin_port);
						    osip_trace (OSIP_INFO1,
								("info: Message from %s:%i\n",
								 name,
								 port));
						    break;
						  case AF_INET6:
						    port = ntohs(((struct sockaddr_in6*)&sa)->sin6_port);
						    osip_trace (OSIP_INFO1,
								("info: Message from [%s]:%i\n",
								 name,
								 port));
						    break;
						  default:
						    osip_trace(OSIP_WARNING,
							       ("Cant find port number"));
						  }
						}
#else
						name = inet_ntoa (sa.sin_addr);
						port = sa.sin_port;
#endif
						osip_trace (OSIP_INFO1,
						    ("info: RECEIVING UDP MESSAGE:\n%s\n",
						     manager->udp_buf));
					
						sipevent = osip_parse (manager->udp_buf);
						if (sipevent != NULL){
						  if (MSG_IS_REQUEST(sipevent->sip))
						    {
						      osipua_fix_via_header(sipevent,
									    name,
									    port);
						    }
						  osipua_distribute_event(manager,sipevent);
						}	
					}
					else	/* if i>0 */
					{
					  osip_trace(OSIP_ERROR,
					     ("UDP listener failed while receiving data!\n"));
					}
				}
			}
			/* messages have been received */
		}
		smutex_lock(manager->mutex);
		osipua_execute(manager);
		smutex_unlock(manager->mutex);
		
	}
	osip_trace (OSIP_INFO1, ("Exiting osipua thread.\n"));
	return NULL;
}

int
osipua_fix_via_header(sipevent_t *evt, char *ip_addr, int port)
{
  generic_param_t *rport;
  via_t *via;
  /* get Top most Via header: */
  if (evt==NULL||evt->sip==NULL)
    return -1;
  via = list_get(evt->sip->vias, 0);
  if (via==NULL||via->host==NULL)
    /* Hey, we could build it? */
    return -1;

  via_param_getbyname(via, "rport", &rport);

  /* detect rport */
  if (rport!=NULL)
    {
      if (rport->gvalue==NULL)
	{
	  rport->gvalue = (char *) smalloc(8);
	  snprintf(rport->gvalue, 8, "%i", port);
	} /* else bug? */
    }

  /* only add the received parameter if the 'sent-by' value does not contains
     this ip address */
  if (0==strcmp(via->host, ip_addr)) /* don't need the received parameter */
    return 0;
  via_set_received(via, sgetcopy(ip_addr));
  return 0;
}


int
udp_send (transaction_t * trn, sip_t * sipmsg, char *host, int port, int sock)
{
#ifdef INET6  
         struct sockaddr_storage addr;  
         struct addrinfo hints, *res;  
         char num[8];  
         int error;  
#else
	struct sockaddr_in addr;
	unsigned long int one_inet_addr;
#endif
	char *message;
	
	if (sipmsg==NULL){
		osip_trace(OSIP_ERROR,("Null message !!"));
		return 0;
	}
	
	msg_2char(sipmsg,&message);
	
	osip_trace(OSIP_INFO1,("Sending message:\n%s",message));

       /* I add that code for compatibility with older version:
          This code is not needed anymore with libosip-0.9.2 and >.
          The "received" parameter IS ALWAYS THERE because linphone
          add it itself to evry incoming requests.
       */
	if (MSG_IS_RESPONSE(sipmsg))
	  {
	    via_t *via;
	    char *host;
	    int port;
	    generic_param_t *maddr;
	    generic_param_t *received;
	    generic_param_t *rport;
	    via = list_get(sipmsg->vias, 0);
	    via_param_getbyname(via, "maddr", &maddr);
	    via_param_getbyname(via, "received", &received);
	    via_param_getbyname(via, "rport", &rport);
	    /* 1: for reliable protocol (not supported)
	       2: check maddr and multicast usage (not supported) */
	    if (maddr!=NULL)
	      host = maddr->gvalue;
	    /* we should check if this is a multicast address and use
	       set the "ttl" in this case. */
	    else if (received!=NULL)
	      host = received->gvalue;
	    else host = via->host;
	    
	    if (rport==NULL||rport->gvalue==NULL)
	      {
		if (via->port!=NULL) port = satoi(via->port);
		else port = 5060;
	      }
	    else
	      port = satoi(rport->gvalue);
	  }

#ifdef INET6  
	memset(&hints, 0, sizeof(hints));  
	hints.ai_family = PF_UNSPEC;  
	hints.ai_socktype = SOCK_DGRAM;  
	hints.ai_flags = AI_NUMERICHOST;  
	snprintf(num, sizeof(num), "%d", port);  
	error = getaddrinfo(host, num,  &hints, &res);  
	if (error) {  
	  osip_trace (OSIP_INFO1,  
		      ("error: %s: %s\n", host, gai_strerror(error)));  
	  return -1;  
	}  
	if (res->ai_addrlen > sizeof(addr)) {  
	  osip_trace (OSIP_INFO1,  
		      ("error: sockaddr too large\n"));  
	  freeaddrinfo(res);
	  return -1;  
	}  
	memcpy(&addr, res->ai_addr, res->ai_addrlen);  
	freeaddrinfo(res);  
#else
	if ((int) (one_inet_addr = inet_addr (host)) == -1)
	{
		osip_trace (OSIP_INFO1,
			    ("error: destination is not an ip address:%s\n",
			     host));
		return -1;
	}
	else
	{
		addr.sin_addr.s_addr = one_inet_addr;
	}

	addr.sin_port = htons ((short) port);
	addr.sin_family = AF_INET;
#endif
	if (sock<=0){
		osip_trace(OSIP_INFO1,("warning: using default manager socket to send a message.\n"));
		sock=def_manager->send_sock;
	}

	if (0 > sendto (sock, (const void *) message, strlen (message), 0,
			(struct sockaddr *) &addr, sizeof (addr)))
	{
		osip_trace(OSIP_ERROR,("udp_send: could not send message using socket %i: %s\n",sock,strerror(errno)));
	}

	
	return 0;
}
