/*
linphone
Copyright (C) 2000  Simon MORLAT (simon.morlat@free.fr)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "linphonecore.h"
#include <mediastream.h>
#include <osipua.h>
#include "../osipua/src/sdphandler.h"
#include "gnome-config.h"
#include "misc.h"

#include <telephonyevents.h>
#include <utils.h>
#include "osipuacb.h"

#include <glib.h>
#include <net/if.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#ifdef INET6  
#include <netdb.h>  
#endif

#define MAX_IF 20
const double bandwidths[5]=
{
	28800.0,	/*28k modem*/
	28800.0, /*56k modem*/
	64000.0, /*64k modem (numeris)*/
	128000.0,	/*adsl or cable*/
	10e6, /* ethernet */
};/* carefull: these are the upstream bandwidth !!!*/

/* relative path where is stored local ring*/
#define LOCAL_RING "rings/orig.wav"
/* same for remote ring (ringback)*/
#define REMOTE_RING_FR "ringback.wav"
#define REMOTE_RING_US "ringback.wav"


/* these are useful messages for the status */
gchar *ready;
gchar *end;
gchar *contacting;
gchar *subscribing;
gchar *connected;
gchar *cancel;
gchar *contacted;


void lc_callback_obj_init(LCCallbackObj *obj,LinphoneCoreCbFunc func,gpointer ud)
{
  obj->_func=func;
  obj->_user_data=ud;
}

gint lc_callback_obj_invoke(LCCallbackObj *obj, LinphoneCore *lc){
  if (obj->_func!=NULL) obj->_func(lc,obj->_user_data);
  return 0;
}

LinphoneDialogParams * linphone_dialog_params_new()
{
	return g_new0(LinphoneDialogParams,1);
}

void linphone_dialog_params_destroy(LinphoneDialogParams *obj)
{
	if (obj->profile!=&av_profile) rtp_profile_destroy(obj->profile);
	g_free(obj);
}


#ifdef LINPHONE_DEPRECATED
/* this is the retrieve_if function for FreeBSD. It uses getifaddrs system call instead of ioctl in order to get interface names and address.*/
/* Thanks 	to Wolfgang for this .*/
#ifdef HAVE_GETIFADDRS
#include <sys/types.h>
#include <sys/socket.h>
#include <ifaddrs.h>
GList *
retrieve_if ()
{
	int i;
	struct ifaddrs *ifp;
	struct ifaddrs *ifpstart;
	GList *interfaces=NULL;

	if (getifaddrs (&ifpstart) < 0)
	{
		return NULL;
	}
	ifp=ifpstart;
	for (i = 0; ifp != NULL; ifp = ifp->ifa_next)
	{
		interface_t *netif;
#ifdef INET6
		char namebuf[BUFSIZ];

		if (!ifp->ifa_addr)
		  continue;

		switch (ifp->ifa_addr->sa_family) {
		case AF_INET:
		        getnameinfo(ifp->ifa_addr, sizeof (struct sockaddr_in) ,
				    namebuf, sizeof(namebuf), NULL, 0, NI_NUMERICHOST);
		        netif=g_new0(interface_t,1);
			netif->index=i;
			netif->family = AF_INET;
			strncpy (netif->name, ifp->ifa_name,20);
			strncpy (netif->ipaddr, namebuf,20);
			interfaces=g_list_append(interfaces,netif);
			i++;
			break;
		case AF_INET6:
		        getnameinfo(ifp->ifa_addr, sizeof (struct sockaddr_in6) ,
				    namebuf, sizeof(namebuf), NULL, 0, NI_NUMERICHOST);
			if (strchr(namebuf, '%') == NULL) {
			  netif=g_new0(interface_t,1);
			  netif->index=i;
			  netif->family = AF_INET6;
			  strncpy (netif->name, ifp->ifa_name,20);
			  strncpy (netif->ipaddr, namebuf, 40);
			  interfaces=g_list_append(interfaces,netif);
			}
			i++;
			break;
		default:
		        continue;
  		}
#else
		if (ifp->ifa_addr->sa_family != AF_INET)
			continue;

		/*printf ("ifc %s %s\n",
			ifp->ifa_name,
			inet_ntoa (((struct sockaddr_in *) ifp->ifa_addr)->
				   sin_addr));*/
			
		netif=g_new0(interface_t,1);
		netif->index=i;
		strncpy (netif->name, ifp->ifa_name,20);
		strncpy (netif->ip4addr, inet_ntoa (((struct sockaddr_in *) ifp->ifa_addr)->sin_addr),20);
		interfaces=g_list_append(interfaces,netif);
		i++;
#endif
	}

	freeifaddrs (ifpstart);

	return interfaces;
}
#else

GList * retrieve_if()
{
	int sock,err,if_count,i;
	struct ifconf netconf;
	char buffer[sizeof(struct ifreq)*MAX_IF];
	GList *interfaces=NULL;
	
	netconf.ifc_len=sizeof(struct ifreq)*MAX_IF;
	netconf.ifc_buf=buffer;
	sock=socket(PF_INET, SOCK_DGRAM,0);
	err=ioctl(sock,SIOCGIFCONF,&netconf);
	if (err<0) printf("Error in ioctl: %i.\n",errno);
	close(sock);
	
	if_count=netconf.ifc_len/sizeof(struct ifreq);
	g_message("Found %i interfaces.\n",if_count);
	for (i=0;i<if_count;i++)
	{
#ifdef INET6
		interface_t *netif;
		char namebuf[BUFSIZ];

		
		

		switch (netconf.ifc_req[i].ifr_addr.sa_family) {
		case AF_INET:
			getnameinfo(&netconf.ifc_req[i].ifr_addr, sizeof (struct sockaddr_in) ,
				    namebuf, sizeof(namebuf), NULL, 0, NI_NUMERICHOST);
			netif=g_new0(interface_t,1);
			netif->index=i;
			netif->family = AF_INET;
			strncpy (netif->name, netconf.ifc_req[i].ifr_name,20);
			strncpy (netif->ipaddr, namebuf,20);
			interfaces=g_list_append(interfaces,netif);
			break;
		case AF_INET6:
			getnameinfo(&netconf.ifc_req[i].ifr_addr, sizeof (struct sockaddr_in6) ,
				    namebuf, sizeof(namebuf), NULL, 0, NI_NUMERICHOST);
			if (strchr(namebuf, '%') == NULL) {
				netif=g_new0(interface_t,1);
			  	netif->index=i;
			  	netif->family = AF_INET6;
			  	strncpy (netif->name, netconf.ifc_req[i].ifr_name,20);
			  	strncpy (netif->ipaddr, namebuf, 40);
			  	interfaces=g_list_append(interfaces,netif);
			}
			break;
		default:
		        continue;
  		}
		
#else
		interface_t *netif;
		if (netconf.ifc_req[i].ifr_addr.sa_family != AF_INET)
			continue;
		netif=g_new0(interface_t,1);
		strncpy(netif->name,netconf.ifc_req[i].ifr_name,20);
		strncpy(netif->ip4addr,inet_ntoa(((struct sockaddr_in*)(&netconf.ifc_req[i].ifr_addr))->sin_addr),20);
		netif->index=i;
		interfaces=g_list_append(interfaces,netif);
#endif
	}
	return interfaces;
}
#endif



gboolean find_interface(net_config_t *config, char *ifname)
{
	GList *interfaces=config->interfaces;
	interface_t *tmpif;
#ifdef INET6
	interface_t *tmp4if = NULL;
	int checkflag = 0;
#endif
	
	tmpif=(interface_t*) interfaces->data;
	while (interfaces!=NULL){
		tmpif=(interface_t*) interfaces->data;
		if (ifname!=NULL){
			if (strcmp(ifname,tmpif->name)==0){
#ifdef INET6
			  if (tmpif->family == AF_INET6) {
				config->sel_if=tmpif;
				return 1;
			  } else {
			        checkflag = 1;
				tmp4if = tmpif;
			  }
#else
				config->sel_if=tmpif;
				return 1;
#endif
			}
		}
		interfaces=g_list_next(interfaces);
	}
#ifdef INET6
	if (checkflag == 1) {
	        config->sel_if=tmp4if;
		return 1;
	}
#endif
	return 0;
}

void find_best_interface(net_config_t *config)
{
	interface_t *tmpif;
	tmpif=(interface_t*) config->interfaces->data;
	if (!find_interface(config,"ppp0")){
		/* if ppp0 is not found, then default to eth0 */
		if (!find_interface(config,"eth0")){
			/* if eth0 is not found, try ed0 (used on FreeBsd) */
			if (!find_interface(config,"ed0")){
				/* if eth0 is not found, then default to lo */
				if (!find_interface(config,"lo")){
					/* if lo is not found, default to first interface in the list */
					config->sel_if=tmpif;
				}
			}
		}
	}
}

#endif /*LINPHONE_DEPRECATED*/

void
net_config_read (LinphoneCore *lc)
{
	gchar *ifname;
	gboolean def=FALSE;
	gchar *tmpstr;
	gint tmp;
	
#ifdef LINPHONE_DEPRECATED
	config->interfaces=retrieve_if();
	/* select the best interface, not lo */
	if (config->interfaces==NULL) {
		g_error("Could not get network interfaces !!");
		return;
	}
	ifname=gnome_config_get_string("net/if_name");
	if ((ifname!=NULL) && (strlen(ifname)!=0)){
		/* try to find it */
		if (!find_interface(config,ifname)){
			g_warning("Could not find network interface %s.",ifname);
			find_best_interface(config);
		}
	}else find_best_interface(config);
#endif /*LINPHONE_DEPRECATED*/	

	tmp=gnome_config_get_int_with_default("net/con_type",&def);
#ifdef __ARM__
	if (def) tmp=CONNECTION_TYPE_ETHERNET;
#else
	if (def) tmp=CONNECTION_TYPE_56KMODEM;
#endif
	linphone_core_set_connection_type(lc,tmp);
	
	tmpstr=gnome_config_get_string_with_default("net/nat_address",&def);
	if (def) tmpstr=NULL;
	if (tmpstr!=NULL && (strlen(tmpstr)<1)) tmpstr=NULL;
	tmp=gnome_config_get_int_with_default("net/use_nat",&def);
	if (def) tmp=0;
	linphone_core_set_nat_address(lc,tmpstr,tmp);		
}

void sound_config_read(LinphoneCore *lc)
{
	gboolean def;
	int devid,tmp;
	char *tmpbuf;
	SndCard *sndcard;
	devid=gnome_config_get_int_with_default("sound/dev_id",&def);
	if (def) {
		/* choose an oss card if availlable, because alsa native support is not well enough stable*/
		int i;
		char *id;
		for(i=0;i<MAX_SND_CARDS;i++){
			sndcard=snd_card_manager_get_card(snd_card_manager,i);
			if (sndcard==NULL){
				devid=0;
				break;
			}
			id=snd_card_get_identifier(sndcard);
			if (strstr(id,"Open Sound System")!=0){
				g_message("Choosing oss device %s.",id);
				devid=i;
				break;
			}
		}
	}
	linphone_core_set_sound_device(lc,devid);
	tmp=gnome_config_get_int_with_default("sound/play_lev",&def);
	if (def) tmp=80;
	linphone_core_set_play_level(lc,tmp);
	tmp=gnome_config_get_int_with_default("sound/rec_lev",&def);
	if (def) tmp=80;
	linphone_core_set_rec_level(lc,tmp);
	tmpbuf=gnome_config_get_string_with_default("sound/source",&def);
	if (def) tmpbuf="m";
	linphone_core_set_sound_source(lc,tmpbuf[0]);
	
	tmpbuf=gnome_config_get_string_with_default("sound/local_ring",&def);
	if (def) tmpbuf=PACKAGE_SOUND_DIR "/" LOCAL_RING;
	else {
		if (!g_file_test(tmpbuf,G_FILE_TEST_EXISTS)){
			tmpbuf=PACKAGE_SOUND_DIR "/" LOCAL_RING;
		}
		if (strstr(tmpbuf,".wav")==NULL){
			/* it currently uses old sound files, so replace them */
			tmpbuf=PACKAGE_SOUND_DIR "/" LOCAL_RING;
		}
	}
	linphone_core_set_ring(lc,tmpbuf);
	tmpbuf=gnome_config_get_string_with_default("sound/remote_ring",&def);
	if (def) tmpbuf=PACKAGE_SOUND_DIR "/" REMOTE_RING_FR;
	else {
		if (!g_file_test(tmpbuf,G_FILE_TEST_EXISTS)){
			tmpbuf=PACKAGE_SOUND_DIR "/" REMOTE_RING_FR;
		}
		if (strstr(tmpbuf,".wav")==NULL){
			/* it currently uses old sound files, so replace them */
			tmpbuf=PACKAGE_SOUND_DIR "/" REMOTE_RING_FR;
		}
	}
	linphone_core_set_ringback(lc,0);
	lc->sound_conf.autokill=1;
	check_sound_device(lc);
}

void registrar_config_read(LinphoneCore *lc)
{
	gboolean def;
	gchar *tmpstr,*tmpstr2;
	gint tmp1,tmp2,tmp3;
	
	tmpstr=gnome_config_get_string_with_default("sip/registrar",&def);
	if (def) tmpstr=NULL;
	tmpstr2=gnome_config_get_string_with_default("sip/addr_of_rec",&def);
	if (def) tmpstr2=NULL;
	linphone_core_set_registrar_addr(lc,tmpstr,tmpstr2);
	tmpstr=gnome_config_get_string_with_default("sip/passwd",&def);
	if (def) tmpstr=NULL;
	linphone_core_set_registrar_passwd(lc,tmpstr);
	tmp1=gnome_config_get_int_with_default("sip/use_registrar",&def);
	if (def) tmp1=0;
	tmp2=gnome_config_get_int_with_default("sip/as_proxy",&def);
	if (def) tmp2=0;
	tmp3=gnome_config_get_int_with_default("sip/expires",&def);
	if (def) tmp3=15*60;
	linphone_core_set_registrar_usage(lc,tmp1,tmp2,tmp3);
	
}

void sip_config_read(LinphoneCore *lc)
{
	gboolean def;
	gchar *hostname,*username;
	gint port;
	username=gnome_config_get_string_with_default("sip/username",&def);
	if (def) {
		gchar *logname=getenv("LOGNAME");
		if (logname!=NULL)
			username=g_strdup(logname);
		else username=g_strdup("toto");
	}
	hostname=gnome_config_get_string_with_default("sip/hostname",&def);
	if (def) {
		gchar *hostname2=getenv("HOST");
		if (hostname2!=NULL)
			hostname=g_strdup(hostname2);
		else hostname=g_strdup("unknown-host");
	}
	port=gnome_config_get_int_with_default("sip/sip_port",&def);
	if (def){
		port=5060;
	}
	linphone_core_set_sip_port(lc,port);
	linphone_core_set_primary_contact(lc,g_strdup_printf("sip:%s@%s",username,hostname));
	registrar_config_read(lc);
}

void rtp_config_read(LinphoneCore *lc)
{
	gboolean def;
	gint port;
	gint jitt_comp;
	port=gnome_config_get_int_with_default("rtp/audio_rtp_port",&def);
	if (def) port=7078;
	linphone_core_set_audio_port(lc,port);
	port=gnome_config_get_int_with_default("rtp/video_rtp_port",&def);
	if (def) port=9078;
	
	jitt_comp=gnome_config_get_int_with_default("rtp/audio_jitt_comp",&def);
	if (def) jitt_comp=60;
	linphone_core_set_audio_jittcomp(lc,jitt_comp);		
	jitt_comp=gnome_config_get_int_with_default("rtp/video_jitt_comp",&def);
	if (def) jitt_comp=60;	
	
}


PayloadType * get_codec(char* type,int index){
	gchar codeckey[50];
	gchar *mime;
	gint rate,enabled;
	gboolean def;
	PayloadType *pt;
	
	snprintf(codeckey,50,"%s_%i/mime",type,index);
	mime=gnome_config_get_string_with_default(codeckey,&def);
	if (def) return NULL;
	
	pt=payload_type_new();
	pt->mime_type=mime;
	snprintf(codeckey,50,"%s_%i/rate",type,index);
	rate=gnome_config_get_int_with_default(codeckey,&def);
	if (def) rate=8000; // ??
	pt->clock_rate=rate;
	
	snprintf(codeckey,50,"%s_%i/enabled",type,index);
	enabled=gnome_config_get_int_with_default(codeckey,&def);
	if (enabled || def) pt->flags|=PAYLOAD_TYPE_ENABLED;
	//g_message("Found codec %s/%i",pt->mime_type,pt->clock_rate);
	return pt;
}

void codecs_config_read(LinphoneCore *lc)
{
	int i;
	PayloadType *pt;
	GList *audio_codecs=NULL;
	GList *video_codecs=NULL;
	for (i=0;;i++){
		pt=get_codec("audio_codec",i);
		if (pt==NULL) break;
		audio_codecs=g_list_append(audio_codecs,(gpointer)pt);
	}
	for (i=0;;i++){
		pt=get_codec("video_codec",i);
		if (pt==NULL) break;
		video_codecs=g_list_append(video_codecs,(gpointer)pt);
	}
	linphone_core_set_audio_codecs(lc,audio_codecs);
	linphone_core_set_video_codecs(lc,video_codecs);
	linphone_core_setup_local_rtp_profile(lc);
}

void video_config_read(LinphoneCore *lc)
{
	gboolean def;
	gint tmp;
	tmp=gnome_config_get_int_with_default("video/enabled",&def);
	if (def) tmp=0;
	tmp=gnome_config_get_int_with_default("video/show_local",&def);
	if (def) tmp=0;
}

void ui_config_read(LinphoneCore *lc)
{
	int len,tmp;
	char *ab_entry;
	GList *address_list=NULL;
	GList *pc_list=NULL; /* placed calls list */
	GList *rc_list=NULL; /* received calls list */
	GList *mc_list=NULL; /* missed calls list */
	GList *buddy_list=NULL;
	/* get address book entries*/
	len= gnome_config_get_int("address_book/entry_count");
	if (len!=-1)
	{
		for (tmp=0;tmp<len;tmp++)
		{
			LinphoneCoreAddressEntry *entry = (LinphoneCoreAddressEntry *)g_malloc(sizeof(LinphoneCoreAddressEntry));
			ab_entry=g_strdup_printf("address_book/entry%i",tmp);
			entry->address = (gpointer)gnome_config_get_string(ab_entry);
			g_free(ab_entry);

			ab_entry=g_strdup_printf("address_book/entrysubscribe%i",tmp);
			entry->subscribe_notify = (gpointer)gnome_config_get_string(ab_entry);
			g_free(ab_entry);

			address_list=g_list_append(address_list,entry);
		}
	}
	linphone_core_set_address_list(lc,address_list);

	len= gnome_config_get_int("placed_calls/entry_count");
	if (len!=-1)
	{
		for (tmp=0;tmp<len;tmp++)
		{
			char *pc_name=g_strdup_printf("placed_calls/entry%i",tmp);
			char *pc_value = (gpointer)gnome_config_get_string(pc_name);
			g_free(pc_name);

			pc_list = g_list_append(pc_list,pc_value);
		}
	}
	linphone_core_set_placed_calls_list(lc,pc_list);

	len= gnome_config_get_int("received_calls/entry_count");
	if (len!=-1)
	{
		for (tmp=0;tmp<len;tmp++)
		{
			char *pc_name=g_strdup_printf("received_calls/entry%i",tmp);
			char *pc_value = (gpointer)gnome_config_get_string(pc_name);
			g_free(pc_name);

			rc_list = g_list_append(rc_list,pc_value);
		}
	}
	linphone_core_set_received_calls_list(lc,rc_list);

	len= gnome_config_get_int("missed_calls/entry_count");
	if (len!=-1)
	{
		for (tmp=0;tmp<len;tmp++)
		{
			char *pc_name=g_strdup_printf("missed_calls/entry%i",tmp);
			char *pc_value = (gpointer)gnome_config_get_string(pc_name);
			g_free(pc_name);

			mc_list = g_list_append(mc_list,pc_value);
		}
	}
	linphone_core_set_missed_calls_list(lc,mc_list);

	len= gnome_config_get_int("buddy_list/entry_count");
	if (len!=-1)
	{
		for (tmp=0;tmp<len;tmp++)
		{
			char *pc_name=g_strdup_printf("buddy_list/entry%i",tmp);
			char *pc_value = (gpointer)gnome_config_get_string(pc_name);
			g_free(pc_name);

			buddy_list = g_list_append(buddy_list,pc_value);
		}
	}
	linphone_core_set_buddy_list(lc,buddy_list);

}

void autoreplier_config_init(autoreplier_config_t *config)
{
	gboolean def;
	config->enabled=gnome_config_get_int_with_default("autoreplier/enabled",&def);
	if (def) config->enabled=0;
	config->after_seconds=gnome_config_get_int_with_default("autoreplier/after_seconds",&def);
	if (def) config->after_seconds=6;
	config->max_users=gnome_config_get_int_with_default("autoreplier/max_users",&def);
	if (def) config->max_users=1;
	config->max_rec_time=gnome_config_get_int_with_default("autoreplier/max_rec_time",&def);
	if (def) config->max_rec_time=0;
	config->max_rec_msg=gnome_config_get_int_with_default("autoreplier/max_rec_msg",&def);
	if (def) config->max_rec_msg=0;
	config->message=gnome_config_get_string_with_default("autoreplier/message",&def);
	if (def) config->message=NULL;
}

void
linphone_core_init (LinphoneCore * lc,LinphoneCoreVTable *vtable, gchar *config_path, gpointer userdata)
{
	int len,err;
	char *prim_contact;
	
	//TRACE_INITIALIZE(TRACE_LEVEL5,stdout);
	osipua_init();
	ortp_init();
	ortp_set_debug_file("oRTP",NULL);
	rtp_profile_set_payload(&av_profile,115,&lpc1015);
#ifdef  BUILD_SPEEX
	rtp_profile_set_payload(&av_profile,110,&speex_nb);
	rtp_profile_set_payload(&av_profile,111,&speex_wb);
#endif
	rtp_profile_set_payload(&av_profile,116,&truespeech);
	rtp_profile_set_payload(&av_profile,101,&telephone_event);
	ms_init();
#ifdef  BUILD_SPEEX
	ms_speex_codec_init();
#endif
	
	memset (lc, 0, sizeof (LinphoneCore));
	lc->data=userdata;
	ready=_("Ready.");
  	end=_("Communication ended.");
	contacting= _("Contacting ");
	contacted=_("is calling you.");
	connected=_("Connected.");
	cancel=_("Call cancelled.");
	subscribing= _("Subscribing to ");
	
	memcpy(&lc->vtable,vtable,sizeof(LinphoneCoreVTable));

	if (config_path!=NULL){
		gnome_config_push_prefix(g_strdup_printf("=%s=",config_path));
	}else gnome_config_push_prefix("/linphone/");
	
	
	lc->ua=osip_ua_new();
  	err=osip_ua_signal_connect(lc->ua,"INVITE_ACCEPTED",invite_accepted_cb);
  	err=osip_ua_signal_connect(lc->ua,"BYE",bye_cb);
  	err=osip_ua_signal_connect(lc->ua,"FAILURE",failure_cb);
  	err=osip_ua_signal_connect(lc->ua,"INVITE",invite_cb);
  	err=osip_ua_signal_connect(lc->ua,"INFORMATIVE",informative_cb);
  	err=osip_ua_signal_connect(lc->ua,"MESSAGE",message_cb);
  	err=osip_ua_signal_connect(lc->ua,"SUBSCRIBE",subscribe_cb);
  	err=osip_ua_signal_connect(lc->ua,"SUBSCRIBE_ACCEPTED",subscribe_accepted_cb);
  	err=osip_ua_signal_connect(lc->ua,"NOTIFY",notify_cb);
	/* LIK: Transfer call related functionality */
	err=osip_ua_signal_connect(lc->ua, "MUTE", mute_func_cb);
	err=osip_ua_signal_connect(lc->ua, "BYETRANSFER", byetransfer_func_cb);
	err=osip_ua_signal_connect(lc->ua, "REFERTRANSFER", refertransfer_func_cb);
	err=osip_ua_signal_connect(lc->ua, "NOTIFYTRANSFER", notifytransfer_func_cb);
  	lc->ua->data=(void*)lc;
	
	/* create a sdp handler */
	lc->sdph=sdp_handler_new();
	sdp_handler_set_write_offer_fcn(SDP_HANDLER(lc->sdph),set_audio_offer,set_video_offer);
	sdp_handler_set_accept_offer_fcn(SDP_HANDLER(lc->sdph),accept_audio_offer,accept_video_offer);
	sdp_handler_set_read_answer_fcn(SDP_HANDLER(lc->sdph),read_audio_answer,read_video_answer);
	
	/* add the sdp handler to the ua */
	osip_ua_add_body_handler(lc->ua,lc->sdph);
	
	/* create a cpim handler and bind it to the ua */
	lc->cpimh=cpim_handler_new();
	osip_ua_add_body_handler(lc->ua,lc->cpimh);
	
	
	sound_config_read(lc);
	sip_config_read(lc);
	net_config_read(lc);
	rtp_config_read(lc);
	codecs_config_read(lc);
	video_config_read(lc);
	//autoreplier_config_init(&lc->autoreplier_conf);
	ui_config_read(lc);
	
	lc->in_main_thread=1;
	lc->lock=g_mutex_new();
}

LinphoneCore *linphone_core_new(LinphoneCoreVTable *vtable,
						gchar *config_path, gpointer userdata)
{
	LinphoneCore *core=g_new(LinphoneCore,1);
	linphone_core_init(core,vtable,config_path,userdata);
	return core;
}

GList *linphone_core_get_audio_codecs(LinphoneCore *lc)
{
	return g_list_copy(lc->codecs_conf.audio_codecs);
}

GList *linphone_core_get_video_codecs(LinphoneCore *lc)
{
	return g_list_copy(lc->codecs_conf.video_codecs);
}

int linphone_core_set_primary_contact(LinphoneCore *lc, gchar *contact)
{
	int error=0;
	from_t *ct;
	from_init(&ct);
	if ( from_parse(ct,contact) <0){
		g_warning("Bad contact url: %s", contact);
		error=1;
		goto end;
	}
	lc->sip_conf.username=g_strdup(ct->url->username);
	lc->sip_conf.hostname=g_strdup(ct->url->host);
	linphone_core_update_contact_info(lc);
	end:
		from_free(ct);
		sfree(ct);
	return error;
}

gchar *linphone_core_get_primary_contact(LinphoneCore *lc)
{
	return g_strdup(lc->sip_conf.contact);
}

from_t *linphone_core_get_primary_contact_parsed(LinphoneCore *lc)
{
	int err;
	from_t *contact;
	from_init(&contact);
	err=from_parse(contact,lc->sip_conf.contact);
	if (err<0) {
		from_free(contact);
		sfree(contact);
		return NULL;
	}
	return contact;
}

int linphone_core_set_audio_codecs(LinphoneCore *lc, GList *codecs)
{
	if (lc->codecs_conf.audio_codecs!=NULL) g_list_free(lc->codecs_conf.audio_codecs);
	lc->codecs_conf.audio_codecs=codecs;
	return 0;
}

int linphone_core_set_video_codecs(LinphoneCore *lc, GList *codecs)
{
	if (lc->codecs_conf.video_codecs!=NULL) g_list_free(lc->codecs_conf.video_codecs);
	lc->codecs_conf.video_codecs=codecs;
	return 0;
}

GList * linphone_core_get_address_list(LinphoneCore *lc)
{
	return g_list_copy(lc->ui_conf.address_list);
}

int linphone_core_set_address_list(LinphoneCore *lc, GList *addresses)
{
	if (lc->ui_conf.address_list!=NULL) g_list_free(lc->ui_conf.address_list);
	lc->ui_conf.address_list=addresses;
	return 0;
}

GList * linphone_core_get_placed_calls_list(LinphoneCore *lc)
{
	return g_list_copy(lc->ui_conf.placed_calls_list);
}

int linphone_core_set_placed_calls_list(LinphoneCore *lc, GList *placed_calls)
{
	if (lc->ui_conf.placed_calls_list!=NULL) g_list_free(lc->ui_conf.placed_calls_list);
	lc->ui_conf.placed_calls_list=placed_calls;
	return 0;
}

GList * linphone_core_get_received_calls_list(LinphoneCore *lc)
{
	return g_list_copy(lc->ui_conf.received_calls_list);
}

int linphone_core_set_received_calls_list(LinphoneCore *lc, GList *received_calls)
{
	if (lc->ui_conf.received_calls_list!=NULL) g_list_free(lc->ui_conf.received_calls_list);
	lc->ui_conf.received_calls_list=received_calls;
	return 0;
}

GList * linphone_core_get_missed_calls_list(LinphoneCore *lc)
{
	return g_list_copy(lc->ui_conf.missed_calls_list);
}

int linphone_core_set_missed_calls_list(LinphoneCore *lc, GList *missed_calls)
{
	if (lc->ui_conf.missed_calls_list!=NULL) g_list_free(lc->ui_conf.missed_calls_list);
	lc->ui_conf.missed_calls_list=missed_calls;
	return 0;
}

GList * linphone_core_get_buddy_list(LinphoneCore *lc)
{
	return g_list_copy(lc->ui_conf.buddy_list);
}

int linphone_core_set_buddy_list(LinphoneCore *lc, GList *addresses)
{
	if (lc->ui_conf.address_list!=NULL) g_list_free(lc->ui_conf.buddy_list);
	lc->ui_conf.buddy_list=addresses;
	return 0;
}

char *linphone_core_get_registrar_addr(LinphoneCore *lc)
{
	return g_strdup(lc->sip_conf.reg_conf.registrar);
}

int linphone_core_set_registrar_addr(LinphoneCore *lc,char *url,char *addr_of_rec)
{
	url_t *surl=NULL,*surl2=NULL;
	int err=0;
	if (url==NULL || strlen(url)==0) return 0;
	url_init(&surl);
	err=url_parse(surl,url);
	if (err<0){
		g_warning("Bad proxy or registrar url:%s",url);
		goto end;
	}
	if (addr_of_rec!=NULL){
		url_init(&surl2);
		err=url_parse(surl2,addr_of_rec);
		if (err<0){
			g_warning("Bad address of record:%s",addr_of_rec);
			goto end;
		}
	}
	if (lc->sip_conf.reg_conf.registrar!=NULL){
		g_free(lc->sip_conf.reg_conf.registrar);
	}
	lc->sip_conf.reg_conf.registrar=g_strdup(url);
	
	if (addr_of_rec!=NULL){
		if (lc->sip_conf.reg_conf.addr_of_rec!=NULL){
			g_free(lc->sip_conf.reg_conf.addr_of_rec);
		}
		lc->sip_conf.reg_conf.addr_of_rec=g_strdup(addr_of_rec);
	}else{
		/* guess a valid address of record */
		if (surl->port!=NULL)
			lc->sip_conf.reg_conf.addr_of_rec=g_strdup_printf("sip:%s@%s:%s",
				lc->sip_conf.username,surl->host,surl->port);
		else lc->sip_conf.reg_conf.addr_of_rec=g_strdup_printf("sip:%s@%s",
				lc->sip_conf.username,surl->host);
	}
	end:
		url_free(surl);
		sfree(surl);
		if (surl2!=NULL){
			url_free(surl2);
			sfree(surl2);
		}
	return err;
}

char *linphone_core_get_addr_of_rec(LinphoneCore *lc)
{
	return g_strdup(lc->sip_conf.reg_conf.addr_of_rec);
}

char *linphone_core_get_registrar_passwd(LinphoneCore *lc)
{
	return g_strdup(lc->sip_conf.reg_conf.passwd);
}

void linphone_core_set_registrar_passwd(LinphoneCore *lc, gchar *passwd)
{
	gchar *tmp=NULL;
	if (passwd!=NULL){
		tmp=g_strdup(passwd);
	}
	if (lc->sip_conf.reg_conf.passwd!=NULL){
		g_free(lc->sip_conf.reg_conf.passwd);
	}
	lc->sip_conf.reg_conf.passwd=tmp;
}


int linphone_core_register(LinphoneCore *lc, char *url){
	if (url!=NULL){
		linphone_core_set_registrar_addr(lc,url,NULL);
	}
	do_registration(lc,1);
	return 0;
}

void linphone_core_get_registrar_usage(LinphoneCore *lc,gboolean *reg, gboolean *proxy, gint *expires)
{
	if (reg!=NULL) *reg= lc->sip_conf.reg_conf.use_registrar;
	if (proxy!=NULL) *proxy=lc->sip_conf.reg_conf.as_proxy;
	if (expires!=NULL) *expires=lc->sip_conf.reg_conf.expires;
}



void linphone_core_set_registrar_usage(LinphoneCore *lc, gboolean use, gboolean outbound_proxy,gint expires)
{
	if (lc->sip_conf.reg_conf.timer==NULL){
		GTimer *timer;
		timer=g_timer_new();
		g_timer_start(timer);
		lc->sip_conf.reg_conf.timer=timer;
	}
	if (expires<=0) lc->sip_conf.reg_conf.expires=15*60;
	else lc->sip_conf.reg_conf.expires=expires;
	if ((lc->sip_conf.reg_conf.registrar!=NULL) && (strlen(lc->sip_conf.reg_conf.registrar)!=0)){
		if (use){
			if (outbound_proxy)
				osip_ua_set_outbound_proxy(lc->ua,lc->sip_conf.reg_conf.registrar,lc->sip_conf.reg_conf.passwd);
			else osip_ua_set_outbound_proxy(lc->ua,NULL,NULL);
			do_registration(lc,1);
		}else{
			osip_ua_set_outbound_proxy(lc->ua,NULL,NULL);
			if (lc->sip_conf.reg_conf.use_registrar){
				/* the proxy is no more used, we need to unregister */
				do_registration(lc,0);
			}
		}
	}
	lc->sip_conf.reg_conf.use_registrar=use;
	lc->sip_conf.reg_conf.as_proxy=outbound_proxy;
}

GList * linphone_core_get_net_interfaces(LinphoneCore *lc)
{
	return g_list_copy(lc->net_conf.interfaces);
}

interface_t * linphone_core_get_selected_interface(LinphoneCore *lc)
{
	return lc->net_conf.sel_if;
}

#ifdef INET6
void linphone_core_select_net_interface(LinphoneCore *lc, gint index)
#else
void linphone_core_select_net_interface(LinphoneCore *lc, gchar *ifname)
#endif
{
	GList *elem=lc->net_conf.interfaces;
	while(elem!=NULL){
		interface_t *iface=(interface_t *)elem->data;
#ifdef INET6
		if (index == iface->index) {
#else
		if (strcmp(ifname,iface->name)==0){
#endif
			lc->net_conf.sel_if=iface;
			linphone_core_update_contact_info(lc);
			return;
		}
		elem=g_list_next(elem);
	}
#ifdef INET6
	g_warning("Could not select interface (index: %d): non existent.",index);
#else
	g_warning("Could not select interface %s: non existent.",ifname);
#endif
}

void linphone_core_set_connection_type(LinphoneCore *lc, int type)
{
	lc->net_conf.con_type=type;
	lc->net_conf.bandwidth=bandwidths[type];
	linphone_core_check_codecs_for_bandwidth(lc);
}

int linphone_core_get_connection_type(LinphoneCore *lc)
{
	return lc->net_conf.con_type;
}

int linphone_core_get_audio_jittcomp(LinphoneCore *lc)
{
	return lc->rtp_conf.audio_jitt_comp;
}

int linphone_core_get_audio_port(LinphoneCore *lc)
{
	return lc->rtp_conf.audio_rtp_port;
}

void linphone_core_set_audio_jittcomp(LinphoneCore *lc, int value)
{
	lc->rtp_conf.audio_jitt_comp=value;
}

void linphone_core_set_audio_port(LinphoneCore *lc, int port)
{
	lc->rtp_conf.audio_rtp_port=port;
}

int linphone_core_get_sip_port(LinphoneCore *lc)
{
	return lc->sip_conf.sip_port;
}

void linphone_core_set_sip_port(LinphoneCore *lc,int port)
{
	lc->sip_conf.sip_port=port;
	linphone_core_update_contact_info(lc);
}

void linphone_core_iterate(LinphoneCore *lc)
{
  check_for_registration(lc);
  check_subscriptions(lc);
  if (lc->preview_finished){
    lc->preview_finished=0;
    ring_stop(lc->ringstream);
    lc->ringstream=NULL;
    lc_callback_obj_invoke(&lc->preview_finished_cb,lc);
  }
}

gchar *sipurl_fixup(LinphoneCore *lc, gchar *url)
{
     /* if it's only digits, we want to send it to the proxy */
     if (strspn(url, "0123456789") == strlen(url)) {
	  if (lc->ua->registrar) {
	       GString *gstring = g_string_new("");
	       g_string_printf(gstring, "%s <sip:%s@%s:%s>", 
			       url, url,
			       lc->ua->registrar->host,
			       (lc->ua->registrar->port ? lc->ua->registrar->port : "5060"));
	       url = g_string_free(gstring, 0);
	  }
     }
     /* if it does not contain with sip:, add that to the url */
     fprintf(stderr, "url=%s strstr=%s\n", url, strstr("sip:", url));
     if (strstr(url, "sip:") == NULL) {
	  GString *gstring = g_string_new("");
	  g_string_printf(gstring, "<sip:%s>", url);
	  url = g_string_free(gstring, 0);
     }
     return url;
}
gboolean sipurl_check(gchar *url)
{
	from_t *surl;
	int err,ret;
	if (url==NULL) return FALSE;
	
	from_init(&surl);
	err=from_parse(surl,url);
	if (err<0) ret=FALSE;
	else ret=TRUE;
	from_free(surl);
	sfree(surl);
	return ret;
}

/* LIK: Transfer related functionality */
int linphone_core_transfer(LinphoneCore *lc, char *url)
{
	gchar *barmsg;
	gint err;
	char *muteaddr = (lc->ua->ua_family == AF_INET) ? "0.0.0.0" : "::0";

	linphone_core_lock(lc);
	if (!sipurl_check(url) ){
		lc->vtable.display_warning(lc,_("Bad formuled sip address. A sip address looks like <sip:username@domainname>"));
		linphone_core_unlock(lc);
		return -1;
	}
	barmsg=g_malloc(strlen(url)+strlen(contacting)+2);
	sprintf(barmsg,"%s %s",contacting,url);
	lc->vtable.display_status(lc,barmsg);

	if (lc->audiostream != NULL && lc->audiostream->timer != NULL) {
	    lc->audiostream->timer->mute = 1;
	}

	/* Put 0.0.0.0 in lc->ua->mute _ip4addr before sending out the
	   invite.  This is where the sdp handler gets the contact
	   information from */
	
	lc->ua->mute_ipaddr = muteaddr;
	osip_trace(8, ("About to transfer to %s", url));

	err=osip_dialog_transfer(lc->call,url,"application/sdp");

	lc->ua->mute_ipaddr = NULL;

	if (err!=0){
	    osip_trace(8, ("URL transfer to %s failed %d", url, err));
	    g_warning("error in invite: %s (%i)\n",strerror(err),err);
	    lc->vtable.display_status(lc,ready);
	}
	linphone_core_unlock(lc);
	return 0;    
}

int linphone_core_invite(LinphoneCore *lc, char *url)
{
     gchar *barmsg;
     gint err;
	
     linphone_core_lock(lc);
     url = sipurl_fixup(lc, url);
     if (!sipurl_check(url) ){
	  lc->vtable.display_warning(lc,_("Bad formuled sip address. A sip address looks like <sip:username@domainname>"));
	  linphone_core_unlock(lc);
	  return -1;
     }
     barmsg=g_malloc(strlen(url)+strlen(contacting)+2);
     sprintf(barmsg,"%s %s",contacting,url);
     lc->vtable.display_status(lc,barmsg);
	
     lc->call=osip_dialog_new(lc->ua);
     err=osip_dialog_invite(lc->call,url,"application/sdp");
     if (err!=0){
	  g_warning("error in invite: %s (%i)\n",strerror(err),err);
	  lc->vtable.display_status(lc,ready);
     }
     linphone_core_unlock(lc);
     return 0;
}

gboolean linphone_core_inc_invite_pending(LinphoneCore*lc){
	if (lc->call==NULL){
		return 0;
	}
	linphone_core_lock(lc);
	if (lc->call->status==DIALOG_INVITED){
		linphone_core_unlock(lc);
		return 1;
	}
	return 0;
}

int linphone_core_accept_dialog(LinphoneCore *lc, const char *url)
{
	LinphoneDialogParams *diaparams;
	StreamParams *audio_params;
	int pos;
	gint jitt_comp;
	
	if (lc->call==NULL){
		return -1;
	}
	linphone_core_lock(lc);
	if (lc->call->status!=DIALOG_INVITED){
		g_warning("No dialog to accept.");
		linphone_core_unlock(lc);
		return -1;
	}
	diaparams=(LinphoneDialogParams *)lc->call->data;
	audio_params=&diaparams->audio_params;
	
	/*stop ringing */
	if (lc->ringstream!=NULL) {
		ring_stop(lc->ringstream);
		lc->ringstream=NULL;
	}
	/* sends a 200 OK */
	osip_dialog_accept_invite(lc->call,lc->current_trn);
	lc->vtable.display_status(lc,connected);
	
	if (try_open_dsp(lc)){
		/* adjust rtp jitter compensation. It must be at least the latency of the sound card */
		jitt_comp=MAX(lc->sound_conf.latency,lc->rtp_conf.audio_jitt_comp);
		lc->audiostream=audio_stream_start_with_sndcard(diaparams->profile,audio_params->localport,
							audio_params->remoteaddr,audio_params->remoteport,
							audio_params->pt,jitt_comp,lc->sound_conf.sndcard);
	}
	linphone_core_unlock(lc);
	return 0;
}

int linphone_core_terminate_dialog(LinphoneCore *lc, const char *url)
{
	OsipDialog *call=lc->call;
	LinphoneDialogParams *diaparams;
	if (call==NULL){
		return -1;
	}
	
	linphone_core_lock(lc);
	diaparams=(LinphoneDialogParams*)call->data;
	lc->call=NULL;
	
	if (call->status==DIALOG_INVITED ) {
		/*stop ringing*/
		if (lc->ringstream!=NULL) {
			ring_stop(lc->ringstream);
			lc->ringstream=NULL;
			restore_sound_daemons();
		}
		/* decline the invitation */
		osip_dialog_reject_invite(call);
		lc->vtable.display_status(lc,end);
	}
	else if (call->status==DIALOG_ESTABLISHED)
	{
    		 /* sends a bye, the call is terminated */
		if (lc->audiostream!=NULL) {
			audio_stream_stop(lc->audiostream);
			lc->audiostream=NULL;
			restore_sound_daemons();
		}
		osip_dialog_bye(call);
		lc->vtable.display_status(lc,end);
	}
	else if (call->status==DIALOG_INVITING)
	{
		/*sends a cancel or a bye ???*/
		if (lc->ringstream!=NULL) {
			ring_stop(lc->ringstream);
			lc->ringstream=NULL;
			restore_sound_daemons();
		}
		osip_dialog_bye(call);
		lc->vtable.display_status(lc,cancel);
	}
	if (diaparams!=NULL) {
		call->data=NULL;
		linphone_dialog_params_destroy(diaparams);
	}
	linphone_core_unlock(lc);
	return 0;
}

int linphone_core_subscribe(LinphoneCore *lc, char *url)
{
	OsipDialog *call_leg;
	gchar *barmsg;
	gint err;
	
	linphone_core_lock(lc);
	url = sipurl_fixup(lc, url);
	if (!sipurl_check(url) ){
		lc->vtable.display_warning(lc,_("Bad formuled sip address. A sip address looks like <sip:username@domainname>"));
		linphone_core_unlock(lc);
		return -1;
	}
	barmsg=g_malloc(strlen(url)+strlen(subscribing)+2);
	sprintf(barmsg,"%s %s",subscribing,url);
	lc->vtable.display_status(lc,barmsg);
	
	call_leg=osip_dialog_new(lc->ua);
	call_leg->expires = 3600;
	err=osip_dialog_subscribe(call_leg, url);
	if (err!=0){
		g_warning("error in subscribe: %s (%i)\n",strerror(err),err);
		lc->vtable.display_status(lc,ready);
	}
	linphone_core_unlock(lc);
	return 0;
}

int linphone_core_unsubscribe(LinphoneCore *lc, char *url)
{
	OsipDialog *call_leg = NULL;
	gchar *barmsg;
	gint err;
	
	linphone_core_lock(lc);
	lc->vtable.display_warning(lc,_("Not yet implemented."));
	linphone_core_unlock(lc);
	return 0;
}

void linphone_core_set_presence_info(LinphoneCore *lc,gint minutes_away,
				     gchar *contact,
				     int presence_mode)
{
	if (minutes_away>0) osip_ua_set_presence_delay(lc->ua,minutes_away*60);
	if (contact!=NULL) {
		osip_ua_set_presence_contact_url(lc->ua,contact);
	}
	osip_ua_set_presence_mode(lc->ua,presence_mode);
}

/* sound functions */
gint linphone_core_get_play_level(LinphoneCore *lc)
{
	return lc->sound_conf.play_lev;
}
gint linphone_core_get_rec_level(LinphoneCore *lc){
	return lc->sound_conf.rec_lev;
}
void linphone_core_set_play_level(LinphoneCore *lc, gint level){
	lc->sound_conf.play_lev=level;
	g_return_if_fail(lc->sound_conf.sndcard!=NULL);
	snd_card_set_level(lc->sound_conf.sndcard,SND_CARD_LEVEL_OUTPUT,level);
}

void linphone_core_set_rec_level(LinphoneCore *lc, gint level)
{
	lc->sound_conf.rec_lev=level;
	g_return_if_fail(lc->sound_conf.sndcard!=NULL);
	snd_card_set_level(lc->sound_conf.sndcard,SND_CARD_LEVEL_INPUT,level);
}
gint linphone_core_set_sound_device(LinphoneCore *lc, gint devid)
{
	SndCard *sndcard;
	int tmp;
	if (devid<0){
		g_warning("Bad devid value: %i",devid);
		devid=0;
	}
	sndcard=snd_card_manager_get_card(snd_card_manager,devid);
	if (sndcard==NULL){
		g_warning("Sound card with index %i does not exist.");
		devid=0;
	}
	lc->sound_conf.dev_id=devid;
	lc->sound_conf.sndcard=sndcard;
	tmp=test_audio_dev(devid);
	if (tmp>0) lc->sound_conf.latency=tmp;
	return 0;
}

gint linphone_core_set_sound_device_from_name(LinphoneCore *lc,gchar *name)
{
	SndCard *card;
	gint index;
	card=snd_card_manager_get_card_with_string(snd_card_manager,name,&index);
	if (card!=NULL){
		g_message("Setting card with id=%i",index);
		lc->sound_conf.dev_id=index;
		lc->sound_conf.sndcard=card;
		return 0;
	}
	return -1;
}

gint linphone_core_get_sound_device(LinphoneCore *lc)
{
	return lc->sound_conf.dev_id;
}

SndCardManager * linphone_core_get_card_manager(LinphoneCore *lc){
	return snd_card_manager;
}

gchar linphone_core_get_sound_source(LinphoneCore *lc)
{
	return lc->sound_conf.source;
}
void linphone_core_set_sound_source(LinphoneCore *lc, gchar source)
{
	lc->sound_conf.source=source;
	g_return_if_fail(lc->sound_conf.sndcard!=NULL);
	snd_card_set_rec_source(lc->sound_conf.sndcard,source);
}

void linphone_core_set_ring(LinphoneCore *lc,gchar *path){
	if (lc->sound_conf.local_ring!=0){
		g_free(lc->sound_conf.local_ring);
	}
	lc->sound_conf.local_ring=g_strdup(path);
}
gchar *linphone_core_get_ring(LinphoneCore *lc){
	return lc->sound_conf.local_ring;
}

static void notify_end_of_ring(MSFilter *f,gint event, gpointer value,gpointer user_data){
  LinphoneCore *lc=(LinphoneCore*)user_data;
  lc->preview_finished=1;
}

gint linphone_core_preview_ring(LinphoneCore *lc, const gchar *ring,LinphoneCoreCbFunc func,gpointer userdata)
{
  if (lc->ringstream!=0){
    g_warning("Cannot start ring now,there's already a ring being played");
    return -1;
  }
  lc_callback_obj_init(&lc->preview_finished_cb,func,userdata);
  lc->preview_finished=0;
  if (try_open_dsp(lc)<0) return;
  lc->ringstream=ring_start_with_cb((char*)ring,2,lc->sound_conf.sndcard,(MSFilterNotifyFunc)notify_end_of_ring,(gpointer)lc);
  return 0;
}


void linphone_core_set_ringback(LinphoneCore *lc,RingBackType type){
	switch(type){
		case RINGBACK_TYPE_FR:
			lc->sound_conf.remote_ring=PACKAGE_SOUND_DIR "/" REMOTE_RING_FR;
		break;
		case RINGBACK_TYPE_US:
			lc->sound_conf.remote_ring=PACKAGE_SOUND_DIR "/" REMOTE_RING_US;
		break;
	}
}
RingBackType linphone_core_get_ringback(LinphoneCore *lc);

void linphone_core_send_dtmf(LinphoneCore *lc,gchar dtmf)
{
	if (lc->audiostream!=NULL){
		send_dtmf(lc->audiostream,dtmf);
	}
}
void linphone_core_force_ip_address(LinphoneCore *lc, gchar *ipaddr)
{
	linphone_core_set_nat_address(lc,ipaddr,1);
}


void linphone_core_set_nat_address(LinphoneCore *lc, gchar *addr, gboolean use)
{
	gchar *tmp=NULL;
	gint err;
	struct addrinfo hints,*res;
	
	if (addr!=NULL && use){
		memset(&hints,0,sizeof(struct addrinfo));
		hints.ai_family=PF_UNSPEC;
		hints.ai_socktype = SOCK_DGRAM;
		err=getaddrinfo(addr,NULL,&hints,&res);
		if (err==0){
			tmp=g_strdup(addr);
			freeaddrinfo(res);
		}else {
			g_warning("Invalid nat address %s",addr);
			return;
		}
	}
	if (lc->net_conf.nat_address!=NULL){
		g_free(lc->net_conf.nat_address);
	}
	lc->net_conf.nat_address=tmp;
	if (use && (tmp!=NULL)){
		gchar *ct;
		/* change OsipUA object main contact */
		if (lc->sip_conf.sip_port==5060){
			ct=g_strdup_printf("sip:%s@%s",lc->sip_conf.username,tmp);
		}else{
			ct=g_strdup_printf("sip:%s@%s:%i",lc->sip_conf.username,tmp,
							lc->sip_conf.sip_port);
		}
		osip_ua_set_contact(lc->ua,ct);
		g_free(ct);
		lc->net_conf.use_nat=TRUE;
	}
	else {
		lc->net_conf.use_nat=FALSE;
		linphone_core_update_contact_info(lc);
	}
}

gchar *linphone_core_get_nat_address(LinphoneCore *lc, gboolean *use)
{
	if (use!=NULL) *use=lc->net_conf.use_nat;
	if (lc->net_conf.nat_address!=NULL) return g_strdup(lc->net_conf.nat_address);
	else return NULL;
}

void net_config_uninit(net_config_t *config)
{
#ifdef LINPHONE_DEPRECATED
	gnome_config_set_string("net/if_name",config->sel_if->name);
	g_list_foreach(config->interfaces,(GFunc)g_free,NULL);
	g_list_free(config->interfaces);
	config->interfaces=NULL;
#endif
	gnome_config_set_int("net/con_type",config->con_type);
	
	gnome_config_set_int("net/use_nat",config->use_nat);
	gnome_config_set_string("net/nat_address",config->nat_address);
	
}

void registrar_config_uninit(registrar_config_t *config)
{
	gnome_config_set_int("sip/use_registrar",config->use_registrar);
	gnome_config_set_int("sip/as_proxy",config->as_proxy);
	gnome_config_set_string("sip/registrar",config->registrar);
	gnome_config_set_string("sip/passwd",config->passwd);
	gnome_config_set_string("sip/addr_of_rec",config->addr_of_rec);
	gnome_config_set_int("sip/expires",config->expires);
	g_timer_destroy(config->timer);
}
void sip_config_uninit(sip_config_t *config)
{
	gnome_config_set_string("sip/username",config->username);
	gnome_config_set_string("sip/hostname", config->hostname);
	gnome_config_set_int("sip/sip_port",config->sip_port);
	registrar_config_uninit(&config->reg_conf);
}

void rtp_config_uninit(rtp_config_t *config)
{
	gnome_config_set_int("rtp/audio_rtp_port",config->audio_rtp_port);
	gnome_config_set_int("rtp/video_rtp_port",config->video_rtp_port);
	gnome_config_set_int("rtp/audio_jitt_comp",config->audio_jitt_comp);
	gnome_config_set_int("rtp/video_jitt_comp",config->audio_jitt_comp);
}

void sound_config_uninit(sound_config_t *config)
{
	char tmpbuf[2];
	gnome_config_set_int("sound/dev_id",config->dev_id);
	gnome_config_set_int("sound/rec_lev",config->rec_lev);
	gnome_config_set_int("sound/play_lev",config->play_lev);
	tmpbuf[0]=config->source;
	tmpbuf[1]='\0';
	gnome_config_set_string("sound/source",tmpbuf);
	gnome_config_set_string("sound/local_ring",config->local_ring);
	gnome_config_set_string("sound/remote_ring",config->remote_ring);
}

void video_config_uninit(video_config_t *config)
{
	gnome_config_set_int("video/enabled",config->enabled);
	gnome_config_set_int("video/show_local",config->show_local);
}

void codecs_config_uninit(codecs_config_t *config)
{
	PayloadType *pt;
	GList *node;
	gchar key[50];
	gint index;
	index=0;
	for(node=config->audio_codecs;node!=NULL;node=g_list_next(node)){
		pt=(PayloadType*)(node->data);
		sprintf(key,"audio_codec_%i/mime",index);
		gnome_config_set_string(key,pt->mime_type);
		sprintf(key,"audio_codec_%i/rate",index);
		gnome_config_set_int(key,pt->clock_rate);
		sprintf(key,"audio_codec_%i/enabled",index);
		gnome_config_set_int(key,payload_type_enabled(pt));
		index++;
	}
	index=0;
	for(node=config->video_codecs;node!=NULL;node=g_list_next(node)){
		pt=(PayloadType*)(node->data);
		sprintf(key,"video_codec_%i/mime",index);
		gnome_config_set_string(key,pt->mime_type);
		sprintf(key,"video_codec_%i/rate",index);
		gnome_config_set_int(key,pt->clock_rate);
		sprintf(key,"video_codec_%i/enabled",index);
		gnome_config_set_int(key,payload_type_enabled(pt));
		index++;
	}
}

void ui_config_uninit(ui_config_t *config)
{
	int len;
	GList *elem;
	gchar ab_entry[100];

	/* record address book entries*/
	len=0;
	elem=config->address_list;
	fprintf(stderr, "ui_config_uninit config->address_list=%p\n", config->address_list);
	while(elem!=NULL)
	{
		LinphoneCoreAddressEntry *entry = (LinphoneCoreAddressEntry *)elem->data;
		snprintf(ab_entry,100,"address_book/entry%i",len);
		gnome_config_set_string(ab_entry,entry->address);
		fprintf(stderr, "ui_config_uninit: entry %d address=%s subscribe_notify=%s\n",
			len, entry->address, entry->subscribe_notify);

		if (entry->subscribe_notify) {
			snprintf(ab_entry,100,"address_book/entrysubscribe%i",len);
			gnome_config_set_string(ab_entry,entry->subscribe_notify);
		}

		elem=g_list_next(elem);
		len++;
	}
	gnome_config_set_int("address_book/entry_count",len);

	len=0;
	elem=config->missed_calls_list;
	while(elem!=NULL)
	{
		char *pc_entry = (char *)elem->data;
		gchar pc_name[100];
		snprintf(pc_name,100,"received_calls/entry%i",len);
		gnome_config_set_string(pc_name, pc_entry);

		elem=g_list_next(elem);
		len++;
	}
	gnome_config_set_int("received_calls/entry_count",len);

	len=0;
	elem=config->missed_calls_list;
	while(elem!=NULL)
	{
		char *pc_entry = (char *)elem->data;
		gchar pc_name[100];
		snprintf(pc_name,100,"missed_calls/entry%i",len);
		gnome_config_set_string(pc_name, pc_entry);

		elem=g_list_next(elem);
		len++;
	}
	gnome_config_set_int("missed_calls/entry_count",len);

	len=0;
	elem=config->placed_calls_list;
	while(elem!=NULL)
	{
		char *pc_entry = (char *)elem->data;
		gchar pc_name[100];
		snprintf(pc_name,100,"placed_calls/entry%i",len);
		gnome_config_set_string(pc_name, pc_entry);

		elem=g_list_next(elem);
		len++;
	}
	gnome_config_set_int("placed_calls/entry_count",len);

	len=0;
	elem=config->buddy_list;
	while(elem!=NULL)
	{
		char *pc_entry = (char *)elem->data;
		gchar pc_name[100];
		snprintf(pc_name,100,"buddy_list/entry%i",len);
		gnome_config_set_string(pc_name, pc_entry);

		elem=g_list_next(elem);
		len++;
	}
	gnome_config_set_int("buddy_list/entry_count",len);
}

void linphone_core_uninit(LinphoneCore *lc)
{
	/* unregister if necessary*/
	fprintf(stderr, "linphone_core_uninit\n");
	if (lc->sip_conf.reg_conf.use_registrar){
		do_registration(lc,0);
		sleep(1);
	}
	/* save all config */
	net_config_uninit(&lc->net_conf);
	sip_config_uninit(&lc->sip_conf);
	rtp_config_uninit(&lc->rtp_conf);
	sound_config_uninit(&lc->sound_conf);
	video_config_uninit(&lc->video_conf);
	codecs_config_uninit(&lc->codecs_conf);
	ui_config_uninit(&lc->ui_conf);
	gnome_config_sync();
	
	ortp_exit();
	osipua_exit();
}

void linphone_core_destroy(LinphoneCore *lc){
	linphone_core_uninit(lc);
	g_free(lc);
}
