/*
linphonec
Copyright (C) 2002  Florian Winterstein (flox@gmx.net)
Copyright (C) 2000  Simon MORLAT (simon.morlat@free.fr)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <linphonecore.h>



LinphoneCore linphonec;
FILE *mylogfile;
FILE *configfile;

void
dummy_log_handler (const gchar * log_domain, GLogLevelFlags log_level,
		   const gchar * message, gpointer user_data)
{
	return;
}

void
sipomatic_log_handler (const gchar * log_domain, GLogLevelFlags log_level,
		       const gchar * message, gpointer user_data)
{
	fprintf (mylogfile, "%s:(GLogLevel=%i)** %s\n", log_domain, log_level,
		 message);
}

static gint64
getelapsed (struct timeval *before, struct timeval *after)
{
	return ((after->tv_sec - before->tv_sec) * 1000) +
		((after->tv_usec - before->tv_usec) / 1000);
}

/*
gpointer tmalloc(int size)
{
	gpointer p;
	struct timeval before,after;
	gettimeofday(&before,NULL);
	p=malloc(size);
	gettimeofday(&after,NULL);
	if (getelapsed(&before,&after)>500){
		g_error("Spending too much time in malloc() of size %i",size);
	}
	return p;
}

GMemVTable dbgtable={
	tmalloc,
	realloc,
	free,
	calloc,
	tmalloc,
	realloc
};
*/
void
stub ()
{
}

void
linphonec_display_something (LinphoneCore * lc, char *something)
{
	fprintf (stdout, "%s\n", something);
}

void
linphonec_display_warning (LinphoneCore * lc, char *something)
{
	fprintf (stdout, "Warning: %s\n", something);
}

void
linphonec_display_url (LinphoneCore * lc, char *something, char *url)
{
	fprintf (stdout, "Warning: %s - %s\n", something, url);
}

LinphoneCoreVTable linphonec_vtable = {
	show:(ShowInterfaceCb) stub,
	inv_recv:(InviteReceivedCb) stub,
	bye_recv:(ByeReceivedCb) stub,
	display_status:linphonec_display_something,
	display_message:linphonec_display_something,
	display_warning:linphonec_display_warning,
	display_url:linphonec_display_url,
	display_question:stub
};

void
print_usage ()
{
	fprintf (stdout, "\n\
usage: linphonec [-c file] [-d level -l logfile]\n\
       linphonec -v\n\
\n\
  -c  file             specify path of configuration file.\n\
  -d  level            be verbose. 0 is no output. 6 is all output\n\
  -l  logfile          specify the log file for your SIP phone\n\
  -v or --version      display version and exits.\n");
}

int
linphonec_main_loop (LinphoneCore * opm)
{
	char input[256], ch;
	char temp[256];
	int i;
	int err;

	ch = 0;

	while (ch != 'q')
	{
		fprintf (stdout, "\nCommand ? ");
		fflush (stdout);
		fgets (input, 255, stdin);
		ch = (char) tolower (input[0]);

		switch (ch)
		{
		case 'r':
			linphone_core_register (opm, NULL);
			break;

		case 'c':
			if (strlen (input) > 2)
			{
				sprintf (temp, "%s", input + 2);	//remove 'c ' from the front

				if (strncmp (temp, "sip:", 4) != 0)
				{
					sprintf (input, "sip:%s", temp);
				}
				else
				{
					sprintf (input, "%s", temp);
				}

				//printf("address with sip prefix:'%s'\n",input);

				if ((strstr (input, "@") == NULL) &&
				    (opm->sip_conf.reg_conf.registrar !=
				     NULL))
				{
					input[strlen (input) - 1] = '\0';	// remove newline
					sprintf (temp, "%s@%s\n", input, opm->sip_conf.reg_conf.registrar + 4);	//add registrar
					sprintf (input, "%s", temp);
				}

				//printf("Final address:'%s'\n",input);
				linphone_core_invite (opm, input);
			}
			break;

		case 't':
			err = linphone_core_terminate_dialog (opm, NULL);
			if (err < 0)
				printf ("There is no active call.");
			break;

		case 'a':
			linphone_core_accept_dialog (opm, NULL);
			if (err < 0)
				printf ("There is no incoming call to accept.");
			break;

		case 'q':
			printf ("bye..\n");
			break;

		case '0':
		case '1':
		case '2':
		case '3':
		case '4':
		case '5':
		case '6':
		case '7':
		case '8':
		case '9':
		case '#':
		case '*':
			if (opm->audiostream != NULL)
				send_dtmf (opm->audiostream, ch);
			break;

		default:
			fprintf (stdout,
				 "\nHelp:\nr\tregister\nc <url>\tmake call\nt\tterminate call\na\tanswer call\nq\tquit\n1-9#*\tsend dtmf\n");
			break;
		}

	}

	return 0;
}

int
main (int argc, char *argv[])
{
	int err;
	char *logfile_name = NULL;
	char *configfile_name = NULL;
	char *p;
	int trace_level = 0;
	int arg_num;

	//g_mem_set_vtable(&dbgtable);

	configfile = NULL;
	mylogfile = NULL;
	arg_num = 1;

#ifdef ENABLE_NLS
	p = bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
	if (p == NULL)
		perror ("bindtextdomain failed");
#ifndef __ARM__
	bind_textdomain_codeset (PACKAGE, "UTF-8");
#endif
	textdomain (GETTEXT_PACKAGE);
#else
	printf ("NLS disabled.\n");
#endif

	while (arg_num < argc)
	{
		int old_arg_num = arg_num;
		if (strncmp ("-d", argv[arg_num], 2) == 0)
		{
			arg_num++;
			if (arg_num < argc)
				trace_level = atoi (argv[arg_num]);
			else
				trace_level = 1;
		}
		else if (strncmp ("-l", argv[arg_num], 2) == 0)
		{
			arg_num++;
			if (arg_num < argc)
				logfile_name = argv[arg_num];
		}
		else if (strncmp ("-c", argv[arg_num], 2) == 0)
		{
			arg_num++;
			if (arg_num < argc)
				configfile_name = argv[arg_num];
		}
		else if ((strncmp ("-v", argv[arg_num], 2) == 0)
			 ||
			 (strncmp
			  ("--version", argv[arg_num],
			   strlen ("--version")) == 0))
		{
			printf ("version: " VERSION "\n");
			exit (0);
		}
		else if (old_arg_num == arg_num)
		{
			fprintf (stderr, "ERROR: bad arguments\n");
			print_usage ();
			exit (0);
		}
		arg_num++;
	}

	if (configfile_name == NULL)
	{
		configfile_name =
			g_strdup_printf ("%s/.linphonec", getenv ("HOME"));
		if (!g_file_test (configfile_name, G_FILE_TEST_EXISTS))
		{
			fprintf (stderr,
				 "WARNING: Could not load config file ~/.linphonec, trying to use ~/.gnome2/linphone\n\n");
			g_free (configfile_name);
			configfile_name =
				g_strdup_printf ("%s/.gnome2/linphone",
						 getenv ("HOME"));
			if (!g_file_test
			    (configfile_name, G_FILE_TEST_EXISTS))
			{
				fprintf (stderr,
					 "ERROR: Could not load ~/.linphonec or ~/.gnome2/linphone . Please edit example configuration file\n"
					 " %s/linphonec/linphonec and copy it to ~/.linphonec\n"
					 "You hust have to configure local address, username, and hostname.\n"
					 "Alternatively, you can run the GTK+ version of linphone to configure it once time\n"
					 "and then retry linphonec.\n",
					 PACKAGE_DATA_DIR);
				exit (1);
			}
		}
	}
	else
	{
		if (!g_file_test (configfile_name, G_FILE_TEST_EXISTS))
		{
			fprintf (stderr, "Cannot open config file %s.\n",
				 configfile_name);
			exit (1);
		}
	}

	if (trace_level > 0)
	{
		if (logfile_name != NULL)
			mylogfile = fopen (logfile_name, "w+");

		if (mylogfile == NULL)
		{
			mylogfile = stdout;
			fprintf (stderr,
				 "INFO: no logfile, logging to stdout\n");
		}
		else
		{
			g_log_set_handler ("MediaStreamer", G_LOG_LEVEL_MASK,
					   sipomatic_log_handler, NULL);
			g_log_set_handler ("oRTP", G_LOG_LEVEL_MASK,
					   sipomatic_log_handler, NULL);
			g_log_set_handler ("oRTP-stats", G_LOG_LEVEL_MASK,
					   sipomatic_log_handler, NULL);
		}
		TRACE_INITIALIZE (trace_level, mylogfile);
	}
	else
	{
		g_log_set_handler ("MediaStreamer",
				   G_LOG_LEVEL_MESSAGE | G_LOG_LEVEL_WARNING,
				   dummy_log_handler, NULL);
		g_log_set_handler ("oRTP",
				   G_LOG_LEVEL_MESSAGE | G_LOG_LEVEL_WARNING,
				   dummy_log_handler, NULL);
		g_log_set_handler ("oRTP-stats", G_LOG_LEVEL_MESSAGE,
				   dummy_log_handler, NULL);
	}
	/* init */
	linphone_core_init (&linphonec, &linphonec_vtable, configfile_name,
			    NULL);

	linphonec_main_loop (&linphonec);

	linphone_core_uninit (&linphonec);

	if (mylogfile != NULL && mylogfile != stdout)
		fclose (mylogfile);

	exit (0);
	return 0;
}
