/*
  The osipua library is a library based on oSIP that implements CallLeg and User Agent
  level.
  Copyright (C) 2001  Simon MORLAT simon.morlat@free.fr
  											Aymeric MOIZARD jack@atosc.org
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include "osipua.h"
#include "utils.h"
#include <osip/port.h>
#include <osip/sdp.h>
 
#ifdef INET6  
#include <sys/types.h>  
#include <sys/socket.h>  
#include <netdb.h>
#endif

list_t ua_list={0,NULL};

osip_t *config=NULL;

OsipManager *def_manager=NULL;


extern FILE *logfile;

/* this is the function that create a default static manager that will be used to create user agents*/

/**
 * osipua_init:
 *
 * Init the osipua library. This creates a default stack manager and
 * starts a receiving thread for udp.
 *
 *
 */
void osipua_init()
{
	int i=0;
#ifdef PARANO_FREE
	alloc_init();
#endif
	if (def_manager!=NULL)  return;
	osip_trace(OSIP_INFO1,("Starting osip stack and osipua layer.\n"));
	def_manager=osip_manager_new();
	/* try to create a socket for sending */
	while ( (osip_manager_set_send_port(def_manager,5060+i)!=0) && (i<20))
	{
		osip_trace(OSIP_INFO1,("info: Trying to bind on port (%i).\n",i+5060+1));
		i++;
	}
	
	osip_manager_start_udp_daemon(def_manager);
	//osip_manager_start_resolver(def_manager);
}


/**
 * osipua_exit:
 *
 * Shutdowns the osipua stack.
 *
 *
 */
void osipua_exit()
{
	if (def_manager==NULL)  return;
	osip_trace(OSIP_INFO1,("Shutting down osip stack and osipua layer.\n"));
	/* this will stop the threads */
	osip_manager_destroy(def_manager);
	osip_global_free();
}

void
osip_ua_set_presence_delay(OsipUA *ua, int delay)
{
  ua->presence_delay=delay;
}

void
osip_ua_set_presence_contact_url(OsipUA *ua, char *url)
{
  if ( ua->presence_contact_url!=NULL) sfree(ua->presence_contact_url);
  ua->presence_contact_url = sgetcopy(url);
}

/* the return code to use:
   actual special codes are fully supported:
   200
   486  * use presence_delay for the Retry-After header
   480  * use presence_url for contact header
   301  * use presence_url for contact header
   302  * use presence_url for contact header
   380  * use presence_url for contact header (only sipurl by now)   
*/
void
osip_ua_set_presence_mode(OsipUA *ua, int mode)
{
  ua->presence_mode = mode;
}

void
osip_ua_set_presence_location(OsipUA *ua, char *location)
{
  ua->presence_location = location;
}


void osip_ua_init(OsipUA *ua)
{
	memset(ua,0,sizeof(OsipUA));
	ua->alias=smalloc(sizeof(list_t));
	ua->contact=NULL;
	ua->max_dialogs=1000;
	list_init(ua->alias);
	list_init(&ua->body_handlers);		
	
	ua->config=def_manager->config;
	ua->manager=def_manager;
	
  	/* init the list of call-legs managed by the ua*/
  	list_init(&ua->call_list);
  	list_init(&ua->body_handlers);
	
	/* add this new user agent entity to the list of the ones managed by the stack*/
	list_add(&ua_list,(void*)ua,-1);
	ua->presence_mode = 200;
	ua->presence_delay=0;
	ua->presence_contact_url=NULL;
}

/**
 * osip_ua_new:
 * @params:  The parameters of the ua.
 *
 * This function creates a user agent according to the parameters
 * in @params.
 *
 * Returns: a OsipUA object.
 */
OsipUA * osip_ua_new()
{
	OsipUA *ua;
	
	if (def_manager==NULL)
	{
		osip_trace(OSIP_ERROR,("You must call osipua_init() before creating a user agent !\n"));
		return NULL;
	}
	/* allocate OsipUA object*/
	ua=(OsipUA*) smalloc(sizeof(OsipUA));
	osip_ua_init(ua);
	
	return(ua);
}

/*set (or change) ua ip address*/
/**
 * osip_ua_set_ip4addr:
 * @ua: a user agent.
 * @ip4addr: an ip address in the dot form (xxx.xxx.xxx.xxx).
 *
 * Set or change the ip address of a user agent.
 *
 *
 */
#ifdef INET6
void osip_ua_set_ipaddr(OsipUA *ua,char *ipaddr, int port, int ua_family)
{
	strncpy(ua->ua_ipaddr,ipaddr,IP_SIZE);
	osip_manager_add_udpport(ua->manager,port);
	ua->ua_family = ua_family;
	ua->ua_port=port;
}
#else
void osip_ua_set_ip4addr(OsipUA *ua,char *ip4addr, int port)
{
	strncpy(ua->ua_ip4addr,ip4addr,IP4_SIZE);
	
  	osip_manager_add_udpport(ua->manager,port);
  	ua->ua_port=port;
}
#endif

void osip_ua_set_max_dialogs(OsipUA *ua, int max)
{
	ua->max_dialogs=max;
}

/* returns 1 if call is handled by ua*/
int osip_ua_call_leg_exists(OsipUA *ua,OsipDialog *call)
{
	int pos;
	OsipDialog *callleg;
	
	for (pos=0;pos<ua->call_list.nb_elt;pos++)
	{
		callleg=(OsipDialog*)list_get(&ua->call_list,pos);
		if (callleg==call) return 1;
  	}
	return 0;	
}


void osip_ua_add_body_handler(OsipUA *ua, BodyHandler *info)
{
	list_add(&ua->body_handlers,(void*)info,0);
	body_handler_init_with_ua(info,ua);
}


int osip_ua_set_registrar( OsipUA *ua, char *registrar, char *passwd)
{
	url_t *rgt;
	int err;
	
	if (registrar==NULL)
	{
		/* means we unset the registrar*/
		if (ua->registrar!=NULL)
		{
			url_free( ua->registrar);
			sfree( ua->registrar);
			ua->registrar=NULL;
		}
		return 0;
	}
	url_init(&rgt);
	err=url_parse(rgt,registrar);
	if (err!=0)
	{
		osip_trace(OSIP_ERROR,("Could not parse registrar address.\n"));
		sfree(rgt);
		return -EINVAL;
	}
	else
	{
		if (ua->registrar!=NULL)
		{
			url_free(ua->registrar);
			sfree(ua->registrar);
		}
		ua->registrar=rgt;
	}
	if (ua->reg_passwd!=NULL)
	{
		sfree(ua->reg_passwd);	
		ua->reg_passwd=NULL;
	}
	if (passwd!=NULL)
	{
		if (strlen(passwd)!=0) ua->reg_passwd=sgetcopy(passwd);
	}
	return 0;
}

/**
 * osip_ua_set_outbound_proxy:
 * @ua: a user agent.
 * @proxy: the sip url of a proxy.
 * @passwd: the passwd to use to access the proxy, NULL if none.
 *
 * Set a default proxy to use with a user agent. All requests will
 * be sent to the proxy.
 *
 */
int osip_ua_set_outbound_proxy(OsipUA *ua, char *proxy, char *passwd)
{	
	osip_ua_set_registrar(ua,proxy,passwd);
	if (proxy!=NULL)
	{
		ua->flags|=OSIP_UA_USE_PROXY;
	}
	else
	{
		ua->flags&=~OSIP_UA_USE_PROXY;
	}
	return 0;	
}
	
/* change identity of an existent user agent*/

/**
 * osip_ua_set_contact:
 * @ua: a user agent.
 * @contact: a sip uri.
 *
 *
 * Set the official identity of the ua.
 *
 * Returns: 0
 */
int osip_ua_set_contact(OsipUA *ua, char *contact)
{
	contact_t *ctt;
	int err;
#ifdef INET6
	struct addrinfo hints, *res = NULL;
#endif
	
	err=contact_init(&ctt);
	err=contact_parse(ctt,contact);
	if (err!=0)
	{
		osip_trace(OSIP_ERROR,("Bad Contact address (%s).\n",contact));
		contact_free(ctt);
		sfree(ctt);
		return -1;
	}
	
	if (ua->contact!=NULL){
		contact_free(ua->contact);
		sfree(ua->contact);
	}
	ua->contact=ctt;
#ifdef INET6
	memset(&hints, 0, sizeof(hints));
	hints.ai_family = PF_UNSPEC;
	hints.ai_socktype = SOCK_DGRAM;
	hints.ai_flags = AI_NUMERICHOST;
	err = getaddrinfo(ctt->url->host, NULL, &hints, &res);
	if (err == 0) {
#else
	if (inet_addr(ctt->url->host)!=-1){
#endif
		int port;
		if (ctt->url->port!=NULL) {
			port=atoi(ctt->url->port);
			if (port<=0) {
				osip_trace(OSIP_ERROR,("Invalid port number %s.",ctt->url->port));
			}
		}
		else port=5060;
#ifdef INET6
		osip_ua_set_ipaddr(ua,ctt->url->host,port, res->ai_family);
		freeaddrinfo(res);
#else
		osip_ua_set_ip4addr(ua,ctt->url->host,port);
#endif
	}
	else osip_trace(OSIP_ERROR,("Primary contact is not ip address: %s !\n",contact));
	return 0;
}

/**
 * osip_ua_destroy:
 * @ua: a user agent.
 *
 *
 * Destroy a user agent.
 *
 * Returns: 0
 */	
int osip_ua_destroy( OsipUA *ua)
{
	int err;
	if (ua==NULL) return(-EFAULT);
	
	if (ua->contact!=NULL)
   {
      contact_free(ua->contact);
      sfree(ua->contact);
   }
   /* remove alias */
	osip_ua_clean_alias(ua);
	/* remove all call-legs managed by the ua from the list of the stack*/
	           /*to do*/
	/* remove the ua from the list managed by the stack*/
	err=list_remove_el(&ua_list,(void*)ua);
	if (err!=0)  osip_trace(OSIP_ERROR,("Could not remove ua from list.\n"));
	sfree(ua);
	return(0);
}

/**
 * osip_ua_signal_connect:
 * @ua: a user agent.
 * @signal: the type of signal.
 * @func: a signal handler of type #OsipUASignalHandler.
 *
 * Setup a signal handler for one of the @OsipUA signals:
 * "INVITE" to be informed of new invites,
 * "INVITE_ACCEPTED" to be informed of the acceptation of an outgoing invite,
 * "FAILURE" to be informed of the failure of an outgoing request,
 * "INFORMATIVE" to be informed a the progress of a call-leg.
 *
 *
 * Returns: 0 if the signal handler was set, a negative value if not.
 */	
int osip_ua_signal_connect(OsipUA *ua,char *signal, OsipUACallbackFunc func)
{
	if (strcmp(signal,"INVITE")==0)
		ua->invite=func;
	else if (strcmp(signal,"INVITE_ACCEPTED")==0)
		ua->invite_accepted=func;
	else if (strcmp(signal,"BYE")==0)
		ua->bye=func;
	else if (strcmp(signal,"FAILURE")==0)
		ua->failure=func;
	else if (strcmp(signal,"INFORMATIVE")==0)
		ua->informative=func;
	else if (strcmp(signal,"SUBSCRIBE")==0)
		ua->subscribe=func;
	else if (strcmp(signal,"SUBSCRIBE_ACCEPTED")==0)
		ua->subscribe_accepted=func;
	else if (strcmp(signal,"NOTIFY")==0)
		ua->notify=func;
	else if (strcmp(signal,"PUBLISH")==0)
		ua->publish=func;
	/* LIK: Transfer call related functionality */
	else if (strcmp(signal,"MUTE")==0)
		ua->mute_function=func;
	else if (strcmp(signal,"BYETRANSFER")==0)
	    ua->byetransfer_function=func;
	else if (strcmp(signal,"REFERTRANSFER")==0)
	    ua->refertransfer_function=func;
	else if (strcmp(signal,"NOTIFYTRANSFER")==0)
	    ua->notifytransfer_function=func;
	else return(-EINVAL);
	return(0);
}

/**
 * osip_ua_add_alias:
 * @ua: a user agent.
 * @contact: a sip uri.
 *
 * Add a new sip URI the user agent is supposed to respond.
 * If an incoming request does not match the official contact or one
 * of the uri in the alias list, then the ua answers 404 Not Found.
 *
 *
 * Returns: 0 if added, -1 if the uri is bad-formuled.
 */	
int osip_ua_add_alias(OsipUA *ua, char *contact)
{
	contact_t *ctt;
	contact_t *ctttmp;
	int err,i;
	
	contact_init(&ctt);
	err=contact_parse(ctt,contact);
	if (err!=0)
	{
		osip_trace(OSIP_ERROR,("Invalid alias sip address: %s\n",contact));
		contact_free(ctt);
		sfree(ctt);
		return -1;
	}
	for (i=0;i<ua->alias->nb_elt;i++)
	{
		ctttmp=(contact_t*)list_get(ua->alias,i);
		if (  from_compare((from_t*) ctttmp, (from_t*) ctt) ==0 )
		{
			/* the alias is already in list so remove it*/
			list_remove(ua->alias,i);
			contact_free(ctttmp);
			sfree(ctttmp);
			break;
		}
	}		
	/* add it in first position */
	list_add(ua->alias,ctt,0);
	return 0;
}

/**
 * osip_ua_remove_alias:
 * @ua: a user agent.
 * @contact: a sip uri.
 *
 * Remove a contact uri from the ua list of alias.
 *
 *
 * Returns: 0 if added, -1 if the uri is bad-formuled or was not in the list.
 */	
int osip_ua_remove_alias(OsipUA *ua, char *contact)
{
	contact_t *ctt;
	contact_t *ctttmp;
	int err,i,found=-1;
	
	contact_init(&ctt);
	err=contact_parse(ctt,contact);
	if (err!=0)
	{
		osip_trace(OSIP_ERROR,("Invalid alias sip address.\n"));
		contact_free(ctt);
		sfree(ctt);
		return -1;
	}
	for (i=0;i<ua->alias->nb_elt;i++)
	{
		ctttmp=(contact_t*)list_get(ua->alias,i);
		if (from_compare((from_t*) ctttmp, (from_t*) ctt) ==0 )
		{
			list_remove(ua->alias,i);
			contact_free(ctttmp);
			sfree(ctttmp);
			found=0;
		}
	}		
	return(found);
}

/**
 *osip_ua_clean_alias:
 *
 * @ua: a user agent
 *
 * Removes and free all alias contact from the user agent alias list.
 */
void osip_ua_clean_alias(OsipUA *ua)
{
	int i;
	contact_t *ctttmp;
	
	for (i=0;i<ua->alias->nb_elt;i++)
	{
		ctttmp=(contact_t*)list_get(ua->alias,i);
		
		list_remove(ua->alias,i);
		contact_free(ctttmp);
		sfree(ctttmp);
	}
}		

/* find the user agent to which the message is destinated*/
/* this function will need to be rewritten: this is shit code*/
OsipUA * osip_ua_find(sip_t *sipmsg)
{
	int pos=0,i;
	OsipUA *ua,*ret_ua=NULL;
	char *ipaddr = NULL;
#ifdef INET6
	struct sockaddr_storage ss;
	struct addrinfo hints, *res = NULL;
	int error;
	char namebuf[BUFSIZ];
#else
	struct in_addr addr;
#endif
	url_t *url=msg_geturi(sipmsg);
#ifndef INET6
	struct hostent *host;
#endif
	char *tmp,*user;
	contact_t *ctt;

	/* reject call with empty username, I think this is not correct for user agents */
	if (url->username==NULL) 
        {
		osip_trace(OSIP_INFO1,("osip_ua_find: user name is NULL\n"));
        	return NULL;
        }
	/* fix (Vijay): should compare rquri instead of To with the alias.*/
	/* resolv to if necessary*/
#ifdef INET6
	memset(&hints, 0, sizeof(hints));
	hints.ai_family = PF_UNSPEC;
	hints.ai_socktype = SOCK_DGRAM;
	hints.ai_flags = AI_NUMERICHOST;

	error = getaddrinfo(url->host, NULL, &hints, &res);
	if (error != 0){
		/* not an ip address */
		osip_trace(OSIP_INFO1,("getaddrinfo: %s : %s\n",url->host,gai_strerror(error)));
	}
	else ipaddr=sgetcopy(url->host);
#else
	addr.s_addr=inet_addr(url->host);
	if ( addr.s_addr==-1)
	{
		host=resolv(url->host);
		if (host!=NULL)
		{
			tmp=inet_ntoa( ((struct in_addr*)host->h_addr)[0]);
			ipaddr=sgetcopy(tmp);
		}
	}
	else ipaddr=sgetcopy(url->host);
#endif
  	/*find an existing ua that may be interested by this new message */
 	for (pos=0;(pos<ua_list.nb_elt) && (ret_ua==NULL);pos++)
  	{
  		ua=(OsipUA*)list_get(&ua_list,pos);
  		user=ua->contact->url->username;
		
  		/* first try to compare only with real identitier of the ua */
		if (strcasecmp(user,url->username)==0)
		{
#ifdef INET6
			if (ipaddr != NULL)
			{
				osip_trace(OSIP_INFO1,("osip_ua_find 1: %s <> %s \n",ua->ua_ipaddr,ipaddr));
				if ((strcasecmp(ua->ua_ipaddr,ipaddr)==0) || (strcasecmp("127.0.0.1",ipaddr)==0) ||
				   	IN6_IS_ADDR_LOOPBACK(res->ai_addr))
				{
				   	 ret_ua=ua;       /* ip addresses match */
				}
			}
			else ret_ua=ua;	/* we accept uncheckable host names... */
			ret_ua=ua;
#else
			if (ipaddr!=NULL)
			{
				osip_trace(OSIP_INFO1,("osip_ua_find 1: %s <> %s \n",ua->ua_ip4addr,ipaddr));
				if ((strcasecmp(ua->ua_ip4addr,ipaddr)==0) || (strcasecmp("127.0.0.1",ipaddr)==0))
				{
					ret_ua=ua;     /* ip addresses match */
				}
			}
			else ret_ua=ua;	/* we accept uncheckable host names... */
#endif
		}
		/* then compare with all contact alias of the ua */
		for (i=0; (i<ua->alias->nb_elt) && (ret_ua==NULL);i++)
		{
			ctt=(contact_t*)list_get(ua->alias,i);
			osip_trace(OSIP_INFO1,("osip_ua_find:  %s <>  %s \n",ctt->url->username,url->username));
			if (strcasecmp(ctt->url->username,url->username)==0)
			{
				osip_trace(OSIP_INFO1,("osip_ua_find:  %s <>  %s \n",ctt->url->host,url->host));
				if (strcasecmp(ctt->url->host, url->host)==0)
				{
					ret_ua=ua;
				}
			}
		}
	}
#ifdef INET6
	if (res!=NULL) freeaddrinfo(res);	
#endif
	if (ipaddr!=NULL) sfree(ipaddr);
	return(ret_ua);	
}


BodyHandler * osip_ua_find_handler(OsipUA *ua, char *body_mime)
{
	int index=0;
	BodyHandler *info=NULL;
	list_t *body_handlers=&ua->body_handlers;
	while (! list_eol(body_handlers,index))
	{
		char *bhmime;
		info=(BodyHandler*)list_get(body_handlers,index);
		bhmime=body_handler_get_mime(info);
		if ( strcmp(body_mime,bhmime)==0) return info;
		index++;
	}
	return NULL;
}		

OsipDialog * osip_ua_get_dialog(OsipUA *ua, int number)
{
	OsipDialog *dialog;
	dialog=(OsipDialog*)list_get(&ua->call_list,number);
	if (dialog!=NULL && dialog->status!=DIALOG_TERMINATED && dialog->status!=DIALOG_CANCELLED)
		return dialog;
	return NULL;
}

void osip_ua_notify_subscribers(OsipUA *ua)
{
	int nelts = ua->call_list.nb_elt;
	int pos;
	char *cpim = "";
	for (pos = 0; pos < nelts; pos++) {
		OsipDialog *call = (OsipDialog*)list_get(&ua->call_list, pos);
		if (call->status == DIALOG_SUBSCRIBED
		    && call->dialog->type == CALLEE) {
			url_t *subscriber_url;
			char *subscriber_url_text = NULL;
			subscriber_url = from_geturl(call->from);
			url_2char(subscriber_url, &subscriber_url_text);
			osip_dialog_notify(call, subscriber_url_text,
					   "presence",
					   "application/cpim-pidf+xml",
					   cpim);
		}
	}
}

