/*
  The osipua library is a library based on oSIP that implements CallLeg and User Agent
  level.
  Copyright (C) 2001  Simon MORLAT simon.morlat@free.fr
  											Aymeric MOIZARD jack@atosc.org
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "osipua.h"
#include "utils.h"
#include "uatransaction.h"
#include "osipmanager.h"
#include <osip/smsg.h>
#include <osip/smsgtypes.h>

int ua_transaction_get_destination(transaction_t *trn, char **destination, int *port)
{
	if (trn->ctx_type==ICT){
		*destination=trn->ict_context->destination;
		*port=trn->ict_context->port;
	}
	else if (trn->ctx_type==NICT){
		*destination=trn->nict_context->destination;
		*port=trn->nict_context->port;
	}else {
		osip_trace(OSIP_ERROR,("Could not get destination for transaction: ctx_type=%i\n",trn->ctx_type));
		*destination=NULL;
		*port=0;
		return -1;
	}
	return 0;
}

void ua_transaction_set_destination(transaction_t *trn,char *destination, int port)
{
	if (trn->ctx_type==ICT){
		ict_set_destination(trn->ict_context,destination,port);
	}
	else if (trn->ctx_type==NICT){
		nict_set_destination(trn->nict_context,destination,port);
	}else {
		osip_trace(OSIP_ERROR,("Could not set destination for transaction: ctx_type=%i\n",trn->ctx_type));
	}
}

transaction_t * ua_transaction_new(OsipDialog *call,sip_t *sipmsg)
{
	transaction_t *transaction;
	OsipUA *ua;
	context_type_t type;
	int port=5060;
	
	if (call==NULL) return NULL;
	
	ua=(OsipUA*)call->ua;
	if (MSG_IS_INVITE(sipmsg))
	{
		type=ICT;
	}else type=NICT;
	transaction_init(&transaction,
			type,
	   		ua->config,
			sipmsg);
	transaction->your_instance=(void*)call;
	if (transaction==NULL) return NULL;
	transaction_set_out_socket(transaction,ua->manager->send_sock);
	/* if an outbound proxy is set for this call-leg, then set it to the transaction */
	if ((ua->registrar!=NULL) && (ua->flags & OSIP_UA_USE_PROXY) )
	{
		
		if (ua->registrar->port!=NULL) port=atoi(ua->registrar->port);
		ua_transaction_set_destination(transaction,sgetcopy(ua->registrar->host),port);
		
	}else{
		route_t *route;
		/* else set the destination in the req uri */
		msg_getroute(sipmsg, 0, &route);
		if (route!=NULL)
		  {
		    port = 5060;
		    if (route->url->port!=NULL)
		      port = satoi(route->url->port);
		    if (MSG_IS_INVITE(sipmsg))
		      ict_set_destination(transaction->ict_context,
					  sgetcopy(route->url->host),
					  port);
		    else
		      nict_set_destination(transaction->nict_context,
					  sgetcopy(route->url->host),
					  port);
		  }
		else
		  {
		    int port = 5060;
		    if (sipmsg->strtline->rquri->port!=NULL)
		      port = satoi(sipmsg->strtline->rquri->port);
		    if (MSG_IS_INVITE(sipmsg))
		    ict_set_destination(transaction->ict_context,
					sgetcopy(sipmsg->strtline->rquri->host),
					port);
		    else
		    nict_set_destination(transaction->nict_context,
					sgetcopy(sipmsg->strtline->rquri->host),
					port);
		  }
	}
	
	if (MSG_IS_INVITE(sipmsg))
	  ua_transaction_set_outgoing_invite_tr(transaction, call);
	else if (MSG_IS_REGISTER(sipmsg))
	  ua_transaction_set_outgoing_register_tr(transaction, call);
	else if (MSG_IS_BYE(sipmsg))
	  ua_transaction_set_outgoing_bye_tr(transaction, call);
	else if (MSG_IS_CANCEL(sipmsg))
	  ua_transaction_set_outgoing_cancel_tr(transaction,call);
	else if (MSG_IS_SUBSCRIBE(sipmsg))
	  ua_transaction_set_outgoing_subscribe_tr(transaction, call);
	else if (MSG_IS_NOTIFY(sipmsg))
	  ua_transaction_set_outgoing_notify_tr(transaction, call);
	
	return transaction;
}

void ua_transaction_set_incoming_invite_tr(transaction_t *trn,OsipDialog *call)
{
  OsipUA *ua = osip_dialog_get_ua (call);
  if (call->inc_invite_tr)
	  call->inc_invite_tr->your_instance = NULL;
  call->inc_invite_tr = trn;
  trn->your_instance=call;
  transaction_set_out_socket(trn,ua->manager->send_sock);
}

void ua_transaction_set_outgoing_invite_tr(transaction_t *trn,OsipDialog *call)
{
  OsipUA *ua = osip_dialog_get_ua (call);
  if (call->out_invite_tr)
	  call->out_invite_tr->your_instance = NULL;
  call->out_invite_tr = trn;
  trn->your_instance=call;
  transaction_set_out_socket(trn,ua->manager->send_sock);
}

void ua_transaction_set_incoming_bye_tr(transaction_t *trn,OsipDialog *call)
{
  OsipUA *ua = osip_dialog_get_ua (call);
  if (call->inc_bye_tr)
	  call->inc_bye_tr->your_instance = NULL;
  call->inc_bye_tr = trn;
  trn->your_instance=call;
  transaction_set_out_socket(trn,ua->manager->send_sock);
}

void ua_transaction_set_outgoing_bye_tr(transaction_t *trn,OsipDialog *call)
{
  OsipUA *ua = osip_dialog_get_ua (call);
  if (call->out_bye_tr)
	  call->out_bye_tr->your_instance = NULL;
  call->out_bye_tr = trn;
  trn->your_instance=call;
  transaction_set_out_socket(trn,ua->manager->send_sock);
}

void ua_transaction_set_outgoing_register_tr(transaction_t *trn,OsipDialog *call)
{
  OsipUA *ua = osip_dialog_get_ua (call);
  if (call->out_register_tr)
	  call->out_register_tr->your_instance = NULL;
  call->out_register_tr = trn;
  trn->your_instance=call;
  transaction_set_out_socket(trn,ua->manager->send_sock);
}

void ua_transaction_set_outgoing_cancel_tr(transaction_t *trn,OsipDialog *call)
{
  OsipUA *ua = osip_dialog_get_ua (call);
  if (call->out_cancel_tr)
	  call->out_cancel_tr->your_instance = NULL;
  call->out_cancel_tr = trn;
  trn->your_instance=call;
  transaction_set_out_socket(trn,ua->manager->send_sock);
}

void ua_transaction_set_incoming_cancel_tr(transaction_t *trn,OsipDialog *call)
{
  OsipUA *ua = osip_dialog_get_ua (call);
  if (call->inc_cancel_tr)
	  call->inc_cancel_tr->your_instance = NULL;
  call->inc_cancel_tr = trn;
  trn->your_instance=call;
  transaction_set_out_socket(trn,ua->manager->send_sock);
}

void ua_transaction_set_incoming_subscribe_tr(transaction_t *trn,OsipDialog *call)
{
  OsipUA *ua = osip_dialog_get_ua (call);
  call->inc_subscribe_tr = trn;
  trn->your_instance=call;
  transaction_set_out_socket(trn,ua->manager->send_sock);
}

void ua_transaction_set_outgoing_subscribe_tr(transaction_t *trn,OsipDialog *call)
{
  OsipUA *ua = osip_dialog_get_ua (call);
  call->out_subscribe_tr = trn;
  trn->your_instance=call;
  transaction_set_out_socket(trn,ua->manager->send_sock);
}

void ua_transaction_set_incoming_notify_tr(transaction_t *trn,OsipDialog *call)
{
  OsipUA *ua = osip_dialog_get_ua (call);
  call->inc_notify_tr = trn;
  trn->your_instance=call;
  transaction_set_out_socket(trn,ua->manager->send_sock);
}

void ua_transaction_set_outgoing_notify_tr(transaction_t *trn,OsipDialog *call)
{
  OsipUA *ua = osip_dialog_get_ua (call);
  call->out_notify_tr = trn;
  trn->your_instance=call;
  transaction_set_out_socket(trn,ua->manager->send_sock);
}

/*
  void ua_transaction_incoming_set_dialog(transaction_t *trn,OsipDialog *call)
  {
  OsipUA *ua=osip_dialog_get_ua(call);
  list_add(&call->incoming_transactions,trn,-1);
  trn->your_instance=call;
  transaction_set_out_socket(trn,ua->manager->send_sock);
  }
*/
/*
  void ua_transaction_outgoing_set_dialog(transaction_t *trn,OsipDialog *call)
  {
  list_add(&call->outgoing_transactions,trn,-1);
  trn->your_instance=call;
  }
*/

int is_in_recv_thread_context(OsipManager *manager)
{
	return (manager->thread_pid==getpid());
}

void ua_transaction_free(transaction_t *transaction)
{
	OsipDialog *call;
	OsipUA *ua;
	int error;
	OsipManager *manager=def_manager;
	
	/* check if the call_leg that the transaction is part of should not be deleted too*/
	call=ua_transaction_get_dialog(transaction);
	if (call==NULL)
	{
		goto free_transaction;
		osip_trace(OSIP_WARNING,("error: dialog==NULL ! "));
		return;
	}
	ua=osip_dialog_get_ua(call);
	manager=ua->manager;

	if (transaction==call->inc_invite_tr)
	  {
	    call->inc_invite_tr = NULL;
	  }
	else if (transaction==call->out_invite_tr)
	  {
	    call->out_invite_tr = NULL;
	  }
	else if (transaction==call->inc_bye_tr)
	  {
	    call->inc_bye_tr = NULL;
	  }
	else if (transaction==call->out_bye_tr)
	  {
	    call->out_bye_tr = NULL;
	  }
	else if (transaction==call->out_register_tr){
	     call->out_register_tr = NULL;
	}else if (transaction==call->out_cancel_tr){
	     call->out_cancel_tr = NULL;
	}else if (transaction==call->inc_cancel_tr){
	     call->inc_cancel_tr = NULL;
	}else if (transaction==call->inc_subscribe_tr){
	     call->inc_subscribe_tr = NULL;
	}else if (transaction==call->out_subscribe_tr){
	     call->out_subscribe_tr = NULL;
	}else if (transaction==call->inc_notify_tr){
	     call->out_notify_tr = NULL;
	}else if (transaction==call->out_notify_tr){
	     call->inc_notify_tr = NULL;
	}
	transaction->your_instance=NULL;

	/* now if it was the last active transaction of the dialog, delete the dialog*/
	if (call->out_invite_tr==NULL && call->inc_invite_tr==NULL &&
	    call->out_bye_tr==NULL && call->inc_bye_tr==NULL &&
	    call->out_subscribe_tr==NULL && call->inc_subscribe_tr==NULL &&
	    call->out_notify_tr==NULL && call->inc_notify_tr==NULL &&
	    call->out_register_tr==NULL && 
	    call->out_cancel_tr == NULL && call->inc_cancel_tr == NULL)
	{
		switch (call->status)
		{
			
			case DIALOG_INVITING:
			case DIALOG_SUBSCRIBING:
				/* if the transaction timed out, notify it */
				if (transaction->last_response==NULL){
				
					error=-ETIMEDOUT;
					/* the invite times out. The call-leg is lost. */
					if (ua->failure!=NULL)
						ua->failure(call,transaction,NULL,(void*)&error);
					/* then destroy the dialog*/
					osip_dialog_destroy(call);
				}
				
			break;
			case DIALOG_CANCELLING:
			case DIALOG_FAKE:
			case DIALOG_CANCELLED:
			case DIALOG_TERMINATED:
				osip_dialog_destroy(call);
			break;
		}
	}
	
	free_transaction:
	if (is_in_recv_thread_context(manager) ){
		/* put the transaction in the free list */
		fifo_add(&manager->garbage_trn,(void*)transaction);
	}else{
		smutex_lock(manager->mutex);
		fifo_add(&manager->garbage_trn,(void*)transaction);
		smutex_unlock(manager->mutex);
	}
}


/* execute a transaction outside the receiving thread */
void ua_transaction_execute(transaction_t *trn, sipevent_t *ev)
{
	sipevent_t *se;
	OsipManager *manager;
	OsipDialog *dialog=ua_transaction_get_dialog(trn);
	OsipUA *ua;
	int cond;
	
	printf("Executing transaction...\n");
	fflush(NULL);
	
	if (dialog==NULL){
		osip_trace(OSIP_WARNING,("ua_transaction_execute: could not get dialog transaction.\n"));
		manager=def_manager;
	}else {
		ua=osip_dialog_get_ua(dialog);
		manager=ua->manager;
	}
	/* check if we are within the process of the receiving thread 
	If you we are in, (probably in one of the osip callbacks), then we can safely run
	fifo_add(trn->transactionff,ev);
	If not the we have to take a lock to prevent the receiving thread from executing
	the transaction while we process it */
	cond=is_in_recv_thread_context(manager);
	
	if (cond){
		fifo_add(trn->transactionff,ev);
		//transaction_execute(trn,ev);
	}
	else{
		smutex_lock(manager->mutex);
		fifo_add(trn->transactionff,ev);
		smutex_unlock(manager->mutex);
	}
	//osip_manager_wake_up(manager);
	
}
