/*
  mbdesktop - a desktop for handhelds etc. 

   Copyright 2002 Matthew Allum

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/


#include "mbdesktop.h"
#include "mbdesktop_view.h"

#ifdef USE_PNG
#define FOLDER_IMG "mbfolder.png"
#else
#define FOLDER_IMG "mbfolder.xpm"
#endif

/* 
   TODO:

     o CODE CLEAN !
       o sort static / defines for mbdesktop
       o sort out callback api / extensability

 */


void
mbdesktop_folder_activate_cb(void *data1, void *data2);

void
mbdesktop_app_item_activate_cb(void *data1, void *data2);

void 
mbdesktop_app_item_activate_sn_cb(void *data1, void *data2);

void 
mbdesktop_app_item_activate_si_cb(void *data1, void *data2);

#ifdef USE_DNOTIFY
static void
mbdesktop_items_reload(MBDesktop *mb);
#endif

Bool
mbdesktop_bg_get_from_theme(MBDesktop *mb);

#ifdef USE_XSETTINGS
static void 
mbdesktop_xsettings_notify_cb (const char       *name,
			       XSettingsAction   action,
			       XSettingsSetting *setting,
			       void             *data);
#endif

MBDesktopItem *
mbdesktop_item_get_first_sibling(MBDesktopItem *item);

struct itemfolderlookup {
  char *name;
  void *data;

  MBDesktopCB activate_cb;
  MBDesktopCB populate_cb;
  MBDesktopCB delete_cb;

  int view;
  int type;
  Bool is_cached;

  /* Private */
  MBDesktopItem *_folder;


} ItemFolders[] = {
  { "Utilities",  (void *)"Utility", mbdesktop_folder_activate_cb, NULL, NULL, 
    VIEW_ICONS, ITEM_TYPE_FOLDER, False, NULL },
  { "Settings", (void *)"System", mbdesktop_folder_activate_cb, NULL, NULL, 
    VIEW_ICONS, ITEM_TYPE_FOLDER, False, NULL },
  { "Games", (void *)"Game", mbdesktop_folder_activate_cb, NULL, NULL, 
    VIEW_ICONS, ITEM_TYPE_FOLDER, False, NULL },
  { "Other", NULL, mbdesktop_folder_activate_cb, NULL, NULL, 
    VIEW_ICONS, ITEM_TYPE_FOLDER, False, NULL },
#define FOLDER_OTHER_IDX 3
  /*
  { "Recent Files", NULL, NULL, ITEM_TYPE_FOLDER },
  { "Today", NULL, NULL, ITEM_TYPE_FOLDER },
  { "Bookmarks", NULL, NULL, ITEM_TYPE_FOLDER },
  { "Tasks", NULL, NULL, ITEM_TYPE_TASK_FOLDER },
  */
  { 
    "Active Tasks", NULL, 
    item_tasks_activate_cb, 
    item_tasks_populate_cb,
    item_tasks_empty_cb, 
    VIEW_ICONS, 
    ITEM_TYPE_TASK_FOLDER, 
    False, 
    NULL 
  },
  { NULL, NULL }
};


MBDesktopItem *
mbdesktop_lookup_folder(char *categories)
{
  int i = 0;
  MBDesktopItem *found_item;

  if (categories == NULL) return NULL;
  
  if (strstr(categories, "pim")) return NULL;

  while( ItemFolders[i].name != NULL )
    {
      if (ItemFolders[i].data
	  && strstr(categories, ItemFolders[i].data))
	{
	  found_item = ItemFolders[i]._folder->item_child;
	  while ( found_item->item_next_sibling != NULL)
	    found_item = found_item->item_next_sibling;
	  return found_item;
	}
      i++;
    }

  return NULL;
}

void
mbdesktop_folder_activate_cb(void *data1, void *data2)
{
  MBDesktop *mb = (MBDesktop *)data1; 
  MBDesktopItem *item = (MBDesktopItem *)data2; 

  switch(item->type)
    {
    case ITEM_TYPE_PREVIOUS:
      mb->kbd_focus_item = item->item_parent;
      mb->scroll_offset_item  = mb->current_head_item 
	= mbdesktop_item_get_first_sibling(item->item_parent);
      break;
    case ITEM_TYPE_FOLDER:
      mb->scroll_offset_item = mb->kbd_focus_item 
	= mb->current_head_item = item->item_child;
      break;
    case ITEM_TYPE_TASK_FOLDER:
      item->populate_cb(mb, item);
      mb->scroll_offset_item = mb->kbd_focus_item 
	= mb->current_head_item = item->item_child;
      break;
    }


  mbdesktop_view_paint(mb, False);

} 


MBDesktopItem *
mbdesktop_item_get_first_sibling(MBDesktopItem *item)
{
  while (item->item_prev_sibling != NULL )
    item = item->item_prev_sibling;
  return item;
}

void 
mbdesktop_app_item_activate_si_cb(void *data1, void *data2)
{
#ifdef USE_LIBSN
  MBDesktop *mb = (MBDesktop *)data1; 
  MBDesktopItem *item = (MBDesktopItem *)data2; 
  Window win_found;

  if (mb_single_instance_is_starting(mb->dpy, item->exec_str))
    return;

  win_found = mb_single_instance_get_window(mb->dpy, item->exec_str);

  if (win_found != None)
    {
      mb_util_window_activate(mb->dpy, win_found);
    }
  else mbdesktop_app_item_activate_sn_cb((void *)mb, (void *)item);

#endif
}


void 
mbdesktop_app_item_activate_sn_cb(void *data1, void *data2)
{
#ifdef USE_LIBSN
  MBDesktop *mb = (MBDesktop *)data1; 
  MBDesktopItem *item = (MBDesktopItem *)data2; 

  SnLauncherContext *context;
  pid_t child_pid = 0;

  context = sn_launcher_context_new (mb->sn_dpy, mb->scr);
  
  sn_launcher_context_set_name (context, item->name);
  if (item->comment)
    sn_launcher_context_set_description (context, item->comment);
  sn_launcher_context_set_binary_name (context, item->exec_str);
  
  sn_launcher_context_initiate (context, "mbdesktop launch", item->exec_str,
				CurrentTime);

  switch ((child_pid = fork ()))
    {
    case -1:
      fprintf (stderr, "Fork failed\n" );
      break;
    case 0:
      sn_launcher_context_setup_child_process (context);
      mb_exec(item->exec_str);
      // execlp(item->exec_str, item->exec_str, NULL);
      fprintf (stderr, "Failed to exec %s \n", item->exec_str);
      _exit (1);
      break;
    }

#endif
}


void
mbdesktop_app_item_activate_cb(void *data1, void *data2)
{
  MBDesktopItem *item = (MBDesktopItem *)data2; 

  switch (fork()) 
    {
    case 0:
      mb_exec(item->exec_str);
      /* execlp("/bin/sh", "sh", "-c", item->exec_str, NULL); */
      fprintf(stderr, "exec failed, cleaning up child\n");
      exit(1);
    case -1:
      fprintf(stderr, "can't fork\n"); 
      break;
    }
} 

MBDesktopItem *
mbdesktop_item_new()
{
  MBDesktopItem *ditem;
  ditem = malloc(sizeof(MBDesktopItem));
  
  ditem->type = ITEM_TYPE_UNKNOWN;
  ditem->name = NULL;
  ditem->comment  = NULL;
  ditem->icon     = NULL;
  ditem->exec_str = NULL;

  ditem->exec_cb = NULL;
  ditem->populate_cb = NULL;

  ditem->item_next_sibling = NULL;
  ditem->item_prev_sibling = NULL;
  ditem->item_child = NULL;
  ditem->item_parent = NULL;

  ditem->view = 0;

  return ditem;
}


MBDesktopItem *
mbdesktop_item_new_with_params(
			       MBDesktop *mb,
			       char *name, 
			       char *comment, 
			       MBPixbufImage *icon,
			       char *exec_str,
			       MBDesktopCB exec_cb,
			       MBDesktopCB populate_cb,
			       int type,
			       int view )
{
  MBDesktopItem *ditem;
  ditem = mbdesktop_item_new();

  if (name) ditem->name           = strdup(name);
  if (comment) ditem->comment     = strdup(comment);
  if (icon) ditem->icon           = mb_pixbuf_img_clone(mb->pixbuf, icon);
  if (exec_str) ditem->exec_str   = strdup(exec_str);
  if (exec_cb) ditem->exec_cb     = exec_cb;
  if (populate_cb) ditem->populate_cb  = populate_cb;
  if (type) ditem->type           = type;
  if (view) ditem->view           = view;

  return ditem;
}

void
mbdesktop_item_set_image(MBDesktop     *mb, 
			 MBDesktopItem *item, 
			 char          *full_img_path)
{
  if (item->icon) mb_pixbuf_img_free(mb->pixbuf, item->icon);
  item->icon = mb_pixbuf_img_new_from_file(mb->pixbuf, full_img_path);
}


void
mbdesktop_init_items(MBDesktop *mb)
{
  DIR *dp;
  struct dirent *dir_entry;
  struct stat stat_info;
  char orig_wd[256];
  MBDesktopItem *cur;

  MBPixbufImage *img;

  /* Now we load in the dotdesktop files */
  
  cur = mb->top_head_item;
  while (cur->item_next_sibling) cur = cur->item_next_sibling;

  if (getcwd(orig_wd, 255) == (char *)NULL)
    {
      fprintf(stderr, "Cant get current directory\n");
      exit(0);
    }
  
  if ((dp = opendir(mb->dd_dir)) == NULL)
    {
      fprintf(stderr, "failed to open %s\n", mb->dd_dir);
      exit(0);
    }
  
  chdir(mb->dd_dir);
  while((dir_entry = readdir(dp)) != NULL)
    {
      lstat(dir_entry->d_name, &stat_info);
      if (!(S_ISDIR(stat_info.st_mode)))
	{
	  MBDotDesktop *dd;
	  dd = mb_dotdesktop_new_from_file(dir_entry->d_name);
	  if (dd)
	    {

	      MBDesktopItem *found_item;

	      if (mb_dotdesktop_get(dd, "Type") 
		  && !strcmp(mb_dotdesktop_get(dd, "Type"), "Application")
		  && mb_dotdesktop_get(dd, "Icon") 
		  && mb_dotdesktop_get(dd, "Name")
		  && mb_dotdesktop_get(dd, "Exec"))
		{
		  char *png_path = NULL;
		  MBDesktopCB activate_cb;
		  unsigned char *icon_name = mb_dotdesktop_get(dd, "Icon");
		  
		  png_path 
		    = mb_dot_desktop_icon_get_full_path( mb->theme_name,
							 32, icon_name );

		  if (png_path == NULL) continue;

		  img = mb_pixbuf_img_new_from_file(mb->pixbuf, png_path);
		  activate_cb = &mbdesktop_app_item_activate_cb;

#ifdef USE_LIBSN
		  if (mb_dotdesktop_get(dd, "SingleInstance")
		      && !strcasecmp(mb_dotdesktop_get(dd, "SingleInstance"), 
				     "true"))
		    {
		      activate_cb = &mbdesktop_app_item_activate_si_cb;
		    }
		    else if (mb_dotdesktop_get(dd, "StartupNotify")
		      && !strcasecmp(mb_dotdesktop_get(dd, "StartupNotify"), 
				     "true"))
		    activate_cb = &mbdesktop_app_item_activate_sn_cb;
#endif
		  
		  if (!img) { free(png_path); continue; }
		  /* Match Category */
		  if ((found_item =  mbdesktop_lookup_folder(mb_dotdesktop_get(dd, "Categories"))) != NULL)
		    {
		      found_item->item_next_sibling 
			= mbdesktop_item_new_with_params(mb,
							 mb_dotdesktop_get(dd,"Name"), 
							 NULL, 
							 img,
							 mb_dotdesktop_get(dd,"Exec"), 
							 activate_cb,
							 NULL,
							 ITEM_TYPE_APP,
							 0);
		      found_item->item_next_sibling->item_prev_sibling 
			= found_item;
		    } else {
		      cur->item_next_sibling 
			= mbdesktop_item_new_with_params(mb,
							 mb_dotdesktop_get(dd,"Name"), 
							 NULL, 
							 img,
							 mb_dotdesktop_get(dd,"Exec"),
							 activate_cb,
							 NULL,
							 ITEM_TYPE_APP, 0 );
		      cur->item_next_sibling->item_prev_sibling = cur;
		      cur = cur->item_next_sibling;
		    }
		  free(png_path);
		  mb_pixbuf_img_free(mb->pixbuf, img);
		}
	      else fprintf(stderr, 
			   "%s has no icon, png or name\n", 
			   dir_entry->d_name);
	      
	      mb_dotdesktop_free(dd);

	    }
	  else
	    fprintf(stderr, "failed to parse %s :( \n", dir_entry->d_name);
	}
    }
  
  closedir(dp);
  chdir(orig_wd);
}

void
mbdesktop_init_folder_items(MBDesktop *mb)
{
  int i = 1;
  MBDesktopItem *cur;
  MBPixbufImage *img;

  if (mb->folder_img_path) free(mb->folder_img_path);

  mb->folder_img_path = mb_dot_desktop_icon_get_full_path(mb->theme_name, 
							  32, FOLDER_IMG );

  img = mb_pixbuf_img_new_from_file(mb->pixbuf, mb->folder_img_path);
  if (!img)
    {
      fprintf(stderr, "Failed to load %s\n", mb->folder_img_path);
      exit(1);
    }
  

  if (mb->ddfolders) mb_dot_desktop_folders_free(mb->ddfolders);

  mb->ddfolders = mb_dot_desktop_folders_new( PKGDATADIR "/vfolders" );
  
  if (mb->ddfolders == NULL )
    {
      fprintf(stderr, "Failed to open folder directory : %s\n", 
	      PKGDATADIR "/vfolders");
      /* exit(0); */
    }

  //  ItemFolders = malloc(sizeof(struct itemfolderlookup)*(mb->ddfolders->n_entries+1));


  mb->top_head_item 
    = mbdesktop_item_new_with_params(mb,
				     ItemFolders[0].name, 
				     NULL, 
				     img,
				     NULL,
				     mbdesktop_folder_activate_cb,
				     NULL,
				     ITEM_TYPE_FOLDER,
				     0);

  ItemFolders[0]._folder = mb->top_head_item;

  ItemFolders[0]._folder->item_child 
    = mbdesktop_item_new_with_params(mb,
				     "Up", 
				     NULL, 
				     img,
				     NULL,
				     mbdesktop_folder_activate_cb,
				     NULL,
				     ITEM_TYPE_PREVIOUS,
				     0);


  ItemFolders[0]._folder->item_child->item_parent = mb->top_head_item;

  cur = mb->top_head_item;

  while ( ItemFolders[i].name != NULL )
    {
      cur->item_next_sibling 
	= mbdesktop_item_new_with_params(mb,
					 ItemFolders[i].name, 
					 NULL, 
					 img,
					 NULL,
					 /* ItemFolders[i].activate_cb, */
					 mbdesktop_folder_activate_cb,
					 ItemFolders[i].populate_cb,
					 ItemFolders[i].type,
					 ItemFolders[i].view);
      ItemFolders[i]._folder = cur->item_next_sibling;
      ItemFolders[i]._folder->item_child 
	= mbdesktop_item_new_with_params(mb,
					 "Up", 
					 NULL, 
					 img,
					 NULL,
					 mbdesktop_folder_activate_cb,
					 NULL,
					 ITEM_TYPE_PREVIOUS, 
					 ItemFolders[i].view);
      ItemFolders[i]._folder->item_child->item_parent = ItemFolders[i]._folder;

      cur->item_next_sibling->item_prev_sibling = cur;
      cur = cur->item_next_sibling;
      i++;
    }

  mb_pixbuf_img_free(mb->pixbuf, img);

  mb->scroll_offset_item = mb->kbd_focus_item 
    = mb->current_head_item = mb->top_head_item;
}

void
mbdesktop_items_free(MBDesktop *mb, MBDesktopItem *item)
{
  MBDesktopItem *tmp_item;
  while(item != NULL)
    {
      tmp_item = item->item_next_sibling;
      if (item->name) free(item->name);
      if (item->comment) free(item->comment);
      if (item->exec_str) free(item->exec_str);
      if (item->icon) mb_pixbuf_img_free(mb->pixbuf, item->icon);

      free(item);
      item = tmp_item;
    }
}

MBDesktopItem *
mbdesktop_item_find(MBDesktop *mb, int x, int y)
{
  MBDesktopItem *item;
  for(item = mb->scroll_offset_item; 
      (item != NULL && item != mb->last_visible_item); 
      item = item->item_next_sibling)
    {
      if (x > item->x && x < (item->x + mb->item_width)
	  && y > item->y && y < (item->y + mb->item_height))
	{
	  return item;
	} 
    }
  return NULL;
}


void 
mbdesktop_scroll_up(MBDesktop *mb)
{
  if (mb->scroll_offset_item->item_prev_sibling)
    {
      int i, items_per_row = mb->workarea_width / mb->item_width;
      for(i = 0; i < items_per_row; i++)
	{
	  mb->scroll_offset_item = mb->scroll_offset_item->item_prev_sibling;
	  if (mb->kbd_focus_item->item_prev_sibling)
	    mb->kbd_focus_item = mb->kbd_focus_item->item_prev_sibling;
	}
    }
}

void 
mbdesktop_scroll_down(MBDesktop *mb)
{
  MBDesktopItem *orig = mb->scroll_offset_item;
  if (mb->scroll_offset_item->item_next_sibling)
    {
      int i, items_per_row = mb->workarea_width / mb->item_width;
      for(i = 0; i < items_per_row; i++)
	{
	  if (mb->scroll_offset_item->item_next_sibling == NULL)
	    {
	      mb->scroll_offset_item = orig;
	      return;
	    } 
	  else
	    mb->scroll_offset_item = mb->scroll_offset_item->item_next_sibling;

	  if (mb->kbd_focus_item->item_next_sibling)
	    mb->kbd_focus_item = mb->kbd_focus_item->item_next_sibling;
	}
    }
}

void
handle_button_event(MBDesktop *mb, XButtonEvent *e)
{
  XEvent ev;
  MBDesktopItem *active_item = NULL;

  active_item = mbdesktop_item_find(mb, e->x, e->y); 
  
  if (active_item)
    {
      Bool canceled = False;
      
      if (XGrabPointer(mb->dpy, mb->win_top_level, False,
		       ButtonPressMask|ButtonReleaseMask|
		       PointerMotionMask|EnterWindowMask|LeaveWindowMask,
		       GrabModeAsync,
		       GrabModeAsync, None, None, CurrentTime)
	  == GrabSuccess)
	{
	  XFillRectangle(mb->dpy, mb->win_top_level, mb->invert_gc,
			 active_item->x, active_item->y + mb->icon_size,
			 mb->item_width, mb->item_height - mb->icon_size);

	  if (mb->have_focus && mb->had_kbd_input)
	    {
	      XDrawRectangle(mb->dpy, mb->win_top_level, 
			     mb->invert_gc,
			     mb->kbd_focus_item->x, 
			     mb->kbd_focus_item->y + mb->icon_size,
			     mb->item_width, mb->item_height - mb->icon_size);
	    }
	  
	  for (;;) 
	    {
	      XMaskEvent(mb->dpy,
			 ButtonPressMask|ButtonReleaseMask|
			 PointerMotionMask, &ev);
	      
	      switch (ev.type)
		{
		case MotionNotify:
		  if (canceled 
		      && ev.xmotion.x > active_item->x
		      && ev.xmotion.x < (active_item->x + mb->item_width)
		      && ev.xmotion.y > active_item->y
		      && ev.xmotion.y < (active_item->y + mb->item_height)
		      )
		    {
		      XFillRectangle(mb->dpy, mb->win_top_level,
				     mb->invert_gc,
				     active_item->x, 
				     active_item->y + mb->icon_size,
				     mb->item_width, mb->item_height - mb->icon_size);
		      canceled = False;
		      break;
		    }
		  if (!canceled 
		      && ( ev.xmotion.x < active_item->x
			   || ev.xmotion.x > (active_item->x + mb->item_width)
			   || ev.xmotion.y < active_item->y
			   || ev.xmotion.y > (active_item->y + mb->item_height)
			   ))
		  {
		    XFillRectangle(mb->dpy, mb->win_top_level,
				   mb->invert_gc,
				   active_item->x, active_item->y + mb->icon_size,
				   mb->item_width, mb->item_height-mb->icon_size);
		    canceled = True;
		    break;
		  }

		  break;
		case ButtonRelease:
		  XUngrabPointer(mb->dpy, CurrentTime);
		  if (!canceled)
		    {
		      XFillRectangle(mb->dpy, mb->win_top_level, 
				     mb->invert_gc,
				     active_item->x, 
				     active_item->y + mb->icon_size,
				     mb->item_width, mb->item_height - mb->icon_size);
		      if (active_item->exec_cb)
			active_item->exec_cb((void *)mb, 
					     (void *)active_item); 
		      return;
		    }
		  else
		    {
		      if (mb->have_focus && mb->had_kbd_input)
			{
			  mb->kbd_focus_item = active_item;
			  XDrawRectangle(mb->dpy, mb->win_top_level, 
					 mb->invert_gc,
					 mb->kbd_focus_item->x, 
					 mb->kbd_focus_item->y + mb->icon_size,
					 mb->item_width, mb->item_height - mb->icon_size);
			}

		      return;
		    }
		}
	    }
	}
    }
  else
    {
      if (e->y < mb->workarea_y + 20)
	{
	  if (e->x > (mb->workarea_width-24))
	    mbdesktop_scroll_up(mb);
	  else if (e->x > (mb->workarea_width-40))
	    mbdesktop_scroll_down(mb);

	  mbdesktop_view_paint(mb, False);
	}
    }

}

void
handle_key_event(MBDesktop *mb, XKeyEvent *e)
{
  MBDesktopItem *active_item = NULL, *tmp_item = NULL;
  KeySym key;
  int i = 0;
  Bool not_scrolled = True;
  int max_items_horiz = mb->workarea_width / mb->item_width;

  if (mb->had_kbd_input == False)
    {
      mb->had_kbd_input = True;
      mbdesktop_view_paint(mb, True);      
      return;
    }

  switch (key = XKeycodeToKeysym (mb->dpy, e->keycode, 0))
    {
    case XK_Left:
      if (mb->kbd_focus_item->item_prev_sibling)
	    {
	      active_item = mb->kbd_focus_item->item_prev_sibling;
	    } else return;
      break;
    case XK_Right:
      if (mb->kbd_focus_item->item_next_sibling)
	{
	  active_item = mb->kbd_focus_item->item_next_sibling;
	} else return;
      break;
    case XK_Up:
      active_item = mb->kbd_focus_item;
      while (i++ < max_items_horiz)
	if ((active_item = active_item->item_prev_sibling) == NULL)
	  return;
      
      
      break;
    case XK_Down:	    
      active_item = mb->kbd_focus_item;
      while (i++ < max_items_horiz)
	if ((active_item = active_item->item_next_sibling) == NULL)
	  return;
      
      break;
    case XK_Return:
    case XK_KP_Enter:
      mb->kbd_focus_item->exec_cb((void *)mb, 
				  (void *)mb->kbd_focus_item);
      return;
    }
 
  if (active_item)
    {
      if ( key == XK_Down || key == XK_Right)
	{
	  /* do we need to scroll ? */
	  if (mb->last_visible_item)
	    {
	      tmp_item = active_item;
	      while ((tmp_item = tmp_item->item_prev_sibling) != NULL)
		if (tmp_item == mb->last_visible_item->item_prev_sibling )
		  {
		    not_scrolled = False;
		    mbdesktop_scroll_down(mb);
		    break;
		  }
	    }
	}
      else
	{
	  /* do we need to scroll ? */
	  if (mb->scroll_offset_item)
	    {
	      tmp_item = active_item;
	      while ((tmp_item = tmp_item->item_next_sibling) != NULL)
		if (tmp_item == mb->scroll_offset_item)
		  {
		    not_scrolled = False;
		    mbdesktop_scroll_up(mb);
		    break;
		  }
	    }
	}
      
      /* Clear the old dashed border */
      if (not_scrolled && mb->had_kbd_input)
	{
	  XDrawRectangle(mb->dpy, mb->win_top_level, 
			 mb->invert_gc,
			 mb->kbd_focus_item->x, 
			 mb->kbd_focus_item->y + mb->icon_size,
			 mb->item_width, mb->item_height - mb->icon_size);
	  mb->kbd_focus_item = active_item;
	  mbdesktop_view_paint(mb, True);
	}
      else
	{
	  mb->kbd_focus_item = active_item;
	  mbdesktop_view_paint(mb, False);
	}
    }
}

Bool
mbdesktop_get_workarea(MBDesktop *mb, int *x, int *y, int *w, int *h)
{
  Atom real_type; int real_format;
  unsigned long items_read, items_left;
  long *coords;

  Atom workarea_atom =
    XInternAtom(mb->dpy, "_NET_WORKAREA",False);

  if (XGetWindowProperty(mb->dpy, mb->root,
			 workarea_atom, 0L, 4L, False,
			 XA_CARDINAL, &real_type, &real_format,
			 &items_read, &items_left,
			 (unsigned char **) &coords) == Success
      && items_read) {
    *x = coords[0];
    *y = coords[1];
    *w = coords[2];
    *h = coords[3];
    XFree(coords);
    return True;
  }
  return False;
}

#ifdef USE_XSETTINGS

#define XSET_UNKNOWN    0
#define XSET_THEME      1
#define XSET_BG         2
#define XSET_FONT       3
#define XSET_TITLE_FONT 4


static Bool
mbdesktop_xsettings_process (MBDesktop        *mb, 
			     const char       *name, 
			     XSettingsSetting *setting
			     )
{
  int i = 0;
  int key = XSET_UNKNOWN;
  
  struct _mb_xsettings { char *name; int value; } mb_xsettings[] = {
    { "Net/ThemeName",           XSET_THEME     },
    { "MATCHBOX/Background",     XSET_BG        },
    { NULL,       -1 } 
  };
  
  while(  mb_xsettings[i].name != NULL )
    {
      if (!strcmp(name, mb_xsettings[i].name)
	  && setting != NULL 	/* XXX set to NULL when action deleted */
	  && setting->type == XSETTINGS_TYPE_STRING )
	{
	  key = mb_xsettings[i].value;
	  break;
	}
      i++;
    }
    
  if (key == XSET_UNKNOWN) {
    /* xsettings_setting_free(setting); cause seg ? */
    return False;
  }

  switch (key)
    {
    case XSET_THEME:
      if (mb->theme_name) free(mb->theme_name);
      mb->theme_name = strdup(setting->data.v_string);
      if (mb->top_head_item != NULL) 	/* Items not yet created */
	{
	  mbdesktop_bg_get_from_theme(mb);
	  mbdesktop_items_reload(mb);
	}
      break;
    case XSET_BG:
      mb->bg_def = strdup(setting->data.v_string);
      if (mb->bg_img != NULL) 	/* Background not yet created */
	{
	  mbdesktop_bg_parse_spec(mb, mb->bg_def);
	  mbdesktop_view_init_bg(mb);
	  mbdesktop_view_paint(mb, False);
	}
      break;
      
    }
  
  /* xsettings_setting_free(setting); */
  return True;
}

static void 
mbdesktop_xsettings_notify_cb (const char       *name,
			       XSettingsAction   action,
			       XSettingsSetting *setting,
			       void             *data)
{
  MBDesktop *mb = (MBDesktop *)data;

  switch (action)
    {
    case XSETTINGS_ACTION_NEW:
    case XSETTINGS_ACTION_CHANGED:
      mbdesktop_xsettings_process (mb, name, setting); 
      break;
    case XSETTINGS_ACTION_DELETED:
        break;
    }

}

#endif


static void
usage(char *name)
{
  fprintf(stderr, 
	  "Usage: %s [options..]\n"
	  "Where options are:\n"
	  "   -display       Display to connect to\n"
	  "  --bg            <background definition>, like;\n\n"
	  "\t\timg-stretched:<filename>\n"
	  "\t\timg-tiled:<filename>\n"
	  "\t\timg-centered:<filename>\n"
	  "\t\tcol-solid:<color definition>\n"
	  "\t\tcol-gradient-vertical:<start color>,<end color>\n"
          "\t\tcol-gradient-horizontal:<start color>,<end color>\n\n"
	  "  --iconsize     <int>         Icon size ( defualt: 32 )\n"
	  "  --font         <font>        Icon font\n"
	  "  --titlefont    <font>        Title font\n"
	  "  --fontcol      <col>         Font color\n"
	  "  --folderimg    <img file>    Alternate folder icon\n"
	  "  --desktopdir   <path>        Path to .desktop files \n"
          "                              ( default: $PREFIX/share/applications )\n"
	  "  --pixmapsdir   <path>        Path to .desktop file pixmpas \n"
          "                              ( default: $PREFIX/share/pixmaps )\n"
	  "Notes;\n"
	  "  <col> is specified as a color name or an rgb def in the form 'rgb:r/g/b' or '#RGB\n"
	  "\n%s is copyright Matthew Allum 2002\n",
	  name, name
	  );
  exit(1);
}

#ifdef USE_DNOTIFY
MBDesktop *_Gmb;

static void 
dnotify_handler(int sig, siginfo_t *si, void *data)
{
  /* event_fd = si->si_fd; */
  mbdesktop_items_reload(_Gmb);
}


#if 0
Bool
mbdesktop_get_theme_name_from_root_prop(MBDesktop *mb)
{
  Atom realType;
  unsigned long n;
  unsigned long extra;
  int format;
  int status;
  char * value;
  struct stat stat_info;
  char *tmp;

  DBG("%s() called\n", __func__);
		  
  status = XGetWindowProperty(mb->dpy, mb->win_root,
			      panel->atoms[ATOM_MB_THEME], 
			      0L, 512L, False,
			      AnyPropertyType, &realType,
			      &format, &n, &extra,
			      (unsigned char **) &value);
	    
  if (status != Success || value == 0
      || *value == 0 || n == 0)
    {
      DBG("%s() no _MB_THEME set on root window\n", __func__ );
      return False;
    } else {
      /* Hack to figure out theme name from root prop - but means
         Matchbox themes will at least change without xsettings   */
      int len = 0 

      tmp = strdup(value);
      XFree(value);

      len = strlen[tmp];
      

    }

  return False;
}
#endif

static void
mbdesktop_items_reload(MBDesktop *mb)
{
  int i = 0;
  MBDesktopItem *item = mb->top_head_item;

  if (mb->folder_img_path) free(mb->folder_img_path);

  mb->folder_img_path = mb_dot_desktop_icon_get_full_path(mb->theme_name, 
							  32, FOLDER_IMG );

  /* free all subfolder items */
  while( ItemFolders[i].name != NULL )
    {
      if ( ItemFolders[i].type == ITEM_TYPE_FOLDER
	  && ItemFolders[i]._folder
	  && ItemFolders[i]._folder->item_child 
	  && ItemFolders[i]._folder->item_child->item_next_sibling != NULL)
	{
	  mbdesktop_items_free(mb, ItemFolders[i]._folder->item_child->item_next_sibling);
	  ItemFolders[i]._folder->item_child->item_next_sibling = NULL;

	}
      /* Make sure all folder icons get set to current theme */
      mbdesktop_item_set_image(mb, ItemFolders[i]._folder, 
			       mb->folder_img_path);
      if (ItemFolders[i]._folder->item_child)
	mbdesktop_item_set_image(mb, ItemFolders[i]._folder->item_child, 
				 mb->folder_img_path);

      i++;
    }

  /* free all toplevel items */
  while (item && item->item_child) item = item->item_next_sibling;

  item->item_prev_sibling->item_next_sibling = NULL;
  mbdesktop_items_free(mb, item);

  mbdesktop_init_items(mb);
  mbdesktop_view_paint(mb, False); 

}
#endif 

Bool
mbdesktop_bg_get_from_theme(MBDesktop *mb)
{
  struct stat s_buf;
  char *theme_path = NULL;
  char theme_conf_fn[256] = { 0 };
  
  if (mb->theme_name == NULL) return False;

  if ((theme_path = mb_util_get_theme_full_path(mb->theme_name)) != NULL)
    {
      snprintf(theme_conf_fn, 256, "%s/background/default.png",
	       theme_path );
      
      if (stat(theme_conf_fn, &s_buf) == 0)
	{
	  mb->bg->type = BG_TILED_PXM;
	  mb->bg->data.filename = strdup(theme_conf_fn);
	  return True;
	}
    }
  return False;
}


void
mbdesktop_bg_free_config(MBDesktop *mb)
{
  if (mb->bg->type == BG_TILED_PXM || mb->bg->type == BG_STRETCHED_PXM)
    free(mb->bg->data.filename);
  free(mb->bg);
  mb->bg = NULL;
}

Bool
mbdesktop_bg_parse_spec(MBDesktop *mb, char *spec)
{
  /*
  img-stretched:filename>
  img-tiled:<filename>
  col-solid:<color definition>
  col-gradient-vertical:<start color>,<end color>
  col-gradient-horizontal:<start color>,<end color>
  */

  XColor tmpxcol;
  int i, mapping_cnt, spec_offset = 0, type = 0;
  char *bg_def = NULL, *p = NULL;

  struct conf_mapping_t {
    char *name;
    int   id;
  } conf_mapping[] = {
    { "img-stretched:",           BG_STRETCHED_PXM  },
    { "img-tiled:",               BG_TILED_PXM      },
    { "img-centered:",            BG_CENTERED_PXM   },
    { "col-solid:",               BG_SOLID          },
    { "col-gradient-vertical:",   BG_GRADIENT_VERT  },
    { "col-gradient-horizontal:", BG_GRADIENT_HORIZ },
  };

  mapping_cnt = (sizeof(conf_mapping)/sizeof(struct conf_mapping_t));

  if (mb->bg != NULL) mbdesktop_bg_free_config(mb);

  mb->bg = malloc(sizeof(MBDesktopBG));
  memset(mb->bg, 0, sizeof(mb->bg));

  if (spec == NULL)
    {
      /* XXX we should probably check theme.desktop too for a bg_def */

      if (!mbdesktop_bg_get_from_theme(mb))
	{
	  bg_def = "#11b2d5"; 	/* Defualt col - red on error */
	  type   = BG_SOLID;
	}
    }
  else
    {
      for (i=0; i<mapping_cnt; i++)
	{
	  spec_offset = strlen(conf_mapping[i].name);
	  if (spec_offset < strlen(spec)
	      && !strncmp(conf_mapping[i].name, spec, spec_offset))
	    {
	      type = conf_mapping[i].id;
	      break;
	    }
	}

      if (!type)
	{
	  /* Assume we've just been passed an image filename */
	  mb->bg->type = BG_STRETCHED_PXM;
	  mb->bg->data.filename = strdup(spec);
	  return True;
	} else bg_def = spec + spec_offset;
    }


  mb->bg->type = type;

  switch(type)
    {
    case BG_SOLID:
      XParseColor(mb->dpy, DefaultColormap(mb->dpy, mb->scr), 
		  bg_def, &tmpxcol);
      mb->bg->data.cols[0] = tmpxcol.red   >> 8;
      mb->bg->data.cols[1] = tmpxcol.green >> 8;
      mb->bg->data.cols[2] = tmpxcol.blue  >> 8;
      break;
    case BG_TILED_PXM:
    case BG_STRETCHED_PXM:
    case BG_CENTERED_PXM:
      mb->bg->data.filename = strdup(bg_def);
      break;
    case BG_GRADIENT_HORIZ:
    case BG_GRADIENT_VERT:
      p = bg_def;
      while(*p != ',' && *p != '\0') p++;
      if (*p == '\0')
	{
	  mbdesktop_bg_free_config(mb);
	  return False; 	/* XXX need to reset on fail */
	}
      *p = '\0';
      XParseColor(mb->dpy, DefaultColormap(mb->dpy, mb->scr), 
		  bg_def, &tmpxcol);
      mb->bg->data.gcols[0] = (tmpxcol.red   >> 8);
      mb->bg->data.gcols[2] = (tmpxcol.green >> 8);
      mb->bg->data.gcols[4] = (tmpxcol.blue  >> 8);
      p++;
      XParseColor(mb->dpy, DefaultColormap(mb->dpy, mb->scr), 
		  p, &tmpxcol);
      mb->bg->data.gcols[1] = (tmpxcol.red   >> 8);
      mb->bg->data.gcols[3] = (tmpxcol.green >> 8);
      mb->bg->data.gcols[5] = (tmpxcol.blue  >> 8);
      break;
    }    

  return True;
}

MBDesktop *
mbdesktop_init(int argc, char **argv)
{
  MBDesktop *mb;
  XColor tmpxcol;
#ifdef USE_XFT
  XRenderColor colortmp;
#endif
  XGCValues gv;

  char *font_def       = FONT_DESC;
  char *font_title_def = FONT_TITLE_DESC;
  char *font_col_def   = FONT_COL;

  int i;

  CARD32 *win_top_level_icon_data = NULL;
  MBPixbufImage *win_top_level_img_icon = NULL;

  Atom window_type_atom, window_type_desktop_atom, atom_net_wm_icon;
  char *display_name = (char *)getenv("DISPLAY");  

#ifdef USE_DNOTIFY
  struct sigaction act;
  int fd;
#endif

  signal(SIGCHLD, SIG_IGN);

  mb = malloc(sizeof(MBDesktop));
  memset(mb, 0, sizeof(MBDesktop));

  mb->dd_dir          = DD_DIR;
  mb->pixmaps_dir     = PIXMAP_PATH;
  mb->icon_size       = 0;
  mb->dd_dir_mtime    = 0;
#ifdef USE_XFT
  mb->xftdraw         = NULL;
#endif

  mb->bg_def        = NULL;



  for (i = 1; i < argc; i++) {
    if (!strcmp ("-display", argv[i]) || !strcmp ("-d", argv[i])) {
      if (++i>=argc) usage (argv[0]);
      display_name = argv[i];
      continue;
    }
    if (!strcmp ("--bg", argv[i])) {
      if (++i>=argc) usage (argv[0]);
      mb->bg_def = strdup(argv[i]);
      continue;
    }
    if (!strcmp("--help", argv[i]) || !strcmp("-h", argv[i])) {
      usage(argv[0]);
    }
    if (!strcmp ("--desktopdir", argv[i])) {
      if (++i>=argc) usage (argv[0]);
      mb->dd_dir = argv[i];
      continue;
    }
    if (!strcmp ("--pixmapsdir", argv[i])) {
      if (++i>=argc) usage (argv[0]);
      mb->pixmaps_dir = argv[i];
      continue;
    }
    if (!strcmp ("--iconsize", argv[i])) {
      if (++i>=argc) usage (argv[0]);
      mb->icon_size = atoi(argv[i]);
      if (!mb->icon_size) usage(argv[0]);
      continue;
    }
    if (!strcmp ("--font", argv[i])) {
      if (++i>=argc) usage (argv[0]);
      font_def = argv[i];
      continue;
    }
    if (!strcmp ("--titlefont", argv[i])) {
      if (++i>=argc) usage (argv[0]);
      font_title_def = argv[i];
      continue;
    }
    if (!strcmp ("--fontcol", argv[i])) {
      if (++i>=argc) usage (argv[0]);
      font_col_def = argv[i];
      continue;
    }
    if (!strcmp ("--folderimg", argv[i])) {
      if (++i>=argc) usage (argv[0]);
      mb->folder_img_path = argv[i];
      continue;
    }
    usage(argv[0]);
  }

  if ((mb->dpy = XOpenDisplay(display_name)) == NULL)
    {
      fprintf(stderr, "mbdesktop: unable to connect to display\n");
      exit(1);
    }


  mb->scr = DefaultScreen(mb->dpy);
  mb->root = RootWindow(mb->dpy, mb->scr); 
  mb->pixbuf = mb_pixbuf_new(mb->dpy, mb->scr);
  mb->root_pxm = None;

  mb->ddfolders = NULL;

  /* Dimentions */
  mb->desktop_width  = DisplayWidth(mb->dpy, mb->scr);
  mb->desktop_height = DisplayHeight(mb->dpy, mb->scr);

  if (mb->icon_size == 0)
    {
      if (mb->desktop_width > 320)
	mb->icon_size = 48;
      else
	mb->icon_size = 32;
    }

  if (mbdesktop_get_workarea(mb, &mb->workarea_x, &mb->workarea_y, 
			     &mb->workarea_width, 
			     &mb->workarea_height ) == False )
    {
      mb->workarea_x = 0;
      mb->workarea_y = 0;
      mb->workarea_width = DisplayWidth(mb->dpy, mb->scr);
      mb->workarea_height = DisplayHeight(mb->dpy, mb->scr);
    }

  mb->had_kbd_input = False;
  mb->theme_name    = NULL;
  mb->bg_img        = NULL;
  mb->bg = NULL;

  mb->top_head_item = NULL;

#ifdef USE_XSETTINGS

  mb->xsettings_client = xsettings_client_new(mb->dpy, mb->scr,
					      mbdesktop_xsettings_notify_cb,
					      NULL, (void *)mb );


  if (mb->xsettings_client == NULL)
    fprintf(stderr,"mbmenu: XSETTINGS manager not found\n");

#endif

  mb->folder_img_path = NULL;

  mbdesktop_bg_parse_spec(mb, mb->bg_def);
  
#ifdef USE_XFT
  /* xft text */
  if ((mb->xftfont = XftFontOpenName(mb->dpy, mb->scr, font_def)) == NULL)
    { fprintf(stderr, "%s: Cant open font %s\n", argv[0], font_def); exit(0); }
  
  if ((mb->xfttitlefont = XftFontOpenName(mb->dpy, mb->scr, 
					  font_title_def)) == NULL)
    { 
      fprintf(stderr, "%s: Cant open font %s\n", argv[0], font_title_def); 
      exit(0); 
    }

  mb->title_offset = mb->xfttitlefont->ascent + mb->xfttitlefont->descent + 4;
  mb->item_height  = mb->icon_size + ( 2.5 * (mb->xftfont->ascent 
					      + mb->xftfont->descent) );

#else
  if ((mb->font = XLoadQueryFont(mb->dpy, font_def)) == NULL)
    { fprintf(stderr, "%s: Cant open font %s\n", argv[0], font_def); exit(0); }
  if ((mb->titlefont = XLoadQueryFont(mb->dpy, font_def)) == NULL)
    { 
      fprintf(stderr, "%s: Cant open font %s\n", argv[0], font_def); 
      exit(0); 
    }

  mb->title_offset = mb->titlefont->ascent + mb->titlefont->descent + 4;
  mb->item_height  = mb->icon_size + ( 2.5 * (mb->font->ascent 
					      + mb->font->descent) );

#endif

  mb->item_width   = mb->icon_size + ( mb->icon_size / 2 );
 
  if (!XParseColor(mb->dpy, DefaultColormap(mb->dpy, mb->scr), 
		   font_col_def, &tmpxcol))
    {
      fprintf(stderr, "%s: Failed to parse %s\n", argv[0], font_col_def); 
      exit(1);
    }

#ifdef USE_XFT
   colortmp.red   = tmpxcol.red;
   colortmp.green = tmpxcol.green;
   colortmp.blue  = tmpxcol.blue;
   colortmp.alpha =  0xffff; 
   XftColorAllocValue(mb->dpy, mb->pixbuf->vis, mb->pixbuf->root_cmap,
		      &colortmp, &mb->xftcol);
#endif

   /* GC's */
   gv.function = GXcopy;
   gv.line_width = 1;
   gv.line_style = LineOnOffDash; 
   gv.fill_style = FillOpaqueStippled;


   mb->gc = XCreateGC(mb->dpy, mb->root,
		      GCFunction|GCLineWidth|GCFillStyle|GCLineStyle, &gv);

#ifndef USE_XFT
   XSetFont(mb->dpy, mb->gc, mb->font->fid);
#endif

   gv.function = GXinvert;
   gv.subwindow_mode = IncludeInferiors;

   mb->invert_gc = XCreateGC(mb->dpy, mb->root,
			     GCFunction|GCSubwindowMode|GCLineWidth
			     |GCFillStyle|GCLineStyle, &gv);

   /* Set up background type */

#ifdef USE_PNG
#define UP_IMG   "mbup.png"
#define DOWN_IMG "mbdown.png"
#else
#define UP_IMG   "mbup.xpm"
#define DOWN_IMG "mbdown.xpm"
#endif


  /* scroll buttons */
  if ((mb->img_scroll_up = mb_pixbuf_img_new_from_file(mb->pixbuf,
						       mb_dot_desktop_icon_get_full_path (mb->theme_name, 16, UP_IMG)))
      == NULL)
    fprintf(stderr, "mbdesktop: warning failed to load %s\n", UP_IMG);

  if ((mb->img_scroll_down = mb_pixbuf_img_new_from_file(mb->pixbuf, 
							 mb_dot_desktop_icon_get_full_path (mb->theme_name, 16, DOWN_IMG)))
      == NULL)
    fprintf(stderr, "mbdesktop: warning failed to load %s\n", DOWN_IMG);

  /* ewmh hints */
  window_type_atom =
    XInternAtom(mb->dpy, "_NET_WM_WINDOW_TYPE" , False);
  window_type_desktop_atom =
    XInternAtom(mb->dpy, "_NET_WM_WINDOW_TYPE_DESKTOP",False);

  mb->win_top_level = XCreateWindow(mb->dpy, mb->root, 
		                    0, 0,  
				    mb->desktop_width, 
				    mb->desktop_height, 0, 
				    CopyFromParent,  
				    CopyFromParent, 
				    mb->pixbuf->vis,
				    0, NULL);


  XStoreName(mb->dpy, mb->win_top_level, "Desktop" );

  XChangeProperty(mb->dpy, mb->win_top_level, window_type_atom, XA_ATOM, 32, 
		  PropModeReplace, 
		  (unsigned char *) &window_type_desktop_atom, 1);

#ifdef USE_PNG
#define MBDESKTOP_ICON "mbdesktop.png"
#else
#define MBDESKTOP_ICON "mbdesktop.xpm"
#endif

  if ((win_top_level_img_icon 
       = mb_pixbuf_img_new_from_file(mb->pixbuf, 
				     PIXMAPSDIR MBDESKTOP_ICON)) != NULL)
    {
      win_top_level_icon_data 
	= malloc(sizeof(CARD32)* ((win_top_level_img_icon->width * 
				   win_top_level_img_icon->height)+2));

      if (win_top_level_icon_data)
	{
	  int i = 0, idx = 0;
	  atom_net_wm_icon = XInternAtom(mb->dpy, "_NET_WM_ICON",False);
	  win_top_level_icon_data[0] = win_top_level_img_icon->width;
	  win_top_level_icon_data[1] = win_top_level_img_icon->height;
	  
	  for( i=2; 
	       i < (win_top_level_img_icon->width * 
		    win_top_level_img_icon->height)+2; 
	       i++)
	    {
	      idx = (i-2)*4;
	      //r = win_top_level_img_icon->rgba[idx] >> 16 ;

	      win_top_level_icon_data[i] = 
		win_top_level_img_icon->rgba[idx] << 16 
		|  win_top_level_img_icon->rgba[idx+1] << 8 
		|  win_top_level_img_icon->rgba[idx+2] 
		|  win_top_level_img_icon->rgba[idx+3] << 24 ;  
	    }
	  
	  XChangeProperty(mb->dpy, mb->win_top_level, 
			  atom_net_wm_icon ,
			  XA_CARDINAL, 32, PropModeReplace,
			  (unsigned char *)win_top_level_icon_data, 
			  (win_top_level_img_icon->width * 
			   win_top_level_img_icon->height )+2);
	  
	  free(win_top_level_icon_data);
	}
    }


#ifdef USE_DNOTIFY

  _Gmb = mb; 			/* arg, global mb for sig callback */

  act.sa_sigaction = dnotify_handler;
  sigemptyset(&act.sa_mask);
  act.sa_flags = SA_SIGINFO;
  sigaction(SIGRTMIN, &act, NULL);
  
  fd = open(mb->dd_dir, O_RDONLY);
  fcntl(fd, F_SETSIG, SIGRTMIN);
  fcntl(fd, F_NOTIFY, DN_RENAME|DN_MODIFY|DN_CREATE|DN_DELETE|DN_MULTISHOT);

#endif 

#ifdef USE_LIBSN
  mb->sn_dpy = sn_display_new (mb->dpy, NULL, NULL);

#endif 

  XMapWindow(mb->dpy, mb->win_top_level);

  XSelectInput(mb->dpy, mb->win_top_level, 
	       ExposureMask | ButtonPressMask | ButtonReleaseMask |
	       KeyPress | KeyRelease | StructureNotifyMask |
	       FocusChangeMask );

  XSelectInput(mb->dpy, mb->root, PropertyChangeMask );


  return mb;
}

void
mbdesktop_main(MBDesktop *mb)
{
  XEvent ev;
  Atom workarea_atom = XInternAtom(mb->dpy, "_NET_WORKAREA",False);
  Atom atom_client_list = XInternAtom(mb->dpy, "_NET_CLIENT_LIST",False);

#ifdef USE_DNOTIFY
  sigset_t block_sigset;
  sigemptyset(&block_sigset);
  sigaddset(&block_sigset, SIGRTMIN);
#endif


  while (1)
    {
	  XNextEvent(mb->dpy, &ev);
#ifdef USE_DNOTIFY 		/* Block dnotify signal */
	  sigprocmask(SIG_BLOCK, &block_sigset, NULL); 
#endif

#ifdef USE_XSETTINGS
	  if (mb->xsettings_client != NULL)
	    xsettings_client_process_event(mb->xsettings_client, &ev);
#endif
	  switch (ev.type) 
	    {
	    case FocusIn:
	      mb->have_focus = True;
	      mbdesktop_view_paint(mb, True);
	      break;

	    case FocusOut:
	      mbdesktop_view_paint(mb, True);
	      mb->have_focus = False;
	      break;

	    case Expose:
	      mbdesktop_view_paint(mb, True);
	      break;
	    case PropertyNotify:
	      if (ev.xproperty.atom == workarea_atom)
		{
		  int wx, wy, ww, wh;
		  if (mbdesktop_get_workarea(mb, &wx, &wy, &ww, &wh))
		    {
		      if (mb->workarea_x != wx 
			  || mb->workarea_y != wy
			  || mb->workarea_width != ww 
			  || mb->workarea_height != wh)
			mbdesktop_view_configure(mb);
		    }
		}
	      else if (ev.xproperty.atom == atom_client_list
		       && mb->current_head_item->item_parent
		       && mb->current_head_item->item_parent->type 
		       == ITEM_TYPE_TASK_FOLDER)
		{
		  item_tasks_populate_cb((void *)mb, 
					 (void *)mb->current_head_item->item_parent);
		  mbdesktop_view_paint(mb, False);
		}
	      break;
	    case ConfigureNotify:
	      
	      if ( ev.xconfigure.width != mb->desktop_width
		   || ev.xconfigure.height != mb->desktop_height)
		{
		  mb->desktop_width = ev.xconfigure.width;
		  mb->desktop_height = ev.xconfigure.height;
		  mbdesktop_view_configure(mb);
		}
	      break;
	    case ButtonPress:
	      handle_button_event(mb, &ev.xbutton);
	      break;
	    case KeyPress:
	      handle_key_event(mb, &ev.xkey);
	      break;
	    }

#ifdef USE_DNOTIFY 		/* Unblock dnotify signal */
	  sigprocmask(SIG_UNBLOCK, &block_sigset, NULL); 
#endif

    }

}

int 
main(int argc, char **argv)
{
  MBDesktop *mb;

#ifdef PANGO
  g_type_init();
#endif

  mb = mbdesktop_init(argc, argv);

  mbdesktop_view_init_bg(mb);

  mbdesktop_init_folder_items(mb);
  mbdesktop_init_items(mb);

  mbdesktop_main(mb);

  return 0;
}
