/* matchbox - a lightweight window manager

   Copyright 2002 Matthew Allum

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/

#include "client_common.h"

/*
   Common functions for use by all client types
*/

void
client_set_state(Client *c, int state)
{
   CARD32 data[2];

   data[0] = state;
   data[1] = None;
   
   XChangeProperty(c->wm->dpy, c->window, c->wm->wm_state, c->wm->wm_state,
		   32, PropModeReplace, (unsigned char *)data, 2);
}

long
client_get_state(Client *c)
{
    Atom real_type; int real_format;
    unsigned long items_read, items_left;
    long *data = NULL, state = WithdrawnState;

    if (XGetWindowProperty(c->wm->dpy, c->window,
			   c->wm->wm_state, 0L, 2L, False,
			   c->wm->wm_state, &real_type, &real_format,
			   &items_read, &items_left,
			   (unsigned char **) &data) == Success
	&& items_read) {
      state = *data;
    }

    if (data)
      XFree(data);

    return state;
}


void
client_deliver_config(Client *c)
{
   XConfigureEvent ce;
   
   ce.type = ConfigureNotify;
   ce.event = c->window;
   ce.window = c->window;
   
   ce.x = c->x;
   ce.y = c->y;
   
   ce.width  = c->width;
   ce.height = c->height;
   ce.border_width = 0;
   ce.above = None;
   ce.override_redirect = 0;
   
   XSendEvent(c->wm->dpy, c->window, False,
	      StructureNotifyMask, (XEvent *)&ce);
}

void
client_deleiver_wm_protocol(Client *c, Atom delivery)
{
    XEvent e;
    e.type = ClientMessage;
    e.xclient.window = c->window;
    e.xclient.message_type = c->wm->wm_protos;
    e.xclient.format = 32;
    e.xclient.data.l[0] = delivery;
    e.xclient.data.l[1] = CurrentTime;
    XSendEvent(c->wm->dpy, c->window, False, NoEventMask, &e);
}

void
client_deliver_delete(Client *c)
{
    int i, n, found = 0;
    Atom *protocols;
    
    if (XGetWMProtocols(c->wm->dpy, c->window, &protocols, &n)) {
        for (i=0; i<n; i++)
	   if (protocols[i] == c->wm->wm_delete) found++;
        XFree(protocols);
    }
    if (found)
      client_deleiver_wm_protocol(c, c->wm->wm_delete);
    else 
      XKillClient(c->wm->dpy, c->window);
}

/* ----- new bits ---------*/

int
client_want_focus(Client *c)
{
   /* TODO: check _WM protocols too ? */
   int ret = 1;
   XWMHints *hints = XGetWMHints(c->wm->dpy, c->window);
   if (hints != NULL)
     if ((hints->flags & InputHint) && (hints->input == False)) ret = 0;
   XFree(hints);

   if (ret) ewmh_set_active(c->wm);
   return ret;
}

Client*
client_get_next(Client* c, Client_type wanted)
{
   client *p;
   for (p=c->next; p != c; p = p->next)
      if (p->type == wanted && p->mapped) return p;
   return c;
}

Client*
client_get_prev(Client* c, Client_type wanted)
{
   client *p;
   for (p=c->prev; p != c; p = p->prev)
      if (p->type == wanted && p->mapped) return p;
   return c;
}

void
client_init_backing(Client* c, int width, int height)
{
  if (c->backing != None) XFreePixmap(c->wm->dpy, c->backing);

  c->backing = XCreatePixmap(c->wm->dpy, c->wm->root, width, height ,
			     DefaultDepth(c->wm->dpy, c->wm->screen));

   /* todo check for error if pixmap cant be created */
#ifdef USE_XFT

  if (c->xftdraw != NULL) XftDrawDestroy(c->xftdraw);

   c->xftdraw = XftDrawCreate(c->wm->dpy, (Drawable) c->backing, 
			      DefaultVisual(c->wm->dpy, c->wm->screen),
			      DefaultColormap(c->wm->dpy, c->wm->screen));
#endif
   
}

void 
client_init_backing_mask(Client *c, int width, int height)
{
  GC shape_gc;
  if (c->backing_mask != None)
    XFreePixmap(c->wm->dpy, c->backing_mask);
      
  c->backing_mask = XCreatePixmap(c->wm->dpy, c->wm->root, width, height, 1);

  shape_gc = XCreateGC( c->wm->dpy, c->backing_mask, 0, 0 );

  XSetForeground(c->wm->dpy, shape_gc, 
		 WhitePixel( c->wm->dpy, c->wm->screen ));

  XFillRectangle(c->wm->dpy, c->backing_mask,shape_gc, 0, 0, width, height);

  XFreeGC(c->wm->dpy, shape_gc);
}


MBClientButton *
client_button_new(Client *c, Window win_parent, 
		  int x, int y, int w, int h,
		  Bool want_inputonly, void *data )
{
  XSetWindowAttributes attr;
  int class = CopyFromParent;

  MBClientButton *b = malloc(sizeof(MBClientButton));
  memset(b, 0, sizeof(MBClientButton));
  
  attr.override_redirect = True;
  attr.event_mask = ExposureMask;
  
  if (want_inputonly ) class = InputOnly;	      
  
  b->x = x; b->y = y; b->w = w; b->h = h; b->data = data;
  
  b->win = XCreateWindow(c->wm->dpy, win_parent, x, y, w, h, 0,
			 CopyFromParent, class, CopyFromParent,
			 CWOverrideRedirect|CWEventMask, &attr);

  XMapWindow(c->wm->dpy, b->win);
  return b;
}

void
client_button_remove(Client *c, int button_action)
{
  struct list_item *l = c->buttons;
  MBClientButton *b = NULL;

  while (l != NULL)
    {
      if (l->id == button_action)
	{
	  b = (MBClientButton *)l->data;
	  dbg("%s() destroying a button ( %li ) for %s\n", __func__, 
	      b->win, c->name); 
	  XDestroyWindow(c->wm->dpy, b->win);
	  b->win = None;
	}
      l = l->next;
    }
}

void
client_buttons_delete_all(Client *c)
{
  struct list_item *l = c->buttons, *p = NULL;
  MBClientButton *b = NULL;
  
  while (l != NULL)
    {
      b = (MBClientButton *)l->data;
      dbg("%s() destroying a button\n", __func__); 
      if (b->win != None)
	XDestroyWindow(c->wm->dpy, b->win);
      free(b);
      p = l->next;
      free(l);
      l = p;
    }
  c->buttons = NULL;
}

MBClientButton *
client_get_button_from_event(Client *c, XButtonEvent *e)
{
  struct list_item *l = c->buttons;
  MBClientButton *b = NULL;

  while (l != NULL)
    {
      b = (MBClientButton *)l->data;
      if (b->win == e->subwindow)
	{
	  return b;
	}
      l = l->next;
    }
  return NULL;
}


struct list_item *
client_get_button_list_item_from_event(Client *c, XButtonEvent *e)
{
  struct list_item *l = c->buttons;
  MBClientButton *b = NULL;

  while (l != NULL)
    {
      b = (MBClientButton *)l->data;
      if (b->win == e->subwindow)
	{
	  return l;
	}
      l = l->next;
    }
  return NULL;
}

int
client_button_do_ops(Client *c, XButtonEvent *e, int frame_type, int w, int h)
{
  int button_action;
  struct list_item* button_item = NULL;
  XEvent ev;

  if ((button_item = client_get_button_list_item_from_event(c, e)) != NULL
       && button_item->id != -1 )
   {

     if (XGrabPointer(c->wm->dpy, e->subwindow, False,
		      ButtonPressMask|ButtonReleaseMask|
		      PointerMotionMask|EnterWindowMask|LeaveWindowMask,
		      GrabModeAsync,
		      GrabModeAsync, None, c->wm->curs, CurrentTime)
	 == GrabSuccess)
       {
	 Bool canceled = False;
	 button_action = button_item->id;

	 theme_frame_button_paint(c->wm->mbtheme, c, button_action,
				  ACTIVE, frame_type, w, h);
	 for (;;) 
	 {
	    XMaskEvent(c->wm->dpy,
		       ButtonPressMask|ButtonReleaseMask|
		       PointerMotionMask|EnterWindowMask|LeaveWindowMask,
		    &ev);
	    
	    switch (ev.type)
	    {
	       case MotionNotify:
		  break;
	       case EnterNotify:
		  theme_frame_button_paint(c->wm->mbtheme, c, button_action,
					   ACTIVE, frame_type, w, h );
		  canceled = False;
		  break;
	       case LeaveNotify:
		  theme_frame_button_paint(c->wm->mbtheme, c, button_action,
					   INACTIVE, frame_type, w, h );
		  canceled = True;
		  break;
	       case ButtonRelease:
		  theme_frame_button_paint(c->wm->mbtheme, c, button_action,
					   INACTIVE, frame_type, w, h );
		  XUngrabPointer(c->wm->dpy, CurrentTime);
		  if (!canceled)
		  {
		    return button_action;
#if 0		    
		     if (frm->buttons[b]->wants_dbl_click)
		     {
			if (c->wm->flags & DBL_CLICK_FLAG)
			  {
			    if ( b == ACTION_MENU_EXTRA) /* HACK */
			      b = ACTION_MENU;
			    else if (b == ACTION_MAX_EXTRA)
			      b = ACTION_MAX;
			    else if (b == ACTION_MIN_EXTRA)
			      b = ACTION_MIN;
			    return b;
			  }
			else
			   return -1;
		     } else {
		       if ( b == ACTION_MENU_EXTRA) /* HACK */
			 b = ACTION_MENU;
		       else if (b == ACTION_MAX_EXTRA)
			 b = ACTION_MAX;
		       else if (b == ACTION_MIN_EXTRA)
			 b = ACTION_MIN;
		       return b;
		     }
#endif		     
		  }
		  else
		     return -1;  /* cancelled  */
	    }
	 }
      }
   }
   return 0;
}


