/* matchbox - a lightweight window manager

   Copyright 2002 Matthew Allum

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/


#include "dialog_client.h"

static void dialog_client_check_for_state_hints(Client *c);
static void dialog_client_drag(Client *c);
static void _get_mouse_position(Wm *w, int *x, int *y);
static void _draw_outline(Client *c, int x, int y, int w, int h);

Client*
dialog_client_new(Wm *w, Window win, Client *trans)
{

   client *c = base_client_new(w, win); 
   c->type = dialog;
   
   c->reparent     = &dialog_client_reparent;
   c->move_resize  = &dialog_client_move_resize;
   c->hide         = &dialog_client_hide;
   
   c->configure    = &dialog_client_configure;
   c->button_press = &dialog_client_button_press;
   c->redraw       = &dialog_client_redraw;
   c->show         = &dialog_client_show;
   c->destroy      = &dialog_client_destroy;
   c->get_coverage = &dialog_client_get_coverage;
   c->trans = trans;

   dialog_client_check_for_state_hints(c);

   return c;
}

static void
dialog_client_check_for_state_hints(Client *c)
{
  if (ewmh_state_check(c, c->wm->atoms[WINDOW_STATE_MODAL]))
    {
      dbg("%s() got modal hint, setting flag\n", __func__);
      c->flags ^= CLIENT_IS_MODAL_FLAG;
    }
}


void
dialog_client_get_coverage(Client *c, int *x, int *y, int *w, int *h)
{
   int frm_size = dialog_client_title_height(c);
   *x = c->x;
   *y = c->y - frm_size;
   *w = c->width;
   *h = c->height + frm_size;
}

void
dialog_client_move_resize(Client *c)
{
   int frm_size     = dialog_client_title_height(c);
   int offset_south = theme_frame_defined_height_get(c->wm->mbtheme, 
						     FRAME_DIALOG_SOUTH);
   int offset_east  = theme_frame_defined_width_get(c->wm->mbtheme, 
						    FRAME_DIALOG_EAST );
   int offset_west  = theme_frame_defined_width_get(c->wm->mbtheme, 
						    FRAME_DIALOG_WEST );

   base_client_move_resize(c);

   XMoveResizeWindow(c->wm->dpy, c->window, 
		     offset_west,
		     frm_size,
		     c->width, c->height);

   if (c->wm->config->dialog_shade && (c->flags & CLIENT_IS_MODAL_FLAG))
     {
       XMoveResizeWindow(c->wm->dpy, 
			 c->title_frame, 
			 c->x - offset_west, 
			 c->y - frm_size, 
			 c->width + offset_east + offset_west,
			 c->height + frm_size + offset_south );
       XMoveWindow(c->wm->dpy, 
		   c->window, 
		   c->x, 
		   c->y);

     }
   else
     {
       XMoveResizeWindow(c->wm->dpy, 
			 c->frame, 
			 c->x - offset_west, 
			 c->y - frm_size, 
			 c->width + offset_west + offset_east,
			 c->height + frm_size + offset_south
			 );
       XResizeWindow(c->wm->dpy, 
			 c->title_frame, 
			 c->width + offset_east + offset_west,
			 c->height + frm_size + offset_south );
     }
}

void
dialog_client_hide(Client *c)
{
  dbg("%s() called for %s\n", __func__, c->name);
  XLowerWindow(c->wm->dpy, c->frame);
}

int
dialog_client_title_height(Client *c)
{
   if (c->flags & CLIENT_TITLE_HIDDEN_FLAG)
      return 0;
   return theme_frame_defined_height_get(c->wm->mbtheme, FRAME_DIALOG);
}

void
dialog_client_show(Client *c)
{
  dbg("%s() called for %s\n", __func__, c->name);
  XFlush(c->wm->dpy);

  client_set_state(c, NormalState); 
  XMapSubwindows(c->wm->dpy, c->frame);
  XMapRaised(c->wm->dpy, c->frame);
  
  if (client_want_focus(c))
    {
      XSetInputFocus(c->wm->dpy, c->window,
		     RevertToPointerRoot, CurrentTime);
      c->wm->focused_client = c;
    }
  c->mapped = True;
}

void
dialog_client_reparent(Client *c)
{
   XSetWindowAttributes attr;

   int offset_north = dialog_client_title_height(c);
   int offset_south = theme_frame_defined_height_get(c->wm->mbtheme, 
						     FRAME_DIALOG_SOUTH);
   int offset_east  = theme_frame_defined_width_get(c->wm->mbtheme, 
						    FRAME_DIALOG_EAST );
   int offset_west  = theme_frame_defined_width_get(c->wm->mbtheme, 
						    FRAME_DIALOG_WEST );

   if (c->flags & CLIENT_TITLE_HIDDEN_FLAG)
   {
      c->frame = c->title_frame = c->window;
      return;
   }

   attr.override_redirect = True;
   attr.background_pixel  = BlackPixel(c->wm->dpy, c->wm->screen);
   attr.event_mask        = ChildMask|ButtonPressMask|ExposureMask;

   if (c->wm->config->dialog_shade && (c->flags & CLIENT_IS_MODAL_FLAG))
     {
        wm_lowlight(c->wm, c);
     }
   else
     {
       c->frame =
	 XCreateWindow(c->wm->dpy, 
		       c->wm->root, 
		       0, 0,
		       c->width + offset_east + offset_west, 
		       c->height + offset_north + offset_south, 
		       0,
		       CopyFromParent, 
		       CopyFromParent, 
		       CopyFromParent,
		       CWOverrideRedirect|CWEventMask|CWBackPixel,
		       &attr);
     }

   attr.background_pixel = BlackPixel(c->wm->dpy, c->wm->screen);
   
   c->title_frame =
     XCreateWindow(c->wm->dpy, 
		   c->frame, 
		   0, 0, 
		   c->width + offset_east + offset_west, 
		   c->height + offset_north + offset_south, 
		   0,
		   CopyFromParent, 
		   CopyFromParent, 
		   CopyFromParent,
		   CWOverrideRedirect|CWBackPixel|CWEventMask, 
		   &attr);
   
   XSetWindowBorderWidth(c->wm->dpy, c->window, 0);

   XAddToSaveSet(c->wm->dpy, c->window);

   XSelectInput(c->wm->dpy, c->window,
		ButtonPressMask|ColormapChangeMask|PropertyChangeMask);
   
   XReparentWindow(c->wm->dpy, 
		   c->window, 
		   c->frame,
		   offset_west, 
		   offset_north);

}


void
dialog_client_configure(Client *c)
{
  int side_frame_width = 0, bottom_frame_width = 0;
  int max_w = 0, max_h = 0, toolbar_y = 0; 


  if (!(c->flags & CLIENT_TITLE_HIDDEN_FLAG))
    {
      side_frame_width = 
	theme_frame_defined_width_get(c->wm->mbtheme, FRAME_DIALOG_EAST)
	+ theme_frame_defined_width_get(c->wm->mbtheme, FRAME_DIALOG_WEST);

      bottom_frame_width =
	theme_frame_defined_height_get(c->wm->mbtheme, FRAME_DIALOG_SOUTH);
    }

  toolbar_y = c->wm->dpy_height - wm_get_offsets_size(c->wm, SOUTH, NULL,True);

  max_w = c->wm->dpy_width - side_frame_width - 16;

   /* make sure the dialog doesn't cover any toolbars */
  max_h = toolbar_y - bottom_frame_width - 32;

  dbg("dialog wants %i x %i, %i x %i", c->x, c->y, c->width, c->height);
  
  if (c->width > max_w) c->width   = max_w;

   if (c->height > max_h) 
     {
       c->height = max_h;
       c->flags |=  CLIENT_SHRUNK_FOR_TB_FLAG;
       c->y = 16;
     }
   
   if (c->y != 16  /* move dialog up above any toolbars  */
       && ( ( c->y + c->height + bottom_frame_width ) > toolbar_y) )
     {
       c->y -= (( c->y + c->height + bottom_frame_width ) - toolbar_y );
     }
   
   /* only repos if at 0,0 or off screen  */
   if ( !c->x || (c->x + c->width) > c->wm->dpy_width || (c->x < 0))
     c->x = (c->wm->dpy_width  - c->width)/2;
   
   if ( !c->y || (c->y + c->height) > c->wm->dpy_height)
     c->y = (c->wm->dpy_height - c->height)/2;
}

void
dialog_client_redraw(Client *c, Bool use_cache)
{
  Bool is_shaped = False;

  int offset_north = dialog_client_title_height(c);
  int offset_south = theme_frame_defined_height_get(c->wm->mbtheme, 
						    FRAME_DIALOG_SOUTH);
  int offset_east  = theme_frame_defined_width_get(c->wm->mbtheme, 
						   FRAME_DIALOG_EAST );
  int offset_west  = theme_frame_defined_width_get(c->wm->mbtheme, 
						   FRAME_DIALOG_WEST );

  int total_w = offset_east  + offset_west + c->width;
  int total_h = offset_north + offset_south + c->height;

  dbg("%s() c->width : %i , offset_east : %i, offset_west : %i\n",
      __func__, c->width, offset_east, offset_west );

  if (c->flags & CLIENT_TITLE_HIDDEN_FLAG) return;

   if (use_cache && c->backing != None) return;

  is_shaped = theme_frame_wants_shaped_window( c->wm->mbtheme, FRAME_DIALOG);
  
  if (c->backing == (Pixmap)NULL)
    client_init_backing(c, total_w, total_h);

  if (is_shaped) client_init_backing_mask(c, total_w, total_h);

  /* Should prevent some flicker */
  XSetForeground(c->wm->dpy, c->wm->mbtheme->gc, 
		 BlackPixel(c->wm->dpy, c->wm->screen));
  XFillRectangle(c->wm->dpy, c->backing, c->wm->mbtheme->gc, 
		 0, 0, total_w, total_h);

  theme_frame_paint(c->wm->mbtheme, c, FRAME_DIALOG, 
		    0, 0, total_w, offset_north); 
    
  theme_frame_paint(c->wm->mbtheme, c, FRAME_DIALOG_WEST, 
		    0, offset_north, offset_west, c->height); 
  
  theme_frame_paint(c->wm->mbtheme, c, FRAME_DIALOG_EAST, 
		    total_w - offset_east, offset_north, 
		    offset_east, c->height); 

  theme_frame_paint(c->wm->mbtheme, c, FRAME_DIALOG_SOUTH, 
		    0, total_h - offset_south, 
		    total_w, offset_south); 

  if (!(c->flags & CLIENT_BORDERS_ONLY_FLAG))
    {
      theme_frame_button_paint(c->wm->mbtheme, c, 
			       BUTTON_ACTION_CLOSE, 
			       INACTIVE, FRAME_DIALOG, 
			       total_w, offset_north);


      if (c->flags & CLIENT_ACCEPT_BUTTON_FLAG)
	theme_frame_button_paint(c->wm->mbtheme, c, BUTTON_ACTION_ACCEPT, 
				 INACTIVE, FRAME_DIALOG,total_w, offset_north);

      if (c->flags & CLIENT_HELP_BUTTON_FLAG)
	theme_frame_button_paint(c->wm->mbtheme, c, BUTTON_ACTION_HELP, 
				 INACTIVE, FRAME_DIALOG,total_w, offset_north);
    }

  /* XXXX ifdef HAVE_SHAPE */
  if (is_shaped)
    {
      if (c->wm->config->dialog_shade && (c->flags & CLIENT_IS_MODAL_FLAG)) 
	{
	  XShapeCombineMask( c->wm->dpy, c->title_frame, 
			     ShapeBounding, 0, 0, 
			     c->backing_mask, ShapeSet);
	}
      else
	{
	  XShapeCombineMask( c->wm->dpy, c->title_frame, 
			     ShapeBounding, 0, 0, 
			     c->backing_mask, ShapeSet);

	  XShapeCombineShape ( c->wm->dpy, 
			       c->frame,
			       ShapeBounding, 0, 0, 
			       c->title_frame,
			       ShapeBounding, ShapeSet);
	    /*
	      XShapeCombineMask( c->wm->dpy, c->frame, ShapeBounding, 0, 0, 
	      c->backing_mask, ShapeSet);
	    */
	}
    }

   XSetWindowBackgroundPixmap(c->wm->dpy, c->title_frame, c->backing);
   XClearWindow(c->wm->dpy, c->title_frame);

   XFlush(c->wm->dpy);
}

void
dialog_client_button_press(Client *c, XButtonEvent *e)
{
  int offset_north = dialog_client_title_height(c);
  int offset_west = theme_frame_defined_width_get(c->wm->mbtheme, 
						    FRAME_DIALOG_WEST);
  int offset_east  = theme_frame_defined_width_get(c->wm->mbtheme, 
						   FRAME_DIALOG_EAST );

  dbg("%s() c->width : %i , offset_east : %i, offset_west : %i\n",
      __func__, c->width, offset_east, offset_west );

   switch (client_button_do_ops(c, e, FRAME_DIALOG, 
				c->width + offset_east + offset_west, 
				offset_north))
   {
      case BUTTON_ACTION_CLOSE:
	 client_deliver_delete(c);
	 break;
      case BUTTON_ACTION_HELP:
	client_deleiver_wm_protocol(c, c->wm->atoms[_NET_WM_CONTEXT_HELP]);
	 break;
      case BUTTON_ACTION_ACCEPT:
	client_deleiver_wm_protocol(c, c->wm->atoms[_NET_WM_CONTEXT_ACCEPT]);
      case -1: 		 /* Cancelled  */
	 break;
      case 0:
	 dialog_client_drag(c);     /* Not on button */
	 break;
   }
}

static void
dialog_client_drag(Client *c) /* drag box */
{
  XEvent ev;
  int x1, y1;
  int old_cx = c->x;
  int old_cy = c->y;
    
  int frm_size     = dialog_client_title_height(c);
  int offset_south = theme_frame_defined_height_get(c->wm->mbtheme, 
						    FRAME_DIALOG_SOUTH);
  int offset_east  = theme_frame_defined_width_get(c->wm->mbtheme, 
						   FRAME_DIALOG_EAST );
  int offset_west  = theme_frame_defined_width_get(c->wm->mbtheme, 
						   FRAME_DIALOG_WEST );

  XMapRaised(c->wm->dpy, c->frame);

  if (XGrabPointer(c->wm->dpy, c->wm->root, False,
		   (ButtonPressMask|ButtonReleaseMask|PointerMotionMask),
		   GrabModeAsync,
		   GrabModeAsync, None, c->wm->curs_drag, CurrentTime)
      != GrabSuccess)
    return;
    
  XGrabServer(c->wm->dpy);

  _get_mouse_position(c->wm, &x1, &y1);

  _draw_outline(c, c->x - offset_west, c->y - frm_size,
		c->width + offset_west + offset_east,
		c->height + frm_size + offset_south);
    
  for (;;) 
    {
    XMaskEvent(c->wm->dpy, ButtonPressMask|ButtonReleaseMask|PointerMotionMask,
	       &ev);
       
    switch (ev.type) 
      {
    case MotionNotify:
      _draw_outline(c, c->x - offset_west, c->y - frm_size,
		    c->width + offset_west + offset_east,
		    c->height + frm_size + offset_south);

      c->x = old_cx + (ev.xmotion.x - x1);
      c->y = old_cy + (ev.xmotion.y - y1);

      _draw_outline(c, c->x - offset_west, c->y - frm_size,
		    c->width + offset_west + offset_east,
		    c->height + frm_size + offset_south);
      break;

    case ButtonRelease:
      dbg("drag, got release");
      _draw_outline(c, c->x - offset_west, c->y - frm_size,
		    c->width + offset_west + offset_east,
		    c->height + frm_size + offset_south);

      if (c->wm->config->dialog_shade && (c->flags & CLIENT_IS_MODAL_FLAG))
	{
	  XMoveResizeWindow(c->wm->dpy, 
			    c->title_frame, 
			    c->x - offset_west, 
			    c->y - frm_size, 
			    c->width + offset_east + offset_west,
			    c->height + frm_size + offset_south);
	  XMoveResizeWindow(c->wm->dpy, 
			    c->window, 
			    c->x, 
			    c->y, 
			    c->width,
			    c->height);
	} else {
	  XMoveWindow(c->wm->dpy, c->frame, c->x,
		      c->y - dialog_client_title_height(c));
	}
      XUngrabPointer(c->wm->dpy, CurrentTime);
      XUngrabServer(c->wm->dpy);
      return;
    }
  }
  client_deliver_config(c);
}

static void
_get_mouse_position(Wm *w, int *x, int *y)
{
    Window mouse_root, mouse_win;
    int win_x, win_y;
    unsigned int mask;

    XQueryPointer(w->dpy, w->root, &mouse_root, &mouse_win,
        x, y, &win_x, &win_y, &mask);
}

static void
_draw_outline(Client *c, int x, int y, int w, int h)
{
  XDrawRectangle(c->wm->dpy, c->wm->root, c->wm->mbtheme->band_gc, x, y, w, h);
}
 
void dialog_client_destroy(Client *c)
{
  /*   if (c->wm->main_client != NULL)
       c->wm->main_client->show(c->wm->main_client); */
   base_client_destroy(c);
   /*
   if (!(c->wm->flags & DESKTOP_RAISED_FLAG)) 
     wm_activate_client(c->wm->main_client);
   */
}
