/*
 *  minisys - tiny sys monitor
 *
 *  cpu reading code based on wmbubblemon
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>

#include "mbtray.h"
#include "mbpixbuf.h"

#define GRAPH_WIDTH  12
#define GRAPH_HEIGHT  9

static Display* dpy;
static Window win_panel, win_root;
static int screen;
static GC gc;
static MBPixbuf *pb;
static MBPixbufImage *img_icon = NULL, 
  *img_backing = NULL, *img_graph = NULL; 
static Pixmap pxm_backing = None;

static void paint( void );

struct {

   /* cpu data  */
   int loadIndex;
   int samples;
   u_int64_t *load, *total;

   /* memory data  */
   u_int64_t mem_used;
   u_int64_t mem_max;
   u_int64_t swap_used;
   u_int64_t swap_max;
   unsigned int swap_percent;  /* swap used, in percent */
   unsigned int mem_percent;   /* memory used, in percent */

} msd;   

/* returns current CPU load in percent, 0 to 100 */
int system_cpu(void)
{
    unsigned int cpuload;
    u_int64_t load, total, oload, ototal;
    u_int64_t ab, ac, ad, ae;
    int i;
    FILE *stat;

    stat = fopen("/proc/stat", "r");
    fscanf(stat, "%*s %Ld %Ld %Ld %Ld", &ab, &ac, &ad, &ae);
    fclose(stat);

    /* Find out the CPU load */
    /* user + sys = load
     * total = total */
    load = ab + ac + ad;	/* cpu.user + cpu.sys; */
    total = ab + ac + ad + ae;	/* cpu.total; */

    /* "i" is an index into a load history */
    i = msd.loadIndex;
    oload = msd.load[i];
    ototal = msd.total[i];

    msd.load[i] = load;
    msd.total[i] = total;
    msd.loadIndex = (i + 1) % msd.samples;

    if (ototal == 0)	
	cpuload = 0;
    else
	cpuload = (100 * (load - oload)) / (total - ototal);

    return cpuload;
}

int system_memory(void)
{
    u_int64_t my_mem_used, my_mem_max;
    u_int64_t my_swap_max;

    static int mem_delay = 0;
    FILE *mem;
    static u_int64_t aa, ab, ac, ad, ae, af, ag, ah;
    /* put this in permanent storage instead of stack */
    static char not_needed[2048];

    if (mem_delay-- <= 0) {
	mem = fopen("/proc/meminfo", "r");
	fgets(not_needed, 2048, mem);
	
	fscanf(mem, "%*s %Ld %Ld %Ld %Ld %Ld %Ld", &aa, &ab, &ac,
	       &ad, &ae, &af);
	fscanf(mem, "%*s %Ld %Ld", &ag, &ah);
	fclose(mem);
	mem_delay = 25;

	/* calculate it */
	my_mem_max = aa;	/* memory.total; */
	my_swap_max = ag;	/* swap.total; */

	my_mem_used = ah + ab - af - ae;

	/* No swap on ipaq 
	if (my_mem_used > my_mem_max) {
	   my_swap_used = my_mem_used - my_mem_max;
	    my_mem_used = my_mem_max;
	} else {
	   my_swap_used = 0;
	}
	*/
	
	msd.mem_used = my_mem_used;
	msd.mem_max = my_mem_max;
	//msd.swap_used = my_swap_used;
	//msd.swap_max = my_swap_max;

	msd.mem_percent = (100 * msd.mem_used) / msd.mem_max;
	//msd.swap_percent = (100 * msd.swap_used) / msd.swap_max;
	/* memory info changed - update things */
	return 1;
	}
    /* nothing new */
    return 0;
}

static void
set_backing( void *user_data )
{
  if (img_backing) mb_pixbuf_img_free(pb, img_backing);

  img_backing = mb_tray_get_bg_img(pb, win_panel);

  if (img_backing == NULL) {
    fprintf(stderr, "minisys failed to get dock background, giving up.\n");
    exit(1);
  }

  paint();
}

static void 
paint( void )
{
  MBPixbufImage *img_tmp = NULL;
  int cpusize, memsize;
  int y;

  if (!img_backing) {
    set_backing( NULL ); return; /* set_backing will recall this */
  }

  system_memory(); 		/* Update reading */

  if (pxm_backing) XFreePixmap(dpy, pxm_backing);

  pxm_backing = XCreatePixmap(dpy, win_panel, 
			      mb_pixbuf_img_get_width(img_icon), 
			      mb_pixbuf_img_get_height(img_icon),
			      pb->depth);

  cpusize = ( system_cpu() * GRAPH_HEIGHT ) / 100; 
  memsize = ( msd.mem_percent * GRAPH_HEIGHT ) / 100; 

  mb_pixbuf_img_copy(pb, img_graph, img_graph, 
		     1, 0, GRAPH_WIDTH-1, GRAPH_HEIGHT, 0, 0);
  for (y = GRAPH_HEIGHT; y >= 0; y--)
    {
      mb_pixbuf_img_set_pixel(img_graph, GRAPH_WIDTH-1, y, 0xff, 0xff, 0xff);
      mb_pixbuf_img_set_pixel_alpha(img_graph, GRAPH_WIDTH-1 , y, 0);
    }

  for (y = GRAPH_HEIGHT; y >= 0; y--)
    if ((GRAPH_HEIGHT-y) < memsize)
      {
	mb_pixbuf_img_set_pixel(img_graph, GRAPH_WIDTH-1, y, 0xff, 0, 0);
	mb_pixbuf_img_set_pixel_alpha(img_graph, GRAPH_WIDTH-1 , y, 75);
      }

  for (y = GRAPH_HEIGHT; y >= 0; y--)
    if ((GRAPH_HEIGHT-y) < cpusize)
      {
	mb_pixbuf_img_set_pixel(img_graph, GRAPH_WIDTH-1, y, 0, 0xff, 0);
	mb_pixbuf_img_set_pixel_alpha(img_graph, GRAPH_WIDTH-1 , y, 100);
	/*
	mb_pixbuf_img_composite_pixel(img_graph, GRAPH_WIDTH-1, y, 0, 
				      0xff, 0, 0);
	*/
      }

  img_tmp = mb_pixbuf_img_clone(pb, img_backing);

  mb_pixbuf_img_composite(pb, img_tmp, img_icon , 0, 0);

  mb_pixbuf_img_composite(pb, img_tmp, img_graph, 4, 2);

  mb_pixbuf_img_render_to_drawable(pb, img_tmp, pxm_backing, 0, 0);

  XSetWindowBackgroundPixmap(dpy, win_panel, pxm_backing);
  XClearWindow(dpy, win_panel);

  mb_pixbuf_img_free(pb, img_tmp);

}


void 
usage(char *progname)
{
  printf("Usage...\n");
}

int 
main(int argc, char **argv)
{
  int i;
  XEvent xevent;
  char *dpy_name = NULL;
  int xfd;
  fd_set fd;

  u_int64_t load = 0, total = 0;

  for (i = 1; i < argc; i++) {
    if (!strcmp ("-display", argv[i]) || !strcmp ("-d", argv[i])) {
      if (++i>=argc) usage (argv[0]);
      dpy_name = argv[i];
      continue;
    }
    usage(argv[0]);
  }

  if ((dpy = XOpenDisplay(dpy_name)) == NULL)
    {
      fprintf(stderr, "Cannot connect to X server on display %s.",
	      dpy_name);
      exit(1);
    }

   /* init load data */
   msd.samples = 16;
   
   if (msd.load) {
      load = msd.load[msd.loadIndex];
      free(msd.load);
   }
   
   if (msd.total) {
      total = msd.total[msd.loadIndex];
      free(msd.total);
   }
   
   msd.loadIndex = 0;
   msd.load = malloc(msd.samples * sizeof(u_int64_t));
   msd.total = malloc(msd.samples * sizeof(u_int64_t));
   for (i = 0; i < msd.samples; i++) {
      msd.load[i] = load;
      msd.total[i] = total;
   }


  screen   = DefaultScreen(dpy);
  win_root = DefaultRootWindow(dpy);
  pb       = mb_pixbuf_new(dpy, screen);

#ifdef USE_PNG
  if (!(img_icon = mb_pixbuf_img_new_from_file(pb, PIXMAPSDIR "/minisys.png")))
#else
  if (!(img_icon = mb_pixbuf_img_new_from_file(pb, PIXMAPSDIR "/minisys.xpm")))
#endif
    {
      fprintf(stderr, "%s: failed to load background image\n", argv[0] );
      exit(1);
    }
  /*
  img_backing = mb_pixbuf_img_new(pb, 
				  mb_pixbuf_img_get_width(img_icon), 
				  mb_pixbuf_img_get_height(img_icon));
  */
  img_graph = mb_pixbuf_img_new(pb, GRAPH_WIDTH, GRAPH_HEIGHT );  

  /*
  for (x=0; x<GRAPH_WIDTH; x++ )
    for (y=0; y<GRAPH_HEIGHT; y++ )
      mb_pixbuf_img_set_pixel_alpha(img_graph, x, y, 0xff);
  */

  win_panel = XCreateSimpleWindow(dpy, win_root, 0, 0,
				  mb_pixbuf_img_get_width(img_icon), 
				  mb_pixbuf_img_get_height(img_icon), 
				  0,
				  BlackPixel(dpy, screen),
				  WhitePixel(dpy, screen));

  gc = XCreateGC(dpy, win_root, 0, NULL);

  XStoreName(dpy, win_panel, "CPU/Memory Monitor");
  
  mb_tray_init_session_info(dpy, win_panel, argv, argc);
  mb_tray_init(dpy, win_panel);
  mb_tray_window_icon_set(dpy, win_panel, img_icon);

  mb_tray_bg_change_cb_set(set_backing, NULL);

  XFlush(dpy);

  XSelectInput(dpy, win_panel, StructureNotifyMask|ExposureMask
	       |ButtonPressMask|ButtonReleaseMask); 

  xfd = ConnectionNumber (dpy);

  while (1)
    {
      char tray_msg[256] = "";
      struct timeval tvt;

      XNextEvent(dpy, &xevent);
      switch (xevent.type) 
	{
	case ReparentNotify:
	  set_backing( NULL );
	  break;
	case ConfigureNotify:
	  set_backing( NULL );
	case Expose:
	  break;
	case ButtonPress:
	  sprintf(tray_msg, "CPU: %i %%, MEM: %i %%\n",
		      system_cpu(), msd.mem_percent);
	  mb_tray_send_message(dpy, win_panel, tray_msg, 5000);
	  
	  break;
	}
      mb_tray_handle_event(dpy, win_panel, &xevent);

      tvt.tv_usec = 100000;
      tvt.tv_sec = 0; 		/* check cpu stats every 1 seconds  */
      
      FD_ZERO (&fd);
      FD_SET (xfd, &fd);

      XFlush(dpy);

      if (select (xfd+1, &fd, NULL, NULL, &tvt) == 0)
	{
	  paint();
	}
    }

}

