/* RFC 2445 iCal Recurrence Object
 * Copyright (C) 2002  Sebastian Rittau <srittau@jroger.in-berlin.de>
 *
 * $Id: mimedir-recurrence.c,v 1.1 2002/08/28 04:57:12 srittau Exp $
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "mimedir-recurrence.h"


#ifndef _
#define _(x) (dgettext(GETTEXT_PACKAGE, (x)))
#endif


static void	 mimedir_recurrence_class_init		(MIMEDirRecurrenceClass	*klass);
static void	 mimedir_recurrence_init		(MIMEDirRecurrence	*recurrence);
static void	 mimedir_recurrence_dispose		(GObject		*object);


struct _MIMEDirRecurrencePriv {
	GList *rules;
};

static GObjectClass *parent_class = NULL;

/*
 * Class and Object Management
 */

GType
mimedir_recurrence_get_type (void)
{
	static GType mimedir_recurrence_type = 0;

	if (!mimedir_recurrence_type) {
		static const GTypeInfo mimedir_recurrence_info = {
			sizeof (MIMEDirRecurrenceClass),
			NULL, /* base_init */
			NULL, /* base_finalize */
			(GClassInitFunc) mimedir_recurrence_class_init,
			NULL, /* class_finalize */
			NULL, /* class_data */
			sizeof (MIMEDirRecurrence),
			1,    /* n_preallocs */
			(GInstanceInitFunc) mimedir_recurrence_init,
		};

		mimedir_recurrence_type = g_type_register_static (G_TYPE_OBJECT,
								  "MIMEDirRecurrence",
								  &mimedir_recurrence_info,
								  0);
	}

	return mimedir_recurrence_type;
}


static void
mimedir_recurrence_class_init (MIMEDirRecurrenceClass *klass)
{
	GObjectClass *gobject_class;

	g_return_if_fail (klass != NULL);
	g_return_if_fail (MIMEDIR_IS_RECURRENCE_CLASS (klass));

	gobject_class = G_OBJECT_CLASS (klass);

	gobject_class->dispose      = mimedir_recurrence_dispose;

	parent_class = g_type_class_peek_parent (klass);
}


static void
mimedir_recurrence_init (MIMEDirRecurrence *recurrence)
{
	MIMEDirRecurrencePriv *priv;

	g_return_if_fail (recurrence != NULL);
	g_return_if_fail (MIMEDIR_IS_RECURRENCE (recurrence));

	priv = g_new0 (MIMEDirRecurrencePriv, 1);
	recurrence->priv = priv;
}


static void
mimedir_recurrence_dispose (GObject *object)
{
	MIMEDirRecurrence *recurrence;

	g_return_if_fail (object != NULL);
	g_return_if_fail (MIMEDIR_IS_RECURRENCE (object));

	recurrence = MIMEDIR_RECURRENCE (object);

	g_free (recurrence->priv);
	recurrence->priv = NULL;

	G_OBJECT_CLASS (parent_class)->dispose (object);
}

/*
 * Public Methods
 */

/**
 * mimedir_recurrence_new:
 *
 * Creates a new iCal recurrence object.
 *
 * Return value: a new recurrence object
 **/
MIMEDirRecurrence *
mimedir_recurrence_new (void)
{
	MIMEDirRecurrence *recur;

	recur = g_object_new (MIMEDIR_TYPE_RECURRENCE, NULL);

	return recur;
}


/**
 * mimedir_recurrence_get_rules:
 *
 * Retrieves the list of recurrence rules for this object
 *
 * Return value: a #GList of #MIMEDirRecurrenceRules
 **/
GList *
mimedir_recurrence_get_rules (MIMEDirRecurrence *recur)
{
	g_return_if_fail (recur != NULL);
	g_return_if_fail (MIMEDIR_IS_RECURRENCE (recur));

	return recur->priv->rules;
}


/**
 * mimedir_recurrence_add_rule:
 *
 * Adds a rule to this recurrence object
 **/
void
mimedir_recurrence_add_rule (MIMEDirRecurrence *recur, MIMEDirRecurrenceRule *rule)
{
	g_return_if_fail (recur != NULL);
	g_return_if_fail (MIMEDIR_IS_RECURRENCE (recur));

	g_return_if_fail (rule != NULL);
	g_return_if_fail (MIMEDIR_IS_RECURRENCE_RULE (rule));
	
	g_object_ref (rule);
	
	recur->priv->rules = g_list_append (recur->priv->rules, rule);
}
