/***************************************************************************
 * graphics.c
 *
 * Copyright (c) 2000 Stefan Hellkvist, Ericsson Radio Systems AB 
 * Email: stefan@hellkvist.org
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *****************************************************************************/

#include "graphics.h"
#include "pattern.h"
#include <string.h>
#include <stdio.h>
#include <X11/Xatom.h>
#include <X11/Xft/Xft.h>

unsigned long getColor( Display *dsp, unsigned short red, unsigned green, unsigned blue );


Display *myDisplay;
Window myWindow;
GC myGC;
int myScreen;
int window_x, window_y, window_width, window_height;

XftDraw *myXftDrw;   /* xft aa bits */
XftFont *myXftFnt;
XftColor myXftCol; 

extern Pattern *current;
XFontStruct *myFont;

#ifdef USE_LIBSN
extern SnLauncheeContext *sn_context;
extern SnDisplay *sn_display;
#endif


#define BAR_HEIGHT 14

static int
x_error_handler (Display     *xdisplay,
                 XErrorEvent *error)
{
  char buf[64];

  XGetErrorText (xdisplay, error->error_code, buf, 63);

  fprintf (stderr, "Unexpected X error: %s serial %ld error_code %d request\
_code %d minor_code %d)\n",
	   buf,
	   error->serial,
	   error->error_code,
	   error->request_code,
	   error->minor_code);

  exit (1);
}

void
assignDefault()
{
    window_x = window_y = 0;
    window_width = DEFAULT_WINDOW_WIDTH;
    window_height = DEFAULT_WINDOW_HEIGHT;
}

int
parseArgs( int argc, char *argv[])
{
    char xsign, ysign;
    if ( argc != 3 || strcmp( "-geometry", argv[1] ) )
	return 1;

    if ( sscanf( argv[2], "%dx%d%c%d%c%d\n", &window_width, 
		 &window_height, 
		 &xsign, &window_x, &ysign, &window_y ) != 6 ) 
    	return 1;
    
    if ( xsign == '-' )
    {
	int screen_width = WidthOfScreen( 
	    ScreenOfDisplay( myDisplay, myScreen ) );
	window_x = screen_width - window_width - window_x;
    }

    if ( ysign == '-' )
    {
	int screen_height = HeightOfScreen( 
	    ScreenOfDisplay( myDisplay, myScreen ) );
	window_y = screen_height - window_height - window_y;
    }
    return 0;
}


void 
initGraphics( int argc, char *argv[] )
{
    XSetWindowAttributes myWindowAttributes;
    int myDepth; 
    XGCValues myGCValues;
    unsigned long myWindowMask; 
    Atom window_type_atom, window_type_toolbar_atom, mwm_atom;
    XWMHints *wm_hints;

#define PROP_MOTIF_WM_HINTS_ELEMENTS    5
#define MWM_HINTS_DECORATIONS          (1L << 1)
#define MWM_DECOR_BORDER               (1L << 1)

   typedef struct
   {
     unsigned long       flags;
     unsigned long       functions;
     unsigned long       decorations;
     long                inputMode;
     unsigned long       status;
   } PropMotifWmHints ;
   
   PropMotifWmHints *mwm_hints;

   XRenderColor colortmp;

#ifdef USE_LIBSN
   Atom atom_NET_STARTUP_ID, atom_UTF8_STRING;
#endif

   myDisplay = XOpenDisplay( "" );
   if (myDisplay == NULL)
     {
       fprintf(stderr, "%s: failed to open display\n", argv[0]);
       exit (0);
     }

    myScreen = DefaultScreen (myDisplay);
    myDepth = DefaultDepth (myDisplay, myScreen);

    if ( parseArgs( argc, argv ) )
	assignDefault();

    window_type_atom =
      XInternAtom(myDisplay, "_NET_WM_WINDOW_TYPE" , False);
    window_type_toolbar_atom =
      XInternAtom(myDisplay, "_NET_WM_WINDOW_TYPE_TOOLBAR",False);
    mwm_atom = XInternAtom(myDisplay, "_MOTIF_WM_HINTS",False);

    

    XSetErrorHandler (x_error_handler);

#ifdef USE_LIBSN

    atom_NET_STARTUP_ID = XInternAtom(myDisplay, "_NET_STARTUP_ID",False);
    atom_UTF8_STRING    = XInternAtom(myDisplay, "UTF8_STRING",False);

    sn_display = sn_display_new (myDisplay, NULL, NULL);

    sn_context 
      = sn_launchee_context_new_from_environment (sn_display, myScreen);


#endif

    myWindowAttributes.border_pixel = 
	BlackPixel (myDisplay, myScreen);
    myWindowAttributes.background_pixel = 
	getColor( myDisplay, 252, 234, 212 );

    myWindowMask = CWBackPixel | CWBorderPixel;

    myWindow = XCreateSimpleWindow(myDisplay, 
				   RootWindow (myDisplay, myScreen), 
				   window_x, 
				   window_y, 
				   window_width, window_height, 0,
				   BlackPixel(myDisplay, myScreen),
				   WhitePixel(myDisplay, myScreen)); 

    XSelectInput( myDisplay, myWindow, 
		  ExposureMask | ButtonPressMask | 
		  ButtonReleaseMask | PointerMotionMask |
		  StructureNotifyMask );

#ifdef USE_LIBSN

    if (sn_context == NULL)
      fprintf (stderr, "mbmerlin: Failed to get Startup notification context\n");
    else
      {
	printf ("Launchee started with window ID \"%s\"\n",
		sn_launchee_context_get_startup_id (sn_context));
	/* Lazily set startup id prop */
	XChangeProperty(myDisplay, myWindow, atom_NET_STARTUP_ID, 
			atom_UTF8_STRING, 8, PropModeReplace, 
			(unsigned char *)sn_launchee_context_get_startup_id (sn_context),
			strlen(sn_launchee_context_get_startup_id (sn_context)));

      }
#endif


    XStoreName(myDisplay, myWindow, "Stroke Recogniser");

    XChangeProperty(myDisplay, myWindow, window_type_atom, XA_ATOM, 32, 
		    PropModeReplace, 
		    (unsigned char *) &window_type_toolbar_atom, 1);

    /* Tell the WM we dont want no borders */
    mwm_hints = malloc(sizeof(PropMotifWmHints));
    memset(mwm_hints, 0, sizeof(PropMotifWmHints));
    
    mwm_hints->flags = MWM_HINTS_DECORATIONS;
    mwm_hints->decorations = 0;
    
    XChangeProperty(myDisplay, myWindow, mwm_atom, 
		    XA_ATOM, 32, PropModeReplace, 
		    (unsigned char *)mwm_hints, 
		    PROP_MOTIF_WM_HINTS_ELEMENTS);
      
    free(mwm_hints);



    wm_hints = XAllocWMHints();
    wm_hints->input = False;
    wm_hints->flags = InputHint;
    XSetWMHints(myDisplay, myWindow, wm_hints );

    myGC = XCreateGC (myDisplay, myWindow, (unsigned long) 0, &myGCValues);
    if (myGC == 0)
    {
	XDestroyWindow(myDisplay, myScreen);
	exit (0);
    }

    myFont = XLoadQueryFont (myDisplay, "fixed");
    if (myFont == (XFontStruct *)NULL)
    {
        fprintf(stderr,"Cannot get font:9x15.\n");
        myFont = NULL;
    }
    else
	XSetFont (myDisplay, myGC, myFont->fid);

    XSetForeground (myDisplay, myGC, BlackPixel( myDisplay, myScreen ) );

    myXftFnt = XftFontOpenName (myDisplay, DefaultScreen(myDisplay), 
				"Sans Serif-8:bold");

    myXftDrw = XftDrawCreate(myDisplay, (Drawable) myWindow, 
			     DefaultVisual(myDisplay, 
					   DefaultScreen(myDisplay)), 
			     DefaultColormap(myDisplay,
					     DefaultScreen(myDisplay)));
    
    colortmp.red   = 0xA000;
    colortmp.green = 0xA000;
    colortmp.blue  = 0xA000;
    colortmp.alpha = 0xffff;
    XftColorAllocValue(myDisplay,
		       DefaultVisual(myDisplay, DefaultScreen(myDisplay)), 
		       DefaultColormap(myDisplay,DefaultScreen(myDisplay)),
		       &colortmp,
		       &myXftCol);

    XMapWindow (myDisplay, myWindow);
    // XLowerWindow( myDisplay, myWindow );

    //paint();
    XFree(wm_hints);
}

unsigned long
getColor( Display *dsp, unsigned short red, unsigned green, unsigned blue )
{
    XColor col;
    Colormap cmap = DefaultColormapOfScreen( ScreenOfDisplay( dsp, 0 ) );
    col.red = (red + 1)*256 - 1;
    col.green = (green + 1)*256 - 1;
    col.blue = (blue + 1)*256 - 1;
    if (!XAllocColor( dsp, cmap, &col ) )
        fprintf(stderr, "Can't alloc color\n" );
    return col.pixel;
}


void
drawText()
{
  XftDrawString8(myXftDrw, &myXftCol, myXftFnt, window_width / 6 - 12 , 12,
		 "abc", strlen("abc"));

  XftDrawString8(myXftDrw, &myXftCol, myXftFnt, 
		 window_width / 2 - 13 + 1, 12,
		 "ABC", strlen("ABC"));

  XftDrawString8(myXftDrw, &myXftCol, myXftFnt, 
		 window_width / 6 * 5 - 11, 
		 12, "123", strlen("123"));

  XSetForeground (myDisplay, myGC, BlackPixel( myDisplay, myScreen ) );
}

void 
paintPattern()
{
    int i;
    if ( current != NULL && current->nPoints > 1 )
    {
	for ( i = 0; i < current->nPoints - 1; i++ )
	    if ( !current->pv[i].isLast ) 
		XDrawLine( myDisplay, myWindow, myGC, 
			   current->pv[i].x, current->pv[i].y,
			   current->pv[i+1].x, current->pv[i+1].y );
    }
}

void
paint( void )
{
    XClearWindow( myDisplay, myWindow );
	
    XSetForeground (myDisplay, myGC, getColor(myDisplay, 222, 222, 222));

	XFillRectangle(myDisplay, myWindow, myGC, 0, 0, window_width, BAR_HEIGHT);	
    XSetForeground (myDisplay, myGC, getColor(myDisplay, 200, 200, 200));	
    XSetLineAttributes( myDisplay, myGC, 1, LineOnOffDash, 
			CapNotLast, JoinMiter ); 	
    /* inner lines */	
    XDrawLine( myDisplay, myWindow, myGC, window_width / 3,
	       0, window_width / 3, window_height );
    XDrawLine( myDisplay, myWindow, myGC, (window_width / 3) * 2,
	       0, (window_width / 3) * 2, window_height );
    XDrawLine( myDisplay, myWindow, myGC, 0,
	       BAR_HEIGHT, window_width, BAR_HEIGHT);
    XDrawLine( myDisplay, myWindow, myGC, 0,
	       (window_height-BAR_HEIGHT) / 3 + BAR_HEIGHT, window_width, (window_height-BAR_HEIGHT) / 3 + BAR_HEIGHT);
    XDrawLine( myDisplay, myWindow, myGC, 0,
	       (window_height-BAR_HEIGHT) * 2 / 3 + BAR_HEIGHT, window_width, (window_height-BAR_HEIGHT) * 2 / 3 + BAR_HEIGHT);
	
    XSetLineAttributes( myDisplay, myGC, 1, LineSolid, 
			CapNotLast, JoinMiter );
	
	/* borders */
    XDrawLine( myDisplay, myWindow, myGC, 0,
	       0, window_width - 1, 0 );
    XDrawLine( myDisplay, myWindow, myGC, 0,
	       0, 0, window_height - 1);
    XDrawLine( myDisplay, myWindow, myGC, 0,
	       window_height - 1, window_width, window_height - 1);
    XDrawLine( myDisplay, myWindow, myGC, window_width - 1,
	       0, window_width - 1, window_height - 1);

	
    XSetForeground (myDisplay, myGC, BlackPixel( myDisplay, myScreen ) );
    paintPattern();

    drawText();
}
