/****************************************************************************/
/* Copyright 1999 Compaq Computer Corporation.                              */
/*                                           .                              */
/* Copying or modifying this code for any purpose is permitted,             */
/* provided that this copyright notice is preserved in its entirety         */
/* in all copies or modifications.  COMPAQ COMPUTER CORPORATION             */
/* MAKES NO WARRANTIES, EXPRESSED OR IMPLIED, AS TO THE USEFULNESS          */
/* OR CORRECTNESS OF THIS CODE OR ITS FITNESS FOR ANY PARTICULAR            */
/* PURPOSE.                                                                 */
/****************************************************************************/

/*++

Copyright (c) 1997-1998  Microsoft Corporation

Module Name:

    ItsyEthWdm.c 

Abstract:

    Wraps up WDM stuff because of the dammn MS header files
	don't let you include ndis.h and wdm.h.


Environment:

    kernel mode only

Notes:

  THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
  KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR
  PURPOSE.

  Copyright (c) 1997-1998 Microsoft Corporation.  All Rights Reserved.


Revision History:

    2/8/98: created

--*/

#include "wdm.h"
#include "stdarg.h"
#include "stdio.h"

#include "usbdi.h"
#include "usbdlib.h"
#include "usbioctl.h"

#include "itsyethdbg.h"
#include "itsyethwdm.h"
#include "itsyethcom.h"
#include "equates.h"
#include <initguid.h>
#include <ndisguid.h>

/*
 * Prototypes
 */
NTSTATUS
ItsyEth_AsyncRecv_Complete(
    IN PDEVICE_OBJECT DeviceObject,
    IN PIRP Irp,
    IN PVOID Context
    );

NTSTATUS
ItsyEth_AsyncSend_Complete(
    IN PDEVICE_OBJECT DeviceObject,
    IN PIRP Irp,
    IN PVOID Context
    );

NTSTATUS
ItsyEth_ResetPipe_Complete(
    IN PDEVICE_OBJECT DeviceObject,
    IN PIRP Irp,
    IN PVOID Context
    );

PIRP
ItsyEth_BuildResetIrp(IN PITSYETH_DEVICE_EXTENSION deviceExtension,
					  IN USHORT Function,
					  IN PUSBD_PIPE_INFORMATION PipeInfo);


PURB
ItsyEth_BuildAsyncRequest(PITSYETH_FRAME Frame,
						  IN PIRP Irp,
						  IN PUSBD_PIPE_INFORMATION PipeHandle,
						  IN BOOLEAN Read);
VOID
ItsyEth_ResetPipe(
    IN PVOID Context
    );

VOID
ItsyEth_ResetPipeIn(
    IN PVOID Context
    );

VOID
ItsyEth_PrintFrameNumber(
    PITSYETH_DEVICE_EXTENSION deviceExtension
    );

/*++

Routine Description:

    Free all the allocated resources, etc.

Arguments:

    DriverObject - pointer to a driver object

Return Value:


--*/
VOID
ItsyEth_Unload(
    IN PDRIVER_OBJECT DriverObject
    )
{
	ItsyEth_KdPrint(DBGLVL_DEFAULT, ("enter ItsyEth_Unload\n"));

    //
    // Free any global resources allocated
    // in DriverEntry.
	// We have few or none because for a PNP device, almost all
	// allocation is done in PnpAddDevice() and all freeing 
	// while handling IRP_MN_REMOVE_DEVICE:
    //
	ITSYETH_ASSERT( gExAllocCount == 0 );

    ItsyEth_KdPrint( DBGLVL_DEFAULT,("exit ItsyUsb_Unload\n"));
}

/*++

Routine Description:

        We keep a pending IO count ( extension->PendingIoCount )  in the device extension.
        The first increment of this count is done on adding the device.
        Subsequently, the count is incremented for each new IRP received and
        decremented when each IRP is completed or passed on.

        Transition to 'one' therefore indicates no IO is pending and signals
        deviceExtension->NoPendingIoEvent. This is needed for processing
        IRP_MN_QUERY_REMOVE_DEVICE

        Transition to 'zero' signals an event ( deviceExtension->RemoveEvent )
        to enable device removal. This is used in processing for IRP_MN_REMOVE_DEVICE
 
Arguments:

        DeviceObject -- ptr to our FDO

Return Value:

        deviceExtension->PendingIoCount


--*/
LONG
ItsyEth_DecrementIoCount(
    IN PITSYETH_DEVICE_EXTENSION deviceExtension,
	IN int Direction
    )
{
    LONG ioCount;
    KIRQL             oldIrql;

	ItsyEth_KdPrint(DBGLVL_HIGH, ("ItsyEth_DecrementIoCount\n"));

//	KeAcquireSpinLock (&deviceExtension->IoCountSpinLock, &oldIrql);

    ioCount = InterlockedDecrement(&deviceExtension->PendingOutIoCount);
#if 0
    if (ioCount==1) 
	{
        // trigger no pending io
        KeSetEvent(&deviceExtension->NoPendingIoEvent, 1, FALSE);
    }

    if (ioCount==0) 
	{
        // trigger remove-device event
        KeSetEvent(&deviceExtension->RemoveEvent, 1, FALSE);
    }
#endif
//	KeReleaseSpinLock (&deviceExtension->IoCountSpinLock, oldIrql);

    ItsyEth_KdPrint( DBGLVL_HIGH,("Exit ItsyEth_DecrementIoCount() Pending io count = %x\n", ioCount));
    return ioCount;
}

/*++

Routine Description:

        We keep a pending IO count ( extension->PendingIoCount )  in the device extension.
        The first increment of this count is done on adding the device.
        Subsequently, the count is incremented for each new IRP received and
        decremented when each IRP is completed or passed on.

 
Arguments:

        DeviceObject -- ptr to our FDO

Return Value:

        None


--*/
VOID
ItsyEth_IncrementIoCount(
    IN PITSYETH_DEVICE_EXTENSION deviceExtension,
	IN int Direction
    )
{
    KIRQL             oldIrql;

	ItsyEth_KdPrint(DBGLVL_HIGH, ("ItsyEth_IncrementIoCount\n"));
    ItsyEth_KdPrint( DBGLVL_MAXIMUM,("Enter ItsyEth_IncrementIoCount() Pending io count = %x\n", deviceExtension->PendingOutIoCount));

//	KeAcquireSpinLock (&deviceExtension->IoCountSpinLock, &oldIrql);

    InterlockedIncrement(&deviceExtension->PendingOutIoCount);

//	KeReleaseSpinLock (&deviceExtension->IoCountSpinLock, oldIrql);

    ItsyEth_KdPrint( DBGLVL_HIGH,("Exit ItsyEth_IncrementIoCount() Pending io count = %x\n", deviceExtension->PendingOutIoCount));
}

/*++

Routine Description:

    Passes a URB to the USBD class driver
	The client device driver passes USB request block (URB) structures 
	to the class driver as a parameter in an IRP with Irp->MajorFunction
	set to IRP_MJ_INTERNAL_DEVICE_CONTROL and the next IRP stack location 
	Parameters.DeviceIoControl.IoControlCode field set to 
	IOCTL_INTERNAL_USB_SUBMIT_URB. 

Arguments:

    DeviceObject - pointer to the physical device object (PDO)

    Urb - pointer to an already-formatted Urb request block

Return Value:

    STATUS_SUCCESS if successful,
    STATUS_UNSUCCESSFUL otherwise

--*/
NTSTATUS
ItsyEth_CallUSBD(IN PDEVICE_OBJECT DeviceObject,
				 IN PDEVICE_OBJECT NextDeviceObject,
                 IN PURB Urb
				 )
{
    NTSTATUS ntStatus = STATUS_SUCCESS;
    PIRP irp;
    KEVENT event;
    IO_STATUS_BLOCK ioStatus;
    PIO_STACK_LOCATION nextStack;
	LARGE_INTEGER timeOut;

    ItsyEth_KdPrint( DBGLVL_MAXIMUM, ("enter BulkUsb_CallUSBD\n"));

    // issue a synchronous request
    KeInitializeEvent(&event, NotificationEvent, FALSE);

    irp = IoBuildDeviceIoControlRequest(
                IOCTL_INTERNAL_USB_SUBMIT_URB,
                NextDeviceObject, //Points to the next-lower driver's device object
                NULL, // optional input bufer; none needed here
                0,	  // input buffer len if used
                NULL, // optional output bufer; none needed here
                0,    // output buffer len if used
                TRUE, // If InternalDeviceControl is TRUE the target driver's Dispatch
				      //  outine for IRP_MJ_INTERNAL_DEVICE_CONTROL or IRP_MJ_SCSI 
					  // is called; otherwise, the Dispatch routine for 
					  // IRP_MJ_DEVICE_CONTROL is called.
                &event,     // event to be signalled on completion
                &ioStatus);  // Specifies an I/O status block to be set when the request is completed the lower driver. 

    //
    // Call the class driver to perform the operation.  If the returned status
    // is PENDING, wait for the request to complete.
    //

    nextStack = IoGetNextIrpStackLocation(irp);
    ITSYETH_ASSERT(nextStack != NULL);

    //
    // pass the URB to the USB driver stack
    //
    nextStack->Parameters.Others.Argument1 = Urb;

    ntStatus = IoCallDriver(NextDeviceObject, irp);

    ItsyEth_KdPrint( DBGLVL_MAXIMUM,("ItsyEth_CallUSBD() return from IoCallDriver USBD %x\n", ntStatus));

    if (ntStatus == STATUS_PENDING) 
	{
		timeOut.QuadPart = -1000000000;  // specified in nano sec
		ntStatus = KeWaitForSingleObject(
			                           &event,
									   Suspended,
									   KernelMode,
									   FALSE,
									   &timeOut);
		if (ntStatus == STATUS_TIMEOUT)
		{
			// cancel the IRP...
			IoCancelIrp(irp);
		}
    } 
	else 
	{
		ItsyEth_KdPrint( DBGLVL_DEFAULT,("ItsyEth_CallUSBD() failed no event %x\n", ntStatus));
        ioStatus.Status = ntStatus;
    }
    ItsyEth_KdPrint( DBGLVL_DEFAULT,("ItsyEth_CallUSBD() URB status = %x status = %x irp status %x\n",
        Urb->UrbHeader.Status, ntStatus, ioStatus.Status));

    //
    // USBD maps the error code for us, but it could be anything if we cancelled the irp
    //
	if (ntStatus != STATUS_TIMEOUT)
	{
		ntStatus = ioStatus.Status;
	}

    ItsyEth_KdPrintCond( DBGLVL_DEFAULT, !NT_SUCCESS( ntStatus ), ("ItsyEth_CallUSBD FAILED (NT %x URB %x IRP %x)\n", 
		                 ntStatus, Urb->UrbHeader.Status, irp->IoStatus.Status));

    return ntStatus;
}

/*++

Routine Description:

    Called from BulkUsb_StageReadWrite() for IRP_MJ_READ or IRP_MJ_WRITE

Arguments:

    DeviceObject - pointer to the FDO ( Functional Device Object )

    Irp - A staged IRP allocated and mapped by this driver in BulkUsb_StageReadWrite()
          to perform a single deviceExtension->MaximumTransferSize IO request

    PipeHandle - handle to the endpoint we're reading or writing

    Read - TRUE for reads, FALSE for writes

Return Value:

    ptr to initialized async urb. ( USB Request Block )

--*/
PURB
ItsyEth_BuildAsyncRequest(
    PITSYETH_FRAME Frame,
    IN PIRP Irp,
    IN PUSBD_PIPE_INFORMATION PipeHandle,
    IN BOOLEAN Read
    )
{
    ULONG siz;
    PURB urb = NULL;

    siz = sizeof(struct _URB_BULK_OR_INTERRUPT_TRANSFER);
    urb = ExAllocatePool(NonPagedPool, siz);

//    ItsyEth_KdPrint( DBGLVL_DEFAULT,("Enter ItsyEth_BuildAsyncRequest() siz = 0x%x urb 0x%x\n Pipehandle 0x%x\n", siz, urb, PipeHandle));

    if (urb) 
	{
        RtlZeroMemory(urb, siz);

        urb->UrbBulkOrInterruptTransfer.Hdr.Length = (USHORT) siz;
        urb->UrbBulkOrInterruptTransfer.Hdr.Function =
                    URB_FUNCTION_BULK_OR_INTERRUPT_TRANSFER;
        urb->UrbBulkOrInterruptTransfer.PipeHandle = PipeHandle->PipeHandle;
        urb->UrbBulkOrInterruptTransfer.TransferFlags =
            Read ? USBD_TRANSFER_DIRECTION_IN : 0;

        // short packet is not treated as an error.
        urb->UrbBulkOrInterruptTransfer.TransferFlags |= USBD_SHORT_TRANSFER_OK;            
                
        //
        // not using linked urb's
        //
        urb->UrbBulkOrInterruptTransfer.UrbLink = NULL;

        urb->UrbBulkOrInterruptTransfer.TransferBuffer = Frame->va;
        urb->UrbBulkOrInterruptTransfer.TransferBufferLength = Frame->length;
		urb->UrbBulkOrInterruptTransfer.TransferBufferMDL = NULL;

        ItsyEth_KdPrint( DBGLVL_MAXIMUM,("BulkUsb_BuildAsyncRequest() Init async urb Length = 0x%x decimal %d, buf = 0x%x\n",
            urb->UrbBulkOrInterruptTransfer.TransferBufferLength,
            urb->UrbBulkOrInterruptTransfer.TransferBufferLength,
            urb->UrbBulkOrInterruptTransfer.TransferBuffer));
    }
//    ItsyEth_KdPrint( DBGLVL_DEFAULT,("exit BulkUsb_BuildAsyncRequest %p\n", urb));
    return urb;
}

void
ItsyEth_Recv(PITSYETH_WDM_HANDLE WdmHandle,
			   PITSYETH_FRAME_HANDLE FrameHandle,
			   void (* Callback)(PVOID ,PVOID, int),
			   PVOID Adapter,
			   PVOID Packet)
{
	PITSYETH_FRAME Frame = (PITSYETH_FRAME) FrameHandle;
	PITSYETH_DEVICE_EXTENSION deviceExtension = (PITSYETH_DEVICE_EXTENSION) WdmHandle;
	PIRP Irp;
	PCHAR buf;
	KIRQL oldIrql;
	PURB urb;
	CCHAR stackSize;
	PIO_STACK_LOCATION nextStack;
	NTSTATUS ntStatus;

	ItsyEth_KdPrint(DBGLVL_HIGH, ("ItsyEth_Recv WdmHandle %p Frame %p\n", WdmHandle, FrameHandle));
	KeAcquireSpinLock(deviceExtension->FastCompleteSpinlock, &oldIrql);
	if ((WdmHandle) && (FrameHandle))
	{
		deviceExtension = (PITSYETH_DEVICE_EXTENSION) WdmHandle;

		Frame->callback = Callback;
		Frame->Adapter = Adapter;
		Frame->Packet = Packet;
		// advance the frame buffer to allow mac addresses to be added later
		buf = Frame->va;
		buf += 10;
		Frame->va = buf;
		Frame->length = MAXIMUM_ETHERNET_PACKET_SIZE - 10;
		ItsyEth_KdPrint(DBGLVL_MAXIMUM, ("receive buffer %p\n", buf));

        stackSize = (CCHAR)(deviceExtension->TopOfStackDeviceObject->StackSize+1);
        Irp = IoAllocateIrp(stackSize, FALSE);
        if (Irp) 
		{
		   urb = ItsyEth_BuildAsyncRequest(Frame,
				                           Irp,
                                           deviceExtension->InPipe,
                                           1);
		   Frame->urb = urb;
        }
		else
		{
			ItsyEth_KdPrint(DBGLVL_DEFAULT, ("ItsyEth_Recv Failed to allocate Irp\n"));
		}
		buf = Frame->va;
		buf = buf - 10;
		Frame->va = buf;
        
        if (urb && Irp) 
		{
            // IoGetNextIrpStackLocation gives a higher level driver access to the next-lower 
            // driver's I/O stack location in an IRP so the caller can set it up for the lower driver.
            nextStack = IoGetNextIrpStackLocation(Irp);
            ITSYETH_ASSERT(nextStack != NULL);

            nextStack->MajorFunction = IRP_MJ_INTERNAL_DEVICE_CONTROL;
            nextStack->Parameters.Others.Argument1 = urb;
            nextStack->Parameters.DeviceIoControl.IoControlCode = IOCTL_INTERNAL_USB_SUBMIT_URB;

            IoSetCompletionRoutine(Irp,
                    ItsyEth_AsyncRecv_Complete,
                    Frame, // pass the context array element to completion routine
                    TRUE,    // invoke on success
                    TRUE,    // invoke on error
                    TRUE);   // invoke on cancellation of the Irp

//            BulkUsb_IncrementIoCount(DeviceObject);

			ntStatus = IoCallDriver(deviceExtension->TopOfStackDeviceObject, Irp);

			// should check the return status of this....
			if (ntStatus != STATUS_PENDING)
			{
				ItsyEth_KdPrint(DBGLVL_DEFAULT, ("ItsyEth_Recv not pending\n"));
			}
		}
	}
	KeReleaseSpinLock(deviceExtension->FastCompleteSpinlock, oldIrql);
	ItsyEth_KdPrint(DBGLVL_HIGH, ("exit ItsyEth_Recv\n"));
}

int InSend=0;
void
ItsyEth_Send(PITSYETH_WDM_HANDLE WdmHandle,
			 PITSYETH_FRAME_HANDLE FrameHandle, 
			 VOID (*Callback)(PVOID, PVOID, int), 
			 PVOID Adapter,
			 PVOID Packet)
{
	PITSYETH_FRAME Frame;
	PITSYETH_DEVICE_EXTENSION deviceExtension = (PITSYETH_DEVICE_EXTENSION) WdmHandle;
	PIRP Irp;
	KIRQL oldIrql;
	PURB urb;
	CCHAR stackSize;
	PIO_STACK_LOCATION nextStack;
	NTSTATUS ntStatus;
	BOOLEAN freeFrame=TRUE;
	LARGE_INTEGER count1, count2;
	LONGLONG        difference;
	KIRQL pre, post;
	
//	InSend = 1;
//	KeQuerySystemTime(&count1);
//	pre = KeGetCurrentIrql();

	ItsyEth_KdPrint(DBGLVL_HIGH, ("ItsyEth_Send\n"));

	KeAcquireSpinLock(deviceExtension->FastCompleteSpinlock, &oldIrql);
//	ItsyEth_KdPrint ( DBGLVL_DEFAULT,("ItsyEth_Send() start\n")); 
	if ((WdmHandle) && (FrameHandle))
	{
		Frame = (PITSYETH_FRAME) FrameHandle;
		deviceExtension = (PITSYETH_DEVICE_EXTENSION) WdmHandle;

		Frame->callback = Callback;
		Frame->Packet = Packet;
		Frame->Adapter = Adapter;
		// must send a non aligned packet so that itsy know's that it ends...
//		if ((Frame->length % deviceExtension->OutPipe->MaximumPacketSize) == 0)
//		{
//			Frame->length++;
//		}
		if ((deviceExtension->DeviceStarted) && (!deviceExtension->DeviceStalled))
		{
			stackSize = (CCHAR)(deviceExtension->TopOfStackDeviceObject->StackSize + 1);
			Irp = IoAllocateIrp(stackSize, FALSE);
        
			if (Irp) 
			{
				// Each new Irp will 'see' the entire buffer, but map it's IO location
				// to a single ChunkSize section within it via IoBuildPartialMdl()
			   urb = ItsyEth_BuildAsyncRequest(Frame,
				                               Irp,
			                                   deviceExtension->OutPipe,
				                               0);
			   Frame->urb = urb;
			}
			if (urb && Irp) 
			{
				// IoGetNextIrpStackLocation gives a higher level driver access to the next-lower 
				// driver's I/O stack location in an IRP so the caller can set it up for the lower driver.
				nextStack = IoGetNextIrpStackLocation(Irp);

	            nextStack->MajorFunction = IRP_MJ_INTERNAL_DEVICE_CONTROL;
		        nextStack->Parameters.Others.Argument1 = urb;
			    nextStack->Parameters.DeviceIoControl.IoControlCode =
				    IOCTL_INTERNAL_USB_SUBMIT_URB;

				IoSetCompletionRoutine(Irp,
					    ItsyEth_AsyncSend_Complete,
						Frame, // pass the context array element to completion routine
						TRUE,    // invoke on success
						TRUE,    // invoke on error
						TRUE);   // invoke on cancellation of the Irp

		        ItsyEth_IncrementIoCount(deviceExtension, 1);
	
				ntStatus = IoCallDriver(deviceExtension->TopOfStackDeviceObject, Irp);

				// should check the return status of this....
				if (ntStatus == STATUS_PENDING)
				{
					freeFrame = FALSE;
				}
				else
				{
					ItsyEth_KdPrint(DBGLVL_DEFAULT, ("ItsyEth_Send freeing\n"));
				}
			}
		}
		else
		{
			ItsyEth_KdPrint(DBGLVL_DEFAULT, ("ItsyEthSend : device not operational\n"));
		}
	}
	else
	{
		ItsyEth_KdPrint(DBGLVL_DEFAULT, ("ItsyEth_Send bad handles\n"));
		// have an unknown state so forget about the frame...
		freeFrame = FALSE; 
	}
	if (freeFrame)
	{
		Frame->callback(Frame->Adapter, Frame->Packet, STATUS_UNSUCCESSFUL);
		ItsyEth_FreeFrame(Frame->WdmHandle, Frame);
	}
//	KeQuerySystemTime(&count2);
//	difference = count2.QuadPart - count1.QuadPart;
//	post = KeGetCurrentIrql();
//	ItsyEth_KdPrint ( DBGLVL_DEFAULT,("ItsyEth_AsyncSend(), tick count %d %d %s\n",
//		pre, post, buffer)); 
//	ItsyEth_KdPrint ( DBGLVL_DEFAULT,("ItsyEth_AsyncSend(), system time %lx %lx %lx %lx\n",
//		(unsigned int) count1.HighPart, 
//		(unsigned int) count1.LowPart, 
//		(unsigned int) count2.HighPart, 
//		(unsigned int) count2.LowPart)); 

	KeReleaseSpinLock(deviceExtension->FastCompleteSpinlock, oldIrql);
	ItsyEth_KdPrint(DBGLVL_HIGH, ("ItsyEth_Send exit\n"));
	
	InSend=0;
}

/*++

Routine Description:

  Completion routine for our staged read/write Irps


Arguments:

    DeviceObject - Pointer to the device object for next lower device
	in the  driver stack; 

    Irp - Irp completed.

    Context - Driver defined context.

Return Value:

    The function value is the final status from the operation.

--*/
int InSendComplete=0;
NTSTATUS
ItsyEth_AsyncRecv_Complete(
    IN PDEVICE_OBJECT DeviceObject,
    IN PIRP Irp,
    IN PVOID Context
    )
{
    NTSTATUS			ntStatus = STATUS_MORE_PROCESSING_REQUIRED;
    PURB				urb;
	PITSYETH_FRAME Frame = Context;
    PDEVICE_OBJECT      deviceObject;
	PITSYETH_DEVICE_EXTENSION   deviceExtension = (PITSYETH_DEVICE_EXTENSION) Frame->WdmHandle;
	KIRQL               oldIrql;
	int  length;
	LARGE_INTEGER count1, count2;
	LONGLONG        difference;
	KIRQL pre, post;

//	KeQuerySystemTime(&count1);
//	pre = KeGetCurrentIrql();

//	if (InSend || InSendComplete)
//	{
//		ItsyEth_KdPrint( DBGLVL_DEFAULT, ("ItsyEth_AsyncRecv_Complete(): while in send %d %d\n", InSend, InSendComplete));
//	}

	KeAcquireSpinLock(deviceExtension->FastCompleteSpinlock, &oldIrql);
//	ItsyEth_KdPrint ( DBGLVL_DEFAULT,("ItsyEth_AsyncRecv_Complete() start\n")); 

    if ( Irp->PendingReturned ) 
	{  
        IoMarkIrpPending(Irp);
    }

    urb = Frame->urb;
    ItsyEth_KdPrint( DBGLVL_HIGH, ("ENTER ItsyEth_AsyncRecv_Complete():  Irp %p, usb %p, Length %d Status 0x%08X\n",
		             Irp, urb,
                     urb->UrbBulkOrInterruptTransfer.TransferBufferLength,
                     urb->UrbHeader.Status));

	if (urb->UrbHeader.Status == USBD_STATUS_SUCCESS)
	{
		// decrement the driver's overall pending irp count
		//    ItsyEth_DecrementIoCount(deviceObject);

		// read the length, if the packet aligns on a 256 byte boundary it won't be the 
		// same as the actual packet length
		length = Frame->va[10] & 0xff;
		length += (Frame->va[11] & 0xff) << 8;
		ItsyEth_KdPrint( DBGLVL_HIGH, ("ItsyEth_AsyncRecv_Complete():  length %x %x %d urb Length %d \n",
		             Frame->va[10],Frame->va[11],length, urb->UrbBulkOrInterruptTransfer.TransferBufferLength));
		AdjustBufferLength(Frame->Packet, length+10);
		Frame->callback(Frame->Adapter, Frame->Packet, STATUS_SUCCESS);
		ExFreePool(Frame->urb);
		ItsyEth_FreeFrame(Frame->WdmHandle, Frame);
	}
	else
	{
		// reset if we are not already resetting and if we haven't been asked to stop.
		if ((deviceExtension->ResetWork == NULL) && (!deviceExtension->StopDeviceRequested))
		{
			ItsyEth_KdPrint( DBGLVL_DEFAULT, ("ItsyEth_AsyncRecv_Complete(): about to schedule IN reset %x\n", urb->UrbHeader.Status));
			deviceExtension->ResetWork = ExAllocatePool(NonPagedPool, sizeof(WORK_QUEUE_ITEM));
			ExInitializeWorkItem(deviceExtension->ResetWork, ItsyEth_ResetPipeIn, Frame);
			ExQueueWorkItem(deviceExtension->ResetWork, CriticalWorkQueue);
		}
		else
		{
			ItsyEth_KdPrint( DBGLVL_DEFAULT, ("ItsyEth_AsyncRecv_Complete(): can't reset, work != NULL\n"));
		}
		ExFreePool(Frame->urb);
		Frame->urb = NULL;
	}

	IoFreeIrp(Irp);

	ItsyEth_KdPrint ( DBGLVL_HIGH,("Exit ItsyEth_AsyncRecv_Complete(), ntStatus = 0x%x\n",ntStatus )); 
//	KeQuerySystemTime(&count2);
//	post = KeGetCurrentIrql();

//	ItsyEth_KdPrint ( DBGLVL_DEFAULT,("ItsyEth_AsyncRecv_Complete(), system time %lx %lx %lx %lx\n",
//		(unsigned int) count1.HighPart, 
//		(unsigned int) count1.LowPart, 
//		(unsigned int) count2.HighPart, 
//		(unsigned int) count2.LowPart)); 
	KeReleaseSpinLock(deviceExtension->FastCompleteSpinlock, oldIrql);
	
    return ntStatus;
}

NTSTATUS
ItsyEth_AsyncSend_Complete(
    IN PDEVICE_OBJECT DeviceObject,
    IN PIRP Irp,
    IN PVOID Context
    )
{
    NTSTATUS			ntStatus = STATUS_MORE_PROCESSING_REQUIRED;
    PURB				urb;
	PITSYETH_FRAME Frame = Context;
	PITSYETH_DEVICE_EXTENSION   deviceExtension = (PITSYETH_DEVICE_EXTENSION) Frame->WdmHandle;
	KIRQL               oldIrql;
	LONG ioCount;
	static int pkts=0;
	LARGE_INTEGER count1, count2;
	KIRQL pre, post;
	LONGLONG        difference;

	KeAcquireSpinLock(deviceExtension->FastCompleteSpinlock, &oldIrql);

	ITSYETH_ASSERT(Irp);
	ITSYETH_ASSERT(deviceExtension);
	ITSYETH_ASSERT(Frame);
	ITSYETH_ASSERT(Frame->urb);

//	ItsyEth_KdPrint ( DBGLVL_DEFAULT,("ItsyEth_AsyncSend_Complete() start\n")); 
//
//	InSendComplete = 1;
//	KeQuerySystemTime(&count1);
//	pre = KeGetCurrentIrql();



	pkts++;
    if ( Irp->PendingReturned )
	{
        IoMarkIrpPending(Irp);
    }

	urb = Frame->urb;
    ItsyEth_KdPrint( DBGLVL_HIGH, ("ENTER ItsyEth_AsyncSend_Complete():  Length %d Status 0x%08X\n",
                     urb->UrbBulkOrInterruptTransfer.TransferBufferLength,
                     urb->UrbHeader.Status));

    // decrement the driver's overall pending irp count
    ioCount = ItsyEth_DecrementIoCount(deviceExtension, 1);
    
    // 
    // IoCallDriver has been called on this Irp;
    // Set the length based on the TransferBufferLength
    // value in the URB
    //
	Frame->callback(Frame->Adapter, Frame->Packet, STATUS_SUCCESS);
	ItsyEth_FreeFrame(Frame->WdmHandle, Frame);

	Irp->IoStatus.Information = urb->UrbBulkOrInterruptTransfer.TransferBufferLength;
	Irp->IoStatus.Status = STATUS_SUCCESS; 
	Irp->IoStatus.Information = Frame->length;

	switch (urb->UrbHeader.Status)
	{
		case  USBD_STATUS_SUCCESS :
			if ((ULONG) Frame->length != urb->UrbBulkOrInterruptTransfer.TransferBufferLength)
			{
				ItsyEth_KdPrint( DBGLVL_DEFAULT, ("ItsyEth_AsyncSend_Complete():  Send problem length %d sent %d\n",
			             urb->UrbBulkOrInterruptTransfer.TransferBufferLength,
				         Frame->length));
			}
			// nothing special, in error cases we don't let ndis know about it.
			break;
		case USBD_STATUS_DEV_NOT_RESPONDING :
		case USBD_STATUS_STALL_PID :
		case USBD_STATUS_ENDPOINT_HALTED :
			ItsyEth_KdPrint( DBGLVL_DEFAULT, ("ItsyEth_AsyncSend_Complete():  Write Error len %d Status 0x%08X\n",
			             urb->UrbBulkOrInterruptTransfer.TransferBufferLength,
				         urb->UrbHeader.Status));
			deviceExtension->DeviceStalled = TRUE;
			deviceExtension->ResetWork = ExAllocatePool(NonPagedPool, sizeof(WORK_QUEUE_ITEM));
			ExInitializeWorkItem(deviceExtension->ResetWork, ItsyEth_ResetPipe, deviceExtension);
			ExQueueWorkItem(deviceExtension->ResetWork, CriticalWorkQueue);
		default :
			// if there are no pending IRPs reset the pipe
			ItsyEth_KdPrint( DBGLVL_DEFAULT, ("ItsyEth Not Reseting Pipes %d %x\n", ioCount, 
				urb->UrbHeader.Status));
			break;
	}
	IoFreeIrp(Irp);
	ExFreePool(urb);

//	KeQuerySystemTime(&count2);
//	difference = count2.QuadPart - count1.QuadPart;
//	post = KeGetCurrentIrql();
//	ItsyEth_KdPrint ( DBGLVL_DEFAULT,("ItsyEth_AsyncSend_Complete(), tick count %d %d\n",
//		pre, post)); 
//	ItsyEth_KdPrint ( DBGLVL_DEFAULT,("ItsyEth_AsyncSend_Complete(), system time %lx %lx %lx %lx\n",
//		(unsigned int) count1.HighPart, 
//		(unsigned int) count1.LowPart, 
//		(unsigned int) count2.HighPart, 
//		(unsigned int) count2.LowPart)); 
//
//	InSendComplete=0;

	KeReleaseSpinLock(deviceExtension->FastCompleteSpinlock, oldIrql);

#if 0
	if (pkts == 100)
	{
		ItsyEth_KdPrint(DBGLVL_DEFAULT, ("ItsyEth_AsyncSend_Complete():  about to reset...\n"));
		deviceExtension->DeviceStalled = TRUE;
		deviceExtension->ResetWork = ExAllocatePool(NonPagedPool, sizeof(WORK_QUEUE_ITEM));
		ExInitializeWorkItem(deviceExtension->ResetWork, ItsyEth_ResetPipe, deviceExtension);
		ExQueueWorkItem(deviceExtension->ResetWork, CriticalWorkQueue);
		pkts = 0;
	}
#endif
    ItsyEth_KdPrint ( DBGLVL_HIGH,("Exit ItsyEth_AsyncSend_Complete(), ntStatus = 0x%x\n",ntStatus )); 

    return ntStatus;
}

PKSPIN_LOCK framelock;
void
ItsyEth_InitFrameList(PITSYETH_DEVICE_EXTENSION deviceExtension)
{
	PITSYETH_FRAME FrameList;
	PCHAR BufferMem;
	int idx;

	if (deviceExtension)
	{
		InitializeListHead(&deviceExtension->FreeFrames);
	    InitializeListHead(&deviceExtension->QueuedFrames);
	    KeInitializeSpinLock(&deviceExtension->FrameListLock);
		framelock = &deviceExtension->FrameListLock;

#define NUMBER_OF_FRAMES 20
		FrameList = ExAllocatePool(NonPagedPool, (NUMBER_OF_FRAMES * sizeof(ITSYETH_FRAME))+
			((NUMBER_OF_FRAMES+1)*MAXIMUM_ETHERNET_PACKET_SIZE));
		BufferMem = (PCHAR) &FrameList[NUMBER_OF_FRAMES];
		if (FrameList)
		{
			for (idx=0;idx<NUMBER_OF_FRAMES;idx++)
			{
				if (framelock != &deviceExtension->FrameListLock)
				{
					ItsyEth_KdPrint(DBGLVL_DEFAULT, ("bad lock %p %p\n", framelock, &deviceExtension->FrameListLock));
				}
				ExInterlockedInsertTailList(&deviceExtension->FreeFrames, 
					                        &FrameList[idx].node,
											&deviceExtension->FrameListLock);

				FrameList[idx].va = BufferMem + (idx * MAXIMUM_ETHERNET_PACKET_SIZE);
				// now align on a 16 byte boundary.
				FrameList[idx].va += 0x10 - ((unsigned int)FrameList[idx].va%0x10);
				FrameList[idx].length = MAXIMUM_ETHERNET_PACKET_SIZE;
				ItsyEth_KdPrint(DBGLVL_MAXIMUM, ("frame va %p\n", FrameList[idx].va));
			}
		}
	}
}

void
ItsyEth_GetFrame(IN  PITSYETH_WDM_HANDLE WdmHandle,
				 OUT PITSYETH_FRAME_HANDLE *handle,
				 OUT PVOID *va)
{
	PITSYETH_DEVICE_EXTENSION deviceExtension;
	PITSYETH_FRAME Frame;

	if (WdmHandle)
	{
		deviceExtension = (PITSYETH_DEVICE_EXTENSION) WdmHandle;
		if (framelock != &deviceExtension->FrameListLock)
		{
			ItsyEth_KdPrint(DBGLVL_DEFAULT, ("bad lock %p %p\n", framelock, &deviceExtension->FrameListLock));
		}
		Frame = (PITSYETH_FRAME) ExInterlockedRemoveHeadList(&deviceExtension->FreeFrames,
													         &deviceExtension->FrameListLock);
		*handle = (PVOID) Frame;
		*va = Frame->va;
		Frame->WdmHandle = WdmHandle;
		ItsyEth_KdPrint(DBGLVL_HIGH, ("get frame va %p\n", Frame->va));
	}
	else
	{
		*handle = NULL;
		*va = NULL;
	}
}

void 
ItsyEth_SetFrameLength(IN PITSYETH_FRAME_HANDLE handle,
					   IN int length)
{
	PITSYETH_FRAME Frame = (PITSYETH_FRAME) handle;

	if (Frame)
	{
		Frame->length = length;
	}
}


void
ItsyEth_FreeFrame(IN PITSYETH_WDM_HANDLE WdmHandle,
			      IN PITSYETH_FRAME_HANDLE Handle)
{
	PITSYETH_DEVICE_EXTENSION deviceExtension;
	PITSYETH_FRAME Frame;

	if ((WdmHandle) && (Handle))
	{
		deviceExtension = (PITSYETH_DEVICE_EXTENSION) WdmHandle;
		Frame = (PITSYETH_FRAME) Handle;
		ItsyEth_KdPrint(DBGLVL_HIGH, ("free frame va %p\n", Frame->va));
		if (framelock != &deviceExtension->FrameListLock)
		{
			ItsyEth_KdPrint(DBGLVL_DEFAULT, ("bad lock %p %p\n", framelock, &deviceExtension->FrameListLock));
		}
		ExInterlockedInsertTailList(&deviceExtension->FreeFrames,
			                        (PLIST_ENTRY) Handle,
  					                &deviceExtension->FrameListLock);
	}
}

int
GetAddress(PITSYETH_DEVICE_EXTENSION deviceExtension)
{
	ULONG Address =0;
	ULONG ResultLength;
	NTSTATUS Status;

	Status = IoGetDeviceProperty(deviceExtension->PhysicalDeviceObject,
		                         DevicePropertyAddress,
								 sizeof(ULONG),
								 (PVOID)&Address,
								 &ResultLength);
	if (Status == STATUS_SUCCESS)
	{
		ItsyEth_KdPrint(DBGLVL_DEFAULT, ("device address is %d\n", Address));
	}
	else
	{
		ItsyEth_KdPrint(DBGLVL_DEFAULT, ("Status %x\n", Status));
	}
	return 0;

#if 0
	NTSTATUS ntStatus, status = STATUS_SUCCESS;
    PIRP Irp;
    KEVENT event;
    IO_STATUS_BLOCK ioStatus;
    PIO_STACK_LOCATION nextStack;
	USB_NODE_CONNECTION_INFORMATION connectionInfo;
	int nBytes;
	USB_HUB_NAME HubName;

    // issue a synchronous request
    KeInitializeEvent(&event, NotificationEvent, FALSE);

	nBytes = sizeof(connectionInfo);
	RtlZeroMemory(&connectionInfo, nBytes);
	connectionInfo.ConnectionIndex = 1;

	// build an IRP for the URB.....
    Irp = IoBuildDeviceIoControlRequest(
                IOCTL_USB_GET_NODE_CONNECTION_INFORMATION,
                deviceExtension->TopOfStackDeviceObject, //Points to the next-lower driver's device object
                &connectionInfo,
                nBytes,
                &connectionInfo,
                nBytes,
                FALSE, // If InternalDeviceControl is TRUE the target driver's Dispatch
				      //  outine for IRP_MJ_INTERNAL_DEVICE_CONTROL or IRP_MJ_SCSI 
					  // is called; otherwise, the Dispatch routine for 
					  // IRP_MJ_DEVICE_CONTROL is called.
                &event,     // event to be signalled on completion
                &ioStatus);  // Specifies an I/O status block to be set when the request is completed the lower driver. 


    //
    // Call the class driver to perform the operation.  If the returned status
    // is PENDING, wait for the request to complete.
    //
	if (Irp)
	{
		//nextStack = IoGetNextIrpStackLocation(Irp);
		//nextStack->Parameters.Others.Argument1 = &HubName;
		//nextStack->Parameters.DeviceIoControl.OutputBufferLength;
		//Irp->AssociatedIrp.SystemBuffer = &HubName;
		ntStatus = IoCallDriver(deviceExtension->TopOfStackDeviceObject, Irp);
		if (ntStatus == STATUS_PENDING) 
		{
			status = KeWaitForSingleObject(&event,
				                           Suspended,
                                           KernelMode,
                                           FALSE,
                                           NULL);
		} 
		else 
		{
	        ioStatus.Status = ntStatus;
		}
		//IoFreeIrp(Irp);
	}
	ItsyEth_KdPrint(DBGLVL_DEFAULT, ("ioStatus %x, connection %s\n", ioStatus.Status, HubName.HubName));
	return 0;
#endif
}

/*++

Routine Description:

	Called as part of sudden device removal handling.
    Cancels any pending transfers for all open pipes. 
	If any pipes are still open, call USBD with URB_FUNCTION_ABORT_PIPE
	Also marks the pipe 'closed' in our saved  configuration info.

Arguments:

    Ptrs to our FDO

Return Value:

    NT status code

--*/
NTSTATUS
ItsyEth_AbortPipes(
    IN PITSYETH_DEVICE_EXTENSION deviceExtension
    )
{
    NTSTATUS ntStatus = STATUS_SUCCESS;
    PURB urb;
	ULONG i;
    PUSBD_INTERFACE_INFORMATION Interface;
	PUSBD_PIPE_INFORMATION PipeInfo;

    Interface = deviceExtension->UsbInterface;

    for (i=0; i<Interface->NumberOfPipes; i++) 
	{
        PipeInfo =  &Interface->Pipes[i]; // PUSBD_PIPE_INFORMATION  PipeInfo;

//		if ( PipeInfo->PipeFlags ) 
//		{ 
			// we set this if open, clear if closed
			ItsyEth_KdPrint( DBGLVL_DEFAULT,("ItsyEth_AbortPipes() Aborting open  Pipe %d\n", i));

			urb = ExAllocatePool(NonPagedPool, sizeof(struct _URB_PIPE_REQUEST));
			if (urb) 
			{
				urb->UrbHeader.Length = (USHORT) sizeof (struct _URB_PIPE_REQUEST);
				urb->UrbHeader.Function = URB_FUNCTION_ABORT_PIPE;
				urb->UrbPipeRequest.PipeHandle =
					PipeInfo->PipeHandle;

				ntStatus = ItsyEth_CallUSBD(deviceExtension->PhysicalDeviceObject,
					                        deviceExtension->TopOfStackDeviceObject,
											urb);

				ExFreePool(urb);
			} 
			else 
			{
				ntStatus = STATUS_INSUFFICIENT_RESOURCES;
				ItsyEth_KdPrint( DBGLVL_HIGH,("ItsyEth_AbortPipes() FAILED urb alloc\n" ));
				break;
			}


			if (!(NT_SUCCESS(ntStatus))) 
			{
				// if we failed, dump out
				break;
			}
			else 
			{
				PipeInfo->PipeFlags = FALSE; // mark the pipe 'closed'
			}

//		} // end, if pipe open
	} // end, for all pipes
    return ntStatus;
}


/*++

Routine Description:

    Abort a pipe, wait for all outstanding IRPs to complete and 
	Reset the pipe.

    NOTES:

    This will reset the host to Data0 and should also reset the device to Data0 
	This will be possibly be run at a higher than PASSIVE level and therefore I 
	cannot use ItsyEthCallUSBD.

Arguments:

    Ptrs to our FDO and a USBD_PIPE_INFORMATION struct

Return Value:

    NT status code

--*/
VOID
ItsyEth_ResetPipe(
    IN PVOID Context
    )
{
    NTSTATUS ntStatus = STATUS_SUCCESS;
	PITSYETH_DEVICE_EXTENSION deviceExtension = (PITSYETH_DEVICE_EXTENSION) Context;
    PUSBD_PIPE_INFORMATION PipeInfo = deviceExtension->OutPipe;
	PIRP irp;
	PURB urb;

    ItsyEth_KdPrint( DBGLVL_DEFAULT,("ItsyEth_ResetPipe() Pipe Request %x %x\n", deviceExtension, PipeInfo));

	// first abort all transactions on the pipe and then reset it.
    urb = ExAllocatePool(NonPagedPool, sizeof(struct _URB_PIPE_REQUEST));
    if (urb) 
	{
		RtlZeroMemory(urb, sizeof(struct _URB_PIPE_REQUEST));
        urb->UrbHeader.Length = (USHORT) sizeof (struct _URB_PIPE_REQUEST);
        urb->UrbHeader.Function = URB_FUNCTION_ABORT_PIPE;
        urb->UrbPipeRequest.PipeHandle = PipeInfo->PipeHandle;
		ntStatus = ItsyEth_CallUSBD(deviceExtension->PhysicalDeviceObject,
								    deviceExtension->TopOfStackDeviceObject,
									urb);

		if (NT_SUCCESS(ntStatus))
		{
			RtlZeroMemory(urb, sizeof(struct _URB_PIPE_REQUEST));
			urb->UrbHeader.Length = (USHORT) sizeof (struct _URB_PIPE_REQUEST);
			urb->UrbHeader.Function = URB_FUNCTION_RESET_PIPE;
			urb->UrbPipeRequest.PipeHandle = PipeInfo->PipeHandle;
			ntStatus = ItsyEth_CallUSBD(deviceExtension->PhysicalDeviceObject,
										deviceExtension->TopOfStackDeviceObject,
										urb);
		}
		ExFreePool(urb);
		deviceExtension->DeviceStalled = FALSE;
	}
	if (deviceExtension->ResetWork)
	{
		ExFreePool(deviceExtension->ResetWork);
		deviceExtension->ResetWork = NULL;
	}
}

/* XXX should redo the reset architecture so it is a bit cleaner */
VOID
ItsyEth_ResetPipeIn(
    IN PVOID Context
    )
{
    NTSTATUS ntStatus = STATUS_SUCCESS;
	PITSYETH_FRAME Frame = (PITSYETH_FRAME) Context;
	PITSYETH_DEVICE_EXTENSION deviceExtension = (PITSYETH_DEVICE_EXTENSION) Frame->WdmHandle;
    PUSBD_PIPE_INFORMATION PipeInfo = deviceExtension->InPipe;
	PIRP irp;
	PURB urb;

    ItsyEth_KdPrint( DBGLVL_DEFAULT, ("ItsyEth_ResetPipeIn() Pipe Request %x %x\n", deviceExtension, PipeInfo));

	// first abort all transactions on the pipe and then reset it.
    urb = ExAllocatePool(NonPagedPool, sizeof(struct _URB_PIPE_REQUEST));
    if (urb) 
	{
		RtlZeroMemory(urb, sizeof(struct _URB_PIPE_REQUEST));
        urb->UrbHeader.Length = (USHORT) sizeof (struct _URB_PIPE_REQUEST);
        urb->UrbHeader.Function = URB_FUNCTION_ABORT_PIPE;
        urb->UrbPipeRequest.PipeHandle = PipeInfo->PipeHandle;
		ntStatus = ItsyEth_CallUSBD(deviceExtension->PhysicalDeviceObject,
								    deviceExtension->TopOfStackDeviceObject,
									urb);

		RtlZeroMemory(urb, sizeof(struct _URB_PIPE_REQUEST));
		urb->UrbHeader.Length = (USHORT) sizeof (struct _URB_PIPE_REQUEST);
		urb->UrbHeader.Function = URB_FUNCTION_RESET_PIPE;
		urb->UrbPipeRequest.PipeHandle = PipeInfo->PipeHandle;
		ntStatus = ItsyEth_CallUSBD(deviceExtension->PhysicalDeviceObject,
									deviceExtension->TopOfStackDeviceObject,
									urb);
		ExFreePool(urb);
		if (NT_SUCCESS(ntStatus))
		{
			deviceExtension->DeviceStalled = FALSE;
		}
	}

	Frame->callback(Frame->Adapter, Frame->Packet, STATUS_UNSUCCESSFUL);
	if (deviceExtension->ResetWork)
	{
		ExFreePool(deviceExtension->ResetWork);
		deviceExtension->ResetWork = NULL;
	}
	ItsyEth_FreeFrame(Frame->WdmHandle, Frame);
}

/* XXX temp hack */
VOID
ItsyEth_PrintFrameNumber(
    PITSYETH_DEVICE_EXTENSION deviceExtension
    )
{
    NTSTATUS ntStatus = STATUS_SUCCESS;
	PURB urb;

	// first abort all transactions on the pipe and then reset it.
    urb = ExAllocatePool(NonPagedPool, sizeof(struct _URB_GET_FRAME_LENGTH));
    if (urb) 
	{
		RtlZeroMemory(urb, sizeof(struct _URB_GET_FRAME_LENGTH));
        urb->UrbHeader.Length = (USHORT) sizeof (struct _URB_GET_FRAME_LENGTH);
        urb->UrbHeader.Function = URB_FUNCTION_GET_CURRENT_FRAME_NUMBER;
		ntStatus = ItsyEth_CallUSBD(deviceExtension->PhysicalDeviceObject,
								    deviceExtension->TopOfStackDeviceObject,
									urb);
		ItsyEth_KdPrint( DBGLVL_DEFAULT, ("ItsyEth_FrameNumber() %d\n", urb->UrbGetCurrentFrameNumber.FrameNumber));
		ExFreePool(urb);
	}
}

