/****************************************************************************/
/* Copyright 1999 Compaq Computer Corporation.                              */
/*                                           .                              */
/* Copying or modifying this code for any purpose is permitted,             */
/* provided that this copyright notice is preserved in its entirety         */
/* in all copies or modifications.  COMPAQ COMPUTER CORPORATION             */
/* MAKES NO WARRANTIES, EXPRESSED OR IMPLIED, AS TO THE USEFULNESS          */
/* OR CORRECTNESS OF THIS CODE OR ITS FITNESS FOR ANY PARTICULAR            */
/* PURPOSE.                                                                 */
/****************************************************************************/

/****************************************************************************
Module Name:
    receive.c

Environment:
    Kernel Mode - Or whatever is the equivalent on WinNT

Abstract:
    This module contains the send routines and send interrupt handling code

*****************************************************************************/

#include <ndis.h>
#include <efilter.h>

#include "equates.h"
#include "itsyethdbg.h"
#include "itsyethsw.h"
#include "itsyethpr.h"
#include "itsyethcom.h"

void
ReceiveCleanup(PVOID AdapterArg, PVOID PacketArg, int);

VOID
ItsyEthInitialiseRecvQueue(PItsyEth_ADAPTER Adapter)
{
	NDIS_STATUS Status;
	
	NdisAllocatePacketPool(&Status, &Adapter->RecvPacketPool, 20, 16);
	if (Status != NDIS_STATUS_SUCCESS)
	{
		ItsyEth_KdPrint(DBGLVL_DEFAULT, ("Allocating Packet Pool Failed\n"));
	}
	NdisAllocateBufferPool(&Status, &Adapter->RecvBufferPool, 20);
	if (Status != NDIS_STATUS_SUCCESS)
	{
		ItsyEth_KdPrint(DBGLVL_DEFAULT, ("Allocating Buffer Pool Failed\n"));
	}

}

//-----------------------------------------------------------------------------
// Procedure: ItsyEthMultipleSend
//
// Description:This function takes an array from NDIS and puts as many as it can
//             in our list to be immediately transferred. Each packet has its
//             status set (NDIS_STATUS_RESOURCES, NDIS_STATUS_PENDED,
//             or NDIS_STATUS_SUCCESS) in the PacketArray individually.
//
//
// Arguments: MiniportAdapterContext (Adapter Structure pointer)
//            PacketArray - an array of pointers to NDIS_PACKET structs
//            PacketCount - number of packets in PacketArray
//
// Returns: nothing
//
//-----------------------------------------------------------------------------
//static PITSYETH_FRAME_HANDLE recvhandle = NULL;
//static PVOID FrameVA;
VOID
ItsyEthStartReceive(PItsyEth_ADAPTER Adapter)
{
    NDIS_STATUS         Status;
	PNDIS_BUFFER        Buffer;
	PVOID               BufferVA;
	UINT                BufferLength, TotalBufferLength;
	PITSYETH_FRAME_HANDLE FrameHandle;
	PVOID FrameVA;


    ItsyEth_KdPrint(DBGLVL_HIGH, ("ItsyEthStartReceive\n"));

	if (!Adapter)
	{
		ItsyEth_KdPrint(DBGLVL_DEFAULT, ("start receive %p\n", Adapter));
	}

	NdisAllocatePacket(&Status, &Adapter->RecvPacket, Adapter->RecvPacketPool);
	if (Status != NDIS_STATUS_SUCCESS)
	{
		ItsyEth_KdPrint(DBGLVL_DEFAULT, ("Allocating Packet Failed\n"));
		return;
	}
	
	NDIS_SET_PACKET_HEADER_SIZE(Adapter->RecvPacket, ETHERNET_HEADER_SIZE);

	ItsyEth_KdPrint(DBGLVL_HIGH, ("StartReceive : Allocating Packet\n"));
	ItsyEth_GetFrame(Adapter->WdmHandle, &FrameHandle, &FrameVA);
	
	ItsyEth_KdPrint(DBGLVL_HIGH, ("Start Receive Frame VA %p\n", FrameVA));
	NdisAllocateBuffer(&Status, 
		               &Buffer, 
					   Adapter->RecvBufferPool, 
					   FrameVA, 
					   MAXIMUM_ETHERNET_PACKET_SIZE);
	if (Status != NDIS_STATUS_SUCCESS)
	{
		ItsyEth_KdPrint(DBGLVL_DEFAULT, ("Allocating Buffer Failed\n"));
		return;
	}
    NdisChainBufferAtFront(Adapter->RecvPacket, Buffer);
	
	ItsyEth_KdPrint(DBGLVL_HIGH, ("Receive Packet %p buffer %p\n", Adapter->RecvPacket, Buffer));

	ItsyEth_Recv(Adapter->WdmHandle, 
                 FrameHandle,
				 ReceiveCleanup,
				 Adapter,
				 Adapter->RecvPacket);
	ItsyEth_KdPrint(DBGLVL_HIGH, ("Receive Packet exit\n"));
    return;
}

VOID
ItsyEthStartReceiveWork(
    IN struct _NDIS_WORK_ITEM *  pWorkItem,
    IN PVOID  Context
	)
{
	ItsyEthStartReceive((PItsyEth_ADAPTER) Context);
}
 

//-----------------------------------------------------------------------------
// Procedure:   ReceiveCleanup
//
// Description: This routine will clean up after a transmitted frame.  It will
//              update the transmit statistic counters, free up TCBs and map
//              regs, and issue a send complete which will unlock the pages.
//
// Arguments:
//      Adapter - ptr to Adapter object instance
//      Packet
//
// Returns:
//      TRUE - If we indicated any loopback packets during this function call
//      FALSE - If we didn't indicate any loopaback packets
//-----------------------------------------------------------------------------
void
ReceiveCleanup(PVOID AdapterArg, 
			   PVOID PacketArg,
			   int Status)
{
	PNDIS_PACKET Packet = PacketArg;
	PItsyEth_ADAPTER Adapter = AdapterArg;
	PNDIS_BUFFER Buffer;
	PCHAR  BufferVA;
	int BufferLength;
	int idx;
	PNDIS_WORK_ITEM startReceive;

	ItsyEth_KdPrint(DBGLVL_HIGH, ("ReceiveCleanup status %x\n", (Status==STATUS_SUCCESS)));
	if (!Packet || !Adapter)
	{
		ItsyEth_KdPrint(DBGLVL_DEFAULT, ("ReceiveCleanup %p %p\n", Adapter, Packet));
	}

//	NdisAllocateMemory(&startReceive, sizeof(NDIS_WORK_ITEM), 0, HighestAcceptableMax);
//	NdisInitializeWorkItem(startReceive, ItsyEthStartReceiveWork, (PVOID) Adapter);
//	NdisScheduleWorkItem(startReceive);

	NdisQueryPacket(Packet,
		            NULL,
					NULL,
					&Buffer,
					NULL);
	NdisQueryBuffer(Buffer,
		            &BufferVA,
					&BufferLength);

	ItsyEth_KdPrint(DBGLVL_HIGH, ("ReceiveCleanup %p %d\n", BufferVA, BufferLength));
	
	// the wdm system doesn't give us the adapter if there is an error
	// the adapter may have already been freed by the halt funciton.
	if (Status == STATUS_SUCCESS)
	{
		// tell NDIS to copy the packet and give me back control immediatly
		NDIS_SET_PACKET_STATUS(Packet, NDIS_STATUS_RESOURCES);

		// put the MAC addresses back...
		for (idx=0;idx<ETH_LENGTH_OF_ADDRESS;idx++)
	    {
			BufferVA[idx] = Adapter->NodeAddress[idx];
			BufferVA[ETH_LENGTH_OF_ADDRESS+idx] = Adapter->DevAddress[idx];
		}

		NdisMIndicateReceivePacket(Adapter->ItsyEthAdapterHandle,
			                       &Packet,
								   1);
	}
	ItsyEthReturnPackets(NULL, Packet);

	// start a new receive pronto!
	ItsyEthStartReceive(Adapter);

	return;
}

//-----------------------------------------------------------------------------
// ItsyEthGetReturnedPackets
//
// PARAMETERS: IN NDIS_HANDLE MiniportAdapterContext
//                 - a context version of our Adapter pointer
//             IN NDIS_PACKET Packet
//                 - the packet that is being freed
//
// DESCRIPTION: This function attempts to return to the receive free list the
//              packet passed to us by NDIS
//
// RETURNS: nothing
//
//-----------------------------------------------------------------------------

VOID
ItsyEthReturnPackets(NDIS_HANDLE  MiniportAdapterContext,
                     PNDIS_PACKET Packet)
{
	PNDIS_BUFFER Buffer;

    ItsyEth_KdPrint(DBGLVL_HIGH, ("ItsyEthReturnPackets\n"));

	NdisQueryPacket(Packet,
		            NULL,
					NULL,
					&Buffer,
					NULL);

	// we always receive into one buffer
	NdisAdjustBufferLength(Buffer, MAXIMUM_ETHERNET_PACKET_SIZE);
	NdisFreeBuffer(Buffer);
	NdisFreePacket(Packet);
}

void
AdjustBufferLength(PVOID PacketArg, int Length)
{
	PNDIS_PACKET Packet = PacketArg;
	PNDIS_BUFFER Buffer=NULL;

	if (!Packet)
	{
		ItsyEth_KdPrint(DBGLVL_DEFAULT, ("AdjustBufferLength %p\n", Packet));
	}
	NdisQueryPacket(Packet,
		            NULL,
					NULL,
					&Buffer,
                    NULL);
	NdisAdjustBufferLength(Buffer, Length);
	return;
}
