/****************************************************************************/
/* Copyright 1999 Compaq Computer Corporation.                              */
/*                                           .                              */
/* Copying or modifying this code for any purpose is permitted,             */
/* provided that this copyright notice is preserved in its entirety         */
/* in all copies or modifications.  COMPAQ COMPUTER CORPORATION             */
/* MAKES NO WARRANTIES, EXPRESSED OR IMPLIED, AS TO THE USEFULNESS          */
/* OR CORRECTNESS OF THIS CODE OR ITS FITNESS FOR ANY PARTICULAR            */
/* PURPOSE.                                                                 */
/****************************************************************************/

/****************************************************************************
** COPYRIGHT (C) 1994-1997 INTEL CORPORATION                               **
** DEVELOPED FOR MICROSOFT BY INTEL CORP., HILLSBORO, OREGON               **
** HTTP://WWW.INTEL.COM/                                                   **
** THIS FILE IS PART OF THE INTEL ETHEREXPRESS PRO/100B(TM) AND            **
** ETHEREXPRESS PRO/100+(TM) NDIS 5.0 MINIPORT SAMPLE DRIVER               **
****************************************************************************/

/****************************************************************************
Module Name:
    request.c

This driver runs on the following hardware:
    - 82557/82558 based PCI 10/100Mb ethernet adapters
    (aka Intel EtherExpress(TM) PRO Adapters)

Environment:
    Kernel Mode - Or whatever is the equivalent on WinNT

Revision History
    - JCB 8/14/97 Example Driver Created
*****************************************************************************/

#include <ndis.h>
#include <efilter.h>

#include "equates.h"
#include "itsyethdbg.h"
#include "itsyethsw.h"
#include "itsyethpr.h"
#include "itsyeth.rc"

//-----------------------------------------------------------------------------
// Procedure:   ItsyEthSetInformation
//
// Description: ItsyEthSetInformation handles a set operation for a single OID.
//              The only operations that really change the configuration of
//              the adapter are set PACKET_FILTER, and SET_MULTICAST.
//
// Arguments:
//      MiniportAdapterContext - The context value returned by the Miniport
//                               when the adapter was initialized (see the call
//                               NdisMSetAttributes. In reality, it is a
//                               pointer to ItsyEth_ADAPTER.
//      Oid - The Oid that is to be set.
//      InformationBuffer -  Holds the data to be set.
//      InformationBufferLength - The length of InformationBuffer.
//
// Result:
//      BytesRead - If the call is successful, returns the number of bytes
//                  read from InformationBuffer.
//      BytesNeeded - If there is not enough data in OvbBuffer to satisfy the
//                    OID, returns the amount of storage needed.
//
// Returns:
//      NDIS_STATUS_SUCCESS
//      NDIS_STATUS_INVALID_LENGTH
//      NDIS_STATUS_INVALID_OID
//      NDIS_STATUS_NOT_SUPPORTED
//      NDIS_STATUS_NOT_ACCEPTED
//-----------------------------------------------------------------------------

NDIS_STATUS
ItsyEthSetInformation(
                   IN NDIS_HANDLE MiniportAdapterContext,
                   IN NDIS_OID Oid,
                   IN PVOID InformationBuffer,
                   IN ULONG InformationBufferLength,
                   OUT PULONG BytesRead,
                   OUT PULONG BytesNeeded
                   )

{
    NDIS_STATUS         Status = NDIS_STATUS_SUCCESS;
    ULONG               PacketFilter;
    PItsyEth_ADAPTER       Adapter;

	ItsyEth_KdPrint(DBGLVL_DEFAULT, ("ItsyEthSetInformation %x\n", Oid));

	Adapter = PITSYETH_ADAPTER_FROM_CONTEXT_HANDLE(MiniportAdapterContext);

    *BytesRead = 0;
    *BytesNeeded = 0;

    switch (Oid)
    {
		case OID_802_3_MULTICAST_LIST:
			// don't support multicast
			return (NDIS_STATUS_NOT_SUPPORTED);
			break;

		case OID_GEN_CURRENT_PACKET_FILTER:
	        // Verify the Length
		    if (InformationBufferLength != 4)
			{
			    return (NDIS_STATUS_INVALID_LENGTH);
			}
	        // Now call the filter package to set the packet filter.
		    NdisMoveMemory((PVOID)&PacketFilter, InformationBuffer, sizeof(ULONG));
	        // Verify bits, if any bits are set that we don't support, leave
		    if (PacketFilter &
				~(NDIS_PACKET_TYPE_DIRECTED |
				NDIS_PACKET_TYPE_MULTICAST |
				NDIS_PACKET_TYPE_BROADCAST ))//|
//				NDIS_PACKET_TYPE_PROMISCUOUS |
//				NDIS_PACKET_TYPE_ALL_MULTICAST))
			{
				ItsyEth_KdPrint(DBGLVL_DEFAULT, ("Filter Type %08x not supported\n", PacketFilter));
				Status = NDIS_STATUS_NOT_SUPPORTED;
				*BytesRead = 4;
				break;
			}
			Status = NDIS_STATUS_SUCCESS;
			*BytesRead = InformationBufferLength;
			break;

		case OID_GEN_CURRENT_LOOKAHEAD:
	        // Verify the Length
		    if (InformationBufferLength != 4)
			{
				return (NDIS_STATUS_INVALID_LENGTH);
			}
	        *BytesRead = 4;
		    Status = NDIS_STATUS_SUCCESS;
			break;

		// stub the next four cases to set up power management stuff
		// if the adapter had supported power management, the commented lines
		// should replace the stub lines. See the power management spec
		case OID_PNP_SET_POWER:
			ItsyEth_KdPrint(DBGLVL_DEFAULT, ("SET: Power State change to 0x%08X!!!\n",InformationBuffer));
			//        *BytesRead = InformationBufferLength;
			//        Status = NDIS_STATUS_SUCCESS;
			*BytesRead = 0;                     //stub
			Status = NDIS_STATUS_NOT_SUPPORTED; //stub
			break;

		case OID_PNP_ADD_WAKE_UP_PATTERN:
			ItsyEth_KdPrint(DBGLVL_DEFAULT, ("SET: Got a WakeUpPattern SET    Call\n"));
			// call a function that would program the adapter's wake
			// up pattern, return success
			//*BytesRead = InformationBufferLength;
			*BytesRead = 0;                     //stub
			Status = NDIS_STATUS_NOT_SUPPORTED; //stub
			break;

		case OID_PNP_REMOVE_WAKE_UP_PATTERN:
			ItsyEth_KdPrint(DBGLVL_DEFAULT, ("SET: Got a WakeUpPattern REMOVE Call\n"));
			// call a function that would remove the adapter's wake
			// up pattern, return success
			//*BytesRead = InformationBufferLength;
			*BytesRead = 0;                     //stub
			Status = NDIS_STATUS_NOT_SUPPORTED; //stub
			break;

		case OID_PNP_ENABLE_WAKE_UP:
			ItsyEth_KdPrint(DBGLVL_DEFAULT, ("SET: Got a EnableWakeUp Call of %08X\n",InformationBuffer));
			// call a function that would enable wake up on the adapter
			// return success
			//*BytesRead = InformationBufferLength;
			*BytesRead = 0;                     //stub
			Status = NDIS_STATUS_NOT_SUPPORTED; //stub
			break;

		default:
			Status = NDIS_STATUS_INVALID_OID;
			break;

    }  // end of switch construct
    return Status;
}


//-----------------------------------------------------------------------------
// Procedure:   ItsyEthQueryInformation
//
// Description: ItsyEthQueryInformation handles a query operation for a single
//              OID. Basically we return information about the current state of
//              the OID in question.
//
// Arguments:
//      MiniportAdapterContext - The context value returned by the Miniport
//                               when the adapter was initialized (see the call
//                               NdisMSetAttributes. In reality, it is a
//                               pointer to ItsyEth_ADAPTER.
//      Oid - The Oid that is to be queried.
//      InformationBuffer -  A pointer to the buffer that holds the result of
//                           the query.
//      InformationBufferLength - The length of InformationBuffer.
//
// Result:
//      BytesWritten - If the call is successful, returns the number of bytes
//                     written into InformationBuffer.
//      BytesNeeded - If there is not enough data in OvbBuffer to satisfy the
//                    OID, returns the amount of storage needed.
//
// Returns:
//      NDIS_STATUS_SUCCESS
//      NDIS_STATUS_NOT_SUPPORTED
//      NDIS_STATUS_BUFFER_TO_SHORT
//-----------------------------------------------------------------------------

NDIS_STATUS
ItsyEthQueryInformation(
                     IN NDIS_HANDLE MiniportAdapterContext,
                     IN NDIS_OID Oid,
                     IN PVOID InformationBuffer,
                     IN ULONG InformationBufferLength,
                     OUT PULONG BytesWritten,
                     OUT PULONG BytesNeeded
                     )

{
    // order is important here because the OIDs should be in order
    // of increasing value
    static
    NDIS_OID        ItsyEthGlobalSupportedOids[] =
    {
        OID_GEN_SUPPORTED_LIST,
        OID_GEN_HARDWARE_STATUS,
        OID_GEN_MEDIA_SUPPORTED,
        OID_GEN_MEDIA_IN_USE,
        OID_GEN_MAXIMUM_LOOKAHEAD,
        OID_GEN_MAXIMUM_FRAME_SIZE,
        OID_GEN_LINK_SPEED,
        OID_GEN_TRANSMIT_BUFFER_SPACE,
        OID_GEN_RECEIVE_BUFFER_SPACE,
        OID_GEN_TRANSMIT_BLOCK_SIZE,
        OID_GEN_RECEIVE_BLOCK_SIZE,
        OID_GEN_VENDOR_ID,
        OID_GEN_VENDOR_DESCRIPTION,
        OID_GEN_CURRENT_PACKET_FILTER,
        OID_GEN_CURRENT_LOOKAHEAD,
        OID_GEN_DRIVER_VERSION,
        OID_GEN_MAXIMUM_TOTAL_SIZE,
        OID_GEN_PROTOCOL_OPTIONS,
        OID_GEN_MAC_OPTIONS,
        OID_GEN_MEDIA_CONNECT_STATUS,
        OID_GEN_MAXIMUM_SEND_PACKETS,
        OID_GEN_SUPPORTED_GUIDS,
        OID_GEN_XMIT_OK,
        OID_GEN_RCV_OK,
        OID_GEN_XMIT_ERROR,
        OID_GEN_RCV_ERROR,
        OID_GEN_RCV_NO_BUFFER,
        OID_GEN_RCV_CRC_ERROR,
        OID_GEN_TRANSMIT_QUEUE_LENGTH,
        OID_802_3_PERMANENT_ADDRESS,
        OID_802_3_CURRENT_ADDRESS,
        OID_802_3_MULTICAST_LIST,
        OID_802_3_MAXIMUM_LIST_SIZE,
        /* for ndis 4 packet priority
        OID_802_3_MAC_OPTIONS,
        */
        OID_802_3_RCV_ERROR_ALIGNMENT,
        OID_802_3_XMIT_ONE_COLLISION,
        OID_802_3_XMIT_MORE_COLLISIONS,
        OID_802_3_XMIT_DEFERRED,
        OID_802_3_XMIT_MAX_COLLISIONS,
        OID_802_3_RCV_OVERRUN,
        OID_802_3_XMIT_UNDERRUN,
        OID_802_3_XMIT_HEARTBEAT_FAILURE,
        OID_802_3_XMIT_TIMES_CRS_LOST,
        OID_802_3_XMIT_LATE_COLLISIONS,
        /* tcp/ip checksum offload */
        /*
        OID_TCP_TASK_OFFLOAD,
        */
        /* powermanagement */
        OID_PNP_CAPABILITIES,
        OID_PNP_SET_POWER,
        OID_PNP_QUERY_POWER,
        OID_PNP_ADD_WAKE_UP_PATTERN,
        OID_PNP_REMOVE_WAKE_UP_PATTERN,
        OID_PNP_ENABLE_WAKE_UP,
        /* end powermanagement */
    };

    // String describing our adapter
    char VendorDescriptor[] = "Compaq Computer Corporation";
    PItsyEth_ADAPTER   Adapter;
    UCHAR           VendorId[4];
    NDIS_MEDIUM     Medium = NdisMedium802_3;
    NDIS_HARDWARE_STATUS HardwareStatus = NdisHardwareStatusReady;
    UINT            GenericUlong;
    USHORT          GenericUShort;
    UCHAR           GenericArray[6];
    NDIS_STATUS     Status = NDIS_STATUS_SUCCESS;
    NDIS_PNP_CAPABILITIES    Power_Management_Capabilities;

    // Common variables for pointing to result of query
    PVOID           MoveSource = (PVOID) (&GenericUlong);
    ULONG           MoveBytes = sizeof(GenericUlong);


	ItsyEth_KdPrint(DBGLVL_DEFAULT, ("ItsyEthQueryInformation context %p, %x\n", 
		MiniportAdapterContext, Oid));

    Adapter = PITSYETH_ADAPTER_FROM_CONTEXT_HANDLE(MiniportAdapterContext);

    // Initialize the result
    *BytesWritten = 0;
    *BytesNeeded = 0;

    // Switch on request type
    switch (Oid)
    {

		case OID_GEN_MAC_OPTIONS:
			// NDIS_MAC_OPTION_COPY_LOOKAHEAD_DATA is set to indicate to the
			// protocol that it can access the lookahead data by any means that
			// it wishes.  On some systems there are fast copy routines that
			// may have trouble accessing shared memory.  Netcard drivers that
			// indicate data out of shared memory, should not have this flag
			// set on these troublesome systems  For the time being this driver
			// will set this flag.  This should be safe because the data area
			// of the RFDs is contained in uncached memory.

			// NOTE: Don't set NDIS_MAC_OPTION_RECEIVE_SERIALIZED if we
			// are doing multipacket (ndis4) style receives.

			GenericUlong = (ULONG)(NDIS_MAC_OPTION_TRANSFERS_NOT_PEND   |
				                   NDIS_MAC_OPTION_COPY_LOOKAHEAD_DATA  |
								   NDIS_MAC_OPTION_NO_LOOPBACK);
			break;


		case OID_GEN_SUPPORTED_LIST:
			MoveSource = (PVOID) (ItsyEthGlobalSupportedOids);
			MoveBytes = sizeof(ItsyEthGlobalSupportedOids);
			break;

		case OID_GEN_HARDWARE_STATUS:
			MoveSource = (PVOID)(&HardwareStatus);
			MoveBytes = sizeof(NDIS_HARDWARE_STATUS);
			break;

		case OID_GEN_MEDIA_SUPPORTED:
		case OID_GEN_MEDIA_IN_USE:
			MoveSource = (PVOID) (&Medium);
			MoveBytes = sizeof(NDIS_MEDIUM);
			break;

		case OID_GEN_MAXIMUM_LOOKAHEAD:
		case OID_GEN_CURRENT_LOOKAHEAD:
		case OID_GEN_MAXIMUM_FRAME_SIZE:
			GenericUlong = MAXIMUM_ETHERNET_PACKET_SIZE - ENET_HEADER_SIZE;
			break;

		case OID_GEN_MAXIMUM_TOTAL_SIZE:
		case OID_GEN_TRANSMIT_BLOCK_SIZE:
		case OID_GEN_RECEIVE_BLOCK_SIZE:
			GenericUlong = (ULONG) MAXIMUM_ETHERNET_PACKET_SIZE;
			break;

	    case OID_GEN_LINK_SPEED:
			// We won't always know what line speed we are actually running at, but
			// report the value that the we detected when the driver loaded.
			GenericUlong =  (((ULONG) Adapter->LineSpeedCur) * 10000);
			break;

	    case OID_GEN_TRANSMIT_BUFFER_SPACE:
			GenericUlong = (ULONG) MAXIMUM_ETHERNET_PACKET_SIZE;
			break;

		case OID_GEN_RECEIVE_BUFFER_SPACE:
			GenericUlong = (ULONG) MAXIMUM_ETHERNET_PACKET_SIZE;
			break;


		case OID_GEN_VENDOR_ID:
			NdisMoveMemory(VendorId, Adapter->NodeAddress, 3);
			VendorId[3] = 0x0;
			MoveSource = (PVOID) VendorId;
			MoveBytes = sizeof(VendorId);
			break;

		case OID_GEN_VENDOR_DESCRIPTION:
			MoveSource = (PVOID) VendorDescriptor;
			MoveBytes = sizeof(VendorDescriptor);
			break;


		case OID_GEN_DRIVER_VERSION:
			GenericUShort = (USHORT) ITSYETH_DRIVER_VERSION;
			MoveSource = (PVOID)(&GenericUShort);
			MoveBytes = sizeof(GenericUShort);
			break;

        // WMI support
		case OID_GEN_SUPPORTED_GUIDS:
			ItsyEth_KdPrint(DBGLVL_DEFAULT, ("OID_GEN_SUPPORTED_GUIDS\n"));
			break;

        /*
        ******************************************************************************
        // Task Offload
        case OID_GEN_TASK_OFFLOAD:
        // here we set up the capabilities that we support
        // later the protocol will come back with a set requesting
        // some subset of the  specific capabilities we support.
        // this is all commented out because it is just a stub
        //


        //             TaskOffload.Size = sizeof(NDIS_TASK_OFFLOAD);
        //             TaskOffload.Task = NDIS_TASK_TCP_IP_CHECKSUM;
        //             // offsetnext should be 0 if this is the last task supported.
        //             TaskOffload.OffsetNextTask = (ULONG) (&TaskOffload + sizeof(NDIS_TASK_OFFLOAD));
        //             TaskOffload.TaskBufferLength = sizeof(NDIS_TASK_TCP_IP_CHECKSUM);
        //
        //             //set up our TaskBuffer
        //             TaskBufferDescriptor.Receive.TcpChecksum = TRUE;
        //             TaskOffload.TaskBuffer = &TaskBufferDescriptor;
        //
        //             MoveSource = (PVOID) &TaskOffload;
        //             MoveBytes = sizeof(NDIS_TASK_OFFLOAD) * NUM_TASKS_SUPPORTED;

          Status = NDIS_STATUS_NOT_SUPPORTED; // stub
          break;

        ******************************************************************************
        */
		case OID_802_3_PERMANENT_ADDRESS:
			ETH_COPY_NETWORK_ADDRESS(
				(PCHAR) GenericArray,
				Adapter->PermanentNodeAddress);
	        MoveSource = (PVOID) (GenericArray);
			MoveBytes = ETH_LENGTH_OF_ADDRESS;
			break;

		case OID_802_3_CURRENT_ADDRESS:
			ETH_COPY_NETWORK_ADDRESS(
				GenericArray,
				Adapter->NodeAddress);
			MoveSource = (PVOID) (GenericArray);
			MoveBytes = ETH_LENGTH_OF_ADDRESS;
			break;

		case OID_802_3_MAXIMUM_LIST_SIZE:
			GenericUlong = (ULONG) MAX_MULTICAST_ADDRESSES;
			break;
 
		case OID_GEN_MAXIMUM_SEND_PACKETS:
			GenericUlong = (ULONG) MAX_ARRAY_SEND_PACKETS;
			break;

		case OID_GEN_MEDIA_CONNECT_STATUS:
			// now we simply return our status (NdisMediaState[Dis]Connected)
			// this line also covers the case where our link status hasn't changed.
			GenericUlong = (ULONG) Adapter->LinkIsActive;
	        break;

		case OID_PNP_CAPABILITIES:
			ItsyEth_KdPrint(DBGLVL_DEFAULT, ("QUERY:Got a Query Capabilities Call\n"));
			// since we are stubbing power management, return only the
			// D0 power state indicating that the lowest power state we
			// support is D0 (full power)
			// Power_Management_Capabilities.WakeUpCapabilities.MinMagicPacketWakeUp   = NdisDeviceStateD0;
			// Power_Management_Capabilities.WakeUpCapabilities.MinPatternWakeUp       = NdisDeviceStateD0;
			// Power_Management_Capabilities.WakeUpCapabilities.MinLinkChangeWakeUp    = NdisDeviceStateD0;

			// MoveSource = (PVOID) &Power_Management_Capabilities;
			// MoveBytes = sizeof(NDIS_PNP_CAPABILITIES);
			Status = NDIS_STATUS_NOT_SUPPORTED; //stub
			break;

		case OID_PNP_QUERY_POWER:
			ItsyEth_KdPrint(DBGLVL_DEFAULT, ("QUERY:Got a Query Power Call of 0x%08X\n",InformationBuffer));
			// Adapter->NextPowerState = (NDIS_DEVICE_POWER_STATE) InformationBuffer;
			// return success indicating we will do nothing to jeapordize the
			// transition...
			// Status is pre-set in this routine to Success
			// but since we aren't actually supporting power manangement
			// return failure.
			Status = NDIS_STATUS_NOT_SUPPORTED; //stub
			break;

	    default:
	        // The query is for a driver statistic, so we need to first
		    // update our statistics in software.

//        DumpStatsCounters(Adapter);
			switch (Oid)
			{

				case OID_GEN_XMIT_OK:
				case OID_GEN_RCV_OK:
				case OID_GEN_XMIT_ERROR:
				case OID_GEN_RCV_ERROR:
				case OID_GEN_RCV_NO_BUFFER:
				case OID_GEN_RCV_CRC_ERROR:
				case OID_GEN_TRANSMIT_QUEUE_LENGTH:
				case OID_802_3_RCV_ERROR_ALIGNMENT:
				case OID_802_3_XMIT_ONE_COLLISION:
				case OID_802_3_XMIT_MORE_COLLISIONS:
				case OID_802_3_XMIT_DEFERRED:
				case OID_802_3_XMIT_MAX_COLLISIONS:
				case OID_802_3_RCV_OVERRUN:
				case OID_802_3_XMIT_UNDERRUN:
				case OID_802_3_XMIT_HEARTBEAT_FAILURE:
				case OID_802_3_XMIT_TIMES_CRS_LOST:
		        case OID_802_3_XMIT_LATE_COLLISIONS:
					GenericUlong = (ULONG) 0;
					// default move is of ULONG size so no need to set anything 
					// up here.
					break;
				default:
					Status = NDIS_STATUS_NOT_SUPPORTED;
					break;
			}
    }

    if (Status == NDIS_STATUS_SUCCESS)
    {
        if (MoveBytes > InformationBufferLength)
        {
            // Not enough room in InformationBuffer. Punt
            *BytesNeeded = MoveBytes;

            Status = NDIS_STATUS_BUFFER_TOO_SHORT;
        }
        else
        {
            // Copy result into InformationBuffer
            *BytesWritten = MoveBytes;
            if (MoveBytes > 0)
                NdisMoveMemory(InformationBuffer, MoveSource, MoveBytes);
        }
    }

    return (Status);
}   /* end of ItsyEthQueryInformation */

