

-- VHDL-AMS model VCO with phase integration 
-- Southampton VHDL-AMS Validation Suite - example 6 
-- author: Tom Kazmierski 
-- Department of Electronics and Computer Science, University of Southampton 
-- Highfield, Southampton SO17 1BJ, United Kingdom 
-- Tel. +44 (01703) 593520   Fax +44 (01703) 592901 
-- e-mail: tjk@ecs.soton.ac.uk 
-- Created: 2 June 1997 
-- Last revised: 5 June 1997 
-- 
-- Description 
-- A differential equation, in which the varying frequency is the 
-- time derivative of phase, is used to model the basic VCO operation. 
-- The phase is modelled as a free quantity local to the architecture. 
-- 

library IEEE, Disciplines; 
use IEEE.math_real.all; 
use Disciplines.electrical_system.all; 

entity VCO is 
  generic( 
          fc: real := 1E6;    -- VCO frequency at Vc 
          df: real := 0.5E6;  -- [Hz/V], frequency characteristic slope 
          Vc: voltage := 0.0 --  centre frequency input voltage 
         ); 
  port( quantity Vin: in voltage; 
                terminal OutTerminal: electrical); 
end entity VCO; 

architecture PhaseIntegrator of VCO is 
    constant TwoPi: real := 6.283118530718; -- 2pi 

    -- Phase is a free quantity: 
    quantity Phase : real; 

    -- define a branch for the output voltage source 
    quantity Vout across Iout through OutTerminal to ground; 

begin 
   -- use break to set the phase initial condition 
   break Phase => 0.0; 

   -- another break statement keeps the phase within 0.. 2pi 
   break Phase => Phase mod TwoPi on Phase'above(TwoPi); 

   -- phase equation 
   Phase'dot == TwoPi*realmax(0.5E6, fc+(Vin-Vc)*df); 

     -- output voltage source equation 
     Vout == 2.5*(1.0+sin(Phase)); 

end architecture PhaseIntegrator; 
