
#include <stdio.h>
#include <assert.h>

#include "tcl.h"
#include "library\PCI.h"
#include "bscan.h"
#include "SA1110BS.h"

typedef unsigned _int32 uint32;

static struct AEROSCANINFO {
    int reset; /* must be set before we do anything interesting */
    int sa1110Instruction;
    int sa1110ScanringLength;
    int sa1110Bypassed;
} aeroScanInfo;

struct Scanring *aeroScanRing = NULL;
void reset_aero_scanring();


static int Aero_write_egpio(ClientData clientData, Tcl_Interp *interp, 
			    int objc, struct Tcl_Obj * CONST objv[])
{
  if (objc == 4) {
    char irdata[32];
    char drdata1[4096];
    char drdata2[4096];
    char drdata3[4096];
    long addr = 0;
    long data = 0;
    long cs = 0;
    int rc;

    assert(aeroScanInfo.reset);

    memset(irdata, 0, 32);
    memset(drdata1, 0, 4096);
    memset(drdata2, 0, 4096);
    memset(drdata3, 0, 4096);

    rc = Tcl_GetIntFromObj(interp, objv[1], &addr);
    if (rc != TCL_OK) {
      Tcl_AppendResult(interp, "Could not get addr from ", 
		       Tcl_GetStringFromObj(objv[1], NULL),
		       NULL);
      return TCL_ERROR;
    }
    rc = Tcl_GetIntFromObj(interp, objv[2], &data);
    if (rc != TCL_OK) {
      Tcl_AppendResult(interp, "Could not get data from ", 
		       Tcl_GetStringFromObj(objv[2], NULL),
		       NULL);
      return TCL_ERROR;
    }
    rc = Tcl_GetIntFromObj(interp, objv[3], &cs);
    if (rc != TCL_OK) {
      Tcl_AppendResult(interp, "Could not get CS from ", 
		       Tcl_GetStringFromObj(objv[3], NULL),
		       NULL);
      return TCL_ERROR;
    }

    /* set up instruction */
    scan_ir(0, (const short*)aeroScanRing->instruction, aeroScanRing->instructionNBits, (short *)irdata);

    reset_aero_scanring();

    enable_SA1110_A();
    enable_SA1110_D();
    enable_SA1110_WE();
    enable_SA1110_OE();
    enable_SA1110_CAS();

    /* enable the EGPIO */
    set_SA1110_RESETO(1);
    set_SA1110_CS(cs);
    /* disable the DRAM */
    set_SA1110_CAS(0xFF);

    set_SA1110_WE(0);
    set_SA1110_OE(1);

    set_SA1110_A(addr);
    set_SA1110_D(data);

    scan_ir(0, (const short*)aeroScanRing->instruction, aeroScanRing->instructionNBits, (short *)irdata);
    scan_dr(0, (const short*)aeroScanRing->data, aeroScanRing->nBits, (short *)drdata1);
    set_SA1110_WE(1);
    set_SA1110_CS(0xFF);
    scan_dr(0, (const short*)aeroScanRing->data, aeroScanRing->nBits, (short *)drdata2);

    /* disable D and A */
    disable_SA1110_D();
    disable_SA1110_A();
    scan_dr(0, (const short*)aeroScanRing->data, aeroScanRing->nBits, (short *)drdata3);

    return TCL_OK;
  } else {
    Tcl_AppendResult(interp, "Wrong number of args to write_flash",
		     NULL);
    return TCL_ERROR;
  }
}

static int Aero_write_flash(ClientData clientData, Tcl_Interp *interp, 
			       int objc, struct Tcl_Obj * CONST objv[])
{
  if (objc == 3) {
    char irdata[32];
    char drdata1[4096];
    char drdata2[4096];
    char drdata3[4096];
    long addr = 0;
    long data = 0;
    int rc;

    assert(aeroScanInfo.reset);

    memset(irdata, 0, 32);
    memset(drdata1, 0, 4096);
    memset(drdata2, 0, 4096);
    memset(drdata3, 0, 4096);

    rc = Tcl_GetIntFromObj(interp, objv[1], &addr);
    if (rc != TCL_OK) {
      Tcl_AppendResult(interp, "Could not get addr from ", 
		       Tcl_GetStringFromObj(objv[1], NULL),
		       NULL);
      return TCL_ERROR;
    }
    rc = Tcl_GetIntFromObj(interp, objv[2], &data);
    if (rc != TCL_OK) {
      Tcl_AppendResult(interp, "Could not get data from ", 
		       Tcl_GetStringFromObj(objv[2], NULL),
		       NULL);
      return TCL_ERROR;
    }


    /* set up instruction */
    scan_ir(0, (const short*)aeroScanRing->instruction, aeroScanRing->instructionNBits, (short *)irdata);

    reset_aero_scanring();

    enable_SA1110_A();
    enable_SA1110_D();
    enable_SA1110_WE();
    enable_SA1110_OE();
    enable_SA1110_CAS();

    /* enable the ROM */
    set_SA1110_CS(0xFE);
    /* disable the DRAM */
    set_SA1110_CAS(0xFF);

    set_SA1110_WE(1);
    set_SA1110_OE(1);

    set_SA1110_A(addr);
    set_SA1110_D(data);

    scan_ir(0, (const short*)aeroScanRing->instruction, aeroScanRing->instructionNBits, (short *)irdata);
    scan_dr(0, (const short*)aeroScanRing->data, aeroScanRing->nBits, (short *)drdata1);
    set_SA1110_WE(0);
    scan_dr(0, (const short*)aeroScanRing->data, aeroScanRing->nBits, (short *)drdata2);

    /* *** drive 1 on corresponding bit on xilinx gpio part */
    set_SA1110_WE(1);
    set_SA1110_CS(0xFF);

    scan_dr(0, (const short*)aeroScanRing->data, aeroScanRing->nBits, (short *)drdata2);

    /* disable D and A */
    disable_SA1110_D();
    disable_SA1110_A();
    scan_dr(0, (const short*)aeroScanRing->data, aeroScanRing->nBits, (short *)drdata3);

    return TCL_OK;
  } else {
    Tcl_AppendResult(interp, "Wrong number of args to write_flash",
		     NULL);
    return TCL_ERROR;
  }
}

/* this is the one that works "old_way" */
static int Aero_read_flash(ClientData clientData, Tcl_Interp *interp, 
			   int objc, struct Tcl_Obj * CONST objv[])
{
  if (objc == 2) {
    char *tmpdata = NULL;
    char buf[1024];
    char irdata[32];
    char drdata1[4096];
    char drdata2[4096];
    char drdata3[4096];
    long addr = 0;
    long data = 0;
    int rc;

    assert(aeroScanInfo.reset);

    memset(irdata, 0, 32);
    memset(drdata1, 0, 4096);
    memset(drdata2, 0, 4096);

    rc = Tcl_GetIntFromObj(interp, objv[1], &addr);
    if (rc != TCL_OK) {
      Tcl_AppendResult(interp, "Could not get addr from ", 
		       Tcl_GetStringFromObj(objv[1], NULL),
		       NULL);
      return TCL_ERROR;
    }

    /* set up EXTEST instruction */
    reset_aero_scanring();

    *(long*)aeroScanRing->instruction =  SA1110_EXTEST; /* EXTEST: 0b000000000 */
    aeroScanRing->instructionNBits = SA1110_INSTRUCTION_LENGTH;
    aeroScanRing->instructionNBytes = 1;

    enable_SA1110_A();
    enable_SA1110_WE();
    enable_SA1110_OE();
    enable_SA1110_CAS();

    /* disable the SDRAM */
    set_SA1110_CAS(0xFF);

    /* enable the ROM */
    set_SA1110_CS(0x3e);

    set_SA1110_WE(1);
    set_SA1110_OE(0);
    set_SA1110_A(addr);

    scan_ir(0, (const short*)aeroScanRing->instruction, aeroScanRing->instructionNBits, (short *)irdata);

    scan_dr(0, (const short*)aeroScanRing->data, aeroScanRing->nBits, (short *)drdata1);
    scan_dr(0, (const short*)aeroScanRing->data, aeroScanRing->nBits, (short *)drdata2);

    /* disable the ROM */
    set_SA1110_CS(0xFF);

    scan_dr(0, (const short*)aeroScanRing->data, aeroScanRing->nBits, (short *)drdata3);

    memcpy(aeroScanRing->data, drdata2, aeroScanRing->nBytes);

    sprintf(buf, "0x%lx", get_SA1110_D());
    Tcl_AppendResult(interp, buf, NULL);
    return TCL_OK;
  } else {
    Tcl_AppendResult(interp, "Wrong number of args to set_oc_output",
		     NULL);
    return TCL_ERROR;
  }
}



static int init_aero_scanring();

void reset_aero_scanring()
{
  assert(aeroScanRing != NULL);
  clear_scanring(aeroScanRing);
  assert(aeroScanInfo.sa1110ScanringLength > 1);
  resetSA1110Scanring();
  
  enable_SA1110_WE();
  enable_SA1110_OE();
  enable_SA1110_CAS();

  /* disable PCMCIA bus */
  set_SA1110_PIOR(1);
  set_SA1110_PIOW(1);
  set_SA1110_POE(1);
  set_SA1110_PWE(1);
  set_SA1110_PCE1(1);
  set_SA1110_PCE2(1);
  set_SA1110_RESETO(1);

  /* disable all the CS's */
  set_SA1110_CS(0xFF);
  
  set_SA1110_OE(1);
  set_SA1110_OE(1);

}

static int init_aero_scanring()
{
   char irdata[32];
   char drdata1[4096];

   aeroScanRing = new_scanring(SA1110_SCANRING_LENGTH);
   assert(aeroScanRing != NULL);
   sa1110Scanring = new_subring(aeroScanRing, 0);

   aeroScanInfo.sa1110Instruction = SA1110_EXTEST;
   aeroScanInfo.sa1110ScanringLength = SA1110_SCANRING_LENGTH;

   aeroScanRing->nBits = aeroScanInfo.sa1110ScanringLength;
   aeroScanRing->nBytes = (aeroScanRing->nBits + 7) >> 3;

  *(long*)aeroScanRing->instruction = aeroScanInfo.sa1110Instruction;
   aeroScanRing->instructionNBits = SA1110_INSTRUCTION_LENGTH;

   // ***   aeroScanRing->instructionNBytes = 2;   
   aeroScanRing->instructionNBytes = (aeroScanRing->instructionNBits + 7) >> 3;   /* *** ??? Is this right? */
   
   /* put both chips in EXTEST mode with outputs disabled or safe values driven */
   reset_aero_scanring();

   scan_ir(0, (const short*)aeroScanRing->instruction, aeroScanRing->instructionNBits, (short *)irdata);
   scan_dr(0, (const short*)aeroScanRing->data, aeroScanRing->nBits, (short *)drdata1);

   aeroScanInfo.reset = 1;

   return TCL_OK;
}




int Aero_Init(Tcl_Interp *interp)
{
   int rc = TCL_OK;

   /* do a hard reset */
   hard_reset(0, _3_20);
   set_scan_clk(0, _30MHZ, 0, CLK_DIV_BY_32);

   aeroScanRing = new_scanring(SA1110_SCANRING_LENGTH);
   init_aero_scanring();
   reset_aero_scanring();

   Tcl_CreateObjCommand(interp, "aero_write_egpio", Aero_write_egpio, NULL, NULL);
   Tcl_CreateObjCommand(interp, "aero_write_flash", Aero_write_flash, NULL, NULL);
   Tcl_CreateObjCommand(interp, "aero_read_flash", Aero_read_flash, NULL, NULL);

   return rc;
}
