
#include <stdio.h>
#include <assert.h>

#include "tcl.h"
#include "library\PCI.h"
#include "bscan.h"
#include "sa110bs.h"
#include "21285bs.h"
#include "xc9572xl_tq100bs.h"
#include "EVERESTbs.h"

typedef unsigned _int32 uint32;

static struct SKFMLB {
    int reset; /* must be set before we do anything interesting */
    int corelogicInstruction;
    int corelogicScanringLength;
    int corelogicBypassed;
    int sa110Instruction;
    int sa110ScanringLength;
    int sa110Bypassed;

    int everestInstruction;
    int everestScanringLength;
    int everestBypassed;

    int xc9572xl_tq100Instruction;
    int xc9572xl_tq100ScanringLength;
    int xc9572xl_tq100Bypassed;
} skfmlb;

static int Corelis_test(ClientData clientData, Tcl_Interp *interp, 
			int objc, struct Tcl_Obj * CONST objv[])
{
  if (test(1) == 1) {
    return TCL_OK;
  } else {
    Tcl_AppendResult(interp, "Corelis test() failed.\n",
		     NULL);
    return TCL_ERROR;
  }
}

static int Corelis_read_io(ClientData clientData, Tcl_Interp *interp, 
			   int objc, struct Tcl_Obj * CONST objv[])
{
  if (objc == 2) {
    int rc = TCL_OK;
    unsigned int port;
    unsigned int input_data;
    rc = Tcl_GetIntFromObj(interp, objv[1], &port);
    if (rc != TCL_OK) {
      Tcl_AppendResult(interp, "Could not get int from port: ", 
		       Tcl_GetStringFromObj(objv[1], NULL),
		       NULL);
      return TCL_ERROR;
    }
    input_data = read_io((char)port);
    Tcl_SetObjResult(interp, Tcl_NewIntObj(input_data));
	return TCL_OK;
  } else {
    Tcl_AppendResult(interp, "Wrong number of args to set_io",
		     NULL);
    return TCL_ERROR;
  }
}


static int Corelis_set_io(ClientData clientData, Tcl_Interp *interp, 
						  int objc, struct Tcl_Obj * CONST objv[])
{
  if (objc == 3) {
    int rc = TCL_OK;
    unsigned int port;
    unsigned int output_data;
    rc = Tcl_GetIntFromObj(interp, objv[1], &port);
    if (rc != TCL_OK) {
      Tcl_AppendResult(interp, "Could not get int from port: ", 
		       Tcl_GetStringFromObj(objv[1], NULL),
		       NULL);
      return TCL_ERROR;
    }
    rc = Tcl_GetIntFromObj(interp, objv[2], &output_data);
    if (rc != TCL_OK) {
      Tcl_AppendResult(interp, "Could not get int from output_data: ", 
		       Tcl_GetStringFromObj(objv[2], NULL),
		       NULL);
      return TCL_ERROR;
    }
    set_io((char)port, (char)output_data);
	return TCL_OK;
  } else {
    Tcl_AppendResult(interp, "Wrong number of args to set_io",
		     NULL);
    return TCL_ERROR;
  }
}

static int Corelis_set_oc_output(ClientData clientData, Tcl_Interp *interp, 
								 int objc, struct Tcl_Obj * CONST objv[])
{
  if (objc == 3) {
    int rc = TCL_OK;
    unsigned int bit;
    unsigned int output_data;
    rc = Tcl_GetIntFromObj(interp, objv[1], &bit);
    if (rc != TCL_OK) {
      Tcl_AppendResult(interp, "Could not get int from bit: ", 
		       Tcl_GetStringFromObj(objv[1], NULL),
		       NULL);
      return TCL_ERROR;
    }
    rc = Tcl_GetIntFromObj(interp, objv[2], &output_data);
    if (rc != TCL_OK) {
      Tcl_AppendResult(interp, "Could not get int from output_data: ", 
		       Tcl_GetStringFromObj(objv[2], NULL),
		       NULL);
      return TCL_ERROR;
    }
    set_oc_output((char)bit, (char)output_data);
    return TCL_OK;
  } else {
    Tcl_AppendResult(interp, "Wrong number of args to set_oc_output",
		     NULL);
    return TCL_ERROR;
  }
}


static int Corelis_tms_reset(ClientData clientData, Tcl_Interp *interp, 
                             int objc, struct Tcl_Obj * CONST objv[])
{
  if (objc == 2) {
    int rc = TCL_OK;
    enum TAPS test_bus;
    rc = Tcl_GetIntFromObj(interp, objv[1], &test_bus);
    if (rc != TCL_OK) {
      Tcl_AppendResult(interp, "Could not get int from ", 
		       Tcl_GetStringFromObj(objv[1], NULL),
		       NULL);
      return TCL_ERROR;
    }
    tms_reset(test_bus);
    return TCL_OK;
  } else {
    Tcl_AppendResult(interp, "Wrong number of args to tms_reset",
		     NULL);
    return TCL_ERROR;
  }  
}

static int Corelis_trst_reset(ClientData clientData, Tcl_Interp *interp, 
                             int objc, struct Tcl_Obj * CONST objv[])
{
  if (objc == 2) {
    int rc = TCL_OK;
    enum TAPS test_bus;
    rc = Tcl_GetIntFromObj(interp, objv[1], &test_bus);
    if (rc != TCL_OK) {
      Tcl_AppendResult(interp, "Could not get int from ", 
		       Tcl_GetStringFromObj(objv[1], NULL),
		       NULL);
      return TCL_ERROR;
    }
    trst_reset(test_bus);
    return TCL_OK;
  } else {
    Tcl_AppendResult(interp, "Wrong number of args to trst_reset",
		     NULL);
    return TCL_ERROR;
  }  
};

static int Corelis_soft_reset(ClientData clientData, Tcl_Interp *interp, 
                             int objc, struct Tcl_Obj * CONST objv[])
{
  if (objc == 2) {
    int rc = TCL_OK;
    enum TAPS test_bus;
    rc = Tcl_GetIntFromObj(interp, objv[1], &test_bus);
    if (rc != TCL_OK) {
      Tcl_AppendResult(interp, "Could not get int from ", 
		       Tcl_GetStringFromObj(objv[1], NULL),
		       NULL);
      return TCL_ERROR;
    }
    soft_reset(test_bus);
    return TCL_OK;
  } else {
    Tcl_AppendResult(interp, "Wrong number of args to soft_reset",
		     NULL);
    return TCL_ERROR;
  }  
}

static int Corelis_hard_reset(ClientData clientData, Tcl_Interp *interp, 
                             int objc, struct Tcl_Obj * CONST objv[])
{
  short int rs422_enable = 0;
  hard_reset(rs422_enable, _3_20);
  return TCL_OK;
}

static int Corelis_set_scan_clk(ClientData clientData, Tcl_Interp *interp, 
                                int objc, struct Tcl_Obj * CONST objv[])
{
   if (objc == 3) {
      long clkspd;
      long divby;
      int rc;
      rc = Tcl_GetIntFromObj(interp, objv[1], &clkspd);
      if (rc != TCL_OK) {
         Tcl_AppendResult(interp, "Could not get clkspd from ", 
                          Tcl_GetStringFromObj(objv[1], NULL),
                          NULL);
         return TCL_ERROR;
      }
      rc = Tcl_GetIntFromObj(interp, objv[2], &divby);
      if (rc != TCL_OK) {
         Tcl_AppendResult(interp, "Could not get divby from ", 
                          Tcl_GetStringFromObj(objv[1], NULL),
                          NULL);
         return TCL_ERROR;
      }
      set_scan_clk(0, clkspd, 0, (short)divby);
      return TCL_OK;
   }   
   return TCL_ERROR;
}

/**
 * move_to_state final_state
 *  where final_state is one of TLR, RTI, PDR, PIR, SDR, SIR
 */
static int Corelis_move_to_state(ClientData clientData, Tcl_Interp *interp, 
                                 int objc, struct Tcl_Obj * CONST objv[])
{
  if (objc == 2) {
    int rc = TCL_OK;
    const char *statename = Tcl_GetStringFromObj(objv[1], NULL);
    long state = 0;
    if (strcmp(statename, "TLR") == 0) {
      state = STATE_TLR;
    } else if (strcmp(statename, "RTI") == 0) {
      state = STATE_RTI;
    } else if (strcmp(statename, "PDR") == 0) {
      state = STATE_PDR;
    } else if (strcmp(statename, "PIR") == 0) {
      state = STATE_PIR;
    } else if (strcmp(statename, "SDR") == 0) {
      state = STATE_SDR;
    } else if (strcmp(statename, "SIR") == 0) {
      state = STATE_SIR;
    } else {
      Tcl_AppendResult(interp, "expected TLR, RTI, PDR, PIR, SDR, or SIR, got ", statename, NULL);
      return TCL_ERROR;
    }
    move_to_state(0, state);
    return TCL_OK;
  } else {
    Tcl_AppendResult(interp, "Wrong number of args to set_oc_output",
		     NULL);
    return TCL_ERROR;
  }
}


struct Scanring *insr = NULL;
struct Scanring *outsr = NULL;

char instring[4096];
char outstring[4096];

/** 
 *  bitsToString expands Scanring sr into null-terminated instring.
 */
void bitsToString(char *str, struct Scanring *sr)
{
  int i, b;
  long nBits = sr->nBits;
  for (i = 0, b=nBits-1; i < nBits; i++, b--) {
    if (get_scanring_bit(sr, b))
      str[i] = '1';
    else
      str[i] = '0';
  }
  str[nBits] = 0;
}

void irBitsToString(char *str, struct Scanring *sr)
{
  int i, b;
  long nBits = sr->instructionNBits;
  for (i = 0, b=nBits-1; i < nBits; i++, b--) {
    if (get_scanring_bit(sr, b))
      str[i] = '1';
    else
      str[i] = '0';
  }
  str[nBits] = 0;
}
/** 
 *  bitsFromString fills scanring from ascii instring
 */
void bitsFromString(Tcl_Interp *interp, const char *str, struct Scanring *sr)
{
  int i, b;
  long nBits = strlen(str);

  sr->nBits = nBits;
  sr->nBytes = nBits/8;
  if (sr->nBits % 8 != 0)
    sr->nBytes++;
  memset(sr->data, 0, sr->nBytes);
  for (i = 0, b=nBits-1; i < nBits; i++, b--) {
    if (str[i] == '1') {
      set_scanring_bit(sr, b, 1);
    } else if (str[i] == '0') {
      set_scanring_bit(sr, b, 0);
    } else {
      char buf[4096];
      sprintf(buf, "-- ending string at character '%c'\n", str[i]);
      Tcl_AppendResult(interp, buf, NULL);
      break;
    }
  }
}

static int Corelis_test_scanring(ClientData clientData, Tcl_Interp *interp, 
                                 int objc, struct Tcl_Obj * CONST objv[])
{
  if (objc == 2) {

    /* from string to bits and back */
    bitsFromString(interp, Tcl_GetStringFromObj(objv[1], NULL), insr);
#if VERBOSE
    Tcl_AppendResult(interp, "Converted '", Tcl_GetStringFromObj(objv[1], NULL), "' to bits\n", NULL);
    sprintf(buf, "%d", insr->nBits);
    Tcl_AppendResult(interp, " nBits=", buf, "\n", NULL);
    sprintf(buf, "%x", insr->data[0]);
    Tcl_AppendResult(interp, " bits=", buf, "\n", NULL);
#endif
    bitsToString(instring, insr);
    Tcl_AppendResult(interp, instring, NULL);
    return TCL_OK;
  } else {
    Tcl_AppendResult(interp, "Wrong number of args to set_oc_output",
		     NULL);
    return TCL_ERROR;
  }
}

static int Corelis_scan_ir(ClientData clientData, Tcl_Interp *interp, 
                                 int objc, struct Tcl_Obj * CONST objv[])
{
  if (objc == 2) {
    /* from string to bits, scan_ir, and bits to string */
    bitsFromString(interp, Tcl_GetStringFromObj(objv[1], NULL), outsr);
    memset(insr->data, 0, outsr->nBytes);
    scan_ir(0, (const short *)outsr->data, outsr->nBits, (short *)insr->data);

    insr->nBits = outsr->nBits;
    insr->nBytes = outsr->nBytes;
    bitsToString(instring, insr);
    Tcl_AppendResult(interp, instring, NULL);
    return TCL_OK;
  } else {
    Tcl_AppendResult(interp, "Wrong number of args to scan_ir",
		     NULL);
    return TCL_ERROR;
  }
}

static int Corelis_scan_dr(ClientData clientData, Tcl_Interp *interp, 
			   int objc, struct Tcl_Obj * CONST objv[])
{
  if (objc == 2) {
    /* from string to bits, scan_ir, and bits to string */
    const char *outstring = Tcl_GetStringFromObj(objv[1], NULL);
    int mode;
    Tcl_Channel outchan = Tcl_GetChannel(interp, "stdout", &mode);
    char buf[4096];

    outsr->nBytes = 0;
    bitsFromString(interp, outstring, outsr);

    if (0) {
      int i;
      sprintf(buf, "nBits=%d nBytes=%d hex(first word=%08x):", 
              outsr->nBits, outsr->nBytes, *(long*)(outsr->data));
      Tcl_Write(outchan, buf, strlen(buf));
      for (i = 0; i <= outsr->nBytes; i++) {
	sprintf(buf, "%d:%04x ", i, outsr->data[i]);
	Tcl_Write(outchan, buf, strlen(buf));
      }
      sprintf(buf, "\n");
      Tcl_Write(outchan, buf, strlen(buf));
    }

    scan_dr(0, (const short *)outsr->data, outsr->nBits, (short *)insr->data);

    insr->nBits = outsr->nBits;
    insr->nBytes = outsr->nBytes;
    bitsToString(instring, insr);
    Tcl_AppendResult(interp, instring, NULL);
    return TCL_OK;
  } else {
    Tcl_AppendResult(interp, "Wrong number of args to scan_dr",
		     NULL);
    return TCL_ERROR;
  }
}

static int Corelis_scan_with_pause(ClientData clientData, Tcl_Interp *interp, 
				   int objc, struct Tcl_Obj * CONST objv[])
{
  if (objc == 2 || objc == 3) {
    int n;
    long bitstride = 64;
    long bytestride = bitstride/8;
    const char *outdata = outsr->data;
    char *indata = insr->data;
    long nBits = outsr->nBits;

    if (objc == 3) {
      Tcl_GetIntFromObj(interp, objv[2], &bitstride);
      bytestride = bitstride/8;
    }

    /* from string to bits, scan_ir, and bits to string */
    bitsFromString(interp, Tcl_GetStringFromObj(objv[1], NULL), outsr);
    memset(insr->data, 0, outsr->nBytes);

    for (n = 0; n+bitstride < nBits; n += bitstride) {
      scan_to_pause_dr(0, (const short *)outdata, bitstride, (short *)indata);
      outdata += bytestride;
      indata += bytestride;
    }
    scan_dr(0, (const short *)outdata, nBits-n, (short *)indata);

    insr->nBits = outsr->nBits;
    insr->nBytes = outsr->nBytes;
    bitsToString(instring, insr);
    Tcl_AppendResult(interp, instring, NULL);
    return TCL_OK;
  } else {
    Tcl_AppendResult(interp, "Wrong number of args to scan_dr_with_pause",
		     NULL);
    return TCL_ERROR;
  }
}

static int Corelis_scan_dr_turbo(ClientData clientData, Tcl_Interp *interp, 
				 int objc, struct Tcl_Obj * CONST objv[])
{
  if (objc == 2) {
    /* from string to bits, scan_ir, and bits to string */
    bitsFromString(interp, Tcl_GetStringFromObj(objv[1], NULL), outsr);
    memset(insr->data, 0, outsr->nBytes);

    scan_dr_turbo(0, (const long *)outsr->data, outsr->nBits, (long *)insr->data);

    insr->nBits = outsr->nBits;
    insr->nBytes = outsr->nBytes;
    bitsToString(instring, insr);
    Tcl_AppendResult(interp, instring, NULL);
    return TCL_OK;
  } else {
    Tcl_AppendResult(interp, "Wrong number of args to scan_dr",
		     NULL);
    return TCL_ERROR;
  }
}

static void appendHexResult(Tcl_Interp *interp, char *name, long data)
{
  char buf[32];
  sprintf(buf, "%s=%lx\n", name, data);
  Tcl_AppendResult(interp, buf, NULL);
}

static int Corelis_read_idcode(ClientData clientData, Tcl_Interp *interp, 
			       int objc, struct Tcl_Obj * CONST objv[])
{
  if (objc == 1) {
    char *tmpdata = NULL;
    char irdata[32];

    memset(irdata, 0, 32);
    insr->nBits = 64;
    insr->nBytes = 8;

    reset_skfmlb_scanring();

    /* set up IDCODE instruction */
    *(long*)outsr->data = (6 << 6) | (4 << 2) | 1;
    outsr->nBits = 11;
    outsr->nBytes = 2;

    scan_ir(0, (const short*)outsr->data, outsr->nBits, (short *)irdata);

    /* now scan out the idcode */
    scan_dr(0, (const short*)insr->data, 64, (short *)insr->data);
    bitsToString(instring, insr);

    /* now put it back in the original state */
    scan_ir(0, (const short*)skfmlbsr->instruction, skfmlbsr->instructionNBits, (short *)irdata);

    Tcl_AppendResult(interp, instring, NULL);

    return TCL_OK;
  } else {
    Tcl_AppendResult(interp, "Wrong number of args to set_oc_output",
		     NULL);
    return TCL_ERROR;
  }
}


static int Corelis_set_tx(ClientData clientData, Tcl_Interp *interp, 
                          int objc, struct Tcl_Obj * CONST objv[])
{
  if (objc == 2) {
    char *tmpdata = NULL;
    char irdata[32];
    char drdata1[4096];
    long tx = 0;
    int rc;

    assert(skfmlb.reset);

    memset(irdata, 0, 32);
    memset(drdata1, 0, 4096);

    rc = Tcl_GetIntFromObj(interp, objv[1], &tx);
    if (rc != TCL_OK) {
      Tcl_AppendResult(interp, "Could not get tx from ", 
		       Tcl_GetStringFromObj(objv[1], NULL),
		       NULL);
      return TCL_ERROR;
    }

    reset_skfmlb_scanring();

    set_corelogic_TX(tx);

    scan_ir(0, (const short*)skfmlbsr->instruction, skfmlbsr->instructionNBits, (short *)irdata);
    scan_dr(0, (const short*)skfmlbsr->data, skfmlbsr->nBits, (short *)drdata1);
    return TCL_OK;
  } else {
    Tcl_AppendResult(interp, "Wrong number of args to set_oc_output",
		     NULL);
    return TCL_ERROR;
  }
}

static int writeFlash(Tcl_Interp *interp, uint32 addr, uint32 data)
{
   char irdata[32];
   char drdata1[4096];
   char drdata2[4096];
   char drdata3[4096];

   assert(skfmlb.reset);

   memset(irdata, 0, 32);
   memset(drdata1, 0, 4096);
   memset(drdata2, 0, 4096);
   memset(drdata3, 0, 4096);

   reset_skfmlb_scanring();

   enable_corelogic_A();
   enable_corelogic_D();
    /* *** enable corresponding bit on xilinx gpio part */
   enable_XC9572XL_TQ100_PB00_05();
   enable_corelogic_ROM_CE_L();

   addr &= ~0x80000000L; /* assert ROM_WE_L, which is on A[31] */
   addr |=  0x40000000L; /* deassert ROM_OE_L, which is on A[30] */

   set_corelogic_A(addr);
   set_corelogic_D(data);

    /* *** drive 0 on corresponding bit on xilinx gpio part */
   set_XC9572XL_TQ100_PB00_05(0);
   set_corelogic_ROM_CE_L(0); /* assert ROM_CE */

   scan_ir(0, (const short*)skfmlbsr->instruction, skfmlbsr->instructionNBits, (short *)irdata);
   scan_dr(0, (const short*)skfmlbsr->data, skfmlbsr->nBits, (short *)drdata1);

    /* *** drive 1 on corresponding bit on xilinx gpio part */
   set_XC9572XL_TQ100_PB00_05(1);
   set_corelogic_ROM_CE_L(1); /* deassert ROM_CE */

   scan_dr(0, (const short*)skfmlbsr->data, skfmlbsr->nBits, (short *)drdata2);

   /* disable D and A */
   disable_corelogic_D();
   disable_corelogic_A();
   scan_dr(0, (const short*)skfmlbsr->data, skfmlbsr->nBits, (short *)drdata3);

   return TCL_OK;
}

static uint32 readFlash(Tcl_Interp *interp, uint32 addr)
{
    char *tmpdata = NULL;
    char irdata[32];
    char drdata1[4096];
    char drdata2[4096];
    char drdata3[4096];

    assert(skfmlb.reset);

    memset(irdata, 0, 32);
    memset(drdata1, 0, 4096);
    memset(drdata2, 0, 4096);

    addr |=  0x80000000L; /* deassert ROM_WE_L, which is on A[31] */
    addr &= ~0x40000000L; /* assert ROM_OE_L, which is on A[30] */

    reset_skfmlb_scanring();

    enable_corelogic_A();
    disable_corelogic_D();
    /* *** enable corresponding bit on xilinx gpio part */
    enable_XC9572XL_TQ100_PB00_05();
    enable_corelogic_ROM_CE_L();

    set_corelogic_A(addr);
    /* *** drive 0 on corresponding bit on xilinx gpio part */
    set_XC9572XL_TQ100_PB00_05(0);
    set_corelogic_ROM_CE_L(0); /* assert ROM_CE */

    scan_ir(0, (const short*)skfmlbsr->instruction, skfmlbsr->instructionNBits, (short *)irdata);

    /* *** drive 0 on corresponding bit on xilinx gpio part */
    set_XC9572XL_TQ100_PB00_05(0);
    set_corelogic_ROM_CE_L(0); /* deassert ROM_CE */

    scan_dr(0, (const short*)skfmlbsr->data, skfmlbsr->nBits, (short *)drdata1);
    scan_dr(0, (const short*)skfmlbsr->data, skfmlbsr->nBits, (short *)drdata2);

    /* *** drive 1 on corresponding bit on xilinx gpio part */
    set_XC9572XL_TQ100_PB00_05(1);
    set_corelogic_ROM_CE_L(1); /* deassert ROM_CE */

    scan_dr(0, (const short*)skfmlbsr->data, skfmlbsr->nBits, (short *)drdata3);

    memcpy(skfmlbsr->data, drdata2, skfmlbsr->nBytes);

    return get_corelogic_D();
}



static int Corelis_write_flash(ClientData clientData, Tcl_Interp *interp, 
			       int objc, struct Tcl_Obj * CONST objv[])
{
  if (objc == 3) {
    char irdata[32];
    char drdata1[4096];
    char drdata2[4096];
    char drdata3[4096];
    long addr = 0;
    long data = 0;
    int rc;

    assert(skfmlb.reset);

    memset(irdata, 0, 32);
    memset(drdata1, 0, 4096);
    memset(drdata2, 0, 4096);
    memset(drdata3, 0, 4096);

    rc = Tcl_GetIntFromObj(interp, objv[1], &addr);
    if (rc != TCL_OK) {
      Tcl_AppendResult(interp, "Could not get addr from ", 
		       Tcl_GetStringFromObj(objv[1], NULL),
		       NULL);
      return TCL_ERROR;
    }
    rc = Tcl_GetIntFromObj(interp, objv[2], &data);
    if (rc != TCL_OK) {
      Tcl_AppendResult(interp, "Could not get data from ", 
		       Tcl_GetStringFromObj(objv[2], NULL),
		       NULL);
      return TCL_ERROR;
    }


    /* set up instruction */
    scan_ir(0, (const short*)skfmlbsr->instruction, skfmlbsr->instructionNBits, (short *)irdata);

    reset_skfmlb_scanring();

    enable_corelogic_A();
    enable_corelogic_D();
    /* *** enable corresponding bit on xilinx gpio part */
    enable_XC9572XL_TQ100_PB00_05();
    enable_corelogic_ROM_CE_L();

    addr &= ~0x80000000L; /* assert ROM_WE_L, which is on A[31] */
    addr |=  0x40000000L; /* deassert ROM_OE_L, which is on A[30] */

    set_corelogic_A(addr);
    set_corelogic_D(data);

    /* *** drive 0 on corresponding bit on xilinx gpio part */
    set_XC9572XL_TQ100_PB00_05(0);
    set_corelogic_ROM_CE_L(0); /* assert ROM_CE */

    scan_ir(0, (const short*)outsr->data, outsr->nBits, (short *)irdata);
    scan_dr(0, (const short*)skfmlbsr->data, skfmlbsr->nBits, (short *)drdata1);
    scan_dr(0, (const short*)skfmlbsr->data, skfmlbsr->nBits, (short *)drdata2);

    /* *** drive 1 on corresponding bit on xilinx gpio part */
    set_XC9572XL_TQ100_PB00_05(1);
    set_corelogic_ROM_CE_L(1); /* deassert ROM_CE */

    scan_dr(0, (const short*)skfmlbsr->data, skfmlbsr->nBits, (short *)drdata2);

    /* disable D and A */
    disable_corelogic_D();
    disable_corelogic_A();
    scan_dr(0, (const short*)skfmlbsr->data, skfmlbsr->nBits, (short *)drdata3);

    return TCL_OK;
  } else {
    Tcl_AppendResult(interp, "Wrong number of args to write_flash",
		     NULL);
    return TCL_ERROR;
  }
}

/*
static int Corelis_read_flash(ClientData clientData, Tcl_Interp *interp, 
			      int objc, struct Tcl_Obj * CONST objv[])
{
   if (objc == 2) {
      char buf[32];
      long addr = 0;
      long data = 0;
      int rc;

      rc = Tcl_GetIntFromObj(interp, objv[1], &addr);
      if (rc != TCL_OK) {
         Tcl_AppendResult(interp, "Could not get addr from ", 
                          Tcl_GetStringFromObj(objv[1], NULL),
                          NULL);
         return TCL_ERROR;
      }

      sprintf(buf, "0x%lx", readFlash(interp, addr));
      Tcl_AppendResult(interp, buf, NULL);
      return TCL_OK;
   } else {
      Tcl_AppendResult(interp, "Wrong number of args to set_oc_output",
                       NULL);
      return TCL_ERROR;
   }
}
*/

/* this is the one that works */
static int Corelis_read_flash_old_way(ClientData clientData, Tcl_Interp *interp, 
			      int objc, struct Tcl_Obj * CONST objv[])
{
  if (objc == 2) {
    char *tmpdata = NULL;
    char buf[32];
    char irdata[32];
    char drdata1[4096];
    char drdata2[4096];
    char drdata3[4096];
    long addr = 0;
    long data = 0;
    int rc;

    assert(skfmlb.reset);

    memset(irdata, 0, 32);
    memset(drdata1, 0, 4096);
    memset(drdata2, 0, 4096);

    rc = Tcl_GetIntFromObj(interp, objv[1], &addr);
    if (rc != TCL_OK) {
      Tcl_AppendResult(interp, "Could not get addr from ", 
		       Tcl_GetStringFromObj(objv[1], NULL),
		       NULL);
      return TCL_ERROR;
    }

    addr |=  0x80000000L; /* deassert ROM_WE_L, which is on A[31] */
    addr &= ~0x40000000L; /* assert ROM_OE_L, which is on A[30] */

    /* set up EXTEST instruction */
    *(long*)outsr->data =  SKFMLB_INSTRUCTION_EXTEST; /* EXTEST: 0b000000000 */
    outsr->nBits = SKFMLB_INSTRUCTION_NBITS;
    outsr->nBytes = 2;

    reset_skfmlb_scanring();

    enable_corelogic_A();
    disable_corelogic_D();
    disable_SA110_A();
    disable_SA110_D();
    /* *** enable corresponding bit on xilinx gpio part */
    enable_XC9572XL_TQ100_PB00_05();
    enable_corelogic_ROM_CE_L();

    set_corelogic_A(addr);
    /* *** drive 0 on corresponding bit on xilinx gpio part */
    set_XC9572XL_TQ100_PB00_05(0);
    set_corelogic_ROM_CE_L(0); /* assert ROM_CE */

    scan_ir(0, (const short*)outsr->data, outsr->nBits, (short *)irdata);

    /* *** drive 0 on corresponding bit on xilinx gpio part */
    set_XC9572XL_TQ100_PB00_05(0);
    set_corelogic_ROM_CE_L(0);

    scan_dr(0, (const short*)skfmlbsr->data, skfmlbsr->nBits, (short *)drdata1);
    scan_dr(0, (const short*)skfmlbsr->data, skfmlbsr->nBits, (short *)drdata2);

    /* *** drive 1 on corresponding bit on xilinx gpio part */
    set_XC9572XL_TQ100_PB00_05(1);
    set_corelogic_ROM_CE_L(1); /* deassert ROM_CE */

    scan_dr(0, (const short*)skfmlbsr->data, skfmlbsr->nBits, (short *)drdata3);

    memcpy(skfmlbsr->data, drdata2, skfmlbsr->nBytes);

    sprintf(buf, "0x%lx", get_SA110_D());
    Tcl_AppendResult(interp, buf, NULL);
    return TCL_OK;
  } else {
    Tcl_AppendResult(interp, "Wrong number of args to set_oc_output",
		     NULL);
    return TCL_ERROR;
  }
}




static uint32 doubleword(uint32 v)
{
   v = v & 0xFFFF; 
   return (v << 16) | v;
}

static int sendProgramFlashWordCommand(Tcl_Interp *interp, uint32 addr, uint32 value)
{
   int count = 0;
   assert((addr & 0x3) == 0);
   writeFlash(interp, 0x555 << 2, doubleword(0xAA));
   writeFlash(interp, 0x2AA << 2, doubleword(0x55));
   writeFlash(interp, 0x555 << 2, doubleword(0xA0));
   writeFlash(interp, addr, value);
   return TCL_OK;
}

static int Corelis_send_program_flash_word_command(ClientData clientData, Tcl_Interp *interp, 
                                                   int objc, struct Tcl_Obj * CONST objv[])
{
  if (objc == 3) {
    long addr = 0;
    long data = 0;
    int rc;

    assert(skfmlb.reset);

    rc = Tcl_GetIntFromObj(interp, objv[1], &addr);
    if (rc != TCL_OK) {
      Tcl_AppendResult(interp, "Could not get addr from ", 
		       Tcl_GetStringFromObj(objv[1], NULL),
		       NULL);
      return TCL_ERROR;
    }
    rc = Tcl_GetIntFromObj(interp, objv[2], &data);
    if (rc != TCL_OK) {
      Tcl_AppendResult(interp, "Could not get data from ", 
		       Tcl_GetStringFromObj(objv[2], NULL),
		       NULL);
      return TCL_ERROR;
    }

    sendProgramFlashWordCommand(interp, addr, data);

    return TCL_OK;
  } else {
    Tcl_AppendResult(interp, "Wrong number of args to send_program_flash_word_command",
		     NULL);
    return TCL_ERROR;
  }
}

static int isSectorAddress(uint32 addr)
{
   /* 128KB per usual sector (64KB per Flash chip) */
   if ((addr&0x00001FFFL) != 0) {
      /* not a multiple of 16K, can't be a sector boundary */
      return 0;
   } else if ((addr&0x0001FFFFL) == 0) {
      /* 128KB sector boundary */
      return 1;
   } else if (addr == 0x00008000L) {
      /* first 16KB sector after 32KB sector in bottom boot block */
      return 1;
   } else if (addr == 0x0000C000L) {
      /* second 16KB sector after 32KB sector in bottom boot block */
      return 1;
   } else if (addr == 0x00010000L) {
      /* second 64KB sector after 32KB sector in bottom boot block */
      return 1;
   } else {
      return 0;
   }
}

static int programFlashWord(Tcl_Interp *interp, uint32 addr, uint32 value)
{
   int count = 0;
   uint32 oldvalue;
   assert((addr & 0x3) == 0);
   writeFlash(interp, 0x555 << 2, doubleword(0xAA));
   writeFlash(interp, 0x2AA << 2, doubleword(0x55));
   writeFlash(interp, 0x555 << 2, doubleword(0xA0));
   writeFlash(interp, addr, value);
   oldvalue = readFlash(interp, addr);
   while (1) {
      uint32 newvalue = readFlash(interp, addr);
      count++;
      if (oldvalue != newvalue) {
         /* toggling */
         oldvalue = newvalue;
         continue;
      }
      if (newvalue == value)
         return TCL_OK;
      if (count > 100000) {
         if (interp != NULL) {
            char msg[4096];
            sprintf(msg,"ProgramFlashWord timed out\n");
            sprintf(msg, "Count=%d addr=%08x value=%08x flash=%08x\n",
                    count, addr, value, readFlash(interp, addr));
            Tcl_AppendResult(interp, msg, NULL);
         }
         return TCL_ERROR;
      }
      oldvalue = newvalue;
   }
   return TCL_OK;
}

static int eraseFlashSector(Tcl_Interp *interp, uint32 addr)
{
   long count = 0;
   long oldvalue;
   writeFlash(interp, 0x555 << 2, doubleword(0xAA));
   writeFlash(interp, 0x2AA << 2, doubleword(0x55));
   writeFlash(interp, 0x555 << 2, doubleword(0x80));
   writeFlash(interp, 0x555 << 2, doubleword(0xAA));
   writeFlash(interp, 0x2AA << 2, doubleword(0x55));
   writeFlash(interp, addr, doubleword(0x30));

   oldvalue = readFlash(interp, addr);

   while (1) {
      long newvalue = readFlash(interp, addr);
      count++;
      if (newvalue != oldvalue) {
         /* toggling */
         oldvalue = newvalue;
         continue;
      }

      if (newvalue == 0xFFFFFFFFL)
         return TCL_OK;

      if (count > 100000) {
         if (interp != NULL) {
            char msg[4096];
            sprintf(msg, "EraseFlashSector timed out\n");
            sprintf(msg, "Count=%d addr=%08lx flash=%08lx\n",
                    count, addr, readFlash(interp, addr));
            Tcl_AppendResult(interp, msg, NULL);
         }
         return TCL_ERROR;
      }

   }
   return TCL_OK;
}

static int programFlashFromFile(Tcl_Interp *interp,
                                const char *filename, uint32 addr, long len)
{
   FILE *input;
   uint32 flashaddr = addr;
   uint32 limit = addr + len;
   int rc = 1;
   int offset = 0;

   input = fopen(filename, "r");
   if (input == NULL) {
      if (interp != NULL) {
         char msg[4096];
         sprintf(msg, "Failed to open file %s with errno=%d\n",
                 filename, errno);
         Tcl_AppendResult(interp, msg, NULL);
      }
      return TCL_ERROR;
   }
   for (flashaddr = addr; flashaddr < limit; flashaddr += 4) {
      uint32 value;
      size_t nitems;
      if ((nitems = fread(&value, sizeof(value), 1, input)) != 1) {
         if (interp != NULL) {
            char msg[4096];
            sprintf(msg, "Failed to read=%d bytes from file=%s at offset=%d nitems=%d\n",
                    sizeof(value), filename, offset, nitems);
            Tcl_AppendResult(interp, msg, NULL);
            return TCL_ERROR;
         }
      }
      offset += sizeof(value);
      if (isSectorAddress(flashaddr)) {
         rc = eraseFlashSector(interp, flashaddr);
         if (rc != TCL_OK) {
            if (interp != NULL) {
               char msg[4096];
               sprintf(msg, "Failed to erase sector %0x\n", flashaddr);
               Tcl_AppendResult(interp, msg, NULL);
            }
            return TCL_ERROR;
         }
      }
      rc = programFlashWord(interp, addr, value);
      if (rc != TCL_OK) {
         if (interp != NULL) {
            char msg[4096];
            sprintf(msg, "Failed to program flashaddr=%0lx value=%lx flashvalue=%lx\n",
                    flashaddr, value, readFlash(interp, flashaddr));
            Tcl_AppendResult(interp, msg, NULL);
         }
         return TCL_ERROR;
      }
   }
   return TCL_OK;
}

static int Corelis_program_flash_from_file(ClientData clientData, Tcl_Interp *interp, 
                                           int objc, struct Tcl_Obj * CONST objv[])
{
   if (objc == 4) {
      char *filename;
      long addr = 0;
      long length = 0;
      int rc;

      filename = Tcl_GetStringFromObj(objv[1], NULL);
      rc = Tcl_GetIntFromObj(interp, objv[2], &addr);
      if (rc != TCL_OK) {
         Tcl_AppendResult(interp, "Could not get flash addr from ", 
                          Tcl_GetStringFromObj(objv[2], NULL),
                          NULL);
         return TCL_ERROR;
      }
      rc = Tcl_GetIntFromObj(interp, objv[3], &length);
      if (rc != TCL_OK) {
         Tcl_AppendResult(interp, "Could not get file length from ", 
                          Tcl_GetStringFromObj(objv[3], NULL),
                          NULL);
         return TCL_ERROR;
      }
      return programFlashFromFile(interp, filename, addr, length);
   } else {
      Tcl_AppendResult(interp, "Wrong number of args to set_oc_output",
                       NULL);
      return TCL_ERROR;
   }
}



void reset_skfmlb_scanring()
{
  clear_scanring(skfmlbsr);
  resetCorelogicScanring();
  assert(skfmlb.sa110ScanringLength > 1);
  resetSa110Scanring();

  /* *** reset xilinx part */
  assert(skfmlb.xc9572xl_tq100ScanringLength > 1);
  resetXC9572XL_TQ100Scanring();

  /* *** reset ethernet part */
  assert(skfmlb.everestScanringLength > 1);
  resetEVERESTScanring();
}

static int reset_skfmlb(Tcl_Interp *interp)
{
   char irdata[32];
   char drdata1[4096];

   assert(skfmlbsr != NULL);
   /* *** Add xilinx info */
   xc9572xl_tq100Scanring = new_subring(skfmlbsr, 0);
   /* *** Add AMD info */
   everestScanring = new_subring(skfmlbsr, XC9572XL_TQ100_SCANRING_LENGTH);   
   corelogicScanring = new_subring(skfmlbsr, XC9572XL_TQ100_SCANRING_LENGTH + EVEREST_SCANRING_LENGTH);
   sa110Scanring = new_subring(skfmlbsr, XC9572XL_TQ100_SCANRING_LENGTH + EVEREST_SCANRING_LENGTH + CORELOGIC_SCANRING_LENGTH);

   skfmlb.sa110Instruction = SA110_EXTEST;
   skfmlb.sa110ScanringLength = SA110_SCANRING_LENGTH;
   skfmlb.corelogicInstruction = CORELOGIC_EXTEST;
   skfmlb.corelogicScanringLength = CORELOGIC_SCANRING_LENGTH;
   /* *** Add xilinx info */
   skfmlb.xc9572xl_tq100Instruction = XC9572XL_TQ100_EXTEST;
   skfmlb.xc9572xl_tq100ScanringLength = XC9572XL_TQ100_SCANRING_LENGTH;
   /* *** Add AMD Info */
   skfmlb.everestInstruction = EVEREST_EXTEST;
   skfmlb.everestScanringLength = EVEREST_SCANRING_LENGTH;

   skfmlbsr->nBits = skfmlb.corelogicScanringLength + skfmlb.sa110ScanringLength + skfmlb.xc9572xl_tq100ScanringLength + skfmlb.everestScanringLength;
   skfmlbsr->nBytes = (skfmlbsr->nBits + 7) >> 3;

/*
  *(long*)skfmlbsr->instruction = 
    ((((( skfmlb.sa110Instruction << CORELOGIC_INSTRUCTION_LENGTH) 
	| skfmlb.corelogicInstruction) << EVEREST_INSTRUCTION_LENGTH) 
        | skfmlb.everestInstruction) << XC9572XL_TQ100_INSTRUCTION_LENGTH) 
        | skfmlb.xc9572xl_tq100Instruction;
*/

  *(long*)skfmlbsr->instruction = skfmlb.sa110Instruction;
  *(long*)skfmlbsr->instruction <<= CORELOGIC_INSTRUCTION_LENGTH;
  *(long*)skfmlbsr->instruction |= skfmlb.corelogicInstruction;
  *(long*)skfmlbsr->instruction <<= EVEREST_INSTRUCTION_LENGTH;
  *(long*)skfmlbsr->instruction |= skfmlb.everestInstruction;
  *(long*)skfmlbsr->instruction <<= XC9572XL_TQ100_INSTRUCTION_LENGTH;
  *(long*)skfmlbsr->instruction |= skfmlb.xc9572xl_tq100Instruction;

  *(long*)skfmlbsr->instruction = 0; /* EXTEST */

   skfmlbsr->instructionNBits = SA110_INSTRUCTION_LENGTH + CORELOGIC_INSTRUCTION_LENGTH + XC9572XL_TQ100_INSTRUCTION_LENGTH + EVEREST_INSTRUCTION_LENGTH;

   // ***   skfmlbsr->instructionNBytes = 2;   
   skfmlbsr->instructionNBytes = (skfmlbsr->instructionNBits + 7) >> 3;   /* *** ??? Is this right? */
   
   /* put both chips in EXTEST mode with outputs disabled or safe values driven */
   reset_skfmlb_scanring();

   scan_ir(0, (const short*)skfmlbsr->instruction, skfmlbsr->instructionNBits, (short *)irdata);
   scan_dr(0, (const short*)skfmlbsr->data, skfmlbsr->nBits, (short *)drdata1);

   skfmlb.reset = 1;


   return TCL_OK;
}

static int Corelis_bypass_sa110(ClientData clientData, Tcl_Interp *interp, 
                                int objc, struct Tcl_Obj * CONST objv[])
{
   assert(skfmlb.reset); /* make sure safe values have been scanned in first */

   sa110Scanring->nBits = 1;
   sa110Scanring->nBytes = 1;
   skfmlb.sa110ScanringLength = 1;

   skfmlbsr->nBits = skfmlb.corelogicScanringLength + skfmlb.sa110ScanringLength + skfmlb.xc9572xl_tq100ScanringLength + skfmlb.everestScanringLength;
   skfmlbsr->nBytes = (skfmlbsr->nBits + 7) >> 3;

   skfmlb.sa110Instruction = SA110_BYPASS;

  *(long*)skfmlbsr->instruction = skfmlb.sa110Instruction;
  *(long*)skfmlbsr->instruction <<= CORELOGIC_INSTRUCTION_LENGTH;
  *(long*)skfmlbsr->instruction |= skfmlb.corelogicInstruction;
  *(long*)skfmlbsr->instruction <<= EVEREST_INSTRUCTION_LENGTH;
  *(long*)skfmlbsr->instruction |= skfmlb.everestInstruction;
  *(long*)skfmlbsr->instruction <<= XC9572XL_TQ100_INSTRUCTION_LENGTH;
  *(long*)skfmlbsr->instruction |= skfmlb.xc9572xl_tq100Instruction;

   return TCL_OK;
}

static int Corelis_bypass_xc9572xl_tq100(ClientData clientData, Tcl_Interp *interp, 
                                int objc, struct Tcl_Obj * CONST objv[])
{
   assert(skfmlb.reset); /* make sure safe values have been scanned in first */

   xc9572xl_tq100Scanring->nBits = 1;
   xc9572xl_tq100Scanring->nBytes = 1;
   skfmlb.xc9572xl_tq100ScanringLength = 1;

   skfmlbsr->nBits = skfmlb.corelogicScanringLength + skfmlb.sa110ScanringLength + skfmlb.xc9572xl_tq100ScanringLength + skfmlb.everestScanringLength;;
   skfmlbsr->nBytes = (skfmlbsr->nBits + 7) >> 3;

   skfmlb.xc9572xl_tq100Instruction = XC9572XL_TQ100_BYPASS;

  
  *(long*)skfmlbsr->instruction = skfmlb.sa110Instruction;
  *(long*)skfmlbsr->instruction <<= CORELOGIC_INSTRUCTION_LENGTH;
  *(long*)skfmlbsr->instruction |= skfmlb.corelogicInstruction;
  *(long*)skfmlbsr->instruction <<= EVEREST_INSTRUCTION_LENGTH;
  *(long*)skfmlbsr->instruction |= skfmlb.everestInstruction;
  *(long*)skfmlbsr->instruction <<= XC9572XL_TQ100_INSTRUCTION_LENGTH;
  *(long*)skfmlbsr->instruction |= skfmlb.xc9572xl_tq100Instruction;
      
   return TCL_OK;
}

static int Corelis_bypass_corelogic(ClientData clientData, Tcl_Interp *interp, 
                                    int objc, struct Tcl_Obj * CONST objv[])
{
   Tcl_AppendResult(interp, "bypass_corelogic is unimplemented", NULL);
   return TCL_ERROR;
}

#ifdef DeadCode
static int Corelis_reset_skfmlb(ClientData clientData, Tcl_Interp *interp, 
                                int objc, struct Tcl_Obj * CONST objv[])
{
   return reset_skfmlb(interp);
}
#endif



static Tcl_Channel inputChannel;

/* sets the binary channel we use for reading binary data */
static int Corelis_set_input_channel(ClientData clientData, Tcl_Interp *interp, 
                                     int objc, struct Tcl_Obj * CONST objv[])
{
  if (objc == 2) {
    int length;
    char *inputChannelName = Tcl_GetStringFromObj(objv[1], &length);
    int mode;
    inputChannel = Tcl_GetChannel(interp, inputChannelName, &mode);
    return TCL_OK;
  } else {
    Tcl_AppendResult(interp, "Wrong number of args",
		     NULL);
    return TCL_ERROR;
  }
}

/* reads 4 bytes from the input channel */
static int Corelis_read_dword(ClientData clientData, Tcl_Interp *interp, 
                              int objc, struct Tcl_Obj * CONST objv[])
{
  if (objc == 1) {
     long value;
     /* read 4 bytes from the channel */
     Tcl_Read(inputChannel, (char*)&value, 4);
     Tcl_SetObjResult(interp, Tcl_NewIntObj(value));
     return TCL_OK;
  } else {
    Tcl_AppendResult(interp, "Wrong number of args",
		     NULL);
    return TCL_ERROR;
  }
}



typedef void (*enable_func_t)(void );
typedef void (*set_func_t)(long);
typedef long (*get_func_t)();

typedef struct {
  const char *name;
  int width;
  int ndrops;
  enable_func_t enable;
  set_func_t    drive;
  get_func_t    read0;
  get_func_t    read1;
  get_func_t    read2;
} netinfo_t;

netinfo_t skfmlb_netinfo[] = {
  /* SA110 Signals */
  { "D", 32, 2,
    enable_SA110_D,
    set_SA110_D,
    get_SA110_D, 
    get_corelogic_D, 
    NULL
  },
  { "A", 32, 2,
    enable_SA110_A,
    set_SA110_A,
    get_corelogic_A, NULL
  },
#if 0
  {
     "ROM_CE", 1, 2,
     enable_corelogic_ROM_CE,
     set_corelogic_ROM_CE,
     get_XC9572XL_TQ100_
  }
#endif
#if 0
  { "ABE", 1, 2,
    enable_corelogic_ABE,
    set_corelogic_ABE,
    get_SA110_ABE, NULL
  },
#endif
  { "DBE", 1, 2,
    enable_corelogic_DBE,
    set_corelogic_DBE,
    get_SA110_DBE, NULL
  },
  { "~RW", 1, 2,
    enable_SA110_NRW,
    set_SA110_NRW,
    get_corelogic_NRW, NULL
  },
  { "LOCK", 1, 2,
    enable_SA110_LOCK,
    set_SA110_LOCK,
    get_corelogic_LOCK, NULL
  },
  { "CLF", 1, 2,
    enable_SA110_CLF,
    set_SA110_CLF,
    get_corelogic_CLF, NULL
  },
  { "NMREQ", 1, 2,
    enable_SA110_NMREQ,
    set_SA110_NMREQ,
    get_corelogic_NMREQ, NULL
  },
  /* PCI Signals */
  { "AD", 32, 2,
    enable_corelogic_AD,
    set_corelogic_AD,
    get_corelogic_AD, get_EVEREST_AD, NULL
  },
  { "CBE", 4, 2,
    enable_corelogic_CBE_L,
    set_corelogic_CBE_L,
    get_corelogic_CBE_L, get_EVEREST_C_BEL, NULL
  },
  { NULL }
};

static void test_net(ClientData clientData, Tcl_Interp *interp, netinfo_t *netinfo)
{
    int i;
    int mode;
    Tcl_Channel outchan = Tcl_GetChannel(interp, "stdout", &mode);
    char buf[1024];
    char irdata[32];
    char drdata1[4096];
    char drdata2[4096];
    char drdata3[4096];
    int passed = 1;

    memset(irdata, 0, 32);
    memset(drdata1, 0, 4096);
    memset(drdata2, 0, 4096);
    memset(drdata3, 0, 4096);

    /* set up instruction */
    scan_ir(0, (const short*)skfmlbsr->instruction, skfmlbsr->instructionNBits, (short *)irdata);

    sprintf(buf, "Testing %s ...\n", netinfo->name);
    Tcl_Write(outchan, buf, strlen(buf));
    passed = 1;
    for (i = -1; i < netinfo->width; i++) {
	long value = (i < 0) ? 0 : ((long)(1L << i));

	reset_skfmlb_scanring();
        if (netinfo->enable != NULL)
           netinfo->enable();
        if (netinfo->drive != NULL)
           netinfo->drive(value);

        /* now scan once to drive the net */
        scan_dr(0, (const short*)skfmlbsr->data, skfmlbsr->nBits, (short *)drdata1);
        /* now scan to read the other pins on the same net */
        scan_dr(0, (const short*)skfmlbsr->data, skfmlbsr->nBits, (short *)skfmlbsr->data);

        /* now read the input pins to see if we got the right value */
        if (netinfo->read0 != NULL) {
          long readback = netinfo->read0();
          if (readback != value) {
            sprintf(buf, "!!! read0 has value=0x%lx, drove value 0x%lx\n", readback, value);
            Tcl_Write(outchan, buf, strlen(buf));
            passed = 0;
          }
        }
        if (netinfo->read1 != NULL) {
          long readback = netinfo->read1();
          if (readback != value) {
            sprintf(buf, "!!! read1 has value=0x%lx, drove value 0x%lx\n", readback, value);
            Tcl_Write(outchan, buf, strlen(buf));
            passed = 0;
          }
        }
        if (netinfo->read2 != NULL) {
          long readback = netinfo->read2();
          if (readback != value) {
            sprintf(buf, "!!! read2 has value=0x%lx, drove value 0x%lx\n", readback, value);
            Tcl_Write(outchan, buf, strlen(buf));
            passed = 0;
          }
        }
    }
    sprintf(buf, (passed) ? "passed\n" : "failed\n");
    Tcl_Write(outchan, buf, strlen(buf));

    reset_skfmlb_scanring();
    scan_dr(0, (const short *)outsr->data, outsr->nBits, (short *)insr->data);

}

static int Corelis_test_pciad(ClientData clientData, Tcl_Interp *interp, 
                              int objc, struct Tcl_Obj * CONST objv[])
{
    int i;
    int mode;
    Tcl_Channel outchan = Tcl_GetChannel(interp, "stdout", &mode);
    char buf[1024];
    char irdata[32];
    char drdata1[4096];
    char drdata2[4096];
    char drdata3[4096];
    int passed = 1;
    netinfo_t *netinfo;

    memset(irdata, 0, 32);
    memset(drdata1, 0, 4096);
    memset(drdata2, 0, 4096);
    memset(drdata3, 0, 4096);

    /* set up instruction */
    scan_ir(0, (const short*)skfmlbsr->instruction, skfmlbsr->instructionNBits, (short *)irdata);

    for (netinfo = &skfmlb_netinfo[0]; netinfo->name != NULL; netinfo++) {
        test_net(clientData, interp, netinfo);
    }

    sprintf(buf, "Testing PCI CBE ...\n");
    Tcl_Write(outchan, buf, strlen(buf));
    passed = 1;
    for (i = 0; i < 4; i++) {
	long val = (long)(1L << i);
	long everest_cbe = 0;
	long corelogic_cbe = 0;

	reset_skfmlb_scanring();
	enable_corelogic_CBE_L();

        set_corelogic_CBE_L(val);
        scan_dr(0, (const short*)skfmlbsr->data, skfmlbsr->nBits, (short *)drdata1);
        scan_dr(0, (const short*)skfmlbsr->data, skfmlbsr->nBits, 
                (short *)skfmlbsr->data);

        everest_cbe = get_EVEREST_C_BEL();
        corelogic_cbe = get_corelogic_CBE_L();
	if (everest_cbe != val || corelogic_cbe != val) {
	  sprintf(buf, "  everest_cbe=%08x corelogic_cbe=%08x\n", 
		  everest_cbe, corelogic_cbe);
	  Tcl_Write(outchan, buf, strlen(buf));
	  passed = 0;
	}
    }
    sprintf(buf, (passed) ? "passed\n" : "failed\n");
    Tcl_Write(outchan, buf, strlen(buf));

    sprintf(buf, "Testing PCI AD\n");
    Tcl_Write(outchan, buf, strlen(buf));
    passed = 1;
    for (i = 0; i < 32; i++) {
	long ad = (long)(1L << i);
	long everest_ad = 0;
	long corelogic_ad = 0;
        long corelogic_A = 0;


	reset_skfmlb_scanring();
	enable_corelogic_AD();
	enable_SA110_A();

        set_corelogic_AD(ad);
        set_SA110_A(ad);
        scan_dr(0, (const short*)skfmlbsr->data, skfmlbsr->nBits, (short *)drdata1);
        scan_dr(0, (const short*)skfmlbsr->data, skfmlbsr->nBits, 
                (short *)skfmlbsr->data);

        everest_ad = get_EVEREST_AD();
        corelogic_ad = get_corelogic_AD();
        corelogic_A = get_corelogic_A();
	if (everest_ad != ad || corelogic_ad != ad || corelogic_A != ad) {
	  sprintf(buf, "  everest_ad=%08x corelogic_ad=%08x corelogic_A=%08x\n", 
		  everest_ad, corelogic_ad, corelogic_A);
	  Tcl_Write(outchan, buf, strlen(buf));
	  passed = 0;
	}
    }
    sprintf(buf, (passed) ? "passed\n" : "failed\n");
    Tcl_Write(outchan, buf, strlen(buf));

    reset_skfmlb_scanring();
    scan_dr(0, (const short *)outsr->data, outsr->nBits, (short *)insr->data);

    return TCL_OK;
}




extern int Aero_Init(Tcl_Interp *interp);

__declspec( dllexport ) int Corelis_Init(Tcl_Interp *interp)
{
   int rc = TCL_OK;
   assert(test(22) == 22);

   Tcl_CreateObjCommand(interp, "set_io", Corelis_set_io, NULL, NULL);
   Tcl_CreateObjCommand(interp, "read_io", Corelis_read_io, NULL, NULL);
   Tcl_CreateObjCommand(interp, "set_oc_output", Corelis_set_oc_output, NULL, NULL);

   /* do a hard reset */
   hard_reset(0, _3_20);
   set_scan_clk(0, _30MHZ, 0, CLK_DIV_BY_32);

   Tcl_CreateObjCommand(interp, "tms_reset", Corelis_tms_reset, NULL, NULL);
   Tcl_CreateObjCommand(interp, "trst_reset", Corelis_trst_reset, NULL, NULL);
   Tcl_CreateObjCommand(interp, "soft_reset", Corelis_soft_reset, NULL, NULL);
   Tcl_CreateObjCommand(interp, "hard_reset", Corelis_hard_reset, NULL, NULL);
   Tcl_CreateObjCommand(interp, "set_scan_clk", Corelis_set_scan_clk, NULL, NULL);
   Tcl_CreateObjCommand(interp, "move_to_state", Corelis_move_to_state, NULL, NULL);

   insr = new_scanring(4096);
   outsr = new_scanring(4096);

   skfmlbsr = new_scanring(SA110_SCANRING_LENGTH + CORELOGIC_SCANRING_LENGTH + XC9572XL_TQ100_SCANRING_LENGTH + EVEREST_SCANRING_LENGTH);

   Tcl_CreateObjCommand(interp, "set_tx", Corelis_set_tx, NULL, NULL);

   Tcl_CreateObjCommand(interp, "bypass_sa110", Corelis_bypass_sa110, NULL, NULL);
   Tcl_CreateObjCommand(interp, "bypass_corelogic", Corelis_bypass_corelogic, NULL, NULL);
   /* *** Add xiling here */
   Tcl_CreateObjCommand(interp, "bypass_xc9572xl_tq100", Corelis_bypass_xc9572xl_tq100, NULL, NULL);

   Tcl_CreateObjCommand(interp, "read_idcode", Corelis_read_idcode, NULL, NULL);
   Tcl_CreateObjCommand(interp, "scan_ir", Corelis_scan_ir, NULL, NULL);
   Tcl_CreateObjCommand(interp, "scan_dr", Corelis_scan_dr, NULL, NULL);
   Tcl_CreateObjCommand(interp, "scan_dr_turbo", Corelis_scan_dr_turbo, NULL, NULL);
   Tcl_CreateObjCommand(interp, "scan_with_pause", Corelis_scan_with_pause, NULL, NULL);

   Tcl_CreateObjCommand(interp, "write_flash", Corelis_write_flash, NULL, NULL);
/*   Tcl_CreateObjCommand(interp, "read_flash", Corelis_read_flash, NULL, NULL); */
   Tcl_CreateObjCommand(interp, "read_flash_old_way", Corelis_read_flash_old_way, NULL, NULL);

   Tcl_CreateObjCommand(interp, "send_program_flash_word_command",
                        Corelis_send_program_flash_word_command, NULL, NULL);
   Tcl_CreateObjCommand(interp, "program_flash_from_file",
                        Corelis_program_flash_from_file, NULL, NULL);

   Tcl_CreateObjCommand(interp, "set_input_channel",
                        Corelis_set_input_channel, NULL, NULL);
   Tcl_CreateObjCommand(interp, "read_dword",
                        Corelis_read_dword, NULL, NULL);

   Tcl_CreateObjCommand(interp, "test_pciad",
                        Corelis_test_pciad, NULL, NULL);

   Aero_Init(interp);

   reset_skfmlb(interp);

   return rc;
}
