/*********************************************************************/
/*                                                                   */
/* Implementation of the                                             */
/*   PCI-1149.1 Scan Function Library                                */
/* for the                                                           */
/*   Xilinx Parallel Cable III JTAG Interface                        */
/*                                                                   */
/* Issues:                                                           */
/*   Some functions are unimplemented either because they are        */
/*   impossible to perform with the parallel cable or they           */
/*   aren't necessary for programming the FLASH.  Note especially    */
/*   that trst_reset() is impossible with the parallel cable.        */
/*   Maybe tms_reset() should be substituted.                        */
/*                                                                   */
/*   Currently uses the DriverLINX Port I/O device driver for        */
/*   parallel port I/O and must be linked against DLPORTIO.LIB.      */
/*   A custom driver would be nice.  This has only been tested       */
/*   on Windows NT/2000.  Under 95/98, it shouldn't require the      */
/*   driver, but may require rewrite of the port access functions.   */
/*                                                                   */
/* Author: Russ Keldorph (keldorph@acm.org)                          */
/*                                                                   */
/*********************************************************************/

#include "dlportio.h"
#include "library/PCI.h"

/* Many of the functions in this file are ignored or not implemented
   and will print a message to that effect when called.  To disable
   the messages, comment out the following line. */
#define VERBOSE

#ifdef VERBOSE
#include <stdio.h>
#endif /* VERBOSE */

/* Maybe these shouldn't be hardcoded...oh well, we shouldn't be using
   a generic port I/O device driver either. */
#define PAR_PORT_BASE (0x378)
#define PAR_PORT_STAT (PAR_PORT_BASE + 1)

/* See Xilinx Parallel Cable schematic for these constants */
#define TDO           (0x10)  /* status port */
#define VCC_SENSE     (0x08)  /* status port */
#define PROG          (0x10)  /* base port */
#define CTRL          (0x08)  /* base port */
#define TMS           (0x04)  /* base port */
#define TCK           (0x02)  /* base port */
#define TDI           (0x01)  /* base port */

/* JTAG state */
static enum STATES g_state = -1;     /* start in invalid state for init */

/* Value output to parallel base port */
static unsigned char c_base = PROG;  /* PROG must be high to read TDO */
                                     /* CTRL must be low to enable output */

/* Prints a message; called by unimplemented functions */
static void
unimplemented(char* name)
{
#ifdef VERBOSE
  fprintf(stderr, "WARNING: %s is not implemented\n", name);
#endif /* VERBOSE */
}

/* Prints a message; called by ignored functions */
static void
ignored(char* name)
{
#ifdef VERBOSE
  fprintf(stderr, "NOTE: %s is ignored\n", name);
#endif /* VERBOSE */
}

/*********************************************************/
/*       Bit Manipulation Functions (Not Exported)       */
/*********************************************************/

/*
 * We should probably define an iterator over these buffers since they
 * are only accessed sequentially, but one ISA bus cycle is so slow it
 * wouldn't make any difference.
 */

/* Sets a bit in a buffer to a given value */
static void
bit_set(unsigned char* buffer, unsigned long bit, int value) {
  unsigned long byte;
  unsigned long bit_in_byte;
  unsigned char mask;

  byte = bit >> 3;
  bit_in_byte = (unsigned char)bit & 7;
  mask = 1 << bit_in_byte;

  if(value) {
    buffer[byte] |= mask;
  } else {
    buffer[byte] &= ~mask;
  }
}

/* Returns the value of a bit in a buffer */
static int
bit_get(unsigned char* buffer, unsigned long bit)
{
  unsigned long byte;
  unsigned long bit_in_byte;
  unsigned char mask;

  byte = bit >> 3;
  bit_in_byte = (unsigned char)bit & 7;
  mask = 1 << bit_in_byte;

  if(buffer[byte] & mask) {
    return 1;
  } else {
    return 0;
  }
}

/*********************************************************/
/*        Hardware Access Functions (Not Exported)       */
/*********************************************************/
/* Pulses the TCK pin high then low.  Doesn't make sure TCK begins low */
static void
tck_tick(void)
{
  c_base |= TCK;
  DlPortWritePortUchar(PAR_PORT_BASE, c_base);
  c_base &= ~TCK;
  DlPortWritePortUchar(PAR_PORT_BASE, c_base);
}

/* Sets TMS to given value */
static void
tms_set(int value)
{
  if(value) {
    c_base |= TMS;
  } else {
    c_base &= ~TMS;
  }
  DlPortWritePortUchar(PAR_PORT_BASE, c_base);
}

/* Sets TDI to given value */
static void
tdi_set(int value)
{
  if(value) {
    c_base |= TDI;
  } else {
    c_base &= ~TDI;
  }
  DlPortWritePortUchar(PAR_PORT_BASE, c_base);
}

/* Gets the value at TDO */
static int
tdo_get(void)
{
  unsigned char c_stat;

  c_stat = DlPortReadPortUchar(PAR_PORT_STAT);

  if(c_stat & TDO) {
    return 1;
  } else {
    return 0;
  }
}

/*********************************************************/
/*                 Access & Test Function                */
/*********************************************************/
/* Library Link Test */
_PRE_ short int  _POST_
test(short int dummy)               // Value to return
{
  if(g_state == -1) {
    tms_reset(0);
  }

  return dummy;
}
 

/*********************************************************/
/*                Hardware Reset Functions               */
/*********************************************************/
/* Holds TMS signal high for 5 TCKS */
_PRE_ void _POST_
tms_reset(enum TAPS test_bus)           // Which TAP to reset 
{
  int i;

  /*
   * This would be a pretty good place to check the VCC_SENSE line on the
   * cable if we wanted to be careful.
   */

  tms_set(1);
  for(i = 0; i < 6; i++) { /* 1 extra tick in case TCK started high */
    tck_tick();
  }
  g_state = STATE_TLR;
}

/* Pulse the TRST signal low then high on the selected TAP */                
_PRE_ short int _POST_
trst_reset(enum TAPS test_bus)     // Which TAP to reset 
{
  unimplemented("trst_reset");
  return 0;
}

                
/* Reset BSC controllers via reset bit*/                
_PRE_ void _POST_
soft_reset(enum TAPS test_bus)     // Which TAP pair to reset 
{
  unimplemented("soft_reset");
}


/* Hard reset the boards functions */                
_PRE_ short _POST_
hard_reset(short int rs422_enable,// En RS422, & TAP 4 recv
	   enum VOLTAGE volts)      // Output voltage level
{
  unimplemented("hard_reset");
  return 0;
}
                
/*********************************************************/
/*                  Scanning Functions                   */
/*********************************************************/                
/* Scan data to and from the JTAG Data Register */
_PRE_ void _POST_
scan_dr(enum TAPS test_bus,                 // Which TAP to Scan
	const unsigned short int *out_data, // Holds data shifted out of card
	unsigned long bit_length,           // Number of bits to be shifted
	unsigned short int *in_data)        // Data to be shifted into card
{
  /* 
   * Casting these to unsigned char only works because Intel is little-
   * endian.  The code that calls these seems to make that same assumption
   * so I won't feel too guilty.
   */
  unsigned char *ocp = (unsigned char*)out_data;
  unsigned char *icp = (unsigned char*)in_data;
  unsigned long bit;

  if(g_state == -1) {
    tms_reset(test_bus);
  }

  if(bit_length > 0) {
    move_to_state(test_bus, STATE_SDR);

    tms_set(0);
    for(bit = 0; bit < bit_length; bit++) {
      bit_set(icp, bit, tdo_get());
      tdi_set(bit_get(ocp, bit));

      /* Here we shift last bit and transition on same CLK pulse */
      if(bit == bit_length - 1) {
	tms_set(1);
      }
      tck_tick();
    }
    tck_tick();
    tms_set(0);
    tck_tick();
    g_state = STATE_RTI;
  }
}
               
/* Scan data to and from the JTAG Data Register with hardware accereration */
_PRE_ void _POST_
scan_dr_turbo(enum TAPS test_bus,             // Which TAP to Scan
	      const unsigned long *out_data,  // Holds data shifted out of card
	      unsigned long bit_length,       // Number of bits to be shifted
	      unsigned long *in_data)         // Data to be shifted into card
{
  unimplemented("scan_dr_turbo");
}

/* Scan data to and from the JTAG Data Register leaving JTAG in Pause-DR state*/
_PRE_ void _POST_
scan_to_pause_dr(enum TAPS test_bus,            // Which TAP to Scan
		 const unsigned short int *out_data, // Holds data shifted out 
		 unsigned long bit_length,      // Number of bits to be shifted
		 unsigned short int *in_data)   // Data to be shifted into card
{
  unsigned char *ocp = (unsigned char*)out_data;
  unsigned char *icp = (unsigned char*)in_data;
  unsigned long bit;

  if(g_state == -1) {
    tms_reset(test_bus);
  }

  if(bit_length > 0) {
    move_to_state(test_bus, STATE_SDR);

    tms_set(0);
    for(bit = 0; bit < bit_length; bit++) {
      bit_set(icp, bit, tdo_get());
      tdi_set(bit_get(ocp, bit));

      /* Here we shift last bit and transition on same CLK pulse */
      if(bit == bit_length - 1) {
	tms_set(1);
      }
      tck_tick();
    }
    tms_set(0);
    tck_tick();
    g_state = STATE_PDR;
  }
}

/* Scan data to and from the JTAG Instruction Register */
_PRE_ void _POST_
scan_ir(enum TAPS test_bus,                 // Which TAP to Scan
	const unsigned short int *out_data, // Holds data shifted out of card
	unsigned long bit_length,           // Number of bits to be shifted
	unsigned short int *in_data)        // Data to be shifted into card
{
  unsigned char *ocp = (unsigned char*)out_data;
  unsigned char *icp = (unsigned char*)in_data;
  unsigned long bit;

  if(g_state == -1) {
    tms_reset(test_bus);
  }

  if(bit_length > 0) {
    move_to_state(test_bus, STATE_SIR);

    tms_set(0);
    for(bit = 0; bit < bit_length; bit++) {
      bit_set(icp, bit, tdo_get());
      tdi_set(bit_get(ocp, bit));

      /* Here we shift last bit and transition on same CLK pulse */
      if(bit == bit_length - 1) {
	tms_set(1);
      }
      tck_tick();
    }
    tck_tick();
    tms_set(0);
    tck_tick();
    g_state = STATE_RTI;
  }
}

/* Scan data to and from the JTAG Instruction Register with hardware accel */
_PRE_ void _POST_
scan_ir_turbo(enum TAPS test_bus,            // Which TAP to Scan
	      const unsigned long *out_data, // Holds data shifted out of card
	      unsigned long bit_length,      // Number of bits to be shifted
	      unsigned long *in_data)        // Data to be shifted into card
{
  unimplemented("scan_ir_turbo");
}

/* Scan data to and from the JTAG Ins Register leaving JTAG in Pause-IR state*/
_PRE_ void _POST_
scan_to_pause_ir(enum TAPS test_bus,           // Which TAP to Scan
		 const unsigned short int *out_data, // Holds data shifted out
		 unsigned long bit_length,     // Number of bits to be shifted
		 unsigned short int *in_data)  // Data to be shifted into card
{
  unsigned char *ocp = (unsigned char*)out_data;
  unsigned char *icp = (unsigned char*)in_data;
  unsigned long bit;

  if(g_state == -1) {
    tms_reset(test_bus);
  }

  if(bit_length > 0) {
    move_to_state(test_bus, STATE_SIR);

    tms_set(0);
    for(bit = 0; bit < bit_length; bit++) {
      bit_set(icp, bit, tdo_get());
      tdi_set(bit_get(ocp, bit));

      /* Here we shift last bit and transition on same CLK pulse */
      if(bit == bit_length - 1) {
	tms_set(1);
      }
      tck_tick();
    }
    tms_set(0);
    tck_tick();
    g_state = STATE_PIR;
  }
}

/* Set JTAG's state machine to a given state */
_PRE_ void _POST_
move_to_state(enum TAPS test_bus,                 // Which TAP to transition
	      enum STATES final_state)            // Final transition state
{
  /*
   * This table tells how to get from any non-transient state to any other.
   * It's indexed by the integer values of the enumerated type STATES, which
   * is bad, but hopefully it won't change.
   */
  static char* trans_tbl[6][6] = {
    {       "",      "0",  "01010", "011010",   "0100",  "01100"},
    {    "111",       "",   "1010",  "11010",    "100",   "1100"},
    {  "11111",    "110",       "","1111010",     "10", "111100"},
    {  "11111",    "110", "111010",       "",  "11100",     "10"},
    {  "11111",    "110",     "10","1111010",       "", "111100"},
    {  "11111",    "110", "111010",     "10",  "11100",       ""}
  };
  char* p;
  char old = 'X';

  if(g_state == -1) {
    tms_reset(test_bus);
  }

  /* assert(final_state >= 0 && final_state <= 5) */

  for(p = trans_tbl[g_state][final_state]; *p != '\0'; p++) {
    if(*p != old) {
      tms_set((*p == '0')?0:1);
    }
    tck_tick();
    old = *p;
  }
  
  g_state = final_state;
}

/* Set the TCK clock speed for specified TAP */
_PRE_ void _POST_
set_scan_clk(unsigned short int clock,   // Which TAP pair to set(0:1&2 1:3&4)
	     enum OSCILLATORS clk_select,     // Which clock oscillator to use
	     short int prescaler,             // Enable clock prescaler
	     unsigned short int clk_divider)  // Integer used as clock divisor
{
  ignored("set_scan_clk");
}

/* Scan data out then back into Data Register */
_PRE_ void _POST_
circulate_dr(enum TAPS test_bus,          // Which TAP to scan
	     unsigned long bit_lengh,     // Number of bits to be shifted
	     unsigned short int *in_data) // Array to hold data shifted to card
{
  unimplemented("circulate_dr");
}

_PRE_ void _POST_
scan_NxCLK(enum TAPS test_bus,  	// Which TAP to scan	
	   enum STATES curr_state,	// Current state	
	   unsigned long num_ticks)     // Number of clocks to scan through
{
  /* assert(curr_state == g_state */

  /* assume TMS is set to the correct value, since all non-transient
     states are entered and repeated with the same value */

  if(g_state == -1) {
    tms_reset(test_bus);
  }

  for(; num_ticks > 0; num_ticks--) {
    tck_tick();
  }
}

/*********************************************************/
/*                   Reading Functions                   */
/*********************************************************/ 

/* * * * * NONE OF THE FOLLOWING ARE IMPLEMENTED * * * * */

/* Read input level of all signals at 8-bit input port */
_PRE_ unsigned char _POST_
read_io(unsigned char port) 				// PORTA or PORTB
{
  return 0;
}

/* Sets the output levels of signals on 8-bit output port */
_PRE_ void _POST_
set_io(unsigned char port,        	// PORTA or PORTB
       unsigned char output_data)       // Data to be output
{
  return;
}

/* Sets the open collector output levels on 8-bit output port A */
_PRE_ void _POST_
set_oc_output(short int bit, 	     // Open collector bit 0 or 1 
	      short int output_data) // 0 - sig driven low, 1 - sig driven high
{
  return;
}

/* Output to a memory location */
_PRE_ long  _POST_ out32(unsigned long addr, unsigned long  data) { return 0; }
_PRE_ short _POST_ out16(unsigned long addr, unsigned short data) { return 0; }
_PRE_ char  _POST_ out8 (unsigned long addr, unsigned char  data) { return 0; }

/* input from a port */
_PRE_ long  _POST_ in32 (unsigned long addr                     ) { return 0; }
_PRE_ short _POST_ in16 (unsigned long addr                     ) { return 0; }
_PRE_ char  _POST_ in8  (unsigned long addr                     ) { return 0; }

